/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * NAME
 *	modify_ldt01.c
 *
 * DESCRIPTION
 *	Testcase to check the error conditions for modify_ldt(2)
 *
 * CALLS
 *	modify_ldt()
 *
 * ALGORITHM
 *	block1:
 *		Invoke modify_ldt() with a func value which is neither
 *		0 or 1. Verify that ENOSYS is set.
 *	block2:
 *		Invoke mprotect() with ptr == NULL. Verify that EINVAL
 *		is set.
 *	block3:
 *		Create an LDT segment.
 *		Try to read from an invalid pointer.
 *		Verify that EFAULT is set.
 *
 * USAGE
 *	modify_ldt01
 *
 * HISTORY
 *	07/2001 Ported by Wayne Boyer
 *
 * RESTRICTIONS
 *	None
 */

#ifdef __i386__
#include <asm/ldt.h>
#include <asm/unistd.h>
#endif
#include <errno.h>
#include "test.h"
#include "usctest.h"

/* Newer ldt.h files use user_desc, instead of modify_ldt_ldt_s */
#ifdef MODIFY_LDT_SPECIALCASE
#define modify_ldt_ldt_s user_desc
#endif

#if !defined(modify_ldt_ldt_s) || !defined(user_desc)
 typedef struct modify_ldt_ldt_t
 {
   unsigned int entry_number;
   unsigned long int base_addr;
   unsigned int limit;
   unsigned int seg_32bit:1;
   unsigned int contents:2;
   unsigned int read_exec_only:1;
   unsigned int limit_in_pages:1;
   unsigned int seg_not_present:1;
   unsigned int useable:1;
   unsigned int empty:25;
 }modify_ldt_s;
#endif

void cleanup(void);
void setup(void);
char *TCID= "modify_ldt01";
int TST_TOTAL = 1;
extern int Tst_count;
int seg[4];

int flag;
#define FAILED 1

#if defined(__i386__)

int create_segment(void *, size_t);

extern int modify_ldt(int, void*, unsigned long);

int main(int ac, char **av)
{
	int lc;                         /* loop counter */
	char *msg;                      /* message returned from parse_opts */

	void *ptr;
	int retval, func;

        /* parse standard options */
        if ((msg = parse_opts(ac, av, (option_t *)NULL, NULL)) != (char *)NULL){
		tst_brkm(TBROK, cleanup, "OPTION PARSING ERROR - %s", msg);
		/*NOTREACHED*/
        }

        setup();                        /* global setup */

	/* The following loop checks looping state if -i option given */
	for (lc = 0; TEST_LOOPING(lc); lc++) {

                /* reset Tst_count in case we are looping */
                Tst_count = 0;

//block1:
		/*
		 * Check for ENOSYS.
		 */
		tst_resm(TINFO, "Enter block 1");
		flag = 0;
		ptr=(void *)malloc(10);
		func = 100;
		retval = modify_ldt(func, ptr, sizeof(ptr));
		if (retval < 0) {
			if (errno != ENOSYS) {
				tst_resm(TFAIL, "modify_ldt() set invalid "
                                         "errno, expected ENOSYS, got: %d",
					 errno);
				flag = FAILED;
			}
		} else {
			tst_resm(TFAIL, "modify_ldt error: "
				 "unexpected return value %d", retval);
			flag = FAILED;
		}

		if (flag) {
			tst_resm(TINFO, "block 1 FAILED");
		} else {
			tst_resm(TINFO, "block 1 PASSED");
		}
		tst_resm(TINFO, "Exit block 1");
		free(ptr);

//block2:
		/*
		 * Check for EINVAL
		 */
		tst_resm(TINFO, "Enter block 2");
		flag = 0;

		ptr = 0;

		retval = modify_ldt(1, ptr, sizeof(ptr));
		if (retval < 0) {
			if (errno != EINVAL) {
				tst_resm(TFAIL, "modify_ldt() set invalid "
                                         "errno, expected EINVAL, got: %d",
					 errno);
				flag = FAILED;
			}
		} else {
			tst_resm(TFAIL, "modify_ldt error: "
				 "unexpected return value %d", retval);
			flag = FAILED;
		}

		if (flag) {
			tst_resm(TINFO, "block 2 FAILED");
		} else {
			tst_resm(TINFO, "block 2 PASSED");
		}
		tst_resm(TINFO, "Exit block 2");

//block3:

		/*
		 * Create a new LDT segment.
		 */
		if (create_segment(seg, sizeof(seg)) == -1) {
			tst_brkm(TINFO, cleanup, "Creation of segment failed");
			/*NOTREACHED*/
		}

		/*
		 * Check for EFAULT
		 */
		ptr = sbrk(0);

		retval = modify_ldt(0, ptr + 0xFFF, sizeof(ptr));
		if (retval < 0) {
			if (errno != EFAULT) {
				tst_resm(TFAIL, "modify_ldt() set invalid "
                                         "errno, expected EFAULT, got: %d",
					 errno);
				flag = FAILED;
			}
		} else {
			tst_resm(TFAIL, "modify_ldt error: "
				 "unexpected return value %d", retval);
			flag = FAILED;
		}

		if (flag) {
			tst_resm(TINFO, "block 3 FAILED");
		} else {
			tst_resm(TINFO, "block 3 PASSED");
		}
		tst_resm(TINFO, "Exit block 3");

        }
        cleanup();
	return(0);
}

#else /* if defined(linux) && defined(__i386__) */

int main()
{
	tst_resm(TINFO, "modify_ldt01 test only for ix86");
	return 0;
}

#endif /* if defined(linux) && defined(__i386__) */

/*
 * setup() - performs all ONE TIME setup for this test
 */
void
setup(void)
{
	/* capture signals */
	tst_sig(FORK, DEF_HANDLER, cleanup);

	/* Pause if that option was specified */
	TEST_PAUSE;
}

/*
 * cleanup() - performs all the ONE TIME cleanup for this test at completion
 * or premature exit.
 */
void
cleanup(void)
{
	/*
	 * print timing status if that option was specified.
	 * print errno log if that option was specified
	 */
	TEST_CLEANUP;

	/* exit with return code appropriate for results */
	tst_exit();
}

#if defined(__i386__)
int
create_segment(void *seg, size_t size)
{
	modify_ldt_s entry;

	entry.entry_number = 0;
	entry.base_addr = (unsigned long)seg;
	entry.limit = size;
	entry.seg_32bit = 1;
	entry.contents = 0;
	entry.read_exec_only = 0;
	entry.limit_in_pages = 0;
	entry.seg_not_present = 0;

	return modify_ldt(1, &entry, sizeof(entry));
}
#endif /* if defined(linux) && defined(__i386__) */
