/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

#ifdef USE_SOUND

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <linux/soundcard.h>	/* linux specific soundcard ioctls */


#define SND_MAX_VAL	127
#define SND_MIN_VAL	0


/* make_snd_buffer( int *divisors, int *durations, int count, int *len )
 * 
 * Takes a list of note frequencies and durations and generates a
 * PC-speaker-like square wave at 8kHz, mono, 8bits.  Sets *len
 * to the length of the returned audio buffer.  The caller is
 * responsible for freeing the returned buffer.  The divisors
 * specify the frequencies and, curiously enough, correspond
 * precisely to the divisors one gives to the timer on a PC in
 * order to set the frequency being output by the PC speaker.
 * 
 * count: number of notes
 * 
 * divisors:  1193180 / divisor = frequency
 * durations: durations in milliseconds
 * 
 * 8000 (samples/sec) / frequency (cy/sec) = (samples/cy)
 * 8000 / (1193180 / divisor) = (samples/cy)
 * (8000 * divisor) / 1193180 = (samples/cy)
 */


char *make_snd_buffer( int *divisors, int *durations, int count, int *len )
{
  char *buffer;
  int *local_divisors;
  int *local_durations;
  int i, total_duration=0, bufidx;
  int dur, div;
  
  local_divisors  = (int *) malloc( count * sizeof(int) );
  local_durations = (int *) malloc( count * sizeof(int) );

  for( i=0; i<count; i++ ) {
    /* Divide by 2 because there are two parts to the square
     * wave: high and low, and we have to alternate between
     * them at twice the actual frequency.
     */
    local_divisors[i]  = (divisors[i] * 8000) / 1193180 / 2;

    /* 125 milliseconds per sample at 8000 Hz. */
    local_durations[i] = durations[i] / 125;
    total_duration += local_durations[i];
  }
  
  buffer = (char *) malloc( total_duration );
  *len = total_duration;
  bufidx=0;

  for( i=0; i<count; i++ ) {
    dur = local_durations[i];

    while( dur >= 0 ) {
      div = local_divisors[i];
      while( --div>=0 && --dur>=0 )
	buffer[bufidx++] = SND_MAX_VAL;

      div = local_divisors[i];
      while( --div>=0 && --dur>=0 )
	buffer[bufidx++] = SND_MIN_VAL;
    }
  }

  free( local_divisors );
  free( local_durations );
  
  return buffer;
}


int sound_handle = -1;


int do_open_sound( void )
{
  int format, stereo, speed;

  if( (sound_handle = open( "/dev/dsp", O_WRONLY )) == -1 )
    return 1;
  
  /* Try to set the audio format. */

  format = AFMT_U8; /* AFMT_U8 is unsigned 8 bit audio */
  if( ioctl( sound_handle, SNDCTL_DSP_SETFMT, &format ) == -1 ) {
    /* Fatal error */
    /* perror("SNDCTL_DSP_SETFMT"); */
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /* Make sure the audio format is the one we tried to set. */

  if( format != AFMT_U8 ) {
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /* Try to set MONO output. */
  
  stereo = 0;	/* 0=mono, 1=stereo */
  if( ioctl( sound_handle, SNDCTL_DSP_STEREO, &stereo ) == -1 ) {
    /* Fatal error */
    /* perror("SNDCTL_DSP_STEREO"); */
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /* Make sure we actually get MONO format. */
  
  if( stereo != 0 ) {
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /* Try to set output to 8kHz. */
  
  speed = 8000;
  if( ioctl( sound_handle, SNDCTL_DSP_SPEED, &speed ) == -1 ) {
    /* Fatal error */
    /* perror("SNDCTL_DSP_SPEED"); */
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /* Check to see that speed is somewhere near what we asked for. */

  if( speed < 7000 || speed > 9000 ) {
    close( sound_handle );
    sound_handle = -1;
    return 1;
  }
  
  /*
   * ioctl(audio_fd, SNDCTL_DSP_POST, 0) is light weight version of
   * SNDCTL_DSP_SYNC. It just tells to the driver that there is likely
   * to be a pause in the output. This makes it possible to the device
   * to handle the pause more intelligently.
   * 
   * You should call SNDCTL_DSP_POST when your program is going to
   * pause continuous output of audio data for relatively long time.
   * This kind of situations are for example the following:
   * 
   *   * After playing a sound effect when a new one is not started
   *     immediately (another way is to output silence until next effect
   *     starts).
   */

  ioctl( sound_handle, SNDCTL_DSP_POST, 0 );
  
  return 0;
}


char *sndbuf_player_die;
char *sndbuf_hive_die;
char *sndbuf_snipe_shoot;

int sndbuflen_player_die, sndbuflen_hive_die, sndbuflen_snipe_shoot;
#endif


int sound_init( int use_sound )
{
#ifdef USE_SOUND
  int player_die_divs[12] = { 0x0640, 0x1f40, 0x1964, 0x0fa0, 0x09c4, 0x03e8,
			      0x03e8, 0x09c4, 0x0fa0, 0x1964, 0x1f40, 0x07d0 };
  int player_die_durs[12] = { 60000, 60000, 60000, 60000, 60000, 60000,
			      60000, 60000, 60000, 60000, 60000, 60000 };

  int hive_die_divs[5] = { 0x1f40, 0x1964, 0x0fa0, 0x09c4, 0x03e8 };
  int hive_die_durs[5] = { 60000, 60000, 60000, 60000, 60000 };

  int snipe_shoot_divs[1] = { 0x0550 };
  int snipe_shoot_durs[1] = { 60000 };
  
  sndbuf_player_die = NULL;
  sndbuf_hive_die = NULL;
  sndbuf_snipe_shoot = NULL;
  sound_handle = -1;
  
  if( use_sound == 0 )
    return 0;

  if( do_open_sound() == 1 )
    return 1;

  sndbuf_snipe_shoot = make_snd_buffer( snipe_shoot_divs, snipe_shoot_durs,
					1, &sndbuflen_snipe_shoot );
  sndbuf_hive_die = make_snd_buffer( hive_die_divs, hive_die_durs,
				     5, &sndbuflen_hive_die );
  sndbuf_player_die = make_snd_buffer( player_die_divs, player_die_durs,
				       12, &sndbuflen_player_die );
#endif

  return 0;
}


void sound_stop( void )
{
#ifdef USE_SOUND
  if( sndbuf_snipe_shoot )
    free( sndbuf_snipe_shoot );
  if( sndbuf_hive_die )
    free( sndbuf_hive_die );
  if( sndbuf_player_die )
    free( sndbuf_player_die );
  if( sound_handle != -1 )
    close( sound_handle );
#endif
}


/* sound_post_player_death( void )
 * 
 * Called when the player dies.
 */

void sound_post_player_death( void )
{
#ifdef USE_SOUND
  if( sound_handle != -1 ) {
    write( sound_handle, sndbuf_player_die, sndbuflen_player_die );
    ioctl( sound_handle, SNDCTL_DSP_POST, NULL );
  }
#endif
}


/* sound_post_enemy_death( void )
 * 
 * Called when an enemy gets shot.
 */

void sound_post_enemy_death( void )
{
#ifdef USE_SOUND
  if( sound_handle != -1 ) {
    write( sound_handle, sndbuf_hive_die, sndbuflen_hive_die );
    ioctl( sound_handle, SNDCTL_DSP_POST, NULL );
  }
#endif
}


/* sound_post_snipe_shot( void )
 * 
 * Called when an enemy fires.
 */

void sound_post_snipe_shot( void )
{
#ifdef USE_SOUND
  if( sound_handle != -1 ) {
    write( sound_handle, sndbuf_snipe_shoot, sndbuflen_snipe_shoot );
    ioctl( sound_handle, SNDCTL_DSP_POST, NULL );
  }
#endif
}


#ifdef STANDALONE
main( void )
{
  int i;
  int fd;
  
  sound_init();

  sound_post_player_die();
  sleep(1);
  sound_post_hive_die();
  sleep(1);
  sound_post_snipe_shoot();

  sound_stop();
}
#endif
