//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <config.h>

#include <algorithm>
#include <assert.h>

#include "vector.h"

#include "smallmap.h"
#include "sdl-draw.h"
#include "timing.h"
#include "GameMap.h"

SmallMap::SmallMap()
{
    input_locked = false;
}

void SmallMap::set_view(Rectangle new_view)
{
    if (view != new_view)
    {
	view = new_view;
	draw();
    }
}

void SmallMap::draw_selection()
{
    // draw the selection rectangle that shows the viewed part of the map
    Vector<int> pos = mapToSurface(view.pos);

    int w = int(view.w * pixels_per_tile);
    int h = int(view.h * pixels_per_tile);

    // this is a bit unfortunate.  we require this catch-all
    // so that our selector box isn't too big for the smallmap
    if (pos.x + w >= surface->w)
      pos.x = surface->w - w - 1;
    if (pos.y + h >= surface->h)
      pos.y = surface->h - h - 1;

    assert(pos.x >= 0 && pos.x + w < surface->w &&
	   pos.y >= 0 && pos.y + h < surface->h);
    
    Uint32 raw = SDL_MapRGB(surface->format, 255, 255, 255);
    draw_rect(surface, pos.x, pos.y, pos.x + w, pos.y + h, raw);
    draw_rect(surface, pos.x+1, pos.y+1, pos.x + w-1, pos.y + h-1, raw);
}

void SmallMap::center_view(Vector<int> p, bool slide, bool from_tile)
{
  if (from_tile == false)
    {
      p.x = int(round(p.x / pixels_per_tile));
      p.y = int(round(p.y / pixels_per_tile));

      p -= view.dim / 2;

      p = clip(Vector<int>(0, 0), p, GameMap::get_dim() - view.dim);
    }
  else
      
    p = clip(Vector<int>(0,0), p - view.dim / 2, GameMap::get_dim() - view.dim);

  if (slide)
    slide_view(Rectangle(p.x, p.y, view.w, view.h));
  else
    set_view(Rectangle(p.x, p.y, view.w, view.h));
	  
  if (Playerlist::isFinished()) //window closed while computer player moves
      return;
  view_changed.emit(view);
}

void SmallMap::after_draw()
{
  OverviewMap::after_draw();
  draw_cities(false);
  draw_selection();
  map_changed.emit(get_surface());
}

void SmallMap::mouse_button_event(MouseButtonEvent e)
{
  if (input_locked)
    return;

  if ((e.button == MouseButtonEvent::LEFT_BUTTON ||
       e.button == MouseButtonEvent::RIGHT_BUTTON)
      && e.state == MouseButtonEvent::PRESSED)
    center_view(e.pos, true, false);
}

void SmallMap::mouse_motion_event(MouseMotionEvent e)
{
  if (input_locked)
    return;

  if (e.pressed[MouseMotionEvent::LEFT_BUTTON] ||
      e.pressed[MouseMotionEvent::RIGHT_BUTTON])
    center_view(e.pos, false, false);
}
int slide (int x, int y)
{
  int skip = 2;
  if (x < y)
    {
      if (x + skip < y)
	x += skip;
      else
	x++;
    }
  else if (x > y)
    {
      if (x - skip > y)
	x -= skip;
      else
	x--;
    }
  return x;
}

void SmallMap::slide_view(Rectangle new_view)
{
  if (view != new_view)
    {
      while (1)
	{
	  if (Playerlist::isFinished()) //window closed while ai player moves
	      return;
	  Rectangle tmp_view(view);
	  tmp_view.x = slide(tmp_view.x, new_view.x);
	  tmp_view.y = slide(tmp_view.y, new_view.y);

	  view = tmp_view;
	  draw();
	  view_slid.emit(view);

	  if (tmp_view.x == new_view.x && tmp_view.y == new_view.y)
	    break;
	}
    }
}
