/* Copyright (C) 2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <time.h>
#include <string.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "global.h"
#include "userinfo.h"
#include "support.h"
#include "callbacks.h"
#include "string_list.h"
#include "subscription.h"
#include "whois.h"
#include "utils.h"

#define INFO_LIFE        15    // days

struct server_t;

user_info_t* user_info_new(char* user) {
  user_info_t* user_info;

  user_info = l_malloc(sizeof(user_info_t));
  user_info->last_server = NULL;
  user_info->nick = l_strdup(user);
  user_info->cur[0] = 0;
  user_info->cur[1] = 0;
  user_info->real_cur[0] = 0;
  user_info->real_cur[1] = 0;
  user_info->max[0] = -1;
  user_info->max[1] = -1;
  user_info->limit[0] = 0;
  user_info->limit[1] = 0;
  user_info->bytes[0] = 0;
  user_info->bytes[1] = 0;
  user_info->ignore_global[0] = 0;
  user_info->ignore_global[1] = 0;
  user_info->linespeed = -1;
  user_info->timestamp = 1;
  user_info->last_seen = global.current_time;
  user_info->shares = -1;
  user_info->remotes = 0;
  user_info->client = C_UNKNOWN;
  global.userinfo = g_list_prepend(global.userinfo, user_info);

  return user_info;
}

void user_info_destroy(user_info_t* userinfo) {
  if (!userinfo) return;
  global.userinfo = g_list_remove(global.userinfo, userinfo);
  if (userinfo->nick) l_free(userinfo->nick);
  if (userinfo->last_server) l_free(userinfo->last_server);
  l_free(userinfo);
}

void user_info_save() {
  GList *dlist;
  char *fname;
  char *fname_new;
  FILE *fd;
  user_info_t *info;

  fname = l_strdup_printf("%s/userinfo.list", global.options.config_dir);
  fname_new = l_strdup_printf("%s/access.list.new", global.options.config_dir);

  if ((fd = fopen(fname_new, "w")) == NULL) {
    g_warning("Could not write [%s]\n", fname);
    l_free(fname);
    l_free(fname_new);
    return;
  }

  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    info = dlist->data;

    if (global.current_time - info->last_seen > 60*60*24*INFO_LIFE) continue;
    qfprintf(fd, "%s %d %d %ld %ld %d %d %lu %S\n",
	     info->nick, info->max[0], info->max[1], info->limit[0], info->limit[1],
	     info->ignore_global[0], info->ignore_global[1], info->last_seen,
             info->last_server?info->last_server:"");
  }
  
  if (!ferror(fd)) rename(fname_new, fname);
  else {
    g_warning("Could not write [%s]\n", fname);
  }

  fclose(fd);
  l_free(fname);
  l_free(fname_new);
}

void user_info_load() {
  FILE *fd;
  char *fname;
  char* user;
  char* max1, *max2;
  char* limit1, *limit2;
  char* ignore1, *ignore2;
  user_info_t* userinfo;
  char* seen;
  char* server;
  time_t last_seen;
  char line[2048];

  //  printf("user_info_load(): %ld\n", time(NULL));
  fname = l_strdup_printf("%s/userinfo.list", global.options.config_dir);
  if ((fd = fopen(fname, "r")) == NULL) {
    l_free(fname);
    return;
  }
  l_free(fname);

  while (mfgets(line, sizeof(line), fd)) {
    user = arg(line, 2);
    max1 = arg(NULL, 2);
    max2 = arg(NULL, 2);
    limit1 = arg(NULL, 2);
    limit2 = arg(NULL, 2);
    ignore1 = arg(NULL, 2);
    ignore2 = arg(NULL, 2);
    seen = arg(NULL, 2);
    server = arg(NULL, 2);
    
    if (!user || !max1 || !max2 || !limit1 || !limit2 || !ignore1 || !ignore2) continue;

    if (seen) last_seen = strtol(seen, NULL, 10);
    else last_seen = global.current_time;
    if (global.current_time - last_seen > 60*60*24*INFO_LIFE) continue;

    userinfo = user_info_new(user);
    userinfo->max[0] = atoi(max1);
    if (userinfo->max[0] == -2) userinfo->max[0] = 0;
    userinfo->max[1] = atoi(max2);
    if (userinfo->max[1] == -2) userinfo->max[1] = 0;
    userinfo->limit[0] = strtol(limit1, NULL, 10);
    userinfo->limit[1] = strtol(limit2, NULL, 10);
    userinfo->ignore_global[0] = atoi(ignore1);
    userinfo->ignore_global[0] = atoi(ignore2);
    userinfo->last_seen = last_seen;
    if (server) userinfo->last_server = l_strdup(server);
  }

  fclose(fd);

  return;
}

user_info_t *user_info_search(char *user) {
  GList *source;
  GList *dlist;
  user_info_t *user_info;

  source = global.userinfo;

  for (dlist = source; dlist; dlist = dlist->next) {
    user_info = dlist->data;
    if (!l_strcasecmp(user_info->nick, user))
      return user_info;
  }

  return NULL;
}

int user_info_priority(user_info_t* userinfo, int* class) {
  subscription_t* sub;

  if (string_list_search(LIST_FRIEND, userinfo->nick)) {
    if (class) *class = PRI_FRIEND;
    return global.options.upload_priority[PRI_FRIEND];
  }

  sub = subscription_user_search(userinfo->nick);
  if (sub && sub->friend_while_subs) {
    if (class) *class = PRI_FRIEND;
    return global.options.upload_priority[PRI_FRIEND];
  }

  if (userinfo->real_cur[0] > 0) {
    if (class) *class = PRI_DOWNLOAD;
    return global.options.upload_priority[PRI_DOWNLOAD];
  }

  if (userinfo->shares == 0) {
    if (class) *class = PRI_NONE;
    return global.options.upload_priority[PRI_NONE];
  } else {
    if (class) *class = PRI_SHARE;
    return global.options.upload_priority[PRI_SHARE];
  }

  return PRI_NONE;
}

void user_info_get(user_info_t* userinfo, net_t* net) {
  // check timestamp.
  if (userinfo->timestamp &&
      global.current_time - userinfo->timestamp > 60*60) {
    userinfo->timestamp = 0;  // mark
    whois_request(net, userinfo->nick, WHOIS_NONE);
  }
}

void user_info_show(user_info_t* userinfo, int download, int active) {
  GtkWidget* area;
  GtkWidget* spin;
  int posx;
  int posy;
  int dx, dy;
  long limit;
  char* ss;
  int width, height;
  int upload = !download;
  char str[1024];
  GtkStyle* style;

  if (!GTK_WIDGET_REALIZED(global.progressbar))
    gtk_widget_realize(global.progressbar);
  style = global.progressbar->style;
  dx = style->klass->xthickness;
  dy = style->klass->ythickness;

  if (download) {
    limit = global.down_width.limit;
    area = lookup_widget(global.win, "drawingarea12");
    spin = lookup_widget(global.win, "spinbutton15");
  } else {
    limit = global.up_width.limit;
    area = lookup_widget(global.win, "drawingarea11");
    spin = lookup_widget(global.win, "spinbutton49");
  }


  if (limit <= 0) limit = 1024*80;

  gtk_object_set_data(GTK_OBJECT(area), "userinfo", userinfo);
  gtk_object_set_data(GTK_OBJECT(spin), "userinfo", userinfo);

  if (!area->window) return;
  width = area->allocation.width;
  height = area->allocation.height;
  
  gtk_paint_box (style, area->window,
		 GTK_STATE_NORMAL, GTK_SHADOW_IN, 
		 NULL, area, "trough",
		 0, 0, width, height);

  if (!userinfo) return;

  if (!active) 
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), (gfloat) (userinfo->max[upload]));

  posx = (width - 2*dx) * userinfo->limit[upload] / limit;
  if (posx > width) posx = width;
  if (posx)
    gtk_paint_box(style, area->window,
		  GTK_STATE_PRELIGHT, GTK_SHADOW_OUT,
		  NULL, area, "bar", dx, dy, posx, height-2*dy);

  if (!active) {
    ss = userinfo->nick;
  } else if (userinfo->limit[upload] == 0) {
    strcpy(str, "No bandwidth limit");
    ss = str;
  } else {
    ss = print_speed(str, userinfo->limit[upload], 1);
  }
  posx = (width - 2*dx - gdk_string_width(style->font, ss))/2;
  posy = (height + style->font->ascent - 2*dy)/2+1;
  gdk_draw_text (area->window, style->font,
		 style->fg_gc[GTK_STATE_NORMAL],
		 posx, posy, ss, strlen (ss));
}

// this function could be dangerous, as it does delete user infos that
// might be still in use somewhere else
// deactivated at the moment.

int user_info_cleanup(gpointer data ATTR_UNUSED) {
  GList* dlist;
  user_info_t* info;
  GList* res = NULL;

  return 0;

  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    info = dlist->data;
    if (global.current_time - info->last_seen > 60*60*24*INFO_LIFE)
      res = g_list_prepend(res, dlist);
  }
  // not finished yet.
  return 1;
}

user_info_t* user_info_detect(net_t* net, char* user) {
  user_info_t *user_info;

  user_info = user_info_search(user);

  if (!user_info) user_info = user_info_new(user);
  user_info_get(user_info, net);

  return user_info;
}

