/* Copyright (C) 2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <time.h>
#include <string.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "global.h"
#include "subscription.h"
#include "support.h"
#include "callbacks.h"
#include "chat.h"
#include "utils.h"

void check_uploads_again();

void subscription_load() {
  FILE *file;
  char* filename;
  char line[2049];
  char* user;
  char* flag;

  subscription_t* sub;

  filename = l_strdup_printf("%s/subscriptions", global.options.config_dir);
  if ((file = fopen(filename, "r")) == NULL) {
    l_free(filename);
    return;
  }
  l_free(filename);

  sub = NULL;
  while (mfgets(line, sizeof(line), file)) {
    if ((line[0] == '#') || (line[0] == ' ') || (line[0] == 0))
      continue;
    
    if (line[0] == '"') {
      filename = arg(line, 2);
      if (sub && filename) {
	filename = l_strdup(filename);
	sub->files = g_list_append(sub->files, filename);
	subscription_file_add_ctree(sub, filename);
      }
    } else if (!strncasecmp(line, "USER ", 5)) {
      user = arg(line+5, 2);
      flag = arg(NULL, 2);
      sub = subscription_new(user);
      sub->friend_while_subs = atoi(flag);
      subscription_user_add_ctree(sub);
    } else {
      g_warning("unknown line in subcriptions [%s]", line);
    }
  }

  fclose(file);
}

void subscription_save() {
  GList *dlist;
  GList *dlist2;
  FILE *file;
  char* filename;
  subscription_t* sub;

  printf("saveing sub\n");
  filename = l_strdup_printf("%s/subscriptions", global.options.config_dir);
  if ((file = fopen(filename, "w")) == NULL) {
    g_warning("Could not write subscriptions");
    l_free(filename);
    return;
  }
  l_free(filename);

  for (dlist = global.subscriptions; dlist; dlist = dlist->next) {
    sub = dlist->data;
    fprintf(file, "USER %s %d\n", sub->user, sub->friend_while_subs);
    for (dlist2 = sub->files; dlist2; dlist2 = dlist2->next) {
      filename = dlist2->data;
      qfprintf(file, "%S\n", filename);
    }
  }

  fclose(file);
}

subscription_t* subscription_new(char* user) {
  subscription_t* sub;

  sub = l_malloc(sizeof(subscription_t));
  sub->user = l_strdup(user);
  sub->friend_while_subs = 1;
  sub->files = NULL;
  global.subscriptions = g_list_append(global.subscriptions, sub);
  return sub;
}

void subscription_destroy(subscription_t* sub) {
  GList* dlist;
  char* filename;

  if (!sub) return;

  for (dlist = sub->files; dlist; dlist = dlist->next) {
    filename = dlist->data;
    l_free(filename);
  }
  g_list_free(sub->files);
  l_free(sub->user);
  l_free(sub);
  global.subscriptions = g_list_remove(global.subscriptions, sub);
  check_uploads_again();

  subscription_save();
}

subscription_t* subscription_user_search(char* user) {
  GList* dlist;
  subscription_t* sub;

  for (dlist = global.subscriptions; dlist; dlist = dlist->next) {
    sub = dlist->data;
    if (!strcasecmp(sub->user, user)) return sub;
  }
  return NULL;
}

void subscription_user_add_ctree(subscription_t* sub) {
  GList* dlist;
  GtkCTreeNode* node;
  GtkCTree* ctree;
  char* file;

  if (!sub) return;
  if (!global.subscription_win) return;
  
  // add him to ctree
  ctree = GTK_CTREE(lookup_widget(global.subscription_win, "ctree7"));
  
  if (sub->friend_while_subs) sprintf(tstr[0], "%s (Always allow upload)", sub->user);
  else strcpy(tstr[0], sub->user);
  
  node = gtk_ctree_insert_node(ctree, NULL, NULL, list, 5,
			       NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(ctree, node, sub);

  // add files to ctree
  for (dlist = sub->files; dlist; dlist = dlist->next) {
    file = dlist->data;
    subscription_file_add_ctree(sub, file);
  }
}

subscription_t* subscription_user_add(char* user) {
  subscription_t* sub;
  
  sub = subscription_user_search(user);
  if (sub) return NULL;
  sub = subscription_new(user);

  subscription_save();

  subscription_user_add_ctree(sub);
  check_uploads_again();
  return sub;
}

void subscription_user_delete_ctree(subscription_t* sub) {
  GtkCTreeNode* node;
  GtkCTree* ctree;

  if (!global.subscription_win) return;
  if (!sub) return;

  ctree = GTK_CTREE(lookup_widget(global.subscription_win, "ctree7"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, sub);
  if (!node) return;

  gtk_ctree_remove_node(ctree, node);
}

void subscription_user_delete(subscription_t* sub) {

  if (!sub) return;

  subscription_destroy(sub);
  subscription_user_delete_ctree(sub);
}

void subscription_update_clist(subscription_t* sub) {
  GtkCTreeNode* node;
  GtkCTree* ctree;

  if (!global.subscription_win) return;
  if (!sub) return;

  ctree = GTK_CTREE(lookup_widget(global.subscription_win, "ctree7"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, sub);
  if (!node) return;

  if (sub->friend_while_subs) sprintf(tstr[0], "%s (Always allow upload)", sub->user);
  else strcpy(tstr[0], sub->user);
  
  gtk_ctree_node_set_text(ctree, node, 0, tstr[0]);
}

void subscription_user_set_friend(subscription_t* sub, int friend) {
  if (!sub) return;

  sub->friend_while_subs = friend;

  subscription_update_clist(sub);
  subscription_save();
}

char* subscription_file_search(subscription_t* sub, char* file) {
  GList* dlist;
  char* tfile;

  for (dlist = sub->files; dlist; dlist = dlist->next) {
    tfile = dlist->data;
    if (!strcasecmp(tfile, file)) return tfile;
  }
  return NULL;
}

subscription_t* subscription_data_search(char* file) {
  GList* dlist;
  GList* dlist0;
  subscription_t* sub;
  char* tfile;

  for (dlist0 = global.subscriptions; dlist0; dlist0 = dlist0->next) {
    sub = dlist0->data;
    for (dlist = sub->files; dlist; dlist = dlist->next) {
      tfile = dlist->data;
      if (tfile == file) return sub;
    }
  }
  return NULL;
}

void subscription_file_add_ctree(subscription_t* sub, char* file) {
  GtkCTreeNode* node;
  GtkCTree* ctree;

  if (!global.subscription_win) return;
  if (!sub) return;
  if (!file) return;

  ctree = GTK_CTREE(lookup_widget(global.subscription_win, "ctree7"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, sub);
  if (!node) return;

  strcpy(tstr[0], file);
  
  node = gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			       NULL, NULL, NULL, NULL, TRUE, FALSE);
  gtk_ctree_node_set_row_data(ctree, node, file);
  
}

void subscription_file_add(subscription_t* sub, char* name) {
  char* tfile;

  if (!sub) return;
  if (subscription_file_search(sub, name)) return;

  tfile = l_strdup(name);
  sub->files = g_list_append(sub->files, tfile);
  subscription_save();
  subscription_file_add_ctree(sub, tfile);
  check_uploads_again();
}

void rec_add_sub_node(GtkCTree * ctree, GtkCTreeNode * node, gpointer data) {
  subscription_t* sub = data;
  file_t *file;
  
  if (!GTK_CTREE_ROW(node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file) {
    subscription_file_add(sub, file->longname);
  }
}

void subscription_folder_add(subscription_t* sub, GtkCTree* ctree, GtkCTreeNode* node) {

  gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) rec_add_sub_node, 
			   (gpointer)sub);
}

void subscription_file_remove_ctree(subscription_t* sub ATTR_UNUSED, char* file) {
  GtkCTreeNode* node;
  GtkCTree* ctree;

  if (!global.subscription_win) return;
  ctree = GTK_CTREE(lookup_widget(global.subscription_win, "ctree7"));

  node = gtk_ctree_find_by_row_data(ctree, NULL, file);
  if (!node) return;

  gtk_ctree_remove_node(ctree, node);
}

void subscription_file_remove(subscription_t* sub, char* file) {
  char* tfile;
  
  if (!sub) return;

  tfile = subscription_file_search(sub, file);
  if (!tfile) return;
  sub->files = g_list_remove(sub->files, tfile);
  check_uploads_again();

  subscription_file_remove_ctree(sub, tfile);
  l_free(tfile);
  subscription_save();
}

subscription_t* subscription_lookup_file(char* user, char* filename, char** ret_file) {
  subscription_t* sub;
  
  *ret_file = NULL;

  sub = subscription_user_search(user);
  if (!sub) return 0;

  *ret_file = subscription_file_search(sub, filename);
  return sub;
}

void subscription_send_info(net_t* net, char* user) {
  subscription_t* sub;
  GList* dlist;
  chat_page_t* page;

  page = chat_page_get_printable();
  chat_print_time_stamp(page, M_PUBLIC);
  chat_print_prefix(page, 1);
  chat_print_text(page, M_PUBLIC, "user", "<");
  chat_print_nick(page, M_PUBLIC, user, net);
  chat_print_text(page, M_PUBLIC, "user", "> ");
  chat_print_text(page, M_PUBLIC, "message", "has requested subscription status!\n");

  sub = subscription_user_search(user);
  if (!sub) {
    send_notice(net, user, "You are not in the subscription list, so download whatever you want", 0);
  } else {
    if (!sub->files)
      send_notice(net, user, "You are in the subscription list, but there are no files subscribed", 0);
    else {
      send_notice(net, user, "You are allowed to download the following files:", 0);
      for (dlist = sub->files; dlist; dlist = dlist->next) {
	send_notice(net, user, (char*)(dlist->data), 0);
      }
    }
  }
}

void on_friend_while_activate(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  subscription_t* sub = user_data;

  if (!sub) return;
  subscription_user_set_friend(sub, !sub->friend_while_subs);
}


void on_play_file5_activate(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  char* file = user_data;
  
  if (!file) return;
  play_file(file);
}

GtkWidget *create_subscription_popup()
{
  GtkWidget *popup = NULL;
  GtkWidget *user_popup = NULL;
  GtkWidget *item;
  GtkCTree *ctree;
  GtkCTreeNode *node;
  subscription_t* sub = NULL;
  char* filename = NULL;
  static net_user_t nu;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);
  if (!node) return NULL;

  nu.net = NULL;
  nu.user = NULL;

  if (GTK_CTREE_ROW(node)->parent != NULL) {
    filename = gtk_ctree_node_get_row_data(ctree, node);
  } else {
    sub = gtk_ctree_node_get_row_data(ctree, node);
    nu.user = sub->user;
  }
  
  if (filename) {
    popup = gtk_menu_new();
    item = gtk_menu_item_new_with_label("Remove File");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_sub_remove_activate), filename);

    item = gtk_menu_item_new_with_label("Open File");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_play_file5_activate), filename);

  } else {
    popup = gtk_menu_new();
    item = gtk_menu_item_new_with_label("Remove user");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_subscribe_activate), &nu );

    item = gtk_check_menu_item_new_with_label("Always allow upload while in list");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    if (sub->friend_while_subs) {
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
    }
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_friend_while_activate), (gpointer)sub);

    item = gtk_menu_item_new();
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_widget_set_sensitive(item, FALSE);

    item = gtk_menu_item_new_with_label("User Menu");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    
    user_popup = create_user_popup(M_SUBSCRIPTION, &nu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);
  }

  return popup;
}

