/**
 * Logback: the reliable, generic, fast and flexible logging framework.
 * Copyright (C) 1999-2011, QOS.ch. All rights reserved.
 *
 * This program and the accompanying materials are dual-licensed under
 * either the terms of the Eclipse Public License v1.0 as published by
 * the Eclipse Foundation
 *
 *   or (per the licensee's choosing)
 *
 * under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation.
 */
package ch.qos.logback.core.joran.conditional;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Stack;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ch.qos.logback.core.Context;
import ch.qos.logback.core.ContextBase;
import ch.qos.logback.core.joran.TrivialConfigurator;
import ch.qos.logback.core.joran.action.Action;
import ch.qos.logback.core.joran.action.NOPAction;
import ch.qos.logback.core.joran.action.ext.StackAction;
import ch.qos.logback.core.joran.spi.JoranException;
import ch.qos.logback.core.joran.spi.Pattern;
import ch.qos.logback.core.status.StatusChecker;
import ch.qos.logback.core.testUtil.RandomUtil;
import ch.qos.logback.core.util.CoreTestConstants;
import ch.qos.logback.core.util.StatusPrinter;

public class IfThenElseTest {

  Context context = new ContextBase();
  TrivialConfigurator tc;
  int diff = RandomUtil.getPositiveInt();
  static final String CONDITIONAL_DIR_PREFIX = CoreTestConstants.JORAN_INPUT_PREFIX
  + "conditional/";


  StackAction stackAction = new StackAction();

  @Before
  public void setUp() throws Exception {
    HashMap<Pattern, Action> rulesMap = new HashMap<Pattern, Action>();
    rulesMap.put(new Pattern("x"), new NOPAction());
    rulesMap.put(new Pattern("x/stack"), stackAction);
    rulesMap.put(new Pattern("*/if"), new IfAction());
    rulesMap.put(new Pattern("*/if/then"), new ThenAction());
    rulesMap.put(new Pattern("*/if/then/*"), new NOPAction());
    rulesMap.put(new Pattern("*/if/else"), new ElseAction());
    rulesMap.put(new Pattern("*/if/else/*"), new NOPAction());    
    
    tc = new TrivialConfigurator(rulesMap);
    tc.setContext(context);
  }

  @After
  public void tearDown() throws Exception {
    StatusPrinter.printIfErrorsOccured(context);
    //StackAction.reset();
  }

  @Test
  public void if0_Then() throws JoranException {
    context.putProperty("Ki1", "Val1");
    tc.doConfigure(CONDITIONAL_DIR_PREFIX+"if0.xml");
    verifyConfig(new String[] {"BEGIN", "a", "END"});
  }

  @Test
  public void if0_Else() throws JoranException {
    tc.doConfigure(CONDITIONAL_DIR_PREFIX+"if0.xml");
    verifyConfig(new String[] {"BEGIN", "b", "END"});
  }

  @Test
  public void ifWithoutElse_True() throws JoranException {
    context.putProperty("Ki1", "Val1");
    tc.doConfigure(CONDITIONAL_DIR_PREFIX+"ifWithoutElse.xml");
    verifyConfig(new String[] {"BEGIN", "a", "END"});
  }

  @Test
  public void ifWithoutElse_False() throws JoranException {
    tc.doConfigure(CONDITIONAL_DIR_PREFIX+"ifWithoutElse.xml");
    verifyConfig(new String[] {"BEGIN", "END"});
    StatusChecker checker = new StatusChecker(context);
    assertTrue(checker.isErrorFree(0));
  }

  
  @Test
  public void nestedIf() throws JoranException {
    tc.doConfigure(CONDITIONAL_DIR_PREFIX+"nestedIf.xml");
    StatusPrinter.printIfErrorsOccured(context);
    verifyConfig(new String[] {"BEGIN", "a", "c", "END"});
  }
  
  void verifyConfig(String[] expected) {
    Stack<String> witness = new Stack<String>();
    witness.addAll(Arrays.asList(expected));
    assertEquals(witness, stackAction.getStack());
  }
  
  
  
}
