/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * A LibreOffice extension to send the menubar structure through DBusMenu
 *
 * Copyright 2011 Canonical, Ltd.
 * Authors:
 *     Alberto Ruiz <alberto.ruiz@codethink.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the the GNU Lesser General Public License version 3, as published by the Free
 * Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR PURPOSE.  See the applicable
 * version of the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with this program. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "FrameJob.h"
#include "DesktopJob.h"
#include "FrameHelper.h"

#include <com/sun/star/awt/XSystemDependentWindowPeer.hpp>
#include <com/sun/star/awt/SystemDependentXWindow.hpp>
#include <com/sun/star/awt/XMenu.hpp>
#include <com/sun/star/awt/XMenuExtended.hpp>
#include <com/sun/star/awt/XMenuBar.hpp>
#include <com/sun/star/awt/XPopupMenu.hpp>
#include <com/sun/star/awt/XPopupMenuExtended.hpp>
#include <com/sun/star/beans/XPropertySet.hpp>
#include <com/sun/star/beans/PropertyValue.hpp>
#include <com/sun/star/container/NoSuchElementException.hpp>
#include <com/sun/star/container/XIndexAccess.hpp>
#include <com/sun/star/document/XEventBroadcaster.hpp>
#include <com/sun/star/frame/XController.hpp>
#include <com/sun/star/frame/XLayoutManager.hpp>
#include <com/sun/star/frame/XDispatch.hpp>
#include <com/sun/star/frame/XDispatchHelper.hpp>
#include <com/sun/star/frame/XDispatchProvider.hpp>
#include <com/sun/star/frame/XFrameActionListener.hpp>
#include <com/sun/star/frame/XStatusListener.hpp>
#include <com/sun/star/frame/FrameAction.hpp>
#include <com/sun/star/lang/EventObject.hpp>
#include <com/sun/star/lang/SystemDependent.hpp>
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#include <com/sun/star/ui/XUIConfigurationManager.hpp>
#include <com/sun/star/ui/XUIConfigurationManagerSupplier.hpp>
#include <com/sun/star/ui/XUIElementSettings.hpp>
#include <com/sun/star/ui/XUIElement.hpp>
#include <com/sun/star/ui/XModuleUIConfigurationManagerSupplier.hpp>
#include <com/sun/star/util/URL.hpp>
#include <com/sun/star/util/XURLTransformer.hpp>

#include <iostream>
#include <fstream>

#include <gio/gio.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/client.h>

#include <rtl/process.h>
#include <osl/diagnose.h>

#define OBJ_PATH_PREFIX "/com/canonical/menu/"

using rtl::OUString;
using rtl::OString;
using rtl::OUStringToOString;

using com::sun::star::awt::KeyEvent;
using com::sun::star::awt::SystemDependentXWindow;
using com::sun::star::awt::XMenu;
using com::sun::star::awt::XMenuExtended;
using com::sun::star::awt::XPopupMenu;
using com::sun::star::awt::XPopupMenuExtended;
using com::sun::star::awt::XMenuBar;
using com::sun::star::awt::XSystemDependentWindowPeer;
using com::sun::star::uno::Sequence;
using com::sun::star::uno::Reference;
using com::sun::star::uno::WeakReference;
using com::sun::star::uno::Any;
using com::sun::star::uno::UNO_QUERY;
using com::sun::star::uno::UNO_QUERY_THROW;
using com::sun::star::uno::XInterface;
using com::sun::star::uno::Exception;
using com::sun::star::uno::RuntimeException;
using com::sun::star::uno::XInterface;
using com::sun::star::lang::IllegalArgumentException;
using com::sun::star::lang::XMultiServiceFactory;
using com::sun::star::lang::SystemDependent::SYSTEM_XWINDOW;
using com::sun::star::lang::EventObject;
using com::sun::star::beans::NamedValue;
using com::sun::star::beans::PropertyValue;
using com::sun::star::beans::XPropertySet;
using com::sun::star::document::XEventBroadcaster;
using com::sun::star::frame::XFrame;
using com::sun::star::frame::XFrameActionListener;
using com::sun::star::frame::FrameActionEvent;
using com::sun::star::frame::XController;
using com::sun::star::frame::XLayoutManager;
using com::sun::star::frame::XModel;
using com::sun::star::frame::XModuleManager;
using com::sun::star::frame::XDispatch;
using com::sun::star::frame::XDispatchProvider;
using com::sun::star::frame::XDispatchHelper;
using com::sun::star::frame::XStatusListener;
using com::sun::star::frame::FeatureStateEvent;
using com::sun::star::ui::XUIElement;
using com::sun::star::ui::XUIElementSettings;
using com::sun::star::ui::XUIConfigurationManagerSupplier;
using com::sun::star::ui::XUIConfigurationManager;
using com::sun::star::ui::XModuleUIConfigurationManagerSupplier;
using com::sun::star::ui::XAcceleratorConfiguration;
using com::sun::star::util::URL;
using com::sun::star::util::XURLTransformer;
using com::sun::star::container::XIndexContainer;
using com::sun::star::container::XIndexAccess;
using com::sun::star::container::XNameAccess;
using com::sun::star::container::NoSuchElementException;


// This is a helper utility to transform an xid to a /com/canonical/menu/<XID>
// DBUS object path
OString
xid_to_object_path (unsigned long xid)
{

	GString *xid_str = g_string_new ("");
	g_string_printf (xid_str, "%d", (guint32)xid);
	
	OString object_path = OUStringToOString (OUString::createFromAscii (OBJ_PATH_PREFIX).concat (OUString::createFromAscii(xid_str->str)),
	                                         RTL_TEXTENCODING_ASCII_US);
	g_string_free (xid_str, TRUE);
	g_debug ("%s", object_path.getStr ());
	return object_path;
}

//-------------------------- GObject callbacks -------------------------------//
//This is called when a registrar becomes available. It registers the hides the menubar.
static void
on_registrar_available (GDBusConnection *connection,
                        const gchar     *name,
                        const gchar     *name_owner,
                        gpointer         user_data)
{
	GError     *error = NULL;
	GDBusProxy *proxy;
	
	FrameHelper *helper = (FrameHelper*)user_data;
	unsigned long xid = helper->getXID();
	
 	proxy = g_dbus_proxy_new_for_bus_sync (G_BUS_TYPE_SESSION,
 	                                       G_DBUS_PROXY_FLAGS_NONE,
 	                                       NULL,
	                                       "com.canonical.AppMenu.Registrar",
	                                       "/com/canonical/AppMenu/Registrar",
	                                       "com.canonical.AppMenu.Registrar",
 	                                       NULL,
 	                                       &error);
	if (error)
	{
		g_warning ("Couldn't get /com/canonical/AppMenu/Registrar proxy");
		return;
	}


	//TODO: Check if window is registered already
	g_dbus_proxy_call_sync (proxy,
	                        "RegisterWindow",
	                        g_variant_new ("(uo)",
	                                       (guint32)xid,
	                                       xid_to_object_path (xid).getStr()),
	                        G_DBUS_CALL_FLAGS_NONE,
	                        -1,
	                        NULL,
	                        &error);
	
	if (error)
	{
		g_warning ("Couldn't call /com/canonical/AppMenu/Registrar.RegisterWindow");
		return;
	}

	//Hide menubar
	Reference < XFrame > xFrame  = helper->getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY);
	xLayoutManager->hideElement (OUString::createFromAscii("private:resource/menubar/menubar"));
	
	return;
}

//This is called when the registrar becomes unavailable. It shows the menubar.
static void
on_registrar_unavailable (GDBusConnection *connection,
                          const gchar     *name,
                          gpointer         user_data)
{
	//TODO: Unregister window?
	
	// Show menubar
	FrameHelper *helper = (FrameHelper*)user_data;
	Reference < XFrame > xFrame  = helper->getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY);
	xLayoutManager->showElement (OUString::createFromAscii("private:resource/menubar/menubar"));
	return;
}


// Item activated. It distpatches the command associated to a given menu item.
void
item_activated (DbusmenuMenuitem *item, guint timestamp, gpointer user_data)
{
	FrameHelper *helper =  (FrameHelper*)user_data;
	OUString command = OUString::createFromAscii(dbusmenu_menuitem_property_get (item, "CommandURL"));
	helper->dispatchCommand (command);
}

// Rebuilds the submenu
gboolean
item_about_to_show (DbusmenuMenuitem *item, gpointer user_data)
{
	//Get the XMenu interface for the MenuBar UIElement                               
	FrameHelper *helper = (FrameHelper*)user_data;
	Reference < XFrame > xFrame  = helper->getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY);
	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY);
	Reference < XPropertySet > menuPropSet (menuBar, UNO_QUERY);
	
	Reference < XMenu > xMenu(menuPropSet->getPropertyValue(OUString::createFromAscii("XMenuBar")),
	                          UNO_QUERY);
	
	//Find xMenu for the first level item
	Reference < XMenu > xSubMenu;
	Reference < XMenuExtended > xMenuEx (xMenu, UNO_QUERY);
	guint16 root_count = xMenu->getItemCount();
	for (guint16 i = 0; i<root_count ;i++)
	{

		guint16 id = xMenu->getItemId (i);
		if (id == 0)
			continue;

		OUString command = xMenuEx->getCommand (id);

		//We must find the element with the same command URL
		if (! OUString::createFromAscii (dbusmenu_menuitem_property_get (item, "CommandURL")).equals (command))
			continue;

		Reference <XPopupMenu> subPopup (xMenu->getPopupMenu (id), UNO_QUERY);
		xSubMenu = Reference <XMenu> (subPopup, UNO_QUERY);
		break;
	}

		
	if (xSubMenu.is () && xSubMenu.get() != NULL)
	{
		helper->rebuildMenu (xSubMenu, item);
		// Special cases //
		
		// This is the window list. We have to build it manually
/*		if (!g_strcmp0 (dbusmenu_menuitem_property_get (item, "CommandURL"), ".uno:WindowList"))
		{
			helper->populateWindowList (item);
		}*/
	}
	return FALSE;
}

Any SAL_CALL FrameJob::execute( const Sequence< NamedValue >& aArguments )
	throw ( IllegalArgumentException, Exception, RuntimeException )
{	
	Sequence< NamedValue > lEnv;
	Reference< XModel >    xModel;
	gint                   nitems = 0;
	sal_Int32              len    = aArguments.getLength();

	for (int i = 0; i<len; i++)
	{
		if (aArguments[i].Name.equalsAscii("Environment"))
		{
			aArguments[i].Value >>= lEnv;
			break;
		}
	}
	
	len = lEnv.getLength ();
	for (int i = 0; i<len; i++)
	{	
		if (lEnv[i].Name.equalsAscii("Model"))
		{
			lEnv[i].Value >>= xModel;
		}
	}
	
	//If we didn't get the model we have to quit	
	if (!xModel.is())
		return Any();
	
	Reference< XController > xController( xModel->getCurrentController(), UNO_QUERY_THROW);
	xFrame = Reference< XFrame > ( xController->getFrame(), UNO_QUERY_THROW);
	
	
	
	exportMenus	(xFrame);
}


void
FrameJob::exportMenus (Reference < XFrame > xFrame)
{	
	//Create dbusmenu server object path string
	DbusmenuServer *server = dbusmenu_server_new (xid_to_object_path(getXID (xFrame)).getStr());
	//Create a new frame helper	to close the server when needed
	FrameHelper *helper = new FrameHelper (mxMSF, xFrame, server);
	xFrame->addFrameActionListener (Reference < XFrameActionListener > (helper));
	
	//Listen to the availability of the registrar
	guint watcher = g_bus_watch_name (G_BUS_TYPE_SESSION,
	                                  "com.canonical.AppMenu.Registrar",
	                                  G_BUS_NAME_WATCHER_FLAGS_NONE,
	                                  on_registrar_available,
	                                  on_registrar_unavailable,
	                                  helper,
	                                  NULL);
	
	helper->setRegistrarWatcher (watcher);
	
	//Getting the XMenu interface for the menubar
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY);
	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY);
	Reference < XPropertySet > menuPropSet (menuBar, UNO_QUERY);
	
	Reference < XMenu > xMenu;
	menuPropSet->getPropertyValue(OUString::createFromAscii("XMenuBar")) >>= xMenu;
	
	
	//Get xUICommands database (to retrieve labels, see FrameJob::getLabelFromCommandURL ())                                          
	Reference < XNameAccess > xNameAccess (mxMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.UICommandDescription")),
	                                       UNO_QUERY);
	Reference < XModuleManager > xModuleManager(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.ModuleManager")),
	                                            UNO_QUERY);
	xNameAccess->getByName(xModuleManager->identify(xFrame)) >>= xUICommands;
		
	//getAcceleratorConfigurations (xModel, xModuleManager);
	
	//Populate dbusmenu items and start the server
	DbusmenuMenuitem *root = getRootMenuitem (xMenu, (gpointer)helper);
	dbusmenu_server_set_root (server, root);

	helper->setRegistrarWatcher (watcher);
}

OUString FrameJob_getImplementationName ()
	throw (RuntimeException)
{
	return OUString ( RTL_CONSTASCII_USTRINGPARAM ( FRAMEJOB_IMPLEMENTATION_NAME ) );
}

sal_Bool SAL_CALL FrameJob_supportsService( const OUString& ServiceName )
	throw (RuntimeException)
{
    return ServiceName.equalsAsciiL( RTL_CONSTASCII_STRINGPARAM ( FRAMEJOB_SERVICE_NAME ) );
}

Sequence< OUString > SAL_CALL FrameJob_getSupportedServiceNames(  )
	throw (RuntimeException)
{
	Sequence < OUString > aRet(1);
    OUString* pArray = aRet.getArray();
    pArray[0] =  OUString ( RTL_CONSTASCII_USTRINGPARAM ( FRAMEJOB_SERVICE_NAME ) );
    return aRet;
}

Reference< XInterface > SAL_CALL FrameJob_createInstance( const Reference< XMultiServiceFactory > & rSMgr)
	throw( Exception )
{
	return (cppu::OWeakObject*) new FrameJob(rSMgr);
}

// XServiceInfo
OUString SAL_CALL FrameJob::getImplementationName()
	throw (RuntimeException)
{
	return FrameJob_getImplementationName();
}

sal_Bool SAL_CALL FrameJob::supportsService( const OUString& rServiceName )
	throw (RuntimeException)
{
    return FrameJob_supportsService( rServiceName );
}

Sequence< OUString > SAL_CALL FrameJob::getSupportedServiceNames()
	throw (RuntimeException)
{
    return FrameJob_getSupportedServiceNames();
}
//----------- Private functions -------------

//Set all the accelerator configuration sources
void
FrameJob::getAcceleratorConfigurations (Reference < XModel >        xModel,
                                     Reference < XModuleManager> xModuleManager)
{
	Reference< XController > xController(xModel->getCurrentController(), UNO_QUERY_THROW);
	Reference< XFrame >      xFrame(xController->getFrame(), UNO_QUERY_THROW);

	//Get document shortcut database
	Reference< XUIConfigurationManagerSupplier > docUISupplier(xModel, UNO_QUERY_THROW);
    Reference< XUIConfigurationManager >         docUIManager = docUISupplier->getUIConfigurationManager();
    Reference< XAcceleratorConfiguration >       docAccelConf(docUIManager->getShortCutManager(), UNO_QUERY_THROW);
   	this->docAccelConf = docAccelConf;

	//Get module shurtcut database
    Reference< XModuleUIConfigurationManagerSupplier > modUISupplier(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.ui.ModuleUIConfigurationManagerSupplier")),
	                                                                 UNO_QUERY_THROW);
	Reference< XUIConfigurationManager >   modUIManager = modUISupplier->getUIConfigurationManager(xModuleManager->identify(xFrame));
	Reference< XAcceleratorConfiguration > modAccelConf(modUIManager->getShortCutManager(), UNO_QUERY_THROW);
	this->modAccelConf = modAccelConf;
	
	//Get global shortcut database
	Reference< XAcceleratorConfiguration > globAccelConf(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.ui.GlobalAcceleratorConfiguration")),
	                                                     UNO_QUERY_THROW);
	this->globAccelConf = globAccelConf;
}

//Find shortcat for a given command
/*Sequence <KeyEvent>
FrameJob::findShortcutForCommand (OUString commandURL)
{
	Sequence < KeyEvent > shortcut;
	try
	{
		return docAccelConf->getKeyEventsByCommand (commandURL);
	}
	catch(const NoSuchElementException&)
	{
		try
		{
		    return modAccelConf->getKeyEventsByCommand (commandURL);
		}
		catch(const NoSuchElementException&)
		{
		    try
		    {
		        return globAccelConf->getKeyEventsByCommand (commandURL);
		    }
		    catch(const NoSuchElementException&) {}
		}
	}
	return shortcut;
}*/

//Gets the XID for a given XFrame
unsigned long
FrameJob::getXID (css::uno::Reference < css::frame::XFrame > xFrame)
{
	Reference< XSystemDependentWindowPeer > xWin( xFrame->getContainerWindow(), UNO_QUERY_THROW );
	
	if (!xWin.is())
		return 0;

	sal_Int8 processID[16];
	rtl_getGlobalProcessId( (sal_uInt8*)processID );
	Sequence <signed char> pidSeq (processID, 16);

	SystemDependentXWindow xWindow;
	xWin->getWindowHandle (pidSeq, SYSTEM_XWINDOW) >>= xWindow;
	
	return xWindow.WindowHandle;
}


//TODO: Add a DbusmenuMenuitem parameter to avoid the GList
GList*
FrameJob::iterateMenuitems (Reference < XMenu > xMenu, gpointer helper)
{
	GList* items = NULL;
	guint16 count = xMenu->getItemCount ();
	
	URL                              aCommandURL;
	Reference < XDispatch >          xDispatch;
	Reference < XDispatchProvider >  xDispProv (xFrame, UNO_QUERY);
    Reference < XURLTransformer >    xTrans(mxMSF->createInstance( rtl::OUString::createFromAscii("com.sun.star.util.URLTransformer" )), UNO_QUERY);

	XStatusListener *xSL = ((FrameHelper*)helper)->getStatusListener();
	
	Reference < XMenuExtended > xMenuEx (xMenu, UNO_QUERY);	

	for (guint16 i = 0; i < count ; i++)
	{
		guint16 id = xMenu->getItemId (i);
		
		if (id == 0)
		{
			DbusmenuMenuitem *sep = dbusmenu_menuitem_new ();
            dbusmenu_menuitem_property_set (sep, DBUSMENU_MENUITEM_PROP_TYPE, DBUSMENU_CLIENT_TYPES_SEPARATOR);

			//This command does nothing, but the signal is useful if the value of the command is replaced
			//when updating the menu structure.
			g_signal_connect(G_OBJECT(sep), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(item_activated), helper);
			
       		dbusmenu_menuitem_property_set (sep, "CommandURL", "slot:0");
            items = g_list_append (items, (gpointer)sep);
            continue;
		}
		//FIXME: Check if item is enabled and discard it if so
		
		DbusmenuMenuitem *item = dbusmenu_menuitem_new ();
		
		/* Replace the tilde with an underscore */
		OString label = OUStringToOString (xMenu->getItemText (id).replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f), RTL_TEXTENCODING_UTF8);
		OString command = OUStringToOString (xMenuEx->getCommand (id), RTL_TEXTENCODING_ASCII_US);
		
		aCommandURL.Complete = xMenuEx->getCommand (id);
		xTrans->parseStrict (aCommandURL);
		Reference < XDispatch > xDispatch  = xDispProv->queryDispatch (aCommandURL, OUString(), 0);
		if (xDispatch.is())
		{
			xDispatch->addStatusListener (xSL, aCommandURL);
		}
		
		dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_LABEL, label.getStr());
		dbusmenu_menuitem_property_set (item, "CommandURL", command.getStr ());
		
		Reference < XPopupMenu > subPopMenu (xMenu->getPopupMenu (id), UNO_QUERY);
		
		if (subPopMenu.is ())
		{
			Reference < XMenu> subXMenu (subPopMenu, UNO_QUERY);

			GList *subitems = iterateMenuitems (subXMenu, helper);
			GList *head = subitems;
			while (subitems)
			{
				dbusmenu_menuitem_child_append (item, (DbusmenuMenuitem*)subitems->data);
				subitems = subitems->next;
			}
			
			g_list_free (head);
		}
		
		g_signal_connect(G_OBJECT(item), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(item_activated), helper);
		
		items = g_list_append (items, (gpointer)item);
	}
	
	return items;
}

DbusmenuMenuitem*
FrameJob::getRootMenuitem (Reference < XMenu > xMenu, gpointer helper)
{
	
	DbusmenuMenuitem *root = dbusmenu_menuitem_new_with_id (0);
	((FrameHelper*)helper)->setRootItem(root);
	((FrameHelper*)helper)->rebuildMenu (xMenu, root);


	
/*	GList *items = iterateMenuitems (xMenu, helper);
	GList *first = items;
	while (items)
	{
		dbusmenu_menuitem_child_append (root, (DbusmenuMenuitem*)items->data);
		//FIXME: The root items have an about_to_show callback to rebuild the menus
		//       We need to figure out if there's a LibreOffice signal available within UNO
		//       that we can use to avoid this ugly hack.

//		g_signal_connect(G_OBJECT(items->data), DBUSMENU_MENUITEM_SIGNAL_ABOUT_TO_SHOW, G_CALLBACK(item_about_to_show), helper);
		
		//We add a status listener per item
		const gchar *cmdString = dbusmenu_menuitem_property_get ((DbusmenuMenuitem*)items->data, "CommandURL");
		commandURL.Complete = OUString::createFromAscii (cmdString);
		xTrans->parseStrict (commandURL);
		Reference < XDispatch > xDispatch  = xDispProv->queryDispatch (commandURL, OUString(), 0);
		if (xDispatch.is())
		{
			xDispatch->addStatusListener (xSL, commandURL);
		}
		
		items = items->next;
	}
	g_list_free(first);*/
	
	return root;
}
