/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * A LibreOffice extension to send the menubar structure through DBusMenu
 *
 * Copyright 2011 Canonical, Ltd.
 * Authors:
 *     Alberto Ruiz <alberto.ruiz@codethink.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the the GNU Lesser General Public License version 3, as published by the Free
 * Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR PURPOSE.  See the applicable
 * version of the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with this program. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "FrameHelper.h"

#include <com/sun/star/awt/SystemDependentXWindow.hpp>
#include <com/sun/star/awt/XSystemDependentWindowPeer.hpp>
#include <com/sun/star/awt/XWindow2.hpp>
#include <com/sun/star/beans/XPropertySet.hpp>
#include <com/sun/star/container/XNameAccess.hpp>
#include <com/sun/star/awt/XPopupMenu.hpp>
#include <com/sun/star/awt/XMenuExtended.hpp>
#include <com/sun/star/frame/XDispatch.hpp>
#include <com/sun/star/frame/XDispatchHelper.hpp>
#include <com/sun/star/frame/XDispatchProvider.hpp>
#include <com/sun/star/frame/XLayoutManager.hpp>
#include <com/sun/star/frame/XModuleManager.hpp>
#include <com/sun/star/frame/status/Visibility.hpp>
#include <com/sun/star/lang/SystemDependent.hpp>
#include <com/sun/star/beans/PropertyValue.hpp>
#include <com/sun/star/util/URL.hpp>
#include <com/sun/star/util/XURLTransformer.hpp>
#include <com/sun/star/ui/XUIElement.hpp>

#include <rtl/process.h>

#include <gio/gio.h>
#include <libdbusmenu-glib/client.h>

using rtl::OUString;
using rtl::OString;
using rtl::OUStringToOString;

using com::sun::star::awt::SystemDependentXWindow;
using com::sun::star::awt::XMenuExtended;
using com::sun::star::awt::XPopupMenu;
using com::sun::star::awt::XSystemDependentWindowPeer;
using com::sun::star::awt::XWindow2;
using com::sun::star::beans::XPropertySet;
using com::sun::star::beans::PropertyValue;
using com::sun::star::container::XNameAccess;
using com::sun::star::container::NoSuchElementException;
using com::sun::star::frame::XDispatch;
using com::sun::star::frame::XDispatchProvider;
using com::sun::star::frame::XDispatchHelper;
using com::sun::star::frame::XModuleManager;
using com::sun::star::frame::XLayoutManager;
using com::sun::star::frame::FeatureStateEvent;
using com::sun::star::frame::status::Visibility;
using com::sun::star::lang::SystemDependent::SYSTEM_XWINDOW;
using com::sun::star::uno::UNO_QUERY;
using com::sun::star::uno::UNO_QUERY_THROW;
using com::sun::star::uno::Sequence;
using com::sun::star::ui::XUIElement;
using com::sun::star::util::URL;
using com::sun::star::util::XURLTransformer;

gboolean
item_about_to_show (DbusmenuMenuitem *item, gpointer user_data);

typedef enum {
	MENU_ITEM_INVALID = 0,
	MENU_ITEM_ENABLED = 1,
	MENU_ITEM_DISABLED = 2
} MenuItemStatus;

// This is used in a hash table with commandurls as keys
class MenuItemInfo {
	gchar*            label;         //Label text in UTF-8 with tildes subst by underscores
	gint              check_state;
	gchar*            check_type;
	gboolean          is_enabled;
	gboolean          is_visible;

  public:
	MenuItemInfo ()
	{
		label = NULL;
		check_state = DBUSMENU_MENUITEM_TOGGLE_STATE_UNKNOWN;
		check_type  = (gchar*)DBUSMENU_MENUITEM_TOGGLE_CHECK;

		is_visible = TRUE;		
		is_enabled = TRUE;
	}
	~MenuItemInfo ()
	{
		if (label)
			g_free(label);
	}
	
	//Setters
	void
	setLabel (gchar* label)
	{
		this->label = g_strdup (label);
	}
	
	void
	setEnabled (gboolean is_enabled)
	{
		this->is_enabled = is_enabled;
	}

	void
	setCheckState (gint check_state)
	{
		this->check_state = check_state;
	}
	
	void
	setCheckType (const gchar* check_type)
	{
		this->check_type = (gchar*)check_type;
	}
	
	void
	setVisible (gboolean is_visible)
	{
		this->is_visible = is_visible;
	}
	
	//Getters
	gchar*
	getLabel ()
	{
		return label;
	}
	
	gboolean
	getEnabled ()
	{
		return is_enabled;
	}

	gint
	getCheckState ()
	{
		return check_state;
	}
	
	const gchar*
	getCheckType ()
	{
		return check_type;
	}
	
	gboolean
	getVisible ()
	{
		return is_visible;
	}
};

void
destroy_menu_item_info (gpointer data)
{
	delete (MenuItemInfo*)data;
}

class MenuItemStatusListener : public cppu::WeakImplHelper1 < XStatusListener >
{
  private:
	guint16 id;
	FrameHelper *helper;	

  public:
  
	MenuItemStatusListener (FrameHelper *helper)
	{
		if (!helper) throw ("FrameHelper cannot be NULL");
		this->helper = helper;
	}
	
	~MenuItemStatusListener ()
	{
	}
	
	virtual void SAL_CALL
	statusChanged(const FeatureStateEvent& Event)
	  throw (RuntimeException)
	{	
		sal_Bool            isChecked;
		Visibility          visible;
		OUString url = Event.FeatureURL.Complete;
		OUString oULabel;
		gchar* c_url = g_utf16_to_utf8 (url.getStr(),
			                            url.getLength(),
			                            NULL, NULL, NULL);
		g_debug ("UPDATE %s", c_url);
		
		GHashTable *commandsInfo = helper->getCommandsInfo ();
		MenuItemInfo *info = (MenuItemInfo*)g_hash_table_lookup (commandsInfo, (gpointer)c_url);
		if (!info)
		{
			info = new MenuItemInfo ();
			g_hash_table_insert (commandsInfo, c_url, info);
			
			//Set the default label
			oULabel = helper->getLabelFromCommandURL(url);
			// Replace tilde with underscore for Dbusmenu Alt accelerators
			oULabel = oULabel.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
	
			// GLib behaves better than OUStringToOString wrt encoding transformation
			gchar* label = g_utf16_to_utf8 (oULabel.getStr(),
			                                oULabel.getLength(),
			                                NULL, NULL, NULL);		
			info->setLabel (label);
			g_free (label);
		}
		else
		{
			//Since we're not introducing it in the hash table, we get rid of this string
			g_free (c_url);
		}
		
		/* For some reason, URLs can slip through as labels? */
		info->setEnabled ((gboolean)Event.IsEnabled);
		
		if ((Event.State >>= oULabel) &&
		    !oULabel.matchAsciiL ("private:", 8, 0) &&
		    !oULabel.matchAsciiL (".uno:", 5, 0))
		{	
			oULabel = oULabel.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
			gchar* label = g_utf16_to_utf8 (oULabel.getStr(),
		                             oULabel.getLength(),
		                             NULL, NULL, NULL);		
			info->setLabel (label);
			g_free (label);
		}
		else if (Event.State >>= isChecked)
		{
			info->setCheckState (isChecked);
		}
		else if (Event.State >>= visible)
		{
			info->setVisible (visible.bVisible);
		}
		else 
		{
		}
	}
	
	virtual void SAL_CALL disposing(const EventObject& aEvent)
	  throw (RuntimeException)
	{}
};


// Item callbacks
void item_activated (DbusmenuMenuitem *item, guint timestamp, gpointer user_data);

void
destroy_menuitem (gpointer data)
{
	g_object_unref (G_OBJECT (data));
}

FrameHelper::FrameHelper(const Reference< XMultiServiceFactory >&  rServiceManager,
                    const Reference< XFrame >&        xFrame,
                    DbusmenuServer*                   server)
{
	xMSF = rServiceManager;
	this->xFrame = xFrame;
	this->server = server;
	
	//Get xUICommands database (to retrieve labels, see FrameJob::getLabelFromCommandURL ())                                          
	Reference < XNameAccess > xNameAccess (xMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.UICommandDescription")),
	                                       UNO_QUERY);
	Reference < XModuleManager > xModuleManager(xMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.ModuleManager")),
	                                            UNO_QUERY);
	xNameAccess->getByName(xModuleManager->identify(xFrame)) >>= xUICommands;
	
	
	xdp = Reference < XDispatchProvider > (xFrame, UNO_QUERY_THROW);
    xTrans = Reference < XURLTransformer > (xMSF->createInstance( rtl::OUString::createFromAscii("com.sun.star.util.URLTransformer" )), UNO_QUERY);
	
	xSL = (XStatusListener*)new MenuItemStatusListener (this);
	commandsInfo = g_hash_table_new_full (g_str_hash,
		                                  g_str_equal,
		                                  g_free,
		                                  destroy_menu_item_info);
	
	root = NULL;
	watcher_set = FALSE;
}

FrameHelper::~FrameHelper()
{
	if (server)
		g_object_unref (server);
		
	if (root)
		g_object_unref (root);

	if (watcher_set)
		g_bus_unwatch_name (watcher);
		
	g_hash_table_destroy (commandsInfo);
}

void
FrameHelper::setRootItem (DbusmenuMenuitem *root)
{
	this->root = root;
}

void
FrameHelper::setRegistrarWatcher (guint watcher)
{
	watcher_set = TRUE;
	this->watcher = watcher;
}

void
FrameHelper::setServer (DbusmenuServer *server)
{
	this->server = server;
}

//Getters
Reference < XFrame >
FrameHelper::getFrame ()
{
	return xFrame;
}

XStatusListener*
FrameHelper::getStatusListener ()
{
	return xSL;
}

GHashTable*
FrameHelper::getCommandsInfo ()
{
	return commandsInfo;
}

unsigned long
FrameHelper::getXID ()
{
	Reference< XSystemDependentWindowPeer > xWin( xFrame->getContainerWindow(), UNO_QUERY_THROW );

	if (!xWin.is())
		return 0;

	sal_Int8 processID[16];
	rtl_getGlobalProcessId( (sal_uInt8*)processID );
	Sequence <signed char> pidSeq (processID, 16);

	SystemDependentXWindow xWindow;
	xWin->getWindowHandle (pidSeq, SYSTEM_XWINDOW) >>= xWindow;

	return xWindow.WindowHandle;
}
	
void
FrameHelper::dispatchCommand (OUString command)
{
	Reference < XDispatchHelper > rDispatchHelper (xMSF->createInstance(OUString(RTL_CONSTASCII_USTRINGPARAM("com.sun.star.frame.DispatchHelper"))),
	                                               UNO_QUERY_THROW);
	Reference < XDispatchHelper > xdh(rDispatchHelper);
	
	
	xdh->executeDispatch (Reference < XDispatchProvider > (xFrame, UNO_QUERY_THROW),
	                      command,
	                      OUString::createFromAscii(""),
	                      0,
	                      Sequence < PropertyValue > ());
}

void
FrameHelper::rebuildMenu (Reference < XMenu >  xMenu,
                          DbusmenuMenuitem    *parent)
{
	g_return_if_fail (parent != NULL);
	GList *items = dbusmenu_menuitem_get_children (parent);	
	guint nitems = g_list_length (items);
	guint16 count = xMenu->getItemCount ();
	
	// One item does not represent always the same command.
	// We do this for performance reasons, as it's really hard to match a command with
	// a single dbusmenuitem given the lack of information provided by the status listener
	if (count > nitems)
	{
		// Add enough items to replicate all 
		for (guint16 i = 0; i < (count - nitems); i++)
		{
			DbusmenuMenuitem *item = dbusmenu_menuitem_new ();
			dbusmenu_menuitem_child_append (parent, item);
			g_signal_connect(G_OBJECT(item), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(item_activated), this);
		}
		items = dbusmenu_menuitem_get_children (parent);	
	}
	if (count < nitems)
	{
		for (guint16 i = nitems - 1; i >= count; i--)
		{
			DbusmenuMenuitem *item = DBUSMENU_MENUITEM (g_list_nth_data(items, i));
			dbusmenu_menuitem_property_set_bool (item, DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
		}
	}
	
	for (guint16 i = 0; i<count; i++)
	{
		gchar * label = NULL;
		gint isEnabled = 0;
		Reference < XMenuExtended > xMenuEx (xMenu, UNO_QUERY);
		guint16 id = xMenu->getItemId (i);
		OUString oUCommand = xMenuEx->getCommand (id);
		
		DbusmenuMenuitem *item = DBUSMENU_MENUITEM(g_list_nth_data(items, i));
		
		if (!item)
			continue;

		if (!DBUSMENU_IS_MENUITEM (item))
			continue;
		
		
		//We set the default properties (in case it was not visible or a separator)
		dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_TYPE, DBUSMENU_CLIENT_TYPES_DEFAULT);
		dbusmenu_menuitem_property_set_bool (item, DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

		if (id == 0)
		{
       		dbusmenu_menuitem_property_set (item, "CommandURL", "slot:0");
            dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_TYPE, DBUSMENU_CLIENT_TYPES_SEPARATOR);
			//Getting rid of any possible children
   			g_list_free_full (dbusmenu_menuitem_take_children (item), destroy_menuitem);
       		continue;
		}
		
		
		// Replace the tilde with an underscore 
	
		OString command = OUStringToOString (oUCommand, RTL_TEXTENCODING_ASCII_US);
		dbusmenu_menuitem_property_set (item, "CommandURL", command.getStr ());
		
		//g_debug ("new item: %s", command.getStr ());

		MenuItemInfo* commInfo = (MenuItemInfo*)g_hash_table_lookup (commandsInfo, (gconstpointer)command.getStr());
		if (!commInfo)
		{
			commInfo = new MenuItemInfo ();
			g_hash_table_insert (commandsInfo, g_strdup (command.getStr()), commInfo);
			
			
			OUString oULabel = getLabelFromCommandURL(oUCommand);
			//Replace tilde with underscore for Dbusmenu Alt accelerators
			oULabel = oULabel.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
	
			// GLib behaves better than OUStringToOString wrt encoding transformation
			gchar* label = g_utf16_to_utf8 (oULabel.getStr(),
			                                oULabel.getLength(),
			                                NULL, NULL, NULL);	
			commInfo->setLabel (label);
					
			g_free (label);
		}

		//Update the check state directly from the data, this is more reliable
		
		Reference < XPopupMenu > popUp (xMenu, UNO_QUERY);
		if (popUp.is() && popUp->isItemChecked (id))
		{
			commInfo->setCheckState (DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED);
		}
		
		dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_LABEL, commInfo->getLabel ());
		dbusmenu_menuitem_property_set_bool (item, DBUSMENU_MENUITEM_PROP_ENABLED, commInfo->getEnabled ());

		if (commInfo->getCheckState () != DBUSMENU_MENUITEM_TOGGLE_STATE_UNKNOWN)
		{
			dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,  commInfo->getCheckType ());
			dbusmenu_menuitem_property_set_int (item, DBUSMENU_MENUITEM_PROP_TOGGLE_STATE, commInfo->getCheckState ());
		}

		// Adding status listener
		URL commandURL;
		commandURL.Complete = oUCommand;
		xTrans->parseStrict (commandURL);

		Reference < XDispatch > xDispatch  = xdp->queryDispatch (commandURL, OUString(), 0);
		if (xDispatch.is())
			xDispatch->addStatusListener (xSL, commandURL);
		
		// Introspect submenus
		Reference < XPopupMenu > subPopMenu (xMenu->getPopupMenu (id), UNO_QUERY);
		if (subPopMenu.is ())
		{
			Reference <XMenu> subMenu (subPopMenu, UNO_QUERY);	
			g_signal_connect(G_OBJECT(item), DBUSMENU_MENUITEM_SIGNAL_ABOUT_TO_SHOW, G_CALLBACK(item_about_to_show), this);			
			rebuildMenu (subMenu, item);
		}
		//TODO: Recreate the "dynamic" popup menus
		else
		{
			//Getting rid of any possible children
			g_list_free_full (dbusmenu_menuitem_take_children (item), destroy_menuitem);
		}		
	}
	
	return;
}

//Gets the menu item Label given a CommandURL
//This is a work around for bug: https://bugs.freedesktop.org/show_bug.cgi?id=34127
OUString
FrameHelper::getLabelFromCommandURL (OUString commandURL)
{
	OUString label;

	Sequence < PropertyValue > commandProps;
	
	if (commandURL.getLength () < 1)
		return label;
	
	try
	{
		xUICommands->getByName (commandURL) >>= commandProps;
	}
	catch (com::sun::star::container::NoSuchElementException e)
	{
		return label;
	}
	
	for (sal_Int32 i = 0; i < commandProps.getLength(); i++)
	{
		if ( commandProps[i].Name.equalsAsciiL (RTL_CONSTASCII_STRINGPARAM ("Label")))
		{
			commandProps[i].Value >>= label;
			label = label.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
			//break;
		}
	}

	return label;
}

/*typedef struct {
	gboolean focused;
	gchar *  title;
} WindowInfo;

void
FrameHelper::populateWindowList(DbusmenuMenuitem *parent)
{
	GList *window_list = NULL;
	Reference < XFramesSupplier > xDesktop(xMSF->createInstance(OUString ( RTL_CONSTASCII_USTRINGPARAM ("com.sun.star.frame.Desktop"))),
	                                       UNO_QUERY);
	
	if (!xDesktop.is())
		return;
	
	//Retreive focus and title info for each toplevel LibreOffice Window
	Sequence < Reference < XFrame > > xFrames(xDesktop->getFrames()->queryFrames(com::sun::star::frame::FrameSearchFlag::ALL));
	for (gint32 i = 0; i < xFrames.getLength (); i++)
	{
		Reference < XFrame > xFrame (xFrames[i], UNO_QUERY);
		if (!xFrame.is ())
			continue;

		if (!xFrame->isTop())
			continue;
				
		Reference < XWindow2 > xWindow (xFrame->getContainerWindow (), UNO_QUERY);
		if (xWindow.is())
		{
			OUString title;
			WindowInfo *info = (WindowInfo*)g_malloc (sizeof (WindowInfo));
			
			Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
			frameProps->getPropertyValue (OUString::createFromAscii ("Title")) >>= title;
			title = title.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
			
			//Set the values on the struct and append to the list
			info->title = g_utf16_to_utf8 (title.getStr(),
			                               title.getLength(),
			                               NULL, NULL, NULL);
			info->focused = xFrame->isActive();
			
			window_list = g_list_append (window_list, (gpointer)info);
		}
	}
	
	//We ensure here that there are enough Menuitems to list all windows
	//Count the number of items after the last separator:
	GList *items = dbusmenu_menuitem_get_children (parent);
	GList *last_sep = g_list_last(items);
	guint last_sep_index = 0;
	while (last_sep)
	{
		last_sep_index++;
		last_sep = last_sep->prev;
		if (!g_strcmp0 (dbusmenu_menuitem_property_get (DBUSMENU_MENUITEM(last_sep->data),
		                                                "CommandURL"),
		                "slot:0"));
		break;
	}
	if (!last_sep)
	{
		//There was some error looking for the last separator
		g_list_free_full (window_list, g_free);
		return;
	}
	//Index of the last separator starting from 1
	last_sep_index = g_list_length (items) - last_sep_index + 1;
	
	//Create new menuitems if there are less than we need
	if ((last_sep_index + g_list_length (window_list)) > g_list_length (items))
	{
		// Add enough items to replicate all
		guint n_new_items = last_sep_index + g_list_length (window_list) - g_list_length (items);
		for (guint i = 0; i < n_new_items; i++)
		{
			DbusmenuMenuitem *item = dbusmenu_menuitem_new ();
			dbusmenu_menuitem_child_append (parent, item);
			g_signal_connect(G_OBJECT(item), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(item_activated), this);
		}
		items = dbusmenu_menuitem_get_children (parent);
	}

	last_sep = g_list_last(items);
	while (last_sep)
	{
		last_sep = last_sep->prev;
		if (!g_strcmp0 (dbusmenu_menuitem_property_get (DBUSMENU_MENUITEM(last_sep->data),
		                                                "CommandURL"),
		                "slot:0"));
		break;
	}
	

	//Set the information on all menuitems

	GList *window_info = window_list;
	GList *window_item = last_sep->next;
	while (window_item && window_info)
	{
		DbusmenuMenuitem *item = DBUSMENU_MENUITEM(window_item->data);
		WindowInfo *info = (WindowInfo*)window_info->data;
		
		//TODO: Find a way to identify the window (setName?)
		dbusmenu_menuitem_property_set (item, "CommandURL", "focusAnotherWindow");
		dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_LABEL, info->title);
		dbusmenu_menuitem_property_set (item, DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,  DBUSMENU_MENUITEM_TOGGLE_RADIO);
		dbusmenu_menuitem_property_set_bool (item, DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
		
		if (info->focused)
			dbusmenu_menuitem_property_set_int (item, DBUSMENU_MENUITEM_PROP_TOGGLE_STATE, DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED);
		else
			dbusmenu_menuitem_property_set_int (item, DBUSMENU_MENUITEM_PROP_TOGGLE_STATE, DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED);
		
		window_info = window_info->next;
		window_item = window_item->next;
	}
	
	g_list_free_full (window_list, g_free);
}*/

void
FrameHelper::rebuildMenuFromRoot ()
{
	Reference < XFrame > xFrame  = getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY);
	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY);
	Reference < XPropertySet > menuPropSet (menuBar, UNO_QUERY);
	
	Reference < XMenu > xMenu(menuPropSet->getPropertyValue(OUString::createFromAscii("XMenuBar")),
	                          UNO_QUERY);

	rebuildMenu (xMenu, root);
}
