/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * A LibreOffice extension to send the menubar structure through DBusMenu
 *
 * Copyright 2011 Canonical, Ltd.
 * Authors:
 *     Alberto Ruiz <alberto.ruiz@codethink.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the the GNU Lesser General Public License version 3, as published by the Free
 * Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR PURPOSE.  See the applicable
 * version of the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with this program. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "MyJob.h"
#include "DesktopJob.h"
#include "FrameHelper.h"

#include <com/sun/star/awt/XSystemDependentWindowPeer.hpp>
#include <com/sun/star/awt/SystemDependentXWindow.hpp>
#include <com/sun/star/awt/XMenuBar.hpp>
#include <com/sun/star/beans/XPropertySet.hpp>
#include <com/sun/star/beans/PropertyValue.hpp>
#include <com/sun/star/container/NoSuchElementException.hpp>
#include <com/sun/star/container/XIndexAccess.hpp>
#include <com/sun/star/document/XEventBroadcaster.hpp>
#include <com/sun/star/frame/XController.hpp>
#include <com/sun/star/frame/XLayoutManager.hpp>
#include <com/sun/star/frame/XDispatchHelper.hpp>
#include <com/sun/star/frame/XDispatchProvider.hpp>
#include <com/sun/star/frame/XFrameActionListener.hpp>
#include <com/sun/star/frame/FrameAction.hpp>
#include <com/sun/star/lang/EventObject.hpp>
#include <com/sun/star/lang/SystemDependent.hpp>
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#include <com/sun/star/ui/XUIConfigurationManager.hpp>
#include <com/sun/star/ui/XUIConfigurationManagerSupplier.hpp>
#include <com/sun/star/ui/XUIElementSettings.hpp>
#include <com/sun/star/ui/XUIElement.hpp>
#include <com/sun/star/ui/XModuleUIConfigurationManagerSupplier.hpp>

#include <iostream>
#include <fstream>

#include <gio/gio.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/client.h>

#include <rtl/process.h>
#include <osl/diagnose.h>

#define OBJ_PATH_PREFIX "/com/canonical/menu/"

using rtl::OUString;
using rtl::OString;
using rtl::OUStringToOString;

using com::sun::star::awt::KeyEvent;
using com::sun::star::awt::SystemDependentXWindow;
using com::sun::star::awt::XMenuBar;
using com::sun::star::awt::XSystemDependentWindowPeer;
using com::sun::star::uno::Sequence;
using com::sun::star::uno::Reference;
using com::sun::star::uno::WeakReference;
using com::sun::star::uno::Any;
using com::sun::star::uno::UNO_QUERY;
using com::sun::star::uno::UNO_QUERY_THROW;
using com::sun::star::uno::XInterface;
using com::sun::star::uno::Exception;
using com::sun::star::uno::RuntimeException;
using com::sun::star::uno::XInterface;
using com::sun::star::lang::IllegalArgumentException;
using com::sun::star::lang::XMultiServiceFactory;
using com::sun::star::lang::SystemDependent::SYSTEM_XWINDOW;
using com::sun::star::lang::EventObject;
using com::sun::star::beans::NamedValue;
using com::sun::star::beans::PropertyValue;
using com::sun::star::beans::XPropertySet;
using com::sun::star::document::XEventBroadcaster;
using com::sun::star::frame::XFrame;
using com::sun::star::frame::XFrameActionListener;
using com::sun::star::frame::FrameActionEvent;
using com::sun::star::frame::XController;
using com::sun::star::frame::XLayoutManager;
using com::sun::star::frame::XModel;
using com::sun::star::frame::XModuleManager;
using com::sun::star::frame::XDispatchProvider;
using com::sun::star::frame::XDispatchHelper;
using com::sun::star::ui::XUIElement;
using com::sun::star::ui::XUIElementSettings;
using com::sun::star::ui::XUIConfigurationManagerSupplier;
using com::sun::star::ui::XUIConfigurationManager;
using com::sun::star::ui::XModuleUIConfigurationManagerSupplier;
using com::sun::star::ui::XAcceleratorConfiguration;
using com::sun::star::container::XIndexContainer;
using com::sun::star::container::XIndexAccess;
using com::sun::star::container::XNameAccess;
using com::sun::star::container::NoSuchElementException;

typedef struct {
	OUString  commandURL;
	OUString  label;
	sal_Int32 type;
	sal_Bool  isVisible;
	Reference < XIndexAccess > itemDescriptorContainer;
	DbusmenuMenuitem *submenu;
} TmpItemDescriptor;

OString
xid_to_object_path (unsigned long xid)
{

	GString *xid_str = g_string_new ("");
	g_string_printf (xid_str, "%d", (guint32)xid);
	
	OString object_path = OUStringToOString (OUString::createFromAscii (OBJ_PATH_PREFIX).concat (OUString::createFromAscii(xid_str->str)),
	                                         RTL_TEXTENCODING_ASCII_US);
	g_string_free (xid_str, TRUE);
	
	g_debug ("%s", object_path.getStr ());
	
	return object_path;
}

//GObject callbacks
static void
on_registrar_available (GDBusConnection *connection,
                        const gchar     *name,
                        const gchar     *name_owner,
                        gpointer         user_data)
{
	GError     *error = NULL;
	GDBusProxy *proxy;
	
	FrameHelper *helper = (FrameHelper*)user_data;
	unsigned long xid = helper->getXID();
	
 	proxy = g_dbus_proxy_new_for_bus_sync (G_BUS_TYPE_SESSION,
 	                                       G_DBUS_PROXY_FLAGS_NONE,
 	                                       NULL,
	                                       "com.canonical.AppMenu.Registrar",
	                                       "/com/canonical/AppMenu/Registrar",
	                                       "com.canonical.AppMenu.Registrar",
 	                                       NULL,
 	                                       &error);
	if (error)
	{
		g_warning ("Couldn't get /com/canonical/AppMenu/Registrar proxy");
		return;
	}

	g_dbus_proxy_call_sync (proxy,
	                        "RegisterWindow",
	                        g_variant_new ("(uo)",
	                                       (guint32)xid,
	                                       xid_to_object_path (xid).getStr()),
	                        G_DBUS_CALL_FLAGS_NONE,
	                        -1,
	                        NULL,
	                        &error);
	
	if (error)
	{
		g_warning ("Couldn't call /com/canonical/AppMenu/Registrar.RegisterWindow");
		return;
	}

	//Hide menubar
	Reference < XFrame > xFrame  = helper->getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY_THROW);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY_THROW);
	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY_THROW);
	                                 
	xLayoutManager->hideElement (OUString::createFromAscii("private:resource/menubar/menubar"));
	
	return;
}

static void
on_registrar_unavailable (GDBusConnection *connection,
                          const gchar     *name,
                          gpointer         user_data)
{
	// Show menubar
	FrameHelper *helper = (FrameHelper*)user_data;
	Reference < XFrame > xFrame  = helper->getFrame ();
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY_THROW);
	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY_THROW);

	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY_THROW);
	  
	xLayoutManager->showElement (OUString::createFromAscii("private:resource/menubar/menubar"));
	return;
}

void
item_activated (DbusmenuMenuitem *item, guint timestamp, gpointer user_data)
{
	FrameHelper *helper =  (FrameHelper*)user_data;
	OUString command = OUString::createFromAscii(dbusmenu_menuitem_property_get (item, "CommandURL"));
	helper->dispatchCommand (command);
}

Any SAL_CALL MyJob::execute( const Sequence< NamedValue >& aArguments )
	throw ( IllegalArgumentException, Exception, RuntimeException )
{	
	Sequence< NamedValue > lEnv;
	Reference< XModel >    xModel;
	gint                   nitems = 0;
	sal_Int32              len    = aArguments.getLength();

	for (int i = 0; i<len; i++)
	{
		if (aArguments[i].Name.equalsAscii("Environment"))
		{
			aArguments[i].Value >>= lEnv;
			break;
		}
	}
	
	len = lEnv.getLength ();
	for (int i = 0; i<len; i++)
	{	
		if (lEnv[i].Name.equalsAscii("Model"))
		{
			lEnv[i].Value >>= xModel;
		}
	}
	
	//If we didn't get the model we have to quit	
	if (!xModel.is())
		return Any();
		
	Reference< XController > xController( xModel->getCurrentController(), UNO_QUERY_THROW);
	xFrame = Reference< XFrame > ( xController->getFrame(), UNO_QUERY_THROW);
	
	//Create dbusmenu server object path string
	DbusmenuServer *server = dbusmenu_server_new (xid_to_object_path(getXID (xFrame)).getStr());
	//Create a new frame helper	to close the server when needed
	FrameHelper *helper = new FrameHelper (mxMSF, xFrame, server);
	
	guint watcher = g_bus_watch_name (G_BUS_TYPE_SESSION,
	                                  "com.canonical.AppMenu.Registrar",
	                                  G_BUS_NAME_WATCHER_FLAGS_NONE,
	                                  on_registrar_available,
	                                  on_registrar_unavailable,
	                                  helper,
	                                  NULL);
	helper->setRegistrarWatcher (watcher);
	
	Reference< XPropertySet > frameProps (xFrame, UNO_QUERY_THROW);

	Reference < XLayoutManager > xLayoutManager(frameProps->getPropertyValue(OUString::createFromAscii("LayoutManager")),
	                                            UNO_QUERY_THROW);

	Reference < XUIElement > menuBar(xLayoutManager->getElement (OUString::createFromAscii("private:resource/menubar/menubar")),
	                                 UNO_QUERY_THROW);
	
	Reference < XUIElementSettings > menuSettings(menuBar, UNO_QUERY_THROW);
	Reference < XIndexContainer > menuSetCont(menuSettings->getSettings(TRUE),
	                                          UNO_QUERY_THROW);                                          
	
	
	//Get xUICommands database (to retrieve labels, see MyJob::getLabelFromCommandURL ())                                          
	Reference < XNameAccess > xNameAccess (mxMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.UICommandDescription")),
	                                       UNO_QUERY_THROW);
	Reference < XModuleManager > xModuleManager(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.frame.ModuleManager")),
	                                            UNO_QUERY_THROW);
	xNameAccess->getByName(xModuleManager->identify(xFrame)) >>= xUICommands;
		
	getAcceleratorConfigurations (xModel, xModuleManager);
	
	//Populate dbusmenu items and start the server
	DbusmenuMenuitem *root = getRootMenuitem (menuSetCont, (gpointer)helper);
	dbusmenu_server_set_root (server, root);
	//TODO: Use XTopWindow?
	xFrame->addFrameActionListener (Reference < XFrameActionListener > (helper));
	helper->setRegistrarWatcher (watcher);

	return Any ();
}

OUString MyJob_getImplementationName ()
	throw (RuntimeException)
{
	return OUString ( RTL_CONSTASCII_USTRINGPARAM ( MYJOB_IMPLEMENTATION_NAME ) );
}

sal_Bool SAL_CALL MyJob_supportsService( const OUString& ServiceName )
	throw (RuntimeException)
{
    return ServiceName.equalsAsciiL( RTL_CONSTASCII_STRINGPARAM ( MYJOB_SERVICE_NAME ) );
}

Sequence< OUString > SAL_CALL MyJob_getSupportedServiceNames(  )
	throw (RuntimeException)
{
	Sequence < OUString > aRet(1);
    OUString* pArray = aRet.getArray();
    pArray[0] =  OUString ( RTL_CONSTASCII_USTRINGPARAM ( MYJOB_SERVICE_NAME ) );
    return aRet;
}

Reference< XInterface > SAL_CALL MyJob_createInstance( const Reference< XMultiServiceFactory > & rSMgr)
	throw( Exception )
{
	return (cppu::OWeakObject*) new MyJob(rSMgr);
}

// XServiceInfo
OUString SAL_CALL MyJob::getImplementationName()
	throw (RuntimeException)
{
	return MyJob_getImplementationName();
}

sal_Bool SAL_CALL MyJob::supportsService( const OUString& rServiceName )
	throw (RuntimeException)
{
    return MyJob_supportsService( rServiceName );
}

Sequence< OUString > SAL_CALL MyJob::getSupportedServiceNames()
	throw (RuntimeException)
{
    return MyJob_getSupportedServiceNames();
}
//----------- Private functions -------------

//Set all the accelerator configuration sources
void
MyJob::getAcceleratorConfigurations (Reference < XModel >        xModel,
                                     Reference < XModuleManager> xModuleManager)
{
	Reference< XController > xController(xModel->getCurrentController(), UNO_QUERY_THROW);
	Reference< XFrame >      xFrame(xController->getFrame(), UNO_QUERY_THROW);

	//Get document shortcut database
	Reference< XUIConfigurationManagerSupplier > docUISupplier(xModel, UNO_QUERY_THROW);
    Reference< XUIConfigurationManager >         docUIManager = docUISupplier->getUIConfigurationManager();
    Reference< XAcceleratorConfiguration >       docAccelConf(docUIManager->getShortCutManager(), UNO_QUERY_THROW);
   	this->docAccelConf = docAccelConf;

	//Get module shurtcut database
    Reference< XModuleUIConfigurationManagerSupplier > modUISupplier(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.ui.ModuleUIConfigurationManagerSupplier")),
	                                                                 UNO_QUERY_THROW);
	Reference< XUIConfigurationManager >   modUIManager = modUISupplier->getUIConfigurationManager(xModuleManager->identify(xFrame));
	Reference< XAcceleratorConfiguration > modAccelConf(modUIManager->getShortCutManager(), UNO_QUERY_THROW);
	this->modAccelConf = modAccelConf;
	
	//Get global shortcut database
	Reference< XAcceleratorConfiguration > globAccelConf(mxMSF->createInstance(OUString::createFromAscii("com.sun.star.ui.GlobalAcceleratorConfiguration")),
	                                                     UNO_QUERY_THROW);
	this->globAccelConf = globAccelConf;
}

//Find shortcat for a given command
/*Sequence <KeyEvent>
MyJob::findShortcutForCommand (OUString commandURL)
{
	Sequence < KeyEvent > shortcut;
	try
	{
		return docAccelConf->getKeyEventsByCommand (commandURL);
	}
	catch(const NoSuchElementException&)
	{
		try
		{
		    return modAccelConf->getKeyEventsByCommand (commandURL);
		}
		catch(const NoSuchElementException&)
		{
		    try
		    {
		        return globAccelConf->getKeyEventsByCommand (commandURL);
		    }
		    catch(const NoSuchElementException&) {}
		}
	}
	return shortcut;
}*/

//Gets the menu item Label given a CommandURL
//This is a work around for bug: https://bugs.freedesktop.org/show_bug.cgi?id=34127
OUString
MyJob::getLabelFromCommandURL (OUString commandURL)
{
	OUString label;

	Sequence < PropertyValue > commandProps;
	if (!(xUICommands->getByName (commandURL) >>= commandProps))
		return label;
		
		
	for (sal_Int32 i = 0; i < commandProps.getLength(); i++)
	{
		if ( commandProps[i].Name.equalsAsciiL (RTL_CONSTASCII_STRINGPARAM ("Label")))
		{
			commandProps[i].Value >>= label;
			label = label.replace ((sal_Unicode)0x007e, (sal_Unicode)0x005f);
			//break;
		}
	}

	return label;
}

//Gets the XID for a given XFrame
unsigned long
MyJob::getXID (css::uno::Reference < css::frame::XFrame > xFrame)
{
	Reference< XSystemDependentWindowPeer > xWin( xFrame->getContainerWindow(), UNO_QUERY_THROW );
	
	if (!xWin.is())
		return 0;

	sal_Int8 processID[16];
	rtl_getGlobalProcessId( (sal_uInt8*)processID );
	Sequence <signed char> pidSeq (processID, 16);

	SystemDependentXWindow xWindow;
	xWin->getWindowHandle (pidSeq, SYSTEM_XWINDOW) >>= xWindow;
	
	return xWindow.WindowHandle;
}

GList*
MyJob::iterateMenuitems (Reference < XIndexContainer > menuSetCont, gint *nitems, gpointer helper)
{
    GList* items = NULL;
    
	for(int i = 0; i < menuSetCont->getCount(); i++)
	{
		TmpItemDescriptor item;
		Sequence < PropertyValue > menuItem;

		item.commandURL = item.label = OUString();
		item.isVisible = item.type = 0;
    	
		menuSetCont->getByIndex(i) >>= menuItem;
		
		for(int j = 0; j < menuItem.getLength(); j++)
		{
			if (menuItem[j].Name.equalsAscii ("CommandURL"))
				menuItem[j].Value >>= item.commandURL;
				
			else if (menuItem[j].Name.equalsAscii ("Type"))
				menuItem[j].Value >>= item.type;
			
			else if (menuItem[j].Name.equalsAscii ("Label"))
				menuItem[j].Value >>= item.label;
			
			else if (menuItem[j].Name.equalsAscii ("isVisible"))
				menuItem[j].Value >>= item.isVisible;

			else if (menuItem[j].Name.equalsAscii ("ItemDescriptorContainer"))
				menuItem[j].Value >>= item.itemDescriptorContainer;
		}
		
		// Separator
		if (item.type != 0)
		{
			//TODO: Check if previous item is a separator too?
			DbusmenuMenuitem *sep = dbusmenu_menuitem_new_with_id (*nitems);
            dbusmenu_menuitem_property_set(sep, DBUSMENU_MENUITEM_PROP_TYPE, DBUSMENU_CLIENT_TYPES_SEPARATOR);
            (*nitems)++;
            
			items = g_list_append (items, (gpointer)sep);
            
			continue;
		}

		//TODO: Use Label property as soon as this is fixed: https://bugs.freedesktop.org/show_bug.cgi?id=34127
		item.label = getLabelFromCommandURL (item.commandURL);
		
		//TODO: Figure out how to find out wether an item is actually not visisble
		OString label = OUStringToOString(item.label, RTL_TEXTENCODING_ASCII_US);
		OString commandURL = OUStringToOString(item.commandURL, RTL_TEXTENCODING_ASCII_US);
		
		DbusmenuMenuitem *dm_item = dbusmenu_menuitem_new_with_id (*nitems);
		dbusmenu_menuitem_property_set (dm_item, DBUSMENU_MENUITEM_PROP_LABEL, label.getStr());
		dbusmenu_menuitem_property_set (dm_item, "CommandURL", commandURL.getStr ());
		(*nitems)++;

		g_signal_connect(G_OBJECT(dm_item), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(item_activated), helper);
		
		items = g_list_append (items, (gpointer)dm_item);
		
		if (item.itemDescriptorContainer.is ())
		{
			Reference < XIndexContainer > subMenuContainer (item.itemDescriptorContainer, UNO_QUERY_THROW);

			if (subMenuContainer.is ())
			{
				GList *subitems = iterateMenuitems (subMenuContainer, nitems, helper);
				GList *first = subitems;
				while (subitems)
				{
					dbusmenu_menuitem_child_append (dm_item, (DbusmenuMenuitem*)subitems->data);
					subitems = subitems->next;
				}
				g_list_free (first);
			}
		}
	}
	
	return items;
}

DbusmenuMenuitem*
MyJob::getRootMenuitem (Reference < XIndexContainer > menuSetCont, gpointer helper)
{
	gint nitems = 1;
	DbusmenuMenuitem *root = dbusmenu_menuitem_new_with_id (0);
	
	GList *items = iterateMenuitems (menuSetCont, &nitems, helper);
	GList *first = items;
	while (items)
	{
		dbusmenu_menuitem_child_append (root, (DbusmenuMenuitem*)items->data);
		items = items->next;
	}
	g_list_free(first);
	
	((FrameHelper*)helper)->setRootItem(root);
	
	return root;
}
