/*
 * lmms.cpp - LMMS-base-system, contains main() and main-window
 *
 * Copyright (c) 2004-2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "qt3support.h"

#ifdef QT4

#include <QApplication>
#include <QFileDialog>
#include <QCloseEvent>
#include <QSplitter>
#include <QSplashScreen>
#include <QMessageBox>
#include <QMenuBar>

#else

#include <qsplitter.h>
#include <qapplication.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qmenubar.h>

#if QT_VERSION >= 0x030200
#include <qsplashscreen.h>
#endif

#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "lmms_main_win.h"
#include "bb_editor.h"
#include "song_editor.h"
#include "piano_roll.h"
#include "embed.h"
#include "about_dialog.h"
#include "file_browser.h"
#include "plugin_browser.h"
#include "side_bar.h"
#include "config_mgr.h"
#include "mixer.h"
#include "project_notes.h"
#include "buffer_allocator.h"
#include "setup_dialog.h"
#include "preset_preview_play_handle.h"
#include "audio_dummy.h"
#include "tool_button.h"


#if QT_VERSION >= 0x030100
QSplashScreen * lmmsMainWin::s_splashScreen = NULL;
extern int splash_alignment_flags;
#endif

lmmsMainWin * lmmsMainWin::s_instanceOfMe = NULL;



lmmsMainWin::lmmsMainWin() :
	QMainWindow(
#ifndef QT4
			0 , NULL, WDestructiveClose
#endif
		),
	m_workspace( NULL ),
	m_templatesMenu( NULL )
{
#ifdef QT4
	setAttribute( Qt::WA_DeleteOnClose );
#endif

	// small workaround, because inst() is called by objects which
	// are created within this function
	s_instanceOfMe = this;
	bool no_mdi = configManager::inst()->value( "app", "gimplikewindows"
								).toInt();

	QWidget * main_widget = new QWidget( this );
	QVBoxLayout * vbox = new QVBoxLayout( main_widget );
	vbox->setSpacing( 0 );
	vbox->setMargin( 0 );


	QWidget * w = new QWidget( main_widget );
	QHBoxLayout * hbox = new QHBoxLayout( w );
	hbox->setSpacing( 0 );
	hbox->setMargin( 0 );

	sideBar * side_bar = new sideBar( Qt::Vertical, w );
	side_bar->setStyle( sideBar::VSNET/*KDEV3ICON*/ );
	side_bar->setPosition( sideBar::DockLeft );

	QSplitter * splitter = new QSplitter( Qt::Horizontal, w );
#if QT_VERSION >= 0x030200
	splitter->setChildrenCollapsible( FALSE );
#endif

	int id = 0;
	QString wdir = configManager::inst()->lmmsWorkingDir();
	side_bar->appendTab( new pluginBrowser( splitter ), ++id );
	side_bar->appendTab( new fileBrowser( wdir+"projects",
							"*.mmp *.xml *.mid",
							tr( "My projects" ),
					embed::getIconPixmap( "project_file" ),
							splitter ), ++id );
	side_bar->appendTab( new fileBrowser( wdir+"samples", "*.wav *.ogg *.au"
					"*.voc *.aif *.aiff *.flac *.raw",
							tr( "My samples" ),
					embed::getIconPixmap( "sound_file" ),
							splitter ), ++id );
	side_bar->appendTab( new fileBrowser( wdir+"presets", "*.cs.xml",
							tr( "My presets" ),
					embed::getIconPixmap( "preset_file" ),
							splitter ), ++id );
	side_bar->appendTab( new fileBrowser( QDir::homePath(), "*",
							tr( "My home" ),
					embed::getIconPixmap( "home" ),
							splitter ), ++id );
	side_bar->appendTab( new fileBrowser( QDir::rootPath(), "*",
							tr( "Root directory" ),
					embed::getIconPixmap( "root" ),
							splitter ), ++id );

	if( no_mdi == FALSE )
	{
		m_workspace = new QWorkspace( splitter );
		m_workspace->setScrollBarsEnabled( TRUE );

#ifdef QT4
/*		m_workspace->setBackground( embed::getIconPixmap(
						"background_artwork" ) );*/
#else
		m_workspace->setPaletteBackgroundPixmap( embed::getIconPixmap(
						"background_artwork" ) );
#endif
	}

	hbox->addWidget( side_bar );
	hbox->addWidget( splitter );


	// create global-toolbar at the top of our window
	m_toolBar = new QWidget( main_widget );
	m_toolBar->setFixedHeight( 64 );
	m_toolBar->move( 0, 0 );
#ifdef QT4
	QPalette pal;
	pal.setBrush( m_toolBar->backgroundRole(), QBrush(
				embed::getIconPixmap( "main_toolbar_bg" ) ) );
	m_toolBar->setPalette( pal );
#else
	m_toolBar->setPaletteBackgroundPixmap(
				embed::getIconPixmap( "main_toolbar_bg" ) );
#endif

	// add layout for organizing quite complex toolbar-layouting
	m_toolBarLayout = new QGridLayout( m_toolBar/*, 2, 1*/ );
	m_toolBarLayout->setMargin( 0 );
	m_toolBarLayout->setSpacing( 0 );

	vbox->addWidget( m_toolBar );
	vbox->addWidget( w );
	setCentralWidget( main_widget );



#if QT_VERSION >= 0x030200
	if( qApp->argc() > 1 )
	{
		s_splashScreen->showMessage( tr( "Loading song..." ),
							splash_alignment_flags,
								Qt::white );
	}
	else
	{
		s_splashScreen->showMessage( tr( "Creating new song..." ),
							splash_alignment_flags,
								Qt::white );
	}
#endif
	songEditor::inst();


#if QT_VERSION >= 0x030200
	s_splashScreen->showMessage( tr( "Creating GUI..." ),
							splash_alignment_flags,
								Qt::white );
#endif
	resetWindowTitle( "" );
	setWindowIcon( embed::getIconPixmap( "icon" ) );


	// create tool-buttons
	toolButton * project_new = new toolButton(
					embed::getIconPixmap( "project_new" ),
					tr( "Create new project" ),
					this, SLOT( createNewProject() ),
							m_toolBar );

	QDir d( configManager::inst()->projectsDir() + "templates" );
	QStringList templates = d.entryList(
#ifdef QT4
						QStringList( "*.mpt" ),
#else
						"*.mpt",
#endif
						QDir::Files | QDir::Readable );
	if( !templates.isEmpty() )
	{
		m_templatesMenu = new QMenu( project_new );
		for( QStringList::iterator it = templates.begin();
						it != templates.end(); ++it )
		{
			m_templatesMenu->addAction(
					embed::getIconPixmap( "project_file" ),
					( *it ).left( ( *it ).length() - 4 ) );
		}
		connect( m_templatesMenu, SIGNAL( activated( int ) ),
			this, SLOT( createNewProjectFromTemplate( int ) ) );
		project_new->setMenu( m_templatesMenu );
#ifdef QT4
		project_new->setPopupMode( toolButton::MenuButtonPopup );
#else
		project_new->setPopupDelay( 0 );
#endif
	}


	toolButton * project_open = new toolButton( 
					embed::getIconPixmap( "project_open" ),
					tr( "Open existing project" ),
					this, SLOT( openProject() ),
								m_toolBar );


	toolButton * project_save = new toolButton( 
					embed::getIconPixmap( "project_save" ),
					tr( "Save current project" ),
					this, SLOT( saveProject() ),
								m_toolBar );


	toolButton * project_export = new toolButton( 
				embed::getIconPixmap( "project_export" ),
					tr( "Export current project" ),
					songEditor::inst(),
							SLOT( exportProject() ),
								m_toolBar );


	m_toolBarLayout->addWidget( project_new, 0, 0 );
	m_toolBarLayout->addWidget( project_open, 0, 1 );
	m_toolBarLayout->addWidget( project_save, 0, 2 );
	m_toolBarLayout->addWidget( project_export, 0, 3 );



	// window-toolbar
	toolButton * bb_editor_window = new toolButton(
					embed::getIconPixmap( "bb_track" ),
					tr( "Show/hide Beat+Bassline Editor" ) +
									" (F6)",
					this, SLOT( toggleBBEditorWin() ),
								m_toolBar );
	bb_editor_window->setShortcut( Qt::Key_F6 );
#ifdef QT4
	bb_editor_window->setWhatsThis(
#else
	QWhatsThis::add( bb_editor_window, 
#endif
		tr( "By pressing this button, you can show or hide the "
			"Beat+Bassline Editor. The Beat+Bassline Editor is "
			"needed for setting beats, opening, adding and "
			"removing channels, cutting, copying and pasting "
			"beat- and bassline-patterns and other things like "
			"that." ) );


	toolButton * piano_roll_window = new toolButton(
						embed::getIconPixmap( "piano" ),
						tr( "Show/hide Piano-Roll" ) +
									" (F7)",
					this, SLOT( togglePianoRollWin() ),
								m_toolBar );
	piano_roll_window->setShortcut( Qt::Key_F7 );
#ifdef QT4
	piano_roll_window->setWhatsThis(
#else
	QWhatsThis::add( piano_roll_window,
#endif
			tr( "By pressing this button, you can show or hide the "
				"Piano-Roll. With the help of the Piano-Roll "
				"you can edit melody-patterns in an easy way."
				) );

	toolButton * song_editor_window = new toolButton(
					embed::getIconPixmap( "songeditor" ),
					tr( "Show/hide Song-Editor" ) + " (F8)",
					this, SLOT( toggleSongEditorWin() ),
								m_toolBar );
	song_editor_window->setShortcut( Qt::Key_F8 );
#ifdef QT4
	song_editor_window->setWhatsThis(
#else
	QWhatsThis::add( song_editor_window,
#endif
		tr( "By pressing this button, you can show or hide the "
			"Song-Editor. With the help of the Song-Editor you can "
			"edit song-playlist and specify when which track "
			"should be played. "
			"You can also insert and move samples (e.g. "
			"rap-samples) directly into the playlist." ) );


	toolButton * effect_board_window = new toolButton(
					embed::getIconPixmap( "effect_board" ),
					tr( "Show/hide EffectBoard" ) + " (F9)",
					this, SLOT( emptySlot() ), m_toolBar );
	effect_board_window->setShortcut( Qt::Key_F9 );
#ifdef QT4
	effect_board_window->setWhatsThis( 
#else
	QWhatsThis::add( effect_board_window,
#endif
		tr( "By pressing this button, you can show or hide the "
			"EffectBoard. The EffectBoard is a very powerful tool "
			"for managing effects for your song. You can insert "
			"effects into different effect-channels." ) );

	toolButton * project_notes_window = new toolButton(
					embed::getIconPixmap( "project_notes" ),
					tr( "Show/hide project notes" ) +
								" (F10)",
					this, SLOT( toggleProjectNotesWin() ),
								m_toolBar );
	project_notes_window->setShortcut( Qt::Key_F10 );
#ifdef QT4
	project_notes_window->setWhatsThis(
#else
	QWhatsThis::add( project_notes_window,
#endif
		tr( "By pressing this button, you can show or hide the "
			"project notes window. In this window you can put "
			"down your project notes.") );

	m_toolBarLayout->addWidget( bb_editor_window, 1, 0 );
	m_toolBarLayout->addWidget( piano_roll_window, 1, 1 );
	m_toolBarLayout->addWidget( song_editor_window, 1, 2 );
	m_toolBarLayout->addWidget( effect_board_window, 1, 3 );
	m_toolBarLayout->addWidget( project_notes_window, 1, 4 );

	m_toolBarLayout->setColumnStretch( 100, 1 );



	// project-popup-menu
	QMenu * project_menu = new QMenu( this );
#ifdef QT4
	menuBar()->addMenu( project_menu )->setText( tr( "&Project" ) );
#else
	menuBar()->insertItem( tr( "&Project" ), project_menu );
#endif
	project_menu->addAction( embed::getIconPixmap( "project_new" ),
					tr( "&New" ),
					this, SLOT( createNewProject() ),
					Qt::CTRL + Qt::Key_N );

	project_menu->addAction( embed::getIconPixmap( "project_open" ),
					tr( "&Open..." ),
					this, SLOT( openProject() ),
					Qt::CTRL + Qt::Key_O );	

	project_menu->addAction( embed::getIconPixmap( "project_save" ),
					tr( "&Save" ),
					this, SLOT( saveProject() ),
					Qt::CTRL + Qt::Key_S );

	project_menu->addAction( embed::getIconPixmap( "project_saveas" ),
					tr( "Save &As..." ),
					this, SLOT( saveProjectAs() ),
					Qt::CTRL + Qt::SHIFT + Qt::Key_S );
#ifdef QT4
	project_menu->addSeparator();
#else
	project_menu->insertSeparator();
#endif
	project_menu->addAction( /*embed::getIconPixmap( "project_import" ),*/
					tr( "Import file" ),
					songEditor::inst(),
					SLOT( importProject() ) );
#ifdef QT4
	project_menu->addSeparator();
#else
	project_menu->insertSeparator();
#endif
	project_menu->addAction( embed::getIconPixmap( "project_export" ),
					tr( "E&xport" ),
					songEditor::inst(),
					SLOT( exportProject() ),
					Qt::CTRL + Qt::Key_E );
#ifdef QT4
	project_menu->addSeparator();
#else
	project_menu->insertSeparator();
#endif
	project_menu->addAction( embed::getIconPixmap( "exit" ), tr( "&Quit" ),
					qApp, SLOT( closeAllWindows() ),
					Qt::CTRL+Qt::Key_Q );


	QMenu * settings_menu = new QMenu( this );
#ifdef QT4
	menuBar()->addMenu( settings_menu )->setText( tr( "&Settings" ) );
#else
	menuBar()->insertItem( tr( "&Settings" ), settings_menu );
#endif
	settings_menu->addAction( embed::getIconPixmap( "setup_general" ),
					tr( "Show settings dialog" ),
					this, SLOT( showSettingsDialog() ) );
	settings_menu->addAction( embed::getIconPixmap( "wizard" ),
					tr( "Show setup wizard" ),
					configManager::inst(), SLOT( exec() ) );

	
	// help-popup-menu
	QMenu * help_menu = new QMenu( this );
#ifdef QT4
	menuBar()->addMenu( help_menu )->setText( tr( "&Help" ) );
#else
	menuBar()->insertItem( tr( "&Help" ), help_menu );
#endif
	help_menu->addAction( embed::getIconPixmap( "help" ), tr( "Help" ),
							this, SLOT( help() ) );
	help_menu->addAction( embed::getIconPixmap( "whatsthis" ),
					tr( "What's this?" ),
					this, SLOT( enterWhatsThisMode() ) );
#ifdef QT4
	help_menu->addSeparator();
#else
	help_menu->insertSeparator();
#endif
	help_menu->addAction( embed::getIconPixmap( "icon" ), tr( "About" ),
						this, SLOT( aboutLMMS() ) );

	// setup-dialog opened before?
	if( !configManager::inst()->value( "app", "configured" ).toInt() )
	{
		// no, so show it that user can setup everything
/*		setupDialog().exec();
		configManager::inst()->setValue( "app", "configured", "1" );*/
	}
	// look whether mixer could use a audio-interface beside audioDummy
	else if( mixer::inst()->audioDevName() == audioDummy::name() )
	{
		// no, so we offer setup-dialog with audio-settings...
		setupDialog sd( setupDialog::AUDIO_SETTINGS );
		sd.exec();
	}
}




lmmsMainWin::~lmmsMainWin()
{
	// first make sure, there're no mixing/audio-device-threads any more
	mixer::inst()->stopProcessing();

	// destroy editors with all their children
	delete songEditor::inst();
	delete bbEditor::inst();

	presetPreviewPlayHandle::cleanUp();


	// destroy mixer
	delete mixer::inst();


	// now we can clean up all allocated buffer
	bufferAllocator::cleanUp( 0 );

	// destroy config-manager (which automatically saves config-file)
	delete configManager::inst();

}




int lmmsMainWin::addWidgetToToolBar( QWidget * _w, int _row, int _col )
{
	int col = ( _col == -1 ) ? m_toolBarLayout->columnCount() + 6 : _col;
	if( _w->height() > 32 || _row == -1 )
	{
#ifdef QT4
		m_toolBarLayout->addWidget( _w, 0, col, 2, 1 );
#else
		m_toolBarLayout->addMultiCellWidget( _w, 0, 1, col, col );
#endif
	}
	else
	{
		m_toolBarLayout->addWidget( _w, _row, col );
	}
	return( col );
}




void lmmsMainWin::addSpacingToToolBar( int _size )
{
	m_toolBarLayout->setColumnMinimumWidth( m_toolBarLayout->columnCount() +
								6, _size );
}




void lmmsMainWin::resetWindowTitle( const QString & _add )
{
	QString title = _add;
	if( _add == "" && songEditor::inst()->projectFileName() != "" )
	{
		title = QFileInfo( songEditor::inst()->projectFileName()
#ifdef QT4
						).completeBaseName();
#else
						).baseName( TRUE );
#endif
	}
	if( title != "" )
	{
		title += " - ";
	}
	setWindowTitle( title + tr( "LMMS %1" ).arg( VERSION ) );
}




void lmmsMainWin::keyPressEvent( QKeyEvent * _ke )
{
	switch( _ke->key() )
	{
		case Qt::Key_Control: m_keyMods.m_ctrl = TRUE; break;
		case Qt::Key_Shift: m_keyMods.m_shift = TRUE; break;
		case Qt::Key_Alt: m_keyMods.m_alt = TRUE; break;
		default:
			QMainWindow::keyPressEvent( _ke );
	}
}




void lmmsMainWin::keyReleaseEvent( QKeyEvent * _ke )
{
	switch( _ke->key() )
	{
		case Qt::Key_Control: m_keyMods.m_ctrl = FALSE; break;
		case Qt::Key_Shift: m_keyMods.m_shift = FALSE; break;
		case Qt::Key_Alt: m_keyMods.m_alt = FALSE; break;
		default:
			QMainWindow::keyReleaseEvent( _ke );
	}
}




void lmmsMainWin::closeEvent( QCloseEvent * _ce )
{
	if( songEditor::inst()->mayChangeProject() == TRUE )
	{
		_ce->accept();
	}
	else
	{
		_ce->ignore();
	}
}




void lmmsMainWin::createNewProject( void )
{
	if( songEditor::inst()->mayChangeProject() == TRUE )
	{
		songEditor::inst()->createNewProject();
	}
}




void lmmsMainWin::createNewProjectFromTemplate( int _idx )
{
#ifdef QT4
	// TODO!!!
#else
	if( m_templatesMenu != NULL &&
				songEditor::inst()->mayChangeProject() == TRUE )
	{
		songEditor::inst()->createNewProjectFromTemplate(
			configManager::inst()->projectsDir() + "templates/" +
				m_templatesMenu->text( _idx ) + ".mpt" );
	}
#endif
}




void lmmsMainWin::openProject( void )
{
	if( songEditor::inst()->mayChangeProject() == TRUE )
	{
#ifdef QT4
		QFileDialog ofd( this, tr( "Open project" ), "",
				tr( "MultiMedia Project (*.mmp *.xml)" ) );
#else
		QFileDialog ofd( QString::null,
				tr( "MultiMedia Project (*.mmp *.xml)" ),
							this, "", TRUE );
		ofd.setWindowTitle( tr( "Open project" ) );
#endif
		ofd.setDirectory( configManager::inst()->projectsDir() );
		ofd.setFileMode( QFileDialog::ExistingFiles );
		if( ofd.exec () == QDialog::Accepted &&
						!ofd.selectedFiles().isEmpty() )
		{
			songEditor::inst()->loadProject(
						ofd.selectedFiles()[0] );
		}
	}
}




bool lmmsMainWin::saveProject( void )
{
	if( songEditor::inst()->projectFileName() == "" )
	{
		return( saveProjectAs() );
	}
	else
	{
		songEditor::inst()->saveProject();
	}
	return( TRUE );
}




bool lmmsMainWin::saveProjectAs( void )
{
#ifdef QT4
	QFileDialog sfd( this, tr( "Save project" ), "",
				tr( "MultiMedia Project (*.mmp);;"
				"MultiMedia Project Template (*.mpt)" ) );
#else
	QFileDialog sfd( QString::null,
				tr( "MultiMedia Project (*.mmp);;"
					"MultiMedia Project Template (*.mpt)" ),
							this, "", TRUE );
	sfd.setWindowTitle( tr( "Save project" ) );
#endif
	sfd.setFileMode( QFileDialog::AnyFile );
	QString f = songEditor::inst()->projectFileName();
	if( f != "" )
	{
		sfd.selectFile( QFileInfo( f ).fileName() );
#ifdef QT4
		sfd.setDirectory( QFileInfo( f ).absolutePath() );
#else
		sfd.setDirectory( QFileInfo( f ).dirPath( TRUE ) );
#endif
	}
	else
	{
		sfd.setDirectory( configManager::inst()->projectsDir() );
	}

	if( sfd.exec () == QFileDialog::Accepted &&
#ifdef QT4
		!sfd.selectedFiles().isEmpty() && sfd.selectedFiles()[0] != ""
#else
		sfd.selectedFile() != ""
#endif
 		)
	{
#ifdef QT4
		songEditor::inst()->saveProjectAs( sfd.selectedFiles()[0] );
#else
		songEditor::inst()->saveProjectAs( sfd.selectedFile() );
#endif
		return( TRUE );
	}
	return( FALSE );
}




void lmmsMainWin::toggleBBEditorWin( void )
{
	if( bbEditor::inst()->isHidden() == TRUE ||
		( m_workspace != NULL &&
		  	m_workspace->activeWindow() != bbEditor::inst() ) )
	{
		bbEditor::inst()->show();
		bbEditor::inst()->setFocus();
	}
	else
	{
		bbEditor::inst()->hide();
	}
}




void lmmsMainWin::toggleSongEditorWin( void )
{
	if( songEditor::inst()->isHidden() == TRUE ||
		( m_workspace != NULL &&
			m_workspace->activeWindow() != songEditor::inst() ) )
	{
		songEditor::inst()->show();
		songEditor::inst()->setFocus();
	}
	else
	{
		songEditor::inst()->hide();
	}
}




void lmmsMainWin::toggleProjectNotesWin( void )
{
	if( songEditor::inst()->getProjectNotesWindow()->isHidden() == TRUE ||
		( m_workspace != NULL && m_workspace->activeWindow() !=
				songEditor::inst()->getProjectNotesWindow() ) )
	{
		songEditor::inst()->getProjectNotesWindow()->show();
		songEditor::inst()->getProjectNotesWindow()->setFocus();
	}
	else
	{
		songEditor::inst()->getProjectNotesWindow()->hide();
	}
}




void lmmsMainWin::togglePianoRollWin( void )
{
	if( pianoRoll::inst()->isHidden() == TRUE ||
		( m_workspace != NULL &&
			m_workspace->activeWindow() != pianoRoll::inst() ) )
	{
		pianoRoll::inst()->show();
		pianoRoll::inst()->setFocus();
	}
	else
	{
		pianoRoll::inst()->hide();
	}
}




void lmmsMainWin::showSettingsDialog( void )
{
	setupDialog().exec();
}




void lmmsMainWin::aboutLMMS( void )
{
	aboutDialog().exec();
}




void lmmsMainWin::help( void )
{
	QMessageBox::information( this, tr( "Help not available" ),
					tr( "Currently there's no help "
						"available in LMMS.\n"
						"Please visit "
						"http://wiki.mindrules.net "
						"for documentation on LMMS." ),
					QMessageBox::Ok );
}



#include "lmms_main_win.moc"

