//===-- aeabi_uldivmod_test.c - Test aeabi_uldivmod -----------------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is dual licensed under the MIT and the University of Illinois Open
// Source Licenses. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This file tests aeabi_uldivmod for the compiler_rt library.
//
//===----------------------------------------------------------------------===//

#include "int_lib.h"
#include <stdio.h>

#if __arm__
// Based on udivmoddi4_test.c

COMPILER_RT_ABI void /* __value_in_regs */ __aeabi_uldivmod(du_int a, du_int b);

int test_aeabi_uldivmod(du_int a, du_int b, du_int expected_q, du_int expected_r)
{
    du_int q, r;
    __asm__(
        "movs r0, %Q[a] \n"
        "movs r1, %R[a] \n"
        "movs r2, %Q[b] \n"
        "movs r3, %R[b] \n"
        "bl __aeabi_uldivmod \n"
        "movs %Q[q], r0\n"
        "movs %R[q], r1\n"
        "movs %Q[r], r2\n"
        "movs %R[r], r3\n"
        : [q] "=r" (q), [r] "=r"(r)
        : [a] "r"(a), [b] "r"(b)
        );
    if (q != expected_q || r != expected_r)
        printf("error in aeabi_uldivmod: %llX / %llX = %llX, R = %llX, expected %llX, %llX\n",
               a, b, q, r, expected_q, expected_r);
    return !(q == expected_q && r == expected_r);
}

char assumption_1[sizeof(di_int) == 2*sizeof(si_int)] = {0};

du_int tests[][4] =
{
{0x0000000000000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000000uLL},
{0x0000000000000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000000000001uLL, 0x0000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000001uLL},
{0x0000000000000002uLL, 0x0000000000000001uLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x0000000000000002uLL, 0x0000000000000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000000000002uLL, 0x0000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000002uLL},
{0x0000000000000003uLL, 0x0000000000000001uLL, 0x0000000000000003uLL, 0x0000000000000000uLL},
{0x0000000000000003uLL, 0x0000000000000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000000000003uLL, 0x0000000000000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000000000003uLL, 0x0000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000003uLL},
{0x0000000000000010uLL, 0x0000000000000001uLL, 0x0000000000000010uLL, 0x0000000000000000uLL},
{0x0000000000000010uLL, 0x0000000000000002uLL, 0x0000000000000008uLL, 0x0000000000000000uLL},
{0x0000000000000010uLL, 0x0000000000000003uLL, 0x0000000000000005uLL, 0x0000000000000001uLL},
{0x0000000000000010uLL, 0x0000000000000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000000000010uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x0000000000000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000000000010uLL},
{0x00000000078644FAuLL, 0x0000000000000001uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL},
{0x00000000078644FAuLL, 0x0000000000000002uLL, 0x0000000003C3227DuLL, 0x0000000000000000uLL},
{0x00000000078644FAuLL, 0x0000000000000003uLL, 0x00000000028216FEuLL, 0x0000000000000000uLL},
{0x00000000078644FAuLL, 0x0000000000000010uLL, 0x000000000078644FuLL, 0x000000000000000AuLL},
{0x00000000078644FAuLL, 0x00000000078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000000078644FAuLL, 0x000000000747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x00000000078644FAuLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x00000000078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000078644FAuLL},
{0x000000000747AE14uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL},
{0x000000000747AE14uLL, 0x0000000000000002uLL, 0x0000000003A3D70AuLL, 0x0000000000000000uLL},
{0x000000000747AE14uLL, 0x0000000000000003uLL, 0x00000000026D3A06uLL, 0x0000000000000002uLL},
{0x000000000747AE14uLL, 0x0000000000000010uLL, 0x0000000000747AE1uLL, 0x0000000000000004uLL},
{0x000000000747AE14uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000000747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000000747AE14uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000000747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000000747AE14uLL},
{0x000000007FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL},
{0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x000000003FFFFFFFuLL, 0x0000000000000001uLL},
{0x000000007FFFFFFFuLL, 0x0000000000000003uLL, 0x000000002AAAAAAAuLL, 0x0000000000000001uLL},
{0x000000007FFFFFFFuLL, 0x0000000000000010uLL, 0x0000000007FFFFFFuLL, 0x000000000000000FuLL},
{0x000000007FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000011uLL, 0x0000000000156B65uLL},
{0x000000007FFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000011uLL, 0x00000000043D70ABuLL},
{0x000000007FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000007FFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x000000007FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000007FFFFFFFuLL},
{0x0000000080000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL, 0x0000000000000000uLL},
{0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000040000000uLL, 0x0000000000000000uLL},
{0x0000000080000000uLL, 0x0000000000000003uLL, 0x000000002AAAAAAAuLL, 0x0000000000000002uLL},
{0x0000000080000000uLL, 0x0000000000000010uLL, 0x0000000008000000uLL, 0x0000000000000000uLL},
{0x0000000080000000uLL, 0x00000000078644FAuLL, 0x0000000000000011uLL, 0x0000000000156B66uLL},
{0x0000000080000000uLL, 0x000000000747AE14uLL, 0x0000000000000011uLL, 0x00000000043D70ACuLL},
{0x0000000080000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000080000000uLL, 0x0000000080000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000080000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x0000000080000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000080000000uLL},
{0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x000000007FFFFFFEuLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000055555554uLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000000FFFFFFFuLL, 0x000000000000000DuLL},
{0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6C9uLL},
{0x00000000FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333341uLL},
{0x00000000FFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000000FFFFFFFDuLL, 0x0000000080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFDuLL},
{0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000055555554uLL, 0x0000000000000002uLL},
{0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000000FFFFFFFuLL, 0x000000000000000EuLL},
{0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CAuLL},
{0x00000000FFFFFFFEuLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333342uLL},
{0x00000000FFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFEuLL, 0x0000000080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFEuLL},
{0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000055555555uLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000000FFFFFFFuLL, 0x000000000000000FuLL},
{0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CBuLL},
{0x00000000FFFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333343uLL},
{0x00000000FFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x00000000FFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000000FFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000000FFFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x00000000FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000000FFFFFFFFuLL},
{0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000080000000uLL, 0x0000000000000000uLL},
{0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000055555555uLL, 0x0000000000000001uLL},
{0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000010000000uLL, 0x0000000000000000uLL},
{0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CCuLL},
{0x0000000100000000uLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333344uLL},
{0x0000000100000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000100000000uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x0000000100000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000100000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000100000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000100000000uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000100000000uLL, 0x0000000100000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000000uLL},
{0x0000000100000001uLL, 0x0000000000000001uLL, 0x0000000100000001uLL, 0x0000000000000000uLL},
{0x0000000100000001uLL, 0x0000000000000002uLL, 0x0000000080000000uLL, 0x0000000000000001uLL},
{0x0000000100000001uLL, 0x0000000000000003uLL, 0x0000000055555555uLL, 0x0000000000000002uLL},
{0x0000000100000001uLL, 0x0000000000000010uLL, 0x0000000010000000uLL, 0x0000000000000001uLL},
{0x0000000100000001uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CDuLL},
{0x0000000100000001uLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333345uLL},
{0x0000000100000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000100000001uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x0000000100000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000100000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000100000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000100000001uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000100000001uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000100000001uLL, 0x0000000100000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000001uLL},
{0x0000000100000002uLL, 0x0000000000000001uLL, 0x0000000100000002uLL, 0x0000000000000000uLL},
{0x0000000100000002uLL, 0x0000000000000002uLL, 0x0000000080000001uLL, 0x0000000000000000uLL},
{0x0000000100000002uLL, 0x0000000000000003uLL, 0x0000000055555556uLL, 0x0000000000000000uLL},
{0x0000000100000002uLL, 0x0000000000000010uLL, 0x0000000010000000uLL, 0x0000000000000002uLL},
{0x0000000100000002uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CEuLL},
{0x0000000100000002uLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333346uLL},
{0x0000000100000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x0000000100000002uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000100000002uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000100000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000100000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000100000002uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000100000002uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000100000002uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000100000002uLL, 0x0000000100000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000002uLL},
{0x0000000100000003uLL, 0x0000000000000001uLL, 0x0000000100000003uLL, 0x0000000000000000uLL},
{0x0000000100000003uLL, 0x0000000000000002uLL, 0x0000000080000001uLL, 0x0000000000000001uLL},
{0x0000000100000003uLL, 0x0000000000000003uLL, 0x0000000055555556uLL, 0x0000000000000001uLL},
{0x0000000100000003uLL, 0x0000000000000010uLL, 0x0000000010000000uLL, 0x0000000000000003uLL},
{0x0000000100000003uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6CFuLL},
{0x0000000100000003uLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333347uLL},
{0x0000000100000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x0000000100000003uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000100000003uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000006uLL},
{0x0000000100000003uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000100000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000100000003uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000100000003uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000100000003uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000100000003uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000100000003uLL, 0x0000000100000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000003uLL},
{0x0000000100000010uLL, 0x0000000000000001uLL, 0x0000000100000010uLL, 0x0000000000000000uLL},
{0x0000000100000010uLL, 0x0000000000000002uLL, 0x0000000080000008uLL, 0x0000000000000000uLL},
{0x0000000100000010uLL, 0x0000000000000003uLL, 0x000000005555555AuLL, 0x0000000000000002uLL},
{0x0000000100000010uLL, 0x0000000000000010uLL, 0x0000000010000001uLL, 0x0000000000000000uLL},
{0x0000000100000010uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000002AD6DCuLL},
{0x0000000100000010uLL, 0x000000000747AE14uLL, 0x0000000000000023uLL, 0x0000000001333354uLL},
{0x0000000100000010uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000012uLL},
{0x0000000100000010uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x0000000000000010uLL},
{0x0000000100000010uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000013uLL},
{0x0000000100000010uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000012uLL},
{0x0000000100000010uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000011uLL},
{0x0000000100000010uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000000000010uLL},
{0x0000000100000010uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x000000000000000FuLL},
{0x0000000100000010uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x000000000000000EuLL},
{0x0000000100000010uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x000000000000000DuLL},
{0x0000000100000010uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000100000010uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x0000000100000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000100000010uLL},
{0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL},
{0x00000001078644FAuLL, 0x0000000000000002uLL, 0x0000000083C3227DuLL, 0x0000000000000000uLL},
{0x00000001078644FAuLL, 0x0000000000000003uLL, 0x0000000057D76C53uLL, 0x0000000000000001uLL},
{0x00000001078644FAuLL, 0x0000000000000010uLL, 0x000000001078644FuLL, 0x000000000000000AuLL},
{0x00000001078644FAuLL, 0x00000000078644FAuLL, 0x0000000000000023uLL, 0x00000000002AD6CCuLL},
{0x00000001078644FAuLL, 0x000000000747AE14uLL, 0x0000000000000024uLL, 0x000000000171CA2AuLL},
{0x00000001078644FAuLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x00000000078644FCuLL},
{0x00000001078644FAuLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x00000000078644FAuLL},
{0x00000001078644FAuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000000078644FDuLL},
{0x00000001078644FAuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000078644FCuLL},
{0x00000001078644FAuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000078644FBuLL},
{0x00000001078644FAuLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x00000000078644FAuLL},
{0x00000001078644FAuLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x00000000078644F9uLL},
{0x00000001078644FAuLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000078644F8uLL},
{0x00000001078644FAuLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000078644F7uLL},
{0x00000001078644FAuLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000078644EAuLL},
{0x00000001078644FAuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000001078644FAuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x00000001078644FAuLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x00000001078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001078644FAuLL},
{0x000000010747AE14uLL, 0x0000000000000001uLL, 0x000000010747AE14uLL, 0x0000000000000000uLL},
{0x000000010747AE14uLL, 0x0000000000000002uLL, 0x0000000083A3D70AuLL, 0x0000000000000000uLL},
{0x000000010747AE14uLL, 0x0000000000000003uLL, 0x0000000057C28F5CuLL, 0x0000000000000000uLL},
{0x000000010747AE14uLL, 0x0000000000000010uLL, 0x0000000010747AE1uLL, 0x0000000000000004uLL},
{0x000000010747AE14uLL, 0x00000000078644FAuLL, 0x0000000000000022uLL, 0x00000000077284E0uLL},
{0x000000010747AE14uLL, 0x000000000747AE14uLL, 0x0000000000000024uLL, 0x0000000001333344uLL},
{0x000000010747AE14uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL, 0x000000000747AE16uLL},
{0x000000010747AE14uLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x000000000747AE14uLL},
{0x000000010747AE14uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x000000000747AE17uLL},
{0x000000010747AE14uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x000000000747AE16uLL},
{0x000000010747AE14uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x000000000747AE15uLL},
{0x000000010747AE14uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL},
{0x000000010747AE14uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x000000000747AE13uLL},
{0x000000010747AE14uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x000000000747AE12uLL},
{0x000000010747AE14uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x000000000747AE11uLL},
{0x000000010747AE14uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x000000000747AE04uLL},
{0x000000010747AE14uLL, 0x00000001078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000010747AE14uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000010747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000010747AE14uLL},
{0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL},
{0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x00000000BFFFFFFFuLL, 0x0000000000000001uLL},
{0x000000017FFFFFFFuLL, 0x0000000000000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000002uLL},
{0x000000017FFFFFFFuLL, 0x0000000000000010uLL, 0x0000000017FFFFFFuLL, 0x000000000000000FuLL},
{0x000000017FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000033uLL, 0x0000000000404231uLL},
{0x000000017FFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000034uLL, 0x000000000570A3EFuLL},
{0x000000017FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000002uLL},
{0x000000017FFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000002uLL, 0x000000007FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x000000017FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x000000017FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x000000017FFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x000000017FFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x000000017FFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFCuLL},
{0x000000017FFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFEFuLL},
{0x000000017FFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB05uLL},
{0x000000017FFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851EBuLL},
{0x000000017FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000017FFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x000000017FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000017FFFFFFFuLL},
{0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000180000000uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x0000000000000002uLL, 0x00000000C0000000uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x0000000000000003uLL, 0x0000000080000000uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x0000000000000010uLL, 0x0000000018000000uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x00000000078644FAuLL, 0x0000000000000033uLL, 0x0000000000404232uLL},
{0x0000000180000000uLL, 0x000000000747AE14uLL, 0x0000000000000034uLL, 0x000000000570A3F0uLL},
{0x0000000180000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000003uLL},
{0x0000000180000000uLL, 0x0000000080000000uLL, 0x0000000000000003uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000180000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000180000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000180000000uLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000000180000000uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x0000000180000000uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x0000000180000000uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x0000000180000000uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFF0uLL},
{0x0000000180000000uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB06uLL},
{0x0000000180000000uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851ECuLL},
{0x0000000180000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000180000000uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000180000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x0000000180000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000180000000uLL},
{0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFDuLL, 0x0000000000000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFDuLL, 0x0000000000000003uLL, 0x00000000AAAAAAA9uLL, 0x0000000000000002uLL},
{0x00000001FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000001FFFFFFFuLL, 0x000000000000000DuLL},
{0x00000001FFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD95uLL},
{0x00000001FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666685uLL},
{0x00000001FFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFDuLL, 0x0000000080000000uLL, 0x0000000000000003uLL, 0x000000007FFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x00000001FFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000001FFFFFFFDuLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000001FFFFFFFDuLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000001FFFFFFFDuLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFAuLL},
{0x00000001FFFFFFFDuLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEDuLL},
{0x00000001FFFFFFFDuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB03uLL},
{0x00000001FFFFFFFDuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851E9uLL},
{0x00000001FFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000001FFFFFFFDuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFDuLL},
{0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFEuLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFEuLL, 0x0000000000000003uLL, 0x00000000AAAAAAAAuLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000001FFFFFFFuLL, 0x000000000000000EuLL},
{0x00000001FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD96uLL},
{0x00000001FFFFFFFEuLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666686uLL},
{0x00000001FFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000002uLL},
{0x00000001FFFFFFFEuLL, 0x0000000080000000uLL, 0x0000000000000003uLL, 0x000000007FFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x00000001FFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x00000001FFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFEuLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000001FFFFFFFEuLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000001FFFFFFFEuLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000001FFFFFFFEuLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEEuLL},
{0x00000001FFFFFFFEuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB04uLL},
{0x00000001FFFFFFFEuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EAuLL},
{0x00000001FFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000001FFFFFFFEuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFEuLL},
{0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFFuLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFFuLL, 0x0000000000000003uLL, 0x00000000AAAAAAAAuLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000001FFFFFFFuLL, 0x000000000000000FuLL},
{0x00000001FFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD97uLL},
{0x00000001FFFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666687uLL},
{0x00000001FFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000003uLL},
{0x00000001FFFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000003uLL, 0x000000007FFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x00000001FFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x00000001FFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000001FFFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000001FFFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000001FFFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEFuLL},
{0x00000001FFFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB05uLL},
{0x00000001FFFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EBuLL},
{0x00000001FFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x00000001FFFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x00000001FFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000001FFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000001FFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x00000001FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000001FFFFFFFFuLL},
{0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000200000000uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x0000000000000002uLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x0000000000000003uLL, 0x00000000AAAAAAAAuLL, 0x0000000000000002uLL},
{0x0000000200000000uLL, 0x0000000000000010uLL, 0x0000000020000000uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD98uLL},
{0x0000000200000000uLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666688uLL},
{0x0000000200000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000004uLL},
{0x0000000200000000uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000006uLL},
{0x0000000200000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x0000000200000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000200000000uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000200000000uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x0000000200000000uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x0000000200000000uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF0uLL},
{0x0000000200000000uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB06uLL},
{0x0000000200000000uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851ECuLL},
{0x0000000200000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000200000000uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000000200000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000200000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000200000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000200000000uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000200000000uLL, 0x0000000200000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000000uLL},
{0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000200000001uLL, 0x0000000000000000uLL},
{0x0000000200000001uLL, 0x0000000000000002uLL, 0x0000000100000000uLL, 0x0000000000000001uLL},
{0x0000000200000001uLL, 0x0000000000000003uLL, 0x00000000AAAAAAABuLL, 0x0000000000000000uLL},
{0x0000000200000001uLL, 0x0000000000000010uLL, 0x0000000020000000uLL, 0x0000000000000001uLL},
{0x0000000200000001uLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD99uLL},
{0x0000000200000001uLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666689uLL},
{0x0000000200000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000005uLL},
{0x0000000200000001uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x0000000000000001uLL},
{0x0000000200000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000007uLL},
{0x0000000200000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x0000000200000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000200000001uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x0000000200000001uLL, 0x0000000100000001uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000200000001uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000200000001uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x0000000200000001uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF1uLL},
{0x0000000200000001uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB07uLL},
{0x0000000200000001uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EDuLL},
{0x0000000200000001uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000200000001uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000200000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000200000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000200000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000200000001uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000200000001uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000200000001uLL, 0x0000000200000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000001uLL},
{0x0000000200000002uLL, 0x0000000000000001uLL, 0x0000000200000002uLL, 0x0000000000000000uLL},
{0x0000000200000002uLL, 0x0000000000000002uLL, 0x0000000100000001uLL, 0x0000000000000000uLL},
{0x0000000200000002uLL, 0x0000000000000003uLL, 0x00000000AAAAAAABuLL, 0x0000000000000001uLL},
{0x0000000200000002uLL, 0x0000000000000010uLL, 0x0000000020000000uLL, 0x0000000000000002uLL},
{0x0000000200000002uLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD9AuLL},
{0x0000000200000002uLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x000000000266668AuLL},
{0x0000000200000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000006uLL},
{0x0000000200000002uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x0000000000000002uLL},
{0x0000000200000002uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000008uLL},
{0x0000000200000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000006uLL},
{0x0000000200000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x0000000200000002uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000200000002uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x0000000200000002uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000200000002uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000200000002uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF2uLL},
{0x0000000200000002uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB08uLL},
{0x0000000200000002uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EEuLL},
{0x0000000200000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000200000002uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000200000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000200000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000200000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000200000002uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000200000002uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000200000002uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000200000002uLL, 0x0000000200000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000002uLL},
{0x0000000200000003uLL, 0x0000000000000001uLL, 0x0000000200000003uLL, 0x0000000000000000uLL},
{0x0000000200000003uLL, 0x0000000000000002uLL, 0x0000000100000001uLL, 0x0000000000000001uLL},
{0x0000000200000003uLL, 0x0000000000000003uLL, 0x00000000AAAAAAABuLL, 0x0000000000000002uLL},
{0x0000000200000003uLL, 0x0000000000000010uLL, 0x0000000020000000uLL, 0x0000000000000003uLL},
{0x0000000200000003uLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055AD9BuLL},
{0x0000000200000003uLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x000000000266668BuLL},
{0x0000000200000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000007uLL},
{0x0000000200000003uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x0000000000000003uLL},
{0x0000000200000003uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000009uLL},
{0x0000000200000003uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000007uLL},
{0x0000000200000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x0000000200000003uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000200000003uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x0000000200000003uLL, 0x0000000100000002uLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000200000003uLL, 0x0000000100000003uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000200000003uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF3uLL},
{0x0000000200000003uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB09uLL},
{0x0000000200000003uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EFuLL},
{0x0000000200000003uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000004uLL},
{0x0000000200000003uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000200000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000006uLL},
{0x0000000200000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000200000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000200000003uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000200000003uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000200000003uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000200000003uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000200000003uLL, 0x0000000200000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000003uLL},
{0x0000000200000010uLL, 0x0000000000000001uLL, 0x0000000200000010uLL, 0x0000000000000000uLL},
{0x0000000200000010uLL, 0x0000000000000002uLL, 0x0000000100000008uLL, 0x0000000000000000uLL},
{0x0000000200000010uLL, 0x0000000000000003uLL, 0x00000000AAAAAAB0uLL, 0x0000000000000000uLL},
{0x0000000200000010uLL, 0x0000000000000010uLL, 0x0000000020000001uLL, 0x0000000000000000uLL},
{0x0000000200000010uLL, 0x00000000078644FAuLL, 0x0000000000000044uLL, 0x000000000055ADA8uLL},
{0x0000000200000010uLL, 0x000000000747AE14uLL, 0x0000000000000046uLL, 0x0000000002666698uLL},
{0x0000000200000010uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000014uLL},
{0x0000000200000010uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x0000000000000010uLL},
{0x0000000200000010uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000016uLL},
{0x0000000200000010uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000014uLL},
{0x0000000200000010uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000012uLL},
{0x0000000200000010uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000000000010uLL},
{0x0000000200000010uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x000000000000000EuLL},
{0x0000000200000010uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x000000000000000CuLL},
{0x0000000200000010uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x000000000000000AuLL},
{0x0000000200000010uLL, 0x0000000100000010uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000200000010uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB16uLL},
{0x0000000200000010uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851FCuLL},
{0x0000000200000010uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000011uLL},
{0x0000000200000010uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000080000010uLL},
{0x0000000200000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000013uLL},
{0x0000000200000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000012uLL},
{0x0000000200000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000011uLL},
{0x0000000200000010uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000000000010uLL},
{0x0000000200000010uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000000000000FuLL},
{0x0000000200000010uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000000000000EuLL},
{0x0000000200000010uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000000000000DuLL},
{0x0000000200000010uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000200000010uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x0000000200000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000200000010uLL},
{0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL},
{0x00000002078644FAuLL, 0x0000000000000002uLL, 0x0000000103C3227DuLL, 0x0000000000000000uLL},
{0x00000002078644FAuLL, 0x0000000000000003uLL, 0x00000000AD2CC1A8uLL, 0x0000000000000002uLL},
{0x00000002078644FAuLL, 0x0000000000000010uLL, 0x000000002078644FuLL, 0x000000000000000AuLL},
{0x00000002078644FAuLL, 0x00000000078644FAuLL, 0x0000000000000045uLL, 0x000000000055AD98uLL},
{0x00000002078644FAuLL, 0x000000000747AE14uLL, 0x0000000000000047uLL, 0x0000000002A4FD6EuLL},
{0x00000002078644FAuLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x00000000078644FEuLL},
{0x00000002078644FAuLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x00000000078644FAuLL},
{0x00000002078644FAuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000007864500uLL},
{0x00000002078644FAuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x00000000078644FEuLL},
{0x00000002078644FAuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x00000000078644FCuLL},
{0x00000002078644FAuLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x00000000078644FAuLL},
{0x00000002078644FAuLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000078644F8uLL},
{0x00000002078644FAuLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000078644F6uLL},
{0x00000002078644FAuLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000078644F4uLL},
{0x00000002078644FAuLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000078644DAuLL},
{0x00000002078644FAuLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000002078644FAuLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x00000001003E96E6uLL},
{0x00000002078644FAuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x00000000878644FBuLL},
{0x00000002078644FAuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x00000000878644FAuLL},
{0x00000002078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000000078644FDuLL},
{0x00000002078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000078644FCuLL},
{0x00000002078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000078644FBuLL},
{0x00000002078644FAuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000000078644FAuLL},
{0x00000002078644FAuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000000078644F9uLL},
{0x00000002078644FAuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000078644F8uLL},
{0x00000002078644FAuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000078644F7uLL},
{0x00000002078644FAuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000078644EAuLL},
{0x00000002078644FAuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000002078644FAuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x00000002078644FAuLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x00000002078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002078644FAuLL},
{0x000000020747AE14uLL, 0x0000000000000001uLL, 0x000000020747AE14uLL, 0x0000000000000000uLL},
{0x000000020747AE14uLL, 0x0000000000000002uLL, 0x0000000103A3D70AuLL, 0x0000000000000000uLL},
{0x000000020747AE14uLL, 0x0000000000000003uLL, 0x00000000AD17E4B1uLL, 0x0000000000000001uLL},
{0x000000020747AE14uLL, 0x0000000000000010uLL, 0x0000000020747AE1uLL, 0x0000000000000004uLL},
{0x000000020747AE14uLL, 0x00000000078644FAuLL, 0x0000000000000045uLL, 0x00000000001716B2uLL},
{0x000000020747AE14uLL, 0x000000000747AE14uLL, 0x0000000000000047uLL, 0x0000000002666688uLL},
{0x000000020747AE14uLL, 0x000000007FFFFFFFuLL, 0x0000000000000004uLL, 0x000000000747AE18uLL},
{0x000000020747AE14uLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x000000000747AE14uLL},
{0x000000020747AE14uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x000000000747AE1AuLL},
{0x000000020747AE14uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x000000000747AE18uLL},
{0x000000020747AE14uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x000000000747AE16uLL},
{0x000000020747AE14uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x000000000747AE14uLL},
{0x000000020747AE14uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x000000000747AE12uLL},
{0x000000020747AE14uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x000000000747AE10uLL},
{0x000000020747AE14uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x000000000747AE0EuLL},
{0x000000020747AE14uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x000000000747ADF4uLL},
{0x000000020747AE14uLL, 0x00000001078644FAuLL, 0x0000000000000001uLL, 0x00000000FFC1691AuLL},
{0x000000020747AE14uLL, 0x000000010747AE14uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x000000020747AE14uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x000000008747AE15uLL},
{0x000000020747AE14uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000008747AE14uLL},
{0x000000020747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x000000000747AE17uLL},
{0x000000020747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x000000000747AE16uLL},
{0x000000020747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x000000000747AE15uLL},
{0x000000020747AE14uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL},
{0x000000020747AE14uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000000747AE13uLL},
{0x000000020747AE14uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000000747AE12uLL},
{0x000000020747AE14uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000000747AE11uLL},
{0x000000020747AE14uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000000747AE04uLL},
{0x000000020747AE14uLL, 0x00000002078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000020747AE14uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000020747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000020747AE14uLL},
{0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000000uLL},
{0x000000027FFFFFFFuLL, 0x0000000000000002uLL, 0x000000013FFFFFFFuLL, 0x0000000000000001uLL},
{0x000000027FFFFFFFuLL, 0x0000000000000003uLL, 0x00000000D5555555uLL, 0x0000000000000000uLL},
{0x000000027FFFFFFFuLL, 0x0000000000000010uLL, 0x0000000027FFFFFFuLL, 0x000000000000000FuLL},
{0x000000027FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000055uLL, 0x00000000006B18FDuLL},
{0x000000027FFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000057uLL, 0x0000000006A3D733uLL},
{0x000000027FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000005uLL, 0x0000000000000004uLL},
{0x000000027FFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000004uLL, 0x000000007FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000080000005uLL},
{0x000000027FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000080000003uLL},
{0x000000027FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000080000001uLL},
{0x000000027FFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x000000007FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x000000007FFFFFFDuLL},
{0x000000027FFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x000000007FFFFFFBuLL},
{0x000000027FFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x000000007FFFFFF9uLL},
{0x000000027FFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x000000007FFFFFDFuLL},
{0x000000027FFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x0000000070F3760BuLL},
{0x000000027FFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x000000007170A3D7uLL},
{0x000000027FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x000000027FFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x000000027FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x000000027FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x000000027FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x000000027FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x000000027FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFCuLL},
{0x000000027FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFEFuLL},
{0x000000027FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB05uLL},
{0x000000027FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851EBuLL},
{0x000000027FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000027FFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x000000027FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000027FFFFFFFuLL},
{0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000280000000uLL, 0x0000000000000000uLL},
{0x0000000280000000uLL, 0x0000000000000002uLL, 0x0000000140000000uLL, 0x0000000000000000uLL},
{0x0000000280000000uLL, 0x0000000000000003uLL, 0x00000000D5555555uLL, 0x0000000000000001uLL},
{0x0000000280000000uLL, 0x0000000000000010uLL, 0x0000000028000000uLL, 0x0000000000000000uLL},
{0x0000000280000000uLL, 0x00000000078644FAuLL, 0x0000000000000055uLL, 0x00000000006B18FEuLL},
{0x0000000280000000uLL, 0x000000000747AE14uLL, 0x0000000000000057uLL, 0x0000000006A3D734uLL},
{0x0000000280000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000005uLL, 0x0000000000000005uLL},
{0x0000000280000000uLL, 0x0000000080000000uLL, 0x0000000000000005uLL, 0x0000000000000000uLL},
{0x0000000280000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000080000006uLL},
{0x0000000280000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000080000004uLL},
{0x0000000280000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000080000002uLL},
{0x0000000280000000uLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x0000000080000000uLL},
{0x0000000280000000uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x000000007FFFFFFEuLL},
{0x0000000280000000uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x000000007FFFFFFCuLL},
{0x0000000280000000uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x000000007FFFFFFAuLL},
{0x0000000280000000uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x000000007FFFFFE0uLL},
{0x0000000280000000uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x0000000070F3760CuLL},
{0x0000000280000000uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x000000007170A3D8uLL},
{0x0000000280000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000280000000uLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000280000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000280000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000280000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000280000000uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000000280000000uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x0000000280000000uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x0000000280000000uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x0000000280000000uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFF0uLL},
{0x0000000280000000uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB06uLL},
{0x0000000280000000uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851ECuLL},
{0x0000000280000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000280000000uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000280000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x0000000280000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000280000000uLL},
{0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFDuLL, 0x0000000000000002uLL, 0x000000017FFFFFFEuLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFDuLL, 0x0000000000000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000002FFFFFFFuLL, 0x000000000000000DuLL},
{0x00000002FFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808461uLL},
{0x00000002FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999C9uLL},
{0x00000002FFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000003uLL},
{0x00000002FFFFFFFDuLL, 0x0000000080000000uLL, 0x0000000000000005uLL, 0x000000007FFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000000000006uLL},
{0x00000002FFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000003uLL},
{0x00000002FFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFDuLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFBuLL},
{0x00000002FFFFFFFDuLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFF9uLL},
{0x00000002FFFFFFFDuLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFF7uLL},
{0x00000002FFFFFFFDuLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFDDuLL},
{0x00000002FFFFFFFDuLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F37609uLL},
{0x00000002FFFFFFFDuLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3D5uLL},
{0x00000002FFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x00000002FFFFFFFDuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000002FFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000002FFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000002FFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000002FFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000002FFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFAuLL},
{0x00000002FFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEDuLL},
{0x00000002FFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB03uLL},
{0x00000002FFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851E9uLL},
{0x00000002FFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000002FFFFFFFDuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFDuLL},
{0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFEuLL, 0x0000000000000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFEuLL, 0x0000000000000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000002FFFFFFFuLL, 0x000000000000000EuLL},
{0x00000002FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808462uLL},
{0x00000002FFFFFFFEuLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CAuLL},
{0x00000002FFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000004uLL},
{0x00000002FFFFFFFEuLL, 0x0000000080000000uLL, 0x0000000000000005uLL, 0x000000007FFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000000000007uLL},
{0x00000002FFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000004uLL},
{0x00000002FFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFEuLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFCuLL},
{0x00000002FFFFFFFEuLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFAuLL},
{0x00000002FFFFFFFEuLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFF8uLL},
{0x00000002FFFFFFFEuLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFDEuLL},
{0x00000002FFFFFFFEuLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760AuLL},
{0x00000002FFFFFFFEuLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3D6uLL},
{0x00000002FFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFEuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x00000002FFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000002FFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000002FFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000002FFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000002FFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000002FFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEEuLL},
{0x00000002FFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB04uLL},
{0x00000002FFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EAuLL},
{0x00000002FFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000002FFFFFFFEuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFEuLL},
{0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFFuLL, 0x0000000000000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFFuLL, 0x0000000000000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000002uLL},
{0x00000002FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000002FFFFFFFuLL, 0x000000000000000FuLL},
{0x00000002FFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808463uLL},
{0x00000002FFFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CBuLL},
{0x00000002FFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000005uLL},
{0x00000002FFFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000005uLL, 0x000000007FFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000000000008uLL},
{0x00000002FFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000005uLL},
{0x00000002FFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000002uLL},
{0x00000002FFFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFDuLL},
{0x00000002FFFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFBuLL},
{0x00000002FFFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFF9uLL},
{0x00000002FFFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFDFuLL},
{0x00000002FFFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760BuLL},
{0x00000002FFFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3D7uLL},
{0x00000002FFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x00000002FFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x00000002FFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000002FFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000002FFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000002FFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000002FFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEFuLL},
{0x00000002FFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB05uLL},
{0x00000002FFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EBuLL},
{0x00000002FFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x00000002FFFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x00000002FFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000002FFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000002FFFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x00000002FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000002FFFFFFFFuLL},
{0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000300000000uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x0000000000000002uLL, 0x0000000180000000uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x0000000000000003uLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x0000000000000010uLL, 0x0000000030000000uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808464uLL},
{0x0000000300000000uLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CCuLL},
{0x0000000300000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000006uLL},
{0x0000000300000000uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000000000009uLL},
{0x0000000300000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000006uLL},
{0x0000000300000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000003uLL},
{0x0000000300000000uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFEuLL},
{0x0000000300000000uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFCuLL},
{0x0000000300000000uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFAuLL},
{0x0000000300000000uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFE0uLL},
{0x0000000300000000uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760CuLL},
{0x0000000300000000uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3D8uLL},
{0x0000000300000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000300000000uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000003uLL},
{0x0000000300000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x0000000300000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000300000000uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000300000000uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000300000000uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x0000000300000000uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x0000000300000000uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF0uLL},
{0x0000000300000000uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB06uLL},
{0x0000000300000000uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851ECuLL},
{0x0000000300000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000300000000uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000000300000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000300000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000300000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000300000000uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000300000000uLL, 0x0000000300000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000000uLL},
{0x0000000300000001uLL, 0x0000000000000001uLL, 0x0000000300000001uLL, 0x0000000000000000uLL},
{0x0000000300000001uLL, 0x0000000000000002uLL, 0x0000000180000000uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x0000000000000003uLL, 0x0000000100000000uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x0000000000000010uLL, 0x0000000030000000uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808465uLL},
{0x0000000300000001uLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CDuLL},
{0x0000000300000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000007uLL},
{0x0000000300000001uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x000000000000000AuLL},
{0x0000000300000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000007uLL},
{0x0000000300000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000004uLL},
{0x0000000300000001uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL},
{0x0000000300000001uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFDuLL},
{0x0000000300000001uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFBuLL},
{0x0000000300000001uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFE1uLL},
{0x0000000300000001uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760DuLL},
{0x0000000300000001uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3D9uLL},
{0x0000000300000001uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000300000001uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000004uLL},
{0x0000000300000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000003uLL},
{0x0000000300000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x0000000300000001uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000300000001uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000300000001uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000300000001uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x0000000300000001uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF1uLL},
{0x0000000300000001uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB07uLL},
{0x0000000300000001uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EDuLL},
{0x0000000300000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000300000001uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000300000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000300000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000300000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000300000001uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000300000001uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000300000001uLL, 0x0000000300000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000001uLL},
{0x0000000300000002uLL, 0x0000000000000001uLL, 0x0000000300000002uLL, 0x0000000000000000uLL},
{0x0000000300000002uLL, 0x0000000000000002uLL, 0x0000000180000001uLL, 0x0000000000000000uLL},
{0x0000000300000002uLL, 0x0000000000000003uLL, 0x0000000100000000uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x0000000000000010uLL, 0x0000000030000000uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808466uLL},
{0x0000000300000002uLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CEuLL},
{0x0000000300000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000008uLL},
{0x0000000300000002uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x000000000000000BuLL},
{0x0000000300000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000008uLL},
{0x0000000300000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000005uLL},
{0x0000000300000002uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x0000000100000001uLL, 0x0000000000000002uLL, 0x0000000100000000uLL},
{0x0000000300000002uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFEuLL},
{0x0000000300000002uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFCuLL},
{0x0000000300000002uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFE2uLL},
{0x0000000300000002uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760EuLL},
{0x0000000300000002uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3DAuLL},
{0x0000000300000002uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x0000000300000002uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000005uLL},
{0x0000000300000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000004uLL},
{0x0000000300000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000003uLL},
{0x0000000300000002uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x0000000300000002uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000300000002uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000300000002uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x0000000300000002uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF2uLL},
{0x0000000300000002uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB08uLL},
{0x0000000300000002uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EEuLL},
{0x0000000300000002uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000300000002uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000300000002uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000300000002uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000300000002uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000300000002uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000300000002uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000300000002uLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000300000002uLL, 0x0000000300000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000002uLL},
{0x0000000300000003uLL, 0x0000000000000001uLL, 0x0000000300000003uLL, 0x0000000000000000uLL},
{0x0000000300000003uLL, 0x0000000000000002uLL, 0x0000000180000001uLL, 0x0000000000000001uLL},
{0x0000000300000003uLL, 0x0000000000000003uLL, 0x0000000100000001uLL, 0x0000000000000000uLL},
{0x0000000300000003uLL, 0x0000000000000010uLL, 0x0000000030000000uLL, 0x0000000000000003uLL},
{0x0000000300000003uLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808467uLL},
{0x0000000300000003uLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999CFuLL},
{0x0000000300000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000009uLL},
{0x0000000300000003uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x0000000000000003uLL},
{0x0000000300000003uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x000000000000000CuLL},
{0x0000000300000003uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000009uLL},
{0x0000000300000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000006uLL},
{0x0000000300000003uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000000000003uLL},
{0x0000000300000003uLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x0000000000000000uLL},
{0x0000000300000003uLL, 0x0000000100000002uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL},
{0x0000000300000003uLL, 0x0000000100000003uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFDuLL},
{0x0000000300000003uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFE3uLL},
{0x0000000300000003uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3760FuLL},
{0x0000000300000003uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3DBuLL},
{0x0000000300000003uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x0000000300000003uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x0000000300000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000006uLL},
{0x0000000300000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000005uLL},
{0x0000000300000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000004uLL},
{0x0000000300000003uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000100000003uLL},
{0x0000000300000003uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x0000000300000003uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000300000003uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000300000003uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFF3uLL},
{0x0000000300000003uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB09uLL},
{0x0000000300000003uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EFuLL},
{0x0000000300000003uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000004uLL},
{0x0000000300000003uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000300000003uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000006uLL},
{0x0000000300000003uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000005uLL},
{0x0000000300000003uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000004uLL},
{0x0000000300000003uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000000300000003uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000000300000003uLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000300000003uLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000300000003uLL, 0x0000000300000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000003uLL},
{0x0000000300000010uLL, 0x0000000000000001uLL, 0x0000000300000010uLL, 0x0000000000000000uLL},
{0x0000000300000010uLL, 0x0000000000000002uLL, 0x0000000180000008uLL, 0x0000000000000000uLL},
{0x0000000300000010uLL, 0x0000000000000003uLL, 0x0000000100000005uLL, 0x0000000000000001uLL},
{0x0000000300000010uLL, 0x0000000000000010uLL, 0x0000000030000001uLL, 0x0000000000000000uLL},
{0x0000000300000010uLL, 0x00000000078644FAuLL, 0x0000000000000066uLL, 0x0000000000808474uLL},
{0x0000000300000010uLL, 0x000000000747AE14uLL, 0x0000000000000069uLL, 0x00000000039999DCuLL},
{0x0000000300000010uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000000000016uLL},
{0x0000000300000010uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x0000000000000010uLL},
{0x0000000300000010uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000000000019uLL},
{0x0000000300000010uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000000000016uLL},
{0x0000000300000010uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000000000013uLL},
{0x0000000300000010uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000000000010uLL},
{0x0000000300000010uLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x000000000000000DuLL},
{0x0000000300000010uLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x000000000000000AuLL},
{0x0000000300000010uLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x0000000000000007uLL},
{0x0000000300000010uLL, 0x0000000100000010uLL, 0x0000000000000002uLL, 0x00000000FFFFFFF0uLL},
{0x0000000300000010uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F0F3761CuLL},
{0x0000000300000010uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F170A3E8uLL},
{0x0000000300000010uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000012uLL},
{0x0000000300000010uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000000000010uLL},
{0x0000000300000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000013uLL},
{0x0000000300000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000012uLL},
{0x0000000300000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000011uLL},
{0x0000000300000010uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000100000010uLL},
{0x0000000300000010uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000010000000FuLL},
{0x0000000300000010uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000010000000EuLL},
{0x0000000300000010uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000010000000DuLL},
{0x0000000300000010uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000300000010uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB16uLL},
{0x0000000300000010uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851FCuLL},
{0x0000000300000010uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000011uLL},
{0x0000000300000010uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000080000010uLL},
{0x0000000300000010uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000013uLL},
{0x0000000300000010uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000012uLL},
{0x0000000300000010uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000011uLL},
{0x0000000300000010uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000000000010uLL},
{0x0000000300000010uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x000000000000000FuLL},
{0x0000000300000010uLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x000000000000000EuLL},
{0x0000000300000010uLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x000000000000000DuLL},
{0x0000000300000010uLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000300000010uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x0000000300000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000300000010uLL},
{0x00000003078644FAuLL, 0x0000000000000001uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL},
{0x00000003078644FAuLL, 0x0000000000000002uLL, 0x0000000183C3227DuLL, 0x0000000000000000uLL},
{0x00000003078644FAuLL, 0x0000000000000003uLL, 0x00000001028216FEuLL, 0x0000000000000000uLL},
{0x00000003078644FAuLL, 0x0000000000000010uLL, 0x000000003078644FuLL, 0x000000000000000AuLL},
{0x00000003078644FAuLL, 0x00000000078644FAuLL, 0x0000000000000067uLL, 0x0000000000808464uLL},
{0x00000003078644FAuLL, 0x000000000747AE14uLL, 0x000000000000006AuLL, 0x0000000003D830B2uLL},
{0x00000003078644FAuLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000007864500uLL},
{0x00000003078644FAuLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x00000000078644FAuLL},
{0x00000003078644FAuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000007864503uLL},
{0x00000003078644FAuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000007864500uLL},
{0x00000003078644FAuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x00000000078644FDuLL},
{0x00000003078644FAuLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x00000000078644FAuLL},
{0x00000003078644FAuLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x00000000078644F7uLL},
{0x00000003078644FAuLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x00000000078644F4uLL},
{0x00000003078644FAuLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x00000000078644F1uLL},
{0x00000003078644FAuLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x00000000078644CAuLL},
{0x00000003078644FAuLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F879BB06uLL},
{0x00000003078644FAuLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F8F6E8D2uLL},
{0x00000003078644FAuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x00000000078644FCuLL},
{0x00000003078644FAuLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x00000000078644FAuLL},
{0x00000003078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000001078644FDuLL},
{0x00000003078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000001078644FCuLL},
{0x00000003078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000001078644FBuLL},
{0x00000003078644FAuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000001078644FAuLL},
{0x00000003078644FAuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000001078644F9uLL},
{0x00000003078644FAuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000001078644F8uLL},
{0x00000003078644FAuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000001078644F7uLL},
{0x00000003078644FAuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000001078644EAuLL},
{0x00000003078644FAuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000003078644FAuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000001003E96E6uLL},
{0x00000003078644FAuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x00000000878644FBuLL},
{0x00000003078644FAuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x00000000878644FAuLL},
{0x00000003078644FAuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000000078644FDuLL},
{0x00000003078644FAuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000078644FCuLL},
{0x00000003078644FAuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000078644FBuLL},
{0x00000003078644FAuLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x00000000078644FAuLL},
{0x00000003078644FAuLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x00000000078644F9uLL},
{0x00000003078644FAuLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x00000000078644F8uLL},
{0x00000003078644FAuLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x00000000078644F7uLL},
{0x00000003078644FAuLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x00000000078644EAuLL},
{0x00000003078644FAuLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000003078644FAuLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x00000003078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x00000003078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003078644FAuLL},
{0x000000030747AE14uLL, 0x0000000000000001uLL, 0x000000030747AE14uLL, 0x0000000000000000uLL},
{0x000000030747AE14uLL, 0x0000000000000002uLL, 0x0000000183A3D70AuLL, 0x0000000000000000uLL},
{0x000000030747AE14uLL, 0x0000000000000003uLL, 0x00000001026D3A06uLL, 0x0000000000000002uLL},
{0x000000030747AE14uLL, 0x0000000000000010uLL, 0x0000000030747AE1uLL, 0x0000000000000004uLL},
{0x000000030747AE14uLL, 0x00000000078644FAuLL, 0x0000000000000067uLL, 0x000000000041ED7EuLL},
{0x000000030747AE14uLL, 0x000000000747AE14uLL, 0x000000000000006AuLL, 0x00000000039999CCuLL},
{0x000000030747AE14uLL, 0x000000007FFFFFFFuLL, 0x0000000000000006uLL, 0x000000000747AE1AuLL},
{0x000000030747AE14uLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x000000000747AE14uLL},
{0x000000030747AE14uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x000000000747AE1DuLL},
{0x000000030747AE14uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x000000000747AE1AuLL},
{0x000000030747AE14uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x000000000747AE17uLL},
{0x000000030747AE14uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x000000000747AE14uLL},
{0x000000030747AE14uLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x000000000747AE11uLL},
{0x000000030747AE14uLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x000000000747AE0EuLL},
{0x000000030747AE14uLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x000000000747AE0BuLL},
{0x000000030747AE14uLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x000000000747ADE4uLL},
{0x000000030747AE14uLL, 0x00000001078644FAuLL, 0x0000000000000002uLL, 0x00000000F83B2420uLL},
{0x000000030747AE14uLL, 0x000000010747AE14uLL, 0x0000000000000002uLL, 0x00000000F8B851ECuLL},
{0x000000030747AE14uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x000000000747AE16uLL},
{0x000000030747AE14uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x000000000747AE14uLL},
{0x000000030747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x000000010747AE17uLL},
{0x000000030747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x000000010747AE16uLL},
{0x000000030747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x000000010747AE15uLL},
{0x000000030747AE14uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x000000010747AE14uLL},
{0x000000030747AE14uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000010747AE13uLL},
{0x000000030747AE14uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000010747AE12uLL},
{0x000000030747AE14uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000010747AE11uLL},
{0x000000030747AE14uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000010747AE04uLL},
{0x000000030747AE14uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000000FFC1691AuLL},
{0x000000030747AE14uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x000000030747AE14uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000008747AE15uLL},
{0x000000030747AE14uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000008747AE14uLL},
{0x000000030747AE14uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x000000000747AE17uLL},
{0x000000030747AE14uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x000000000747AE16uLL},
{0x000000030747AE14uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x000000000747AE15uLL},
{0x000000030747AE14uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL},
{0x000000030747AE14uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x000000000747AE13uLL},
{0x000000030747AE14uLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x000000000747AE12uLL},
{0x000000030747AE14uLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x000000000747AE11uLL},
{0x000000030747AE14uLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x000000000747AE04uLL},
{0x000000030747AE14uLL, 0x00000003078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000030747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000030747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000030747AE14uLL},
{0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000000uLL},
{0x000000037FFFFFFFuLL, 0x0000000000000002uLL, 0x00000001BFFFFFFFuLL, 0x0000000000000001uLL},
{0x000000037FFFFFFFuLL, 0x0000000000000003uLL, 0x000000012AAAAAAAuLL, 0x0000000000000001uLL},
{0x000000037FFFFFFFuLL, 0x0000000000000010uLL, 0x0000000037FFFFFFuLL, 0x000000000000000FuLL},
{0x000000037FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000077uLL, 0x000000000095EFC9uLL},
{0x000000037FFFFFFFuLL, 0x000000000747AE14uLL, 0x000000000000007BuLL, 0x00000000008F5C63uLL},
{0x000000037FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000007uLL, 0x0000000000000006uLL},
{0x000000037FFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000006uLL, 0x000000007FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000080000008uLL},
{0x000000037FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000080000005uLL},
{0x000000037FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000080000002uLL},
{0x000000037FFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x000000007FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x000000007FFFFFFCuLL},
{0x000000037FFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x000000007FFFFFF9uLL},
{0x000000037FFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x000000007FFFFFF6uLL},
{0x000000037FFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x000000007FFFFFCFuLL},
{0x000000037FFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000003uLL, 0x00000000696D3111uLL},
{0x000000037FFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000003uLL, 0x000000006A28F5C3uLL},
{0x000000037FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000080000001uLL},
{0x000000037FFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x000000007FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000180000002uLL},
{0x000000037FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000180000001uLL},
{0x000000037FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000180000000uLL},
{0x000000037FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x000000037FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000017FFFFFFDuLL},
{0x000000037FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000017FFFFFFCuLL},
{0x000000037FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000017FFFFFEFuLL},
{0x000000037FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x000000017879BB05uLL},
{0x000000037FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000178B851EBuLL},
{0x000000037FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x000000037FFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x000000037FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x000000037FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x000000037FFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x000000037FFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x000000037FFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFCuLL},
{0x000000037FFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFEFuLL},
{0x000000037FFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB05uLL},
{0x000000037FFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851EBuLL},
{0x000000037FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000037FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x000000037FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000037FFFFFFFuLL},
{0x0000000380000000uLL, 0x0000000000000001uLL, 0x0000000380000000uLL, 0x0000000000000000uLL},
{0x0000000380000000uLL, 0x0000000000000002uLL, 0x00000001C0000000uLL, 0x0000000000000000uLL},
{0x0000000380000000uLL, 0x0000000000000003uLL, 0x000000012AAAAAAAuLL, 0x0000000000000002uLL},
{0x0000000380000000uLL, 0x0000000000000010uLL, 0x0000000038000000uLL, 0x0000000000000000uLL},
{0x0000000380000000uLL, 0x00000000078644FAuLL, 0x0000000000000077uLL, 0x000000000095EFCAuLL},
{0x0000000380000000uLL, 0x000000000747AE14uLL, 0x000000000000007BuLL, 0x00000000008F5C64uLL},
{0x0000000380000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000007uLL, 0x0000000000000007uLL},
{0x0000000380000000uLL, 0x0000000080000000uLL, 0x0000000000000007uLL, 0x0000000000000000uLL},
{0x0000000380000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000080000009uLL},
{0x0000000380000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000080000006uLL},
{0x0000000380000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000080000003uLL},
{0x0000000380000000uLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x0000000080000000uLL},
{0x0000000380000000uLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x000000007FFFFFFDuLL},
{0x0000000380000000uLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x000000007FFFFFFAuLL},
{0x0000000380000000uLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x000000007FFFFFF7uLL},
{0x0000000380000000uLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x000000007FFFFFD0uLL},
{0x0000000380000000uLL, 0x00000001078644FAuLL, 0x0000000000000003uLL, 0x00000000696D3112uLL},
{0x0000000380000000uLL, 0x000000010747AE14uLL, 0x0000000000000003uLL, 0x000000006A28F5C4uLL},
{0x0000000380000000uLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000080000002uLL},
{0x0000000380000000uLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x0000000080000000uLL},
{0x0000000380000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000180000003uLL},
{0x0000000380000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000180000002uLL},
{0x0000000380000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000180000001uLL},
{0x0000000380000000uLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x0000000180000000uLL},
{0x0000000380000000uLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL},
{0x0000000380000000uLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x0000000380000000uLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x000000017FFFFFFDuLL},
{0x0000000380000000uLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x000000017FFFFFF0uLL},
{0x0000000380000000uLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x000000017879BB06uLL},
{0x0000000380000000uLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x0000000178B851ECuLL},
{0x0000000380000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x0000000380000000uLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x0000000380000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000080000003uLL},
{0x0000000380000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000080000002uLL},
{0x0000000380000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000001uLL},
{0x0000000380000000uLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000000380000000uLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x0000000380000000uLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x0000000380000000uLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x0000000380000000uLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFF0uLL},
{0x0000000380000000uLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB06uLL},
{0x0000000380000000uLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851ECuLL},
{0x0000000380000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000000380000000uLL, 0x0000000380000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000000380000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x0000000380000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000000380000000uLL},
{0x00000003FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFDuLL, 0x0000000000000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFDuLL, 0x0000000000000003uLL, 0x0000000155555554uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000003FFFFFFFuLL, 0x000000000000000DuLL},
{0x00000003FFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000000000088uLL, 0x0000000000AB5B2DuLL},
{0x00000003FFFFFFFDuLL, 0x000000000747AE14uLL, 0x000000000000008CuLL, 0x0000000004CCCD0DuLL},
{0x00000003FFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000005uLL},
{0x00000003FFFFFFFDuLL, 0x0000000080000000uLL, 0x0000000000000007uLL, 0x000000007FFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000004uLL, 0x0000000000000009uLL},
{0x00000003FFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000005uLL},
{0x00000003FFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFDuLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFAuLL},
{0x00000003FFFFFFFDuLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF7uLL},
{0x00000003FFFFFFFDuLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF4uLL},
{0x00000003FFFFFFFDuLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x00000000FFFFFFCDuLL},
{0x00000003FFFFFFFDuLL, 0x00000001078644FAuLL, 0x0000000000000003uLL, 0x00000000E96D310FuLL},
{0x00000003FFFFFFFDuLL, 0x000000010747AE14uLL, 0x0000000000000003uLL, 0x00000000EA28F5C1uLL},
{0x00000003FFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFDuLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x00000003FFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000001FFFFFFFEuLL},
{0x00000003FFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFCuLL},
{0x00000003FFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFBuLL},
{0x00000003FFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFAuLL},
{0x00000003FFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000001FFFFFFEDuLL},
{0x00000003FFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000001F879BB03uLL},
{0x00000003FFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000001F8B851E9uLL},
{0x00000003FFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x00000003FFFFFFFDuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000003FFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000003FFFFFFFDuLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000003FFFFFFFDuLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000003FFFFFFFDuLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFAuLL},
{0x00000003FFFFFFFDuLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEDuLL},
{0x00000003FFFFFFFDuLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB03uLL},
{0x00000003FFFFFFFDuLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851E9uLL},
{0x00000003FFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000003FFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFDuLL},
{0x00000003FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFEuLL, 0x0000000000000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFEuLL, 0x0000000000000003uLL, 0x0000000155555554uLL, 0x0000000000000002uLL},
{0x00000003FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000003FFFFFFFuLL, 0x000000000000000EuLL},
{0x00000003FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000000000088uLL, 0x0000000000AB5B2EuLL},
{0x00000003FFFFFFFEuLL, 0x000000000747AE14uLL, 0x000000000000008CuLL, 0x0000000004CCCD0EuLL},
{0x00000003FFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000006uLL},
{0x00000003FFFFFFFEuLL, 0x0000000080000000uLL, 0x0000000000000007uLL, 0x000000007FFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000AuLL},
{0x00000003FFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000006uLL},
{0x00000003FFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000002uLL},
{0x00000003FFFFFFFEuLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFBuLL},
{0x00000003FFFFFFFEuLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF8uLL},
{0x00000003FFFFFFFEuLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF5uLL},
{0x00000003FFFFFFFEuLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x00000000FFFFFFCEuLL},
{0x00000003FFFFFFFEuLL, 0x00000001078644FAuLL, 0x0000000000000003uLL, 0x00000000E96D3110uLL},
{0x00000003FFFFFFFEuLL, 0x000000010747AE14uLL, 0x0000000000000003uLL, 0x00000000EA28F5C2uLL},
{0x00000003FFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000100000000uLL},
{0x00000003FFFFFFFEuLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000004uLL},
{0x00000003FFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000002uLL},
{0x00000003FFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFDuLL},
{0x00000003FFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFCuLL},
{0x00000003FFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFBuLL},
{0x00000003FFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000001FFFFFFEEuLL},
{0x00000003FFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000001F879BB04uLL},
{0x00000003FFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000001F8B851EAuLL},
{0x00000003FFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL},
{0x00000003FFFFFFFEuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x00000003FFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000003FFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFEuLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000003FFFFFFFEuLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000003FFFFFFFEuLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000003FFFFFFFEuLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEEuLL},
{0x00000003FFFFFFFEuLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB04uLL},
{0x00000003FFFFFFFEuLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EAuLL},
{0x00000003FFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000003FFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFEuLL},
{0x00000003FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000000000000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFFuLL, 0x0000000000000003uLL, 0x0000000155555555uLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000003FFFFFFFuLL, 0x000000000000000FuLL},
{0x00000003FFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000088uLL, 0x0000000000AB5B2FuLL},
{0x00000003FFFFFFFFuLL, 0x000000000747AE14uLL, 0x000000000000008CuLL, 0x0000000004CCCD0FuLL},
{0x00000003FFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000007uLL},
{0x00000003FFFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000007uLL, 0x000000007FFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000BuLL},
{0x00000003FFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000007uLL},
{0x00000003FFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000003uLL},
{0x00000003FFFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000003uLL, 0x00000000FFFFFFFCuLL},
{0x00000003FFFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF9uLL},
{0x00000003FFFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000003uLL, 0x00000000FFFFFFF6uLL},
{0x00000003FFFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000003uLL, 0x00000000FFFFFFCFuLL},
{0x00000003FFFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000003uLL, 0x00000000E96D3111uLL},
{0x00000003FFFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000003uLL, 0x00000000EA28F5C3uLL},
{0x00000003FFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000000000002uLL, 0x0000000100000001uLL},
{0x00000003FFFFFFFFuLL, 0x0000000180000000uLL, 0x0000000000000002uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000002uLL, 0x0000000000000005uLL},
{0x00000003FFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000002uLL, 0x0000000000000003uLL},
{0x00000003FFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000002uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFEuLL},
{0x00000003FFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFDuLL},
{0x00000003FFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000001uLL, 0x00000001FFFFFFFCuLL},
{0x00000003FFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000001uLL, 0x00000001FFFFFFEFuLL},
{0x00000003FFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000001uLL, 0x00000001F879BB05uLL},
{0x00000003FFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000001uLL, 0x00000001F8B851EBuLL},
{0x00000003FFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000180000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000001uLL, 0x000000017FFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000100000002uLL},
{0x00000003FFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000100000001uLL},
{0x00000003FFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000100000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000003FFFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000003FFFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000003FFFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEFuLL},
{0x00000003FFFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB05uLL},
{0x00000003FFFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EBuLL},
{0x00000003FFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x00000003FFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000003FFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000003FFFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x00000003FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000003FFFFFFFFuLL},
{0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000001000000000uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x0000000000000002uLL, 0x0000000800000000uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x0000000000000003uLL, 0x0000000555555555uLL, 0x0000000000000001uLL},
{0x0000001000000000uLL, 0x0000000000000010uLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x00000000078644FAuLL, 0x0000000000000220uLL, 0x0000000002AD6CC0uLL},
{0x0000001000000000uLL, 0x000000000747AE14uLL, 0x0000000000000232uLL, 0x0000000004A3D818uLL},
{0x0000001000000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x0000000000000020uLL},
{0x0000001000000000uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000000000030uLL},
{0x0000001000000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000000000020uLL},
{0x0000001000000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000000000010uLL},
{0x0000001000000000uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x0000000100000001uLL, 0x000000000000000FuLL, 0x00000000FFFFFFF1uLL},
{0x0000001000000000uLL, 0x0000000100000002uLL, 0x000000000000000FuLL, 0x00000000FFFFFFE2uLL},
{0x0000001000000000uLL, 0x0000000100000003uLL, 0x000000000000000FuLL, 0x00000000FFFFFFD3uLL},
{0x0000001000000000uLL, 0x0000000100000010uLL, 0x000000000000000FuLL, 0x00000000FFFFFF10uLL},
{0x0000001000000000uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000008F21F55AuLL},
{0x0000001000000000uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x0000000092CCCCD4uLL},
{0x0000001000000000uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010000000AuLL},
{0x0000001000000000uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x0000000100000000uLL},
{0x0000001000000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000000000018uLL},
{0x0000001000000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000000000010uLL},
{0x0000001000000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000008uLL},
{0x0000001000000000uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x0000000200000001uLL, 0x0000000000000007uLL, 0x00000001FFFFFFF9uLL},
{0x0000001000000000uLL, 0x0000000200000002uLL, 0x0000000000000007uLL, 0x00000001FFFFFFF2uLL},
{0x0000001000000000uLL, 0x0000000200000003uLL, 0x0000000000000007uLL, 0x00000001FFFFFFEBuLL},
{0x0000001000000000uLL, 0x0000000200000010uLL, 0x0000000000000007uLL, 0x00000001FFFFFF90uLL},
{0x0000001000000000uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001CB541D2AuLL},
{0x0000001000000000uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001CD0A3D74uLL},
{0x0000001000000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000100000006uLL},
{0x0000001000000000uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000100000000uLL},
{0x0000001000000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000010000000FuLL},
{0x0000001000000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010000000AuLL},
{0x0000001000000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000100000005uLL},
{0x0000001000000000uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000100000000uLL},
{0x0000001000000000uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000000FFFFFFFBuLL},
{0x0000001000000000uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF6uLL},
{0x0000001000000000uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF1uLL},
{0x0000001000000000uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000000FFFFFFB0uLL},
{0x0000001000000000uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000DA60A71EuLL},
{0x0000001000000000uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000DB99999CuLL},
{0x0000001000000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000200000004uLL},
{0x0000001000000000uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000200000000uLL},
{0x0000001000000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000CuLL},
{0x0000001000000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000008uLL},
{0x0000001000000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000004uLL},
{0x0000001000000000uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001000000000uLL, 0x0000001000000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000000uLL},
{0x0000001000000001uLL, 0x0000000000000001uLL, 0x0000001000000001uLL, 0x0000000000000000uLL},
{0x0000001000000001uLL, 0x0000000000000002uLL, 0x0000000800000000uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x0000000000000003uLL, 0x0000000555555555uLL, 0x0000000000000002uLL},
{0x0000001000000001uLL, 0x0000000000000010uLL, 0x0000000100000000uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x00000000078644FAuLL, 0x0000000000000220uLL, 0x0000000002AD6CC1uLL},
{0x0000001000000001uLL, 0x000000000747AE14uLL, 0x0000000000000232uLL, 0x0000000004A3D819uLL},
{0x0000001000000001uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x0000000000000021uLL},
{0x0000001000000001uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000000000031uLL},
{0x0000001000000001uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000000000021uLL},
{0x0000001000000001uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000000000011uLL},
{0x0000001000000001uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x0000000100000001uLL, 0x000000000000000FuLL, 0x00000000FFFFFFF2uLL},
{0x0000001000000001uLL, 0x0000000100000002uLL, 0x000000000000000FuLL, 0x00000000FFFFFFE3uLL},
{0x0000001000000001uLL, 0x0000000100000003uLL, 0x000000000000000FuLL, 0x00000000FFFFFFD4uLL},
{0x0000001000000001uLL, 0x0000000100000010uLL, 0x000000000000000FuLL, 0x00000000FFFFFF11uLL},
{0x0000001000000001uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000008F21F55BuLL},
{0x0000001000000001uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x0000000092CCCCD5uLL},
{0x0000001000000001uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010000000BuLL},
{0x0000001000000001uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x0000000100000001uLL},
{0x0000001000000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000000000019uLL},
{0x0000001000000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000000000011uLL},
{0x0000001000000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000009uLL},
{0x0000001000000001uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x0000000200000001uLL, 0x0000000000000007uLL, 0x00000001FFFFFFFAuLL},
{0x0000001000000001uLL, 0x0000000200000002uLL, 0x0000000000000007uLL, 0x00000001FFFFFFF3uLL},
{0x0000001000000001uLL, 0x0000000200000003uLL, 0x0000000000000007uLL, 0x00000001FFFFFFECuLL},
{0x0000001000000001uLL, 0x0000000200000010uLL, 0x0000000000000007uLL, 0x00000001FFFFFF91uLL},
{0x0000001000000001uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001CB541D2BuLL},
{0x0000001000000001uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001CD0A3D75uLL},
{0x0000001000000001uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000100000007uLL},
{0x0000001000000001uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000100000001uLL},
{0x0000001000000001uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x0000000100000010uLL},
{0x0000001000000001uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010000000BuLL},
{0x0000001000000001uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000100000006uLL},
{0x0000001000000001uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000100000001uLL},
{0x0000001000000001uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000000FFFFFFFCuLL},
{0x0000001000000001uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF7uLL},
{0x0000001000000001uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF2uLL},
{0x0000001000000001uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000000FFFFFFB1uLL},
{0x0000001000000001uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000DA60A71FuLL},
{0x0000001000000001uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000DB99999DuLL},
{0x0000001000000001uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000200000005uLL},
{0x0000001000000001uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000200000001uLL},
{0x0000001000000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000DuLL},
{0x0000001000000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000009uLL},
{0x0000001000000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000005uLL},
{0x0000001000000001uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000001000000001uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001000000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000001uLL},
{0x0000001000000002uLL, 0x0000000000000001uLL, 0x0000001000000002uLL, 0x0000000000000000uLL},
{0x0000001000000002uLL, 0x0000000000000002uLL, 0x0000000800000001uLL, 0x0000000000000000uLL},
{0x0000001000000002uLL, 0x0000000000000003uLL, 0x0000000555555556uLL, 0x0000000000000000uLL},
{0x0000001000000002uLL, 0x0000000000000010uLL, 0x0000000100000000uLL, 0x0000000000000002uLL},
{0x0000001000000002uLL, 0x00000000078644FAuLL, 0x0000000000000220uLL, 0x0000000002AD6CC2uLL},
{0x0000001000000002uLL, 0x000000000747AE14uLL, 0x0000000000000232uLL, 0x0000000004A3D81AuLL},
{0x0000001000000002uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x0000000000000022uLL},
{0x0000001000000002uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x0000000000000002uLL},
{0x0000001000000002uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000000000032uLL},
{0x0000001000000002uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000000000022uLL},
{0x0000001000000002uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000000000012uLL},
{0x0000001000000002uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000000000002uLL},
{0x0000001000000002uLL, 0x0000000100000001uLL, 0x000000000000000FuLL, 0x00000000FFFFFFF3uLL},
{0x0000001000000002uLL, 0x0000000100000002uLL, 0x000000000000000FuLL, 0x00000000FFFFFFE4uLL},
{0x0000001000000002uLL, 0x0000000100000003uLL, 0x000000000000000FuLL, 0x00000000FFFFFFD5uLL},
{0x0000001000000002uLL, 0x0000000100000010uLL, 0x000000000000000FuLL, 0x00000000FFFFFF12uLL},
{0x0000001000000002uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000008F21F55CuLL},
{0x0000001000000002uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x0000000092CCCCD6uLL},
{0x0000001000000002uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010000000CuLL},
{0x0000001000000002uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x0000000100000002uLL},
{0x0000001000000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x000000000000001AuLL},
{0x0000001000000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000000000012uLL},
{0x0000001000000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x000000000000000AuLL},
{0x0000001000000002uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000000000002uLL},
{0x0000001000000002uLL, 0x0000000200000001uLL, 0x0000000000000007uLL, 0x00000001FFFFFFFBuLL},
{0x0000001000000002uLL, 0x0000000200000002uLL, 0x0000000000000007uLL, 0x00000001FFFFFFF4uLL},
{0x0000001000000002uLL, 0x0000000200000003uLL, 0x0000000000000007uLL, 0x00000001FFFFFFEDuLL},
{0x0000001000000002uLL, 0x0000000200000010uLL, 0x0000000000000007uLL, 0x00000001FFFFFF92uLL},
{0x0000001000000002uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001CB541D2CuLL},
{0x0000001000000002uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001CD0A3D76uLL},
{0x0000001000000002uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000100000008uLL},
{0x0000001000000002uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000100000002uLL},
{0x0000001000000002uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x0000000100000011uLL},
{0x0000001000000002uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010000000CuLL},
{0x0000001000000002uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000100000007uLL},
{0x0000001000000002uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000100000002uLL},
{0x0000001000000002uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000000FFFFFFFDuLL},
{0x0000001000000002uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF8uLL},
{0x0000001000000002uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF3uLL},
{0x0000001000000002uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000000FFFFFFB2uLL},
{0x0000001000000002uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000DA60A720uLL},
{0x0000001000000002uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000DB99999EuLL},
{0x0000001000000002uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000200000006uLL},
{0x0000001000000002uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000200000002uLL},
{0x0000001000000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000EuLL},
{0x0000001000000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x000000000000000AuLL},
{0x0000001000000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000006uLL},
{0x0000001000000002uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000001000000002uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000001000000002uLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001000000002uLL, 0x0000001000000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000002uLL},
{0x0000001000000003uLL, 0x0000000000000001uLL, 0x0000001000000003uLL, 0x0000000000000000uLL},
{0x0000001000000003uLL, 0x0000000000000002uLL, 0x0000000800000001uLL, 0x0000000000000001uLL},
{0x0000001000000003uLL, 0x0000000000000003uLL, 0x0000000555555556uLL, 0x0000000000000001uLL},
{0x0000001000000003uLL, 0x0000000000000010uLL, 0x0000000100000000uLL, 0x0000000000000003uLL},
{0x0000001000000003uLL, 0x00000000078644FAuLL, 0x0000000000000220uLL, 0x0000000002AD6CC3uLL},
{0x0000001000000003uLL, 0x000000000747AE14uLL, 0x0000000000000232uLL, 0x0000000004A3D81BuLL},
{0x0000001000000003uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x0000000000000023uLL},
{0x0000001000000003uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x0000000000000003uLL},
{0x0000001000000003uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000000000033uLL},
{0x0000001000000003uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000000000023uLL},
{0x0000001000000003uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000000000013uLL},
{0x0000001000000003uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000000000003uLL},
{0x0000001000000003uLL, 0x0000000100000001uLL, 0x000000000000000FuLL, 0x00000000FFFFFFF4uLL},
{0x0000001000000003uLL, 0x0000000100000002uLL, 0x000000000000000FuLL, 0x00000000FFFFFFE5uLL},
{0x0000001000000003uLL, 0x0000000100000003uLL, 0x000000000000000FuLL, 0x00000000FFFFFFD6uLL},
{0x0000001000000003uLL, 0x0000000100000010uLL, 0x000000000000000FuLL, 0x00000000FFFFFF13uLL},
{0x0000001000000003uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000008F21F55DuLL},
{0x0000001000000003uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x0000000092CCCCD7uLL},
{0x0000001000000003uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010000000DuLL},
{0x0000001000000003uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x0000000100000003uLL},
{0x0000001000000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x000000000000001BuLL},
{0x0000001000000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000000000013uLL},
{0x0000001000000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x000000000000000BuLL},
{0x0000001000000003uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000000000003uLL},
{0x0000001000000003uLL, 0x0000000200000001uLL, 0x0000000000000007uLL, 0x00000001FFFFFFFCuLL},
{0x0000001000000003uLL, 0x0000000200000002uLL, 0x0000000000000007uLL, 0x00000001FFFFFFF5uLL},
{0x0000001000000003uLL, 0x0000000200000003uLL, 0x0000000000000007uLL, 0x00000001FFFFFFEEuLL},
{0x0000001000000003uLL, 0x0000000200000010uLL, 0x0000000000000007uLL, 0x00000001FFFFFF93uLL},
{0x0000001000000003uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001CB541D2DuLL},
{0x0000001000000003uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001CD0A3D77uLL},
{0x0000001000000003uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000100000009uLL},
{0x0000001000000003uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000100000003uLL},
{0x0000001000000003uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x0000000100000012uLL},
{0x0000001000000003uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010000000DuLL},
{0x0000001000000003uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000100000008uLL},
{0x0000001000000003uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000100000003uLL},
{0x0000001000000003uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000000FFFFFFFEuLL},
{0x0000001000000003uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF9uLL},
{0x0000001000000003uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000000FFFFFFF4uLL},
{0x0000001000000003uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000000FFFFFFB3uLL},
{0x0000001000000003uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000DA60A721uLL},
{0x0000001000000003uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000DB99999FuLL},
{0x0000001000000003uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000200000007uLL},
{0x0000001000000003uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000200000003uLL},
{0x0000001000000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000000FuLL},
{0x0000001000000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x000000000000000BuLL},
{0x0000001000000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000007uLL},
{0x0000001000000003uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x0000001000000003uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x0000001000000003uLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000001000000003uLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001000000003uLL, 0x0000001000000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000003uLL},
{0x0000001000000010uLL, 0x0000000000000001uLL, 0x0000001000000010uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x0000000000000002uLL, 0x0000000800000008uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x0000000000000003uLL, 0x000000055555555AuLL, 0x0000000000000002uLL},
{0x0000001000000010uLL, 0x0000000000000010uLL, 0x0000000100000001uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x00000000078644FAuLL, 0x0000000000000220uLL, 0x0000000002AD6CD0uLL},
{0x0000001000000010uLL, 0x000000000747AE14uLL, 0x0000000000000232uLL, 0x0000000004A3D828uLL},
{0x0000001000000010uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x0000000000000030uLL},
{0x0000001000000010uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x0000000000000010uLL},
{0x0000001000000010uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000000000040uLL},
{0x0000001000000010uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000000000030uLL},
{0x0000001000000010uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000000000020uLL},
{0x0000001000000010uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000000000010uLL},
{0x0000001000000010uLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x0000000100000002uLL, 0x000000000000000FuLL, 0x00000000FFFFFFF2uLL},
{0x0000001000000010uLL, 0x0000000100000003uLL, 0x000000000000000FuLL, 0x00000000FFFFFFE3uLL},
{0x0000001000000010uLL, 0x0000000100000010uLL, 0x000000000000000FuLL, 0x00000000FFFFFF20uLL},
{0x0000001000000010uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000008F21F56AuLL},
{0x0000001000000010uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x0000000092CCCCE4uLL},
{0x0000001000000010uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010000001AuLL},
{0x0000001000000010uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x0000000100000010uLL},
{0x0000001000000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000000000028uLL},
{0x0000001000000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000000000020uLL},
{0x0000001000000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000000000018uLL},
{0x0000001000000010uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000000000010uLL},
{0x0000001000000010uLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x0000000000000008uLL},
{0x0000001000000010uLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x0000000200000003uLL, 0x0000000000000007uLL, 0x00000001FFFFFFFBuLL},
{0x0000001000000010uLL, 0x0000000200000010uLL, 0x0000000000000007uLL, 0x00000001FFFFFFA0uLL},
{0x0000001000000010uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001CB541D3AuLL},
{0x0000001000000010uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001CD0A3D84uLL},
{0x0000001000000010uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000100000016uLL},
{0x0000001000000010uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000100000010uLL},
{0x0000001000000010uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000010000001FuLL},
{0x0000001000000010uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010000001AuLL},
{0x0000001000000010uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000100000015uLL},
{0x0000001000000010uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000100000010uLL},
{0x0000001000000010uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x000000010000000BuLL},
{0x0000001000000010uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x0000000100000006uLL},
{0x0000001000000010uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x0000000100000001uLL},
{0x0000001000000010uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000000FFFFFFC0uLL},
{0x0000001000000010uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000DA60A72EuLL},
{0x0000001000000010uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000DB9999ACuLL},
{0x0000001000000010uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000200000014uLL},
{0x0000001000000010uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000200000010uLL},
{0x0000001000000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000000001CuLL},
{0x0000001000000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000000000018uLL},
{0x0000001000000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000000000014uLL},
{0x0000001000000010uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000000000010uLL},
{0x0000001000000010uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x000000000000000FuLL},
{0x0000001000000010uLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x000000000000000EuLL},
{0x0000001000000010uLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x000000000000000DuLL},
{0x0000001000000010uLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001000000010uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x0000001000000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001000000010uLL},
{0x00000010078644FAuLL, 0x0000000000000001uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL},
{0x00000010078644FAuLL, 0x0000000000000002uLL, 0x0000000803C3227DuLL, 0x0000000000000000uLL},
{0x00000010078644FAuLL, 0x0000000000000003uLL, 0x0000000557D76C53uLL, 0x0000000000000001uLL},
{0x00000010078644FAuLL, 0x0000000000000010uLL, 0x000000010078644FuLL, 0x000000000000000AuLL},
{0x00000010078644FAuLL, 0x00000000078644FAuLL, 0x0000000000000221uLL, 0x0000000002AD6CC0uLL},
{0x00000010078644FAuLL, 0x000000000747AE14uLL, 0x0000000000000233uLL, 0x0000000004E26EFEuLL},
{0x00000010078644FAuLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x000000000786451AuLL},
{0x00000010078644FAuLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x00000000078644FAuLL},
{0x00000010078644FAuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000000786452AuLL},
{0x00000010078644FAuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000000786451AuLL},
{0x00000010078644FAuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000000786450AuLL},
{0x00000010078644FAuLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x00000000078644FAuLL},
{0x00000010078644FAuLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x00000000078644EAuLL},
{0x00000010078644FAuLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x00000000078644DAuLL},
{0x00000010078644FAuLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x00000000078644CAuLL},
{0x00000010078644FAuLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x00000000078643FAuLL},
{0x00000010078644FAuLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x0000000096A83A54uLL},
{0x00000010078644FAuLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x000000009A5311CEuLL},
{0x00000010078644FAuLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x0000000107864504uLL},
{0x00000010078644FAuLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x00000001078644FAuLL},
{0x00000010078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000007864512uLL},
{0x00000010078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000000786450AuLL},
{0x00000010078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000007864502uLL},
{0x00000010078644FAuLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x00000000078644FAuLL},
{0x00000010078644FAuLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x00000000078644F2uLL},
{0x00000010078644FAuLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x00000000078644EAuLL},
{0x00000010078644FAuLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x00000000078644E2uLL},
{0x00000010078644FAuLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x000000000786447AuLL},
{0x00000010078644FAuLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001D2DA6224uLL},
{0x00000010078644FAuLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001D490826EuLL},
{0x00000010078644FAuLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000107864500uLL},
{0x00000010078644FAuLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x00000001078644FAuLL},
{0x00000010078644FAuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x0000000107864509uLL},
{0x00000010078644FAuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x0000000107864504uLL},
{0x00000010078644FAuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x00000001078644FFuLL},
{0x00000010078644FAuLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x00000001078644FAuLL},
{0x00000010078644FAuLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000001078644F5uLL},
{0x00000010078644FAuLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000001078644F0uLL},
{0x00000010078644FAuLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000001078644EBuLL},
{0x00000010078644FAuLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000001078644AAuLL},
{0x00000010078644FAuLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000E1E6EC18uLL},
{0x00000010078644FAuLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000E31FDE96uLL},
{0x00000010078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x00000002078644FEuLL},
{0x00000010078644FAuLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x00000002078644FAuLL},
{0x00000010078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x0000000007864506uLL},
{0x00000010078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000007864502uLL},
{0x00000010078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x00000000078644FEuLL},
{0x00000010078644FAuLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x00000000078644FAuLL},
{0x00000010078644FAuLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x00000000078644F9uLL},
{0x00000010078644FAuLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x00000000078644F8uLL},
{0x00000010078644FAuLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x00000000078644F7uLL},
{0x00000010078644FAuLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x00000000078644EAuLL},
{0x00000010078644FAuLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000010078644FAuLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x00000010078644FAuLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x00000010078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010078644FAuLL},
{0x000000100747AE14uLL, 0x0000000000000001uLL, 0x000000100747AE14uLL, 0x0000000000000000uLL},
{0x000000100747AE14uLL, 0x0000000000000002uLL, 0x0000000803A3D70AuLL, 0x0000000000000000uLL},
{0x000000100747AE14uLL, 0x0000000000000003uLL, 0x0000000557C28F5CuLL, 0x0000000000000000uLL},
{0x000000100747AE14uLL, 0x0000000000000010uLL, 0x0000000100747AE1uLL, 0x0000000000000004uLL},
{0x000000100747AE14uLL, 0x00000000078644FAuLL, 0x0000000000000221uLL, 0x00000000026ED5DAuLL},
{0x000000100747AE14uLL, 0x000000000747AE14uLL, 0x0000000000000233uLL, 0x0000000004A3D818uLL},
{0x000000100747AE14uLL, 0x000000007FFFFFFFuLL, 0x0000000000000020uLL, 0x000000000747AE34uLL},
{0x000000100747AE14uLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x000000000747AE14uLL},
{0x000000100747AE14uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000000747AE44uLL},
{0x000000100747AE14uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000000747AE34uLL},
{0x000000100747AE14uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000000747AE24uLL},
{0x000000100747AE14uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x000000000747AE14uLL},
{0x000000100747AE14uLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x000000000747AE04uLL},
{0x000000100747AE14uLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x000000000747ADF4uLL},
{0x000000100747AE14uLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x000000000747ADE4uLL},
{0x000000100747AE14uLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x000000000747AD14uLL},
{0x000000100747AE14uLL, 0x00000001078644FAuLL, 0x000000000000000FuLL, 0x000000009669A36EuLL},
{0x000000100747AE14uLL, 0x000000010747AE14uLL, 0x000000000000000FuLL, 0x000000009A147AE8uLL},
{0x000000100747AE14uLL, 0x000000017FFFFFFFuLL, 0x000000000000000AuLL, 0x000000010747AE1EuLL},
{0x000000100747AE14uLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x000000010747AE14uLL},
{0x000000100747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x000000000747AE2CuLL},
{0x000000100747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000000747AE24uLL},
{0x000000100747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x000000000747AE1CuLL},
{0x000000100747AE14uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x000000000747AE14uLL},
{0x000000100747AE14uLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x000000000747AE0CuLL},
{0x000000100747AE14uLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x000000000747AE04uLL},
{0x000000100747AE14uLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x000000000747ADFCuLL},
{0x000000100747AE14uLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x000000000747AD94uLL},
{0x000000100747AE14uLL, 0x00000002078644FAuLL, 0x0000000000000007uLL, 0x00000001D29BCB3EuLL},
{0x000000100747AE14uLL, 0x000000020747AE14uLL, 0x0000000000000007uLL, 0x00000001D451EB88uLL},
{0x000000100747AE14uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x000000010747AE1AuLL},
{0x000000100747AE14uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x000000010747AE14uLL},
{0x000000100747AE14uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000010747AE23uLL},
{0x000000100747AE14uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000010747AE1EuLL},
{0x000000100747AE14uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x000000010747AE19uLL},
{0x000000100747AE14uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x000000010747AE14uLL},
{0x000000100747AE14uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x000000010747AE0FuLL},
{0x000000100747AE14uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x000000010747AE0AuLL},
{0x000000100747AE14uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x000000010747AE05uLL},
{0x000000100747AE14uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x000000010747ADC4uLL},
{0x000000100747AE14uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000000E1A85532uLL},
{0x000000100747AE14uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000000E2E147B0uLL},
{0x000000100747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x000000020747AE18uLL},
{0x000000100747AE14uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x000000020747AE14uLL},
{0x000000100747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000000747AE20uLL},
{0x000000100747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x000000000747AE1CuLL},
{0x000000100747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x000000000747AE18uLL},
{0x000000100747AE14uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL},
{0x000000100747AE14uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x000000000747AE13uLL},
{0x000000100747AE14uLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x000000000747AE12uLL},
{0x000000100747AE14uLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x000000000747AE11uLL},
{0x000000100747AE14uLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x000000000747AE04uLL},
{0x000000100747AE14uLL, 0x00000010078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000100747AE14uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000100747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000100747AE14uLL},
{0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x000000107FFFFFFFuLL, 0x0000000000000000uLL},
{0x000000107FFFFFFFuLL, 0x0000000000000002uLL, 0x000000083FFFFFFFuLL, 0x0000000000000001uLL},
{0x000000107FFFFFFFuLL, 0x0000000000000003uLL, 0x000000057FFFFFFFuLL, 0x0000000000000002uLL},
{0x000000107FFFFFFFuLL, 0x0000000000000010uLL, 0x0000000107FFFFFFuLL, 0x000000000000000FuLL},
{0x000000107FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000231uLL, 0x0000000002C2D825uLL},
{0x000000107FFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000244uLL, 0x0000000001999AAFuLL},
{0x000000107FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000021uLL, 0x0000000000000020uLL},
{0x000000107FFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000020uLL, 0x000000007FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000008000002FuLL},
{0x000000107FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000008000001FuLL},
{0x000000107FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000008000000FuLL},
{0x000000107FFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x000000007FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x000000007FFFFFEFuLL},
{0x000000107FFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x000000007FFFFFDFuLL},
{0x000000107FFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x000000007FFFFFCFuLL},
{0x000000107FFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x000000007FFFFEFFuLL},
{0x000000107FFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000010uLL, 0x00000000079BB05FuLL},
{0x000000107FFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000010uLL, 0x000000000B851EBFuLL},
{0x000000107FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x000000000000000BuLL, 0x000000000000000AuLL},
{0x000000107FFFFFFFuLL, 0x0000000180000000uLL, 0x000000000000000AuLL, 0x000000017FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000080000017uLL},
{0x000000107FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000008000000FuLL},
{0x000000107FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000080000007uLL},
{0x000000107FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x000000007FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x000000007FFFFFF7uLL},
{0x000000107FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x000000007FFFFFEFuLL},
{0x000000107FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x000000007FFFFFE7uLL},
{0x000000107FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x000000007FFFFF7FuLL},
{0x000000107FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000008uLL, 0x0000000043CDD82FuLL},
{0x000000107FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000008uLL, 0x0000000045C28F5FuLL},
{0x000000107FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000180000005uLL},
{0x000000107FFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x000000017FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000018000000EuLL},
{0x000000107FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x0000000180000009uLL},
{0x000000107FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000180000004uLL},
{0x000000107FFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x000000017FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x000000017FFFFFFAuLL},
{0x000000107FFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x000000017FFFFFF5uLL},
{0x000000107FFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x000000017FFFFFF0uLL},
{0x000000107FFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x000000017FFFFFAFuLL},
{0x000000107FFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x000000015A60A71DuLL},
{0x000000107FFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x000000015B99999BuLL},
{0x000000107FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000280000003uLL},
{0x000000107FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x000000027FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000008000000BuLL},
{0x000000107FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000080000007uLL},
{0x000000107FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000080000003uLL},
{0x000000107FFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x000000107FFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x000000107FFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFCuLL},
{0x000000107FFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFEFuLL},
{0x000000107FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB05uLL},
{0x000000107FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851EBuLL},
{0x000000107FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x000000107FFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x000000107FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x000000107FFFFFFFuLL},
{0x0000001080000000uLL, 0x0000000000000001uLL, 0x0000001080000000uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x0000000000000002uLL, 0x0000000840000000uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x0000000000000003uLL, 0x0000000580000000uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x0000000000000010uLL, 0x0000000108000000uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x00000000078644FAuLL, 0x0000000000000231uLL, 0x0000000002C2D826uLL},
{0x0000001080000000uLL, 0x000000000747AE14uLL, 0x0000000000000244uLL, 0x0000000001999AB0uLL},
{0x0000001080000000uLL, 0x000000007FFFFFFFuLL, 0x0000000000000021uLL, 0x0000000000000021uLL},
{0x0000001080000000uLL, 0x0000000080000000uLL, 0x0000000000000021uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x00000000FFFFFFFDuLL, 0x0000000000000010uLL, 0x0000000080000030uLL},
{0x0000001080000000uLL, 0x00000000FFFFFFFEuLL, 0x0000000000000010uLL, 0x0000000080000020uLL},
{0x0000001080000000uLL, 0x00000000FFFFFFFFuLL, 0x0000000000000010uLL, 0x0000000080000010uLL},
{0x0000001080000000uLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x0000000080000000uLL},
{0x0000001080000000uLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x000000007FFFFFF0uLL},
{0x0000001080000000uLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x000000007FFFFFE0uLL},
{0x0000001080000000uLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x000000007FFFFFD0uLL},
{0x0000001080000000uLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x000000007FFFFF00uLL},
{0x0000001080000000uLL, 0x00000001078644FAuLL, 0x0000000000000010uLL, 0x00000000079BB060uLL},
{0x0000001080000000uLL, 0x000000010747AE14uLL, 0x0000000000000010uLL, 0x000000000B851EC0uLL},
{0x0000001080000000uLL, 0x000000017FFFFFFFuLL, 0x000000000000000BuLL, 0x000000000000000BuLL},
{0x0000001080000000uLL, 0x0000000180000000uLL, 0x000000000000000BuLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000080000018uLL},
{0x0000001080000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x0000000080000010uLL},
{0x0000001080000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000080000008uLL},
{0x0000001080000000uLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x0000000080000000uLL},
{0x0000001080000000uLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x000000007FFFFFF8uLL},
{0x0000001080000000uLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x000000007FFFFFF0uLL},
{0x0000001080000000uLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x000000007FFFFFE8uLL},
{0x0000001080000000uLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x000000007FFFFF80uLL},
{0x0000001080000000uLL, 0x00000002078644FAuLL, 0x0000000000000008uLL, 0x0000000043CDD830uLL},
{0x0000001080000000uLL, 0x000000020747AE14uLL, 0x0000000000000008uLL, 0x0000000045C28F60uLL},
{0x0000001080000000uLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000180000006uLL},
{0x0000001080000000uLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x0000000180000000uLL},
{0x0000001080000000uLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000018000000FuLL},
{0x0000001080000000uLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x000000018000000AuLL},
{0x0000001080000000uLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000180000005uLL},
{0x0000001080000000uLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x0000000180000000uLL},
{0x0000001080000000uLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x000000017FFFFFFBuLL},
{0x0000001080000000uLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x000000017FFFFFF6uLL},
{0x0000001080000000uLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x000000017FFFFFF1uLL},
{0x0000001080000000uLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x000000017FFFFFB0uLL},
{0x0000001080000000uLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x000000015A60A71EuLL},
{0x0000001080000000uLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x000000015B99999CuLL},
{0x0000001080000000uLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000280000004uLL},
{0x0000001080000000uLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x0000000280000000uLL},
{0x0000001080000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000008000000CuLL},
{0x0000001080000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000080000008uLL},
{0x0000001080000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000080000004uLL},
{0x0000001080000000uLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x0000001080000000uLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x0000001080000000uLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x0000001080000000uLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x0000001080000000uLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFF0uLL},
{0x0000001080000000uLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB06uLL},
{0x0000001080000000uLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851ECuLL},
{0x0000001080000000uLL, 0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0000001080000000uLL, 0x0000001080000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0000001080000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x0000001080000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0000001080000000uLL},
{0x00000010FFFFFFFDuLL, 0x0000000000000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFDuLL, 0x0000000000000002uLL, 0x000000087FFFFFFEuLL, 0x0000000000000001uLL},
{0x00000010FFFFFFFDuLL, 0x0000000000000003uLL, 0x00000005AAAAAAA9uLL, 0x0000000000000002uLL},
{0x00000010FFFFFFFDuLL, 0x0000000000000010uLL, 0x000000010FFFFFFFuLL, 0x000000000000000DuLL},
{0x00000010FFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000000000242uLL, 0x0000000002D84389uLL},
{0x00000010FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000000000255uLL, 0x0000000005D70B59uLL},
{0x00000010FFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x0000000000000022uLL, 0x000000000000001FuLL},
{0x00000010FFFFFFFDuLL, 0x0000000080000000uLL, 0x0000000000000021uLL, 0x000000007FFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000011uLL, 0x0000000000000030uLL},
{0x00000010FFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000011uLL, 0x000000000000001FuLL},
{0x00000010FFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000011uLL, 0x000000000000000EuLL},
{0x00000010FFFFFFFDuLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x00000000FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x00000000FFFFFFEDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x00000000FFFFFFDDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x00000000FFFFFFCDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x00000000FFFFFEFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000001078644FAuLL, 0x0000000000000010uLL, 0x00000000879BB05DuLL},
{0x00000010FFFFFFFDuLL, 0x000000010747AE14uLL, 0x0000000000000010uLL, 0x000000008B851EBDuLL},
{0x00000010FFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x000000000000000BuLL, 0x0000000080000008uLL},
{0x00000010FFFFFFFDuLL, 0x0000000180000000uLL, 0x000000000000000BuLL, 0x000000007FFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000100000015uLL},
{0x00000010FFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000010000000DuLL},
{0x00000010FFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000100000005uLL},
{0x00000010FFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x00000000FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x00000000FFFFFFF5uLL},
{0x00000010FFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x00000000FFFFFFEDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x00000000FFFFFFE5uLL},
{0x00000010FFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x00000000FFFFFF7DuLL},
{0x00000010FFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000000000008uLL, 0x00000000C3CDD82DuLL},
{0x00000010FFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000000000008uLL, 0x00000000C5C28F5DuLL},
{0x00000010FFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000200000003uLL},
{0x00000010FFFFFFFDuLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x00000001FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000020000000CuLL},
{0x00000010FFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x0000000200000007uLL},
{0x00000010FFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000200000002uLL},
{0x00000010FFFFFFFDuLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x00000001FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF8uLL},
{0x00000010FFFFFFFDuLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF3uLL},
{0x00000010FFFFFFFDuLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000001FFFFFFEEuLL},
{0x00000010FFFFFFFDuLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000001FFFFFFADuLL},
{0x00000010FFFFFFFDuLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000001DA60A71BuLL},
{0x00000010FFFFFFFDuLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000001DB999999uLL},
{0x00000010FFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000300000001uLL},
{0x00000010FFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x00000002FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x0000000100000009uLL},
{0x00000010FFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000100000005uLL},
{0x00000010FFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000100000001uLL},
{0x00000010FFFFFFFDuLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000010FFFFFFFDuLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000010FFFFFFFDuLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFAuLL},
{0x00000010FFFFFFFDuLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEDuLL},
{0x00000010FFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB03uLL},
{0x00000010FFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851E9uLL},
{0x00000010FFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000010FFFFFFFDuLL, 0x0000001080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFDuLL},
{0x00000010FFFFFFFEuLL, 0x0000000000000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFEuLL, 0x0000000000000002uLL, 0x000000087FFFFFFFuLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFEuLL, 0x0000000000000003uLL, 0x00000005AAAAAAAAuLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFEuLL, 0x0000000000000010uLL, 0x000000010FFFFFFFuLL, 0x000000000000000EuLL},
{0x00000010FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000000000242uLL, 0x0000000002D8438AuLL},
{0x00000010FFFFFFFEuLL, 0x000000000747AE14uLL, 0x0000000000000255uLL, 0x0000000005D70B5AuLL},
{0x00000010FFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x0000000000000022uLL, 0x0000000000000020uLL},
{0x00000010FFFFFFFEuLL, 0x0000000080000000uLL, 0x0000000000000021uLL, 0x000000007FFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000011uLL, 0x0000000000000031uLL},
{0x00000010FFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000011uLL, 0x0000000000000020uLL},
{0x00000010FFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000011uLL, 0x000000000000000FuLL},
{0x00000010FFFFFFFEuLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x00000000FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x00000000FFFFFFEEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x00000000FFFFFFDEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x00000000FFFFFFCEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x00000000FFFFFEFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000001078644FAuLL, 0x0000000000000010uLL, 0x00000000879BB05EuLL},
{0x00000010FFFFFFFEuLL, 0x000000010747AE14uLL, 0x0000000000000010uLL, 0x000000008B851EBEuLL},
{0x00000010FFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x000000000000000BuLL, 0x0000000080000009uLL},
{0x00000010FFFFFFFEuLL, 0x0000000180000000uLL, 0x000000000000000BuLL, 0x000000007FFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000100000016uLL},
{0x00000010FFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000010000000EuLL},
{0x00000010FFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000100000006uLL},
{0x00000010FFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x00000000FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x00000000FFFFFFF6uLL},
{0x00000010FFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x00000000FFFFFFEEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x00000000FFFFFFE6uLL},
{0x00000010FFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x00000000FFFFFF7EuLL},
{0x00000010FFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000000000008uLL, 0x00000000C3CDD82EuLL},
{0x00000010FFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000000000008uLL, 0x00000000C5C28F5EuLL},
{0x00000010FFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000200000004uLL},
{0x00000010FFFFFFFEuLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x00000001FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000020000000DuLL},
{0x00000010FFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x0000000200000008uLL},
{0x00000010FFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000200000003uLL},
{0x00000010FFFFFFFEuLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x00000001FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF9uLL},
{0x00000010FFFFFFFEuLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF4uLL},
{0x00000010FFFFFFFEuLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000001FFFFFFEFuLL},
{0x00000010FFFFFFFEuLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000001FFFFFFAEuLL},
{0x00000010FFFFFFFEuLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000001DA60A71CuLL},
{0x00000010FFFFFFFEuLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000001DB99999AuLL},
{0x00000010FFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000300000002uLL},
{0x00000010FFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x00000002FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000010000000AuLL},
{0x00000010FFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000100000006uLL},
{0x00000010FFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000100000002uLL},
{0x00000010FFFFFFFEuLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000010FFFFFFFEuLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000010FFFFFFFEuLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x00000010FFFFFFFEuLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEEuLL},
{0x00000010FFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB04uLL},
{0x00000010FFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EAuLL},
{0x00000010FFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000010FFFFFFFEuLL, 0x0000001080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000010FFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFEuLL},
{0x00000010FFFFFFFFuLL, 0x0000000000000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFFuLL, 0x0000000000000002uLL, 0x000000087FFFFFFFuLL, 0x0000000000000001uLL},
{0x00000010FFFFFFFFuLL, 0x0000000000000003uLL, 0x00000005AAAAAAAAuLL, 0x0000000000000001uLL},
{0x00000010FFFFFFFFuLL, 0x0000000000000010uLL, 0x000000010FFFFFFFuLL, 0x000000000000000FuLL},
{0x00000010FFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000000000242uLL, 0x0000000002D8438BuLL},
{0x00000010FFFFFFFFuLL, 0x000000000747AE14uLL, 0x0000000000000255uLL, 0x0000000005D70B5BuLL},
{0x00000010FFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x0000000000000022uLL, 0x0000000000000021uLL},
{0x00000010FFFFFFFFuLL, 0x0000000080000000uLL, 0x0000000000000021uLL, 0x000000007FFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x0000000000000011uLL, 0x0000000000000032uLL},
{0x00000010FFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x0000000000000011uLL, 0x0000000000000021uLL},
{0x00000010FFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x0000000000000011uLL, 0x0000000000000010uLL},
{0x00000010FFFFFFFFuLL, 0x0000000100000000uLL, 0x0000000000000010uLL, 0x00000000FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000100000001uLL, 0x0000000000000010uLL, 0x00000000FFFFFFEFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000100000002uLL, 0x0000000000000010uLL, 0x00000000FFFFFFDFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000100000003uLL, 0x0000000000000010uLL, 0x00000000FFFFFFCFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000100000010uLL, 0x0000000000000010uLL, 0x00000000FFFFFEFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000001078644FAuLL, 0x0000000000000010uLL, 0x00000000879BB05FuLL},
{0x00000010FFFFFFFFuLL, 0x000000010747AE14uLL, 0x0000000000000010uLL, 0x000000008B851EBFuLL},
{0x00000010FFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x000000000000000BuLL, 0x000000008000000AuLL},
{0x00000010FFFFFFFFuLL, 0x0000000180000000uLL, 0x000000000000000BuLL, 0x000000007FFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000000000008uLL, 0x0000000100000017uLL},
{0x00000010FFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000000000008uLL, 0x000000010000000FuLL},
{0x00000010FFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000000000008uLL, 0x0000000100000007uLL},
{0x00000010FFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000000000008uLL, 0x00000000FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000000000008uLL, 0x00000000FFFFFFF7uLL},
{0x00000010FFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000000000008uLL, 0x00000000FFFFFFEFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000000000008uLL, 0x00000000FFFFFFE7uLL},
{0x00000010FFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000000000008uLL, 0x00000000FFFFFF7FuLL},
{0x00000010FFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000000000008uLL, 0x00000000C3CDD82FuLL},
{0x00000010FFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000000000008uLL, 0x00000000C5C28F5FuLL},
{0x00000010FFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x0000000000000006uLL, 0x0000000200000005uLL},
{0x00000010FFFFFFFFuLL, 0x0000000280000000uLL, 0x0000000000000006uLL, 0x00000001FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x0000000000000005uLL, 0x000000020000000EuLL},
{0x00000010FFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x0000000000000005uLL, 0x0000000200000009uLL},
{0x00000010FFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x0000000000000005uLL, 0x0000000200000004uLL},
{0x00000010FFFFFFFFuLL, 0x0000000300000000uLL, 0x0000000000000005uLL, 0x00000001FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0000000300000001uLL, 0x0000000000000005uLL, 0x00000001FFFFFFFAuLL},
{0x00000010FFFFFFFFuLL, 0x0000000300000002uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF5uLL},
{0x00000010FFFFFFFFuLL, 0x0000000300000003uLL, 0x0000000000000005uLL, 0x00000001FFFFFFF0uLL},
{0x00000010FFFFFFFFuLL, 0x0000000300000010uLL, 0x0000000000000005uLL, 0x00000001FFFFFFAFuLL},
{0x00000010FFFFFFFFuLL, 0x00000003078644FAuLL, 0x0000000000000005uLL, 0x00000001DA60A71DuLL},
{0x00000010FFFFFFFFuLL, 0x000000030747AE14uLL, 0x0000000000000005uLL, 0x00000001DB99999BuLL},
{0x00000010FFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000000000004uLL, 0x0000000300000003uLL},
{0x00000010FFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000000000004uLL, 0x00000002FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000000000004uLL, 0x000000010000000BuLL},
{0x00000010FFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000000000004uLL, 0x0000000100000007uLL},
{0x00000010FFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000000000004uLL, 0x0000000100000003uLL},
{0x00000010FFFFFFFFuLL, 0x0000001000000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0000001000000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x00000010FFFFFFFFuLL, 0x0000001000000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x00000010FFFFFFFFuLL, 0x0000001000000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x00000010FFFFFFFFuLL, 0x0000001000000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEFuLL},
{0x00000010FFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB05uLL},
{0x00000010FFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EBuLL},
{0x00000010FFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x00000010FFFFFFFFuLL, 0x0000001080000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x00000010FFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x00000010FFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x00000010FFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x00000010FFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x00000010FFFFFFFFuLL},
{0x078644FA00000000uLL, 0x0000000000000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000000000002uLL, 0x03C3227D00000000uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000000000003uLL, 0x028216FE00000000uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000000000010uLL, 0x0078644FA0000000uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x000000000747AE14uLL, 0x000000010898F295uLL, 0x0000000004D8C65CuLL},
{0x078644FA00000000uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000000F0C89F4uLL},
{0x078644FA00000000uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001692CEEEuLL},
{0x078644FA00000000uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000000F0C89F4uLL},
{0x078644FA00000000uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000078644FAuLL},
{0x078644FA00000000uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000F879BB07uLL},
{0x078644FA00000000uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F0F3760EuLL},
{0x078644FA00000000uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000E96D3115uLL},
{0x078644FA00000000uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x00000000879BB070uLL},
{0x078644FA00000000uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x000000006C09ACD2uLL},
{0x078644FA00000000uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000082B63E28uLL},
{0x078644FA00000000uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000005042DFCuLL},
{0x078644FA00000000uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000000B496777uLL},
{0x078644FA00000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000078644FAuLL},
{0x078644FA00000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000003C3227DuLL},
{0x078644FA00000000uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000200000001uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD84uLL},
{0x078644FA00000000uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001F879BB08uLL},
{0x078644FA00000000uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001F4B6988CuLL},
{0x078644FA00000000uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001C3CDD840uLL},
{0x078644FA00000000uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000181754582uLL},
{0x078644FA00000000uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x000000019B4CF184uLL},
{0x078644FA00000000uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x00000001830281FDuLL},
{0x078644FA00000000uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000180000000uLL},
{0x078644FA00000000uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000078644FAuLL},
{0x078644FA00000000uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000005042DFCuLL},
{0x078644FA00000000uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000000028216FEuLL},
{0x078644FA00000000uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x0000000300000001uLL, 0x00000000028216FDuLL, 0x00000002FD7DE903uLL},
{0x078644FA00000000uLL, 0x0000000300000002uLL, 0x00000000028216FDuLL, 0x00000002FAFBD206uLL},
{0x078644FA00000000uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002F879BB09uLL},
{0x078644FA00000000uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002D7DE9030uLL},
{0x078644FA00000000uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x0000000266ACEAE4uLL},
{0x078644FA00000000uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000CC648250uLL},
{0x078644FA00000000uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000282265CD9uLL},
{0x078644FA00000000uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000280000000uLL},
{0x078644FA00000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000205A4B3BAuLL},
{0x078644FA00000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000203C3227CuLL},
{0x078644FA00000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000201E1913EuLL},
{0x078644FA00000000uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A00000000uLL},
{0x078644FA00000000uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x00000009FF879BB1uLL},
{0x078644FA00000000uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x00000009FF0F3762uLL},
{0x078644FA00000000uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x00000009FE96D313uLL},
{0x078644FA00000000uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009F879BB10uLL},
{0x078644FA00000000uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000001E3B0AC8uLL},
{0x078644FA00000000uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000438F2016CuLL},
{0x078644FA00000000uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C0074BE5CuLL},
{0x078644FA00000000uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C00000000uLL},
{0x078644FA00000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000000153EE0EuLL},
{0x078644FA00000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000000E29EB4uLL},
{0x078644FA00000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000000714F5AuLL},
{0x078644FA00000000uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA00000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA00000000uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FA00000000uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFEuLL},
{0x078644FA00000000uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFDuLL},
{0x078644FA00000000uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFF0uLL},
{0x078644FA00000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5F879BB06uLL},
{0x078644FA00000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E5F8B851ECuLL},
{0x078644FA00000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000001uLL},
{0x078644FA00000000uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E580000000uLL},
{0x078644FA00000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E500000003uLL},
{0x078644FA00000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E500000002uLL},
{0x078644FA00000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E500000001uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000000uLL},
{0x078644FA00000001uLL, 0x0000000000000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL},
{0x078644FA00000001uLL, 0x0000000000000002uLL, 0x03C3227D00000000uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x0000000000000003uLL, 0x028216FE00000000uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x0000000000000010uLL, 0x0078644FA0000000uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x000000000747AE14uLL, 0x000000010898F295uLL, 0x0000000004D8C65DuLL},
{0x078644FA00000001uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000000F0C89F5uLL},
{0x078644FA00000001uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001692CEEFuLL},
{0x078644FA00000001uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000000F0C89F5uLL},
{0x078644FA00000001uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000078644FBuLL},
{0x078644FA00000001uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000F879BB08uLL},
{0x078644FA00000001uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F0F3760FuLL},
{0x078644FA00000001uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000E96D3116uLL},
{0x078644FA00000001uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x00000000879BB071uLL},
{0x078644FA00000001uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x000000006C09ACD3uLL},
{0x078644FA00000001uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000082B63E29uLL},
{0x078644FA00000001uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000005042DFDuLL},
{0x078644FA00000001uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000000B496778uLL},
{0x078644FA00000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000078644FBuLL},
{0x078644FA00000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000003C3227EuLL},
{0x078644FA00000001uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x0000000200000001uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD85uLL},
{0x078644FA00000001uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001F879BB09uLL},
{0x078644FA00000001uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001F4B6988DuLL},
{0x078644FA00000001uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001C3CDD841uLL},
{0x078644FA00000001uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000181754583uLL},
{0x078644FA00000001uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x000000019B4CF185uLL},
{0x078644FA00000001uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x00000001830281FEuLL},
{0x078644FA00000001uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000180000001uLL},
{0x078644FA00000001uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000078644FBuLL},
{0x078644FA00000001uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000005042DFDuLL},
{0x078644FA00000001uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000000028216FFuLL},
{0x078644FA00000001uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x0000000300000001uLL, 0x00000000028216FDuLL, 0x00000002FD7DE904uLL},
{0x078644FA00000001uLL, 0x0000000300000002uLL, 0x00000000028216FDuLL, 0x00000002FAFBD207uLL},
{0x078644FA00000001uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002F879BB0AuLL},
{0x078644FA00000001uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002D7DE9031uLL},
{0x078644FA00000001uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x0000000266ACEAE5uLL},
{0x078644FA00000001uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000CC648251uLL},
{0x078644FA00000001uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000282265CDAuLL},
{0x078644FA00000001uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000280000001uLL},
{0x078644FA00000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000205A4B3BBuLL},
{0x078644FA00000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000203C3227DuLL},
{0x078644FA00000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000201E1913FuLL},
{0x078644FA00000001uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A00000001uLL},
{0x078644FA00000001uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x00000009FF879BB2uLL},
{0x078644FA00000001uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x00000009FF0F3763uLL},
{0x078644FA00000001uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x00000009FE96D314uLL},
{0x078644FA00000001uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009F879BB11uLL},
{0x078644FA00000001uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000001E3B0AC9uLL},
{0x078644FA00000001uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000438F2016DuLL},
{0x078644FA00000001uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C0074BE5DuLL},
{0x078644FA00000001uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C00000001uLL},
{0x078644FA00000001uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000000153EE0FuLL},
{0x078644FA00000001uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000000E29EB5uLL},
{0x078644FA00000001uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000000714F5BuLL},
{0x078644FA00000001uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FA00000001uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA00000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FA00000001uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA00000001uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FA00000001uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFEuLL},
{0x078644FA00000001uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFF1uLL},
{0x078644FA00000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5F879BB07uLL},
{0x078644FA00000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E5F8B851EDuLL},
{0x078644FA00000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000002uLL},
{0x078644FA00000001uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E580000001uLL},
{0x078644FA00000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E500000004uLL},
{0x078644FA00000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E500000003uLL},
{0x078644FA00000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E500000002uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000001uLL},
{0x078644FA00000002uLL, 0x0000000000000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL},
{0x078644FA00000002uLL, 0x0000000000000002uLL, 0x03C3227D00000001uLL, 0x0000000000000000uLL},
{0x078644FA00000002uLL, 0x0000000000000003uLL, 0x028216FE00000000uLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x0000000000000010uLL, 0x0078644FA0000000uLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x000000000747AE14uLL, 0x000000010898F295uLL, 0x0000000004D8C65EuLL},
{0x078644FA00000002uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000000F0C89F6uLL},
{0x078644FA00000002uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001692CEF0uLL},
{0x078644FA00000002uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000000F0C89F6uLL},
{0x078644FA00000002uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000078644FCuLL},
{0x078644FA00000002uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000F879BB09uLL},
{0x078644FA00000002uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F0F37610uLL},
{0x078644FA00000002uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000E96D3117uLL},
{0x078644FA00000002uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x00000000879BB072uLL},
{0x078644FA00000002uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x000000006C09ACD4uLL},
{0x078644FA00000002uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000082B63E2AuLL},
{0x078644FA00000002uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000005042DFEuLL},
{0x078644FA00000002uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000000B496779uLL},
{0x078644FA00000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000078644FCuLL},
{0x078644FA00000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000003C3227FuLL},
{0x078644FA00000002uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x0000000200000001uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD86uLL},
{0x078644FA00000002uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001F879BB0AuLL},
{0x078644FA00000002uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001F4B6988EuLL},
{0x078644FA00000002uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001C3CDD842uLL},
{0x078644FA00000002uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000181754584uLL},
{0x078644FA00000002uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x000000019B4CF186uLL},
{0x078644FA00000002uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x00000001830281FFuLL},
{0x078644FA00000002uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000180000002uLL},
{0x078644FA00000002uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000078644FCuLL},
{0x078644FA00000002uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000005042DFEuLL},
{0x078644FA00000002uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x0000000002821700uLL},
{0x078644FA00000002uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x0000000300000001uLL, 0x00000000028216FDuLL, 0x00000002FD7DE905uLL},
{0x078644FA00000002uLL, 0x0000000300000002uLL, 0x00000000028216FDuLL, 0x00000002FAFBD208uLL},
{0x078644FA00000002uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002F879BB0BuLL},
{0x078644FA00000002uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002D7DE9032uLL},
{0x078644FA00000002uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x0000000266ACEAE6uLL},
{0x078644FA00000002uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000CC648252uLL},
{0x078644FA00000002uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000282265CDBuLL},
{0x078644FA00000002uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000280000002uLL},
{0x078644FA00000002uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000205A4B3BCuLL},
{0x078644FA00000002uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000203C3227EuLL},
{0x078644FA00000002uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000201E19140uLL},
{0x078644FA00000002uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A00000002uLL},
{0x078644FA00000002uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x00000009FF879BB3uLL},
{0x078644FA00000002uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x00000009FF0F3764uLL},
{0x078644FA00000002uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x00000009FE96D315uLL},
{0x078644FA00000002uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009F879BB12uLL},
{0x078644FA00000002uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000001E3B0ACAuLL},
{0x078644FA00000002uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000438F2016EuLL},
{0x078644FA00000002uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C0074BE5EuLL},
{0x078644FA00000002uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C00000002uLL},
{0x078644FA00000002uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000000153EE10uLL},
{0x078644FA00000002uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000000E29EB6uLL},
{0x078644FA00000002uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000000714F5CuLL},
{0x078644FA00000002uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x078644FA00000002uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FA00000002uLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA00000002uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E600000002uLL},
{0x078644FA00000002uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FA00000002uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA00000002uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FA00000002uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFF2uLL},
{0x078644FA00000002uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5F879BB08uLL},
{0x078644FA00000002uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E5F8B851EEuLL},
{0x078644FA00000002uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000003uLL},
{0x078644FA00000002uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E580000002uLL},
{0x078644FA00000002uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E500000005uLL},
{0x078644FA00000002uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E500000004uLL},
{0x078644FA00000002uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E500000003uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000002uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000002uLL},
{0x078644FA00000003uLL, 0x0000000000000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL},
{0x078644FA00000003uLL, 0x0000000000000002uLL, 0x03C3227D00000001uLL, 0x0000000000000001uLL},
{0x078644FA00000003uLL, 0x0000000000000003uLL, 0x028216FE00000001uLL, 0x0000000000000000uLL},
{0x078644FA00000003uLL, 0x0000000000000010uLL, 0x0078644FA0000000uLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x000000000747AE14uLL, 0x000000010898F295uLL, 0x0000000004D8C65FuLL},
{0x078644FA00000003uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000000F0C89F7uLL},
{0x078644FA00000003uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001692CEF1uLL},
{0x078644FA00000003uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000000F0C89F7uLL},
{0x078644FA00000003uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000078644FDuLL},
{0x078644FA00000003uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000F879BB0AuLL},
{0x078644FA00000003uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F0F37611uLL},
{0x078644FA00000003uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000E96D3118uLL},
{0x078644FA00000003uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x00000000879BB073uLL},
{0x078644FA00000003uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x000000006C09ACD5uLL},
{0x078644FA00000003uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000082B63E2BuLL},
{0x078644FA00000003uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000005042DFFuLL},
{0x078644FA00000003uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000000B49677AuLL},
{0x078644FA00000003uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000078644FDuLL},
{0x078644FA00000003uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000003C32280uLL},
{0x078644FA00000003uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x0000000200000001uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD87uLL},
{0x078644FA00000003uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001F879BB0BuLL},
{0x078644FA00000003uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001F4B6988FuLL},
{0x078644FA00000003uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001C3CDD843uLL},
{0x078644FA00000003uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000181754585uLL},
{0x078644FA00000003uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x000000019B4CF187uLL},
{0x078644FA00000003uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x0000000183028200uLL},
{0x078644FA00000003uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000180000003uLL},
{0x078644FA00000003uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000078644FDuLL},
{0x078644FA00000003uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000005042DFFuLL},
{0x078644FA00000003uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x0000000002821701uLL},
{0x078644FA00000003uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x0000000300000001uLL, 0x00000000028216FDuLL, 0x00000002FD7DE906uLL},
{0x078644FA00000003uLL, 0x0000000300000002uLL, 0x00000000028216FDuLL, 0x00000002FAFBD209uLL},
{0x078644FA00000003uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002F879BB0CuLL},
{0x078644FA00000003uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002D7DE9033uLL},
{0x078644FA00000003uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x0000000266ACEAE7uLL},
{0x078644FA00000003uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000CC648253uLL},
{0x078644FA00000003uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000282265CDCuLL},
{0x078644FA00000003uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000280000003uLL},
{0x078644FA00000003uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000205A4B3BDuLL},
{0x078644FA00000003uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000203C3227FuLL},
{0x078644FA00000003uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000201E19141uLL},
{0x078644FA00000003uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A00000003uLL},
{0x078644FA00000003uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x00000009FF879BB4uLL},
{0x078644FA00000003uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x00000009FF0F3765uLL},
{0x078644FA00000003uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x00000009FE96D316uLL},
{0x078644FA00000003uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009F879BB13uLL},
{0x078644FA00000003uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000001E3B0ACBuLL},
{0x078644FA00000003uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000438F2016FuLL},
{0x078644FA00000003uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C0074BE5FuLL},
{0x078644FA00000003uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C00000003uLL},
{0x078644FA00000003uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000000153EE11uLL},
{0x078644FA00000003uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000000E29EB7uLL},
{0x078644FA00000003uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000000714F5DuLL},
{0x078644FA00000003uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000000000003uLL},
{0x078644FA00000003uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x078644FA00000003uLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FA00000003uLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA00000003uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E600000003uLL},
{0x078644FA00000003uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E600000002uLL},
{0x078644FA00000003uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FA00000003uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA00000003uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFF3uLL},
{0x078644FA00000003uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5F879BB09uLL},
{0x078644FA00000003uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E5F8B851EFuLL},
{0x078644FA00000003uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000004uLL},
{0x078644FA00000003uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E580000003uLL},
{0x078644FA00000003uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E500000006uLL},
{0x078644FA00000003uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E500000005uLL},
{0x078644FA00000003uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E500000004uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000003uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000003uLL},
{0x078644FA00000010uLL, 0x0000000000000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL},
{0x078644FA00000010uLL, 0x0000000000000002uLL, 0x03C3227D00000008uLL, 0x0000000000000000uLL},
{0x078644FA00000010uLL, 0x0000000000000003uLL, 0x028216FE00000005uLL, 0x0000000000000001uLL},
{0x078644FA00000010uLL, 0x0000000000000010uLL, 0x0078644FA0000001uLL, 0x0000000000000000uLL},
{0x078644FA00000010uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x000000000747AE14uLL, 0x000000010898F295uLL, 0x0000000004D8C66CuLL},
{0x078644FA00000010uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000000F0C8A04uLL},
{0x078644FA00000010uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001692CEFEuLL},
{0x078644FA00000010uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000000F0C8A04uLL},
{0x078644FA00000010uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x000000000786450AuLL},
{0x078644FA00000010uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000F879BB17uLL},
{0x078644FA00000010uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F0F3761EuLL},
{0x078644FA00000010uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000E96D3125uLL},
{0x078644FA00000010uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x00000000879BB080uLL},
{0x078644FA00000010uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x000000006C09ACE2uLL},
{0x078644FA00000010uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000082B63E38uLL},
{0x078644FA00000010uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000005042E0CuLL},
{0x078644FA00000010uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000000B496787uLL},
{0x078644FA00000010uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x000000000786450AuLL},
{0x078644FA00000010uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000003C3228DuLL},
{0x078644FA00000010uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x0000000200000001uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD94uLL},
{0x078644FA00000010uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001F879BB18uLL},
{0x078644FA00000010uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001F4B6989CuLL},
{0x078644FA00000010uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001C3CDD850uLL},
{0x078644FA00000010uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000181754592uLL},
{0x078644FA00000010uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x000000019B4CF194uLL},
{0x078644FA00000010uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x000000018302820DuLL},
{0x078644FA00000010uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000180000010uLL},
{0x078644FA00000010uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x000000000786450AuLL},
{0x078644FA00000010uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000005042E0CuLL},
{0x078644FA00000010uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x000000000282170EuLL},
{0x078644FA00000010uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x0000000300000001uLL, 0x00000000028216FDuLL, 0x00000002FD7DE913uLL},
{0x078644FA00000010uLL, 0x0000000300000002uLL, 0x00000000028216FDuLL, 0x00000002FAFBD216uLL},
{0x078644FA00000010uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002F879BB19uLL},
{0x078644FA00000010uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002D7DE9040uLL},
{0x078644FA00000010uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x0000000266ACEAF4uLL},
{0x078644FA00000010uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000CC648260uLL},
{0x078644FA00000010uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000282265CE9uLL},
{0x078644FA00000010uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000280000010uLL},
{0x078644FA00000010uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000205A4B3CAuLL},
{0x078644FA00000010uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000203C3228CuLL},
{0x078644FA00000010uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000201E1914EuLL},
{0x078644FA00000010uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A00000010uLL},
{0x078644FA00000010uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x00000009FF879BC1uLL},
{0x078644FA00000010uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x00000009FF0F3772uLL},
{0x078644FA00000010uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x00000009FE96D323uLL},
{0x078644FA00000010uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009F879BB20uLL},
{0x078644FA00000010uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000001E3B0AD8uLL},
{0x078644FA00000010uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000438F2017CuLL},
{0x078644FA00000010uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C0074BE6CuLL},
{0x078644FA00000010uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C00000010uLL},
{0x078644FA00000010uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000000153EE1EuLL},
{0x078644FA00000010uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000000E29EC4uLL},
{0x078644FA00000010uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000000714F6AuLL},
{0x078644FA00000010uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000000000010uLL},
{0x078644FA00000010uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x000000000000000FuLL},
{0x078644FA00000010uLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x000000000000000EuLL},
{0x078644FA00000010uLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x000000000000000DuLL},
{0x078644FA00000010uLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA00000010uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E600000010uLL},
{0x078644FA00000010uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E60000000FuLL},
{0x078644FA00000010uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E60000000EuLL},
{0x078644FA00000010uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E60000000DuLL},
{0x078644FA00000010uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA00000010uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5F879BB16uLL},
{0x078644FA00000010uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E5F8B851FCuLL},
{0x078644FA00000010uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000011uLL},
{0x078644FA00000010uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E580000010uLL},
{0x078644FA00000010uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E500000013uLL},
{0x078644FA00000010uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E500000012uLL},
{0x078644FA00000010uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E500000011uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA00000010uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA00000010uLL},
{0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000000000002uLL, 0x03C3227D03C3227DuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000000000003uLL, 0x028216FE028216FEuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000000000010uLL, 0x0078644FA078644FuLL, 0x000000000000000AuLL},
{0x078644FA078644FAuLL, 0x00000000078644FAuLL, 0x0000000100000001uLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x000000000747AE14uLL, 0x000000010898F296uLL, 0x0000000005175D42uLL},
{0x078644FA078644FAuLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x000000001692CEEEuLL},
{0x078644FA078644FAuLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001E1913E8uLL},
{0x078644FA078644FAuLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000001692CEEEuLL},
{0x078644FA078644FAuLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x000000000F0C89F4uLL},
{0x078644FA078644FAuLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F879BB08uLL},
{0x078644FA078644FAuLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000F0F3760FuLL},
{0x078644FA078644FAuLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x000000008F21F56AuLL},
{0x078644FA078644FAuLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x00000000738FF1CCuLL},
{0x078644FA078644FAuLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x000000008A3C8322uLL},
{0x078644FA078644FAuLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x000000000C8A72F6uLL},
{0x078644FA078644FAuLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x0000000012CFAC71uLL},
{0x078644FA078644FAuLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x000000000F0C89F4uLL},
{0x078644FA078644FAuLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x000000000B496777uLL},
{0x078644FA078644FAuLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x0000000003C3227DuLL},
{0x078644FA078644FAuLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001FC3CDD86uLL},
{0x078644FA078644FAuLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001CB541D3AuLL},
{0x078644FA078644FAuLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000188FB8A7CuLL},
{0x078644FA078644FAuLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x00000001A2D3367EuLL},
{0x078644FA078644FAuLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x000000018A88C6F7uLL},
{0x078644FA078644FAuLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x00000001878644FAuLL},
{0x078644FA078644FAuLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x000000000F0C89F4uLL},
{0x078644FA078644FAuLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x000000000C8A72F6uLL},
{0x078644FA078644FAuLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x000000000A085BF8uLL},
{0x078644FA078644FAuLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x0000000005042DFCuLL},
{0x078644FA078644FAuLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x00000000028216FEuLL},
{0x078644FA078644FAuLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002DF64D52AuLL},
{0x078644FA078644FAuLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x000000026E332FDEuLL},
{0x078644FA078644FAuLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000D3EAC74AuLL},
{0x078644FA078644FAuLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000289ACA1D3uLL},
{0x078644FA078644FAuLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x00000002878644FAuLL},
{0x078644FA078644FAuLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x000000020D2AF8B4uLL},
{0x078644FA078644FAuLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x000000020B496776uLL},
{0x078644FA078644FAuLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x000000020967D638uLL},
{0x078644FA078644FAuLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A078644FAuLL},
{0x078644FA078644FAuLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000A070DE0ABuLL},
{0x078644FA078644FAuLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000A06957C5CuLL},
{0x078644FA078644FAuLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000A061D180DuLL},
{0x078644FA078644FAuLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000A0000000AuLL},
{0x078644FA078644FAuLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x0000000025C14FC2uLL},
{0x078644FA078644FAuLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000440784666uLL},
{0x078644FA078644FAuLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C07FB0356uLL},
{0x078644FA078644FAuLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C078644FAuLL},
{0x078644FA078644FAuLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x0000000008DA3308uLL},
{0x078644FA078644FAuLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x000000000868E3AEuLL},
{0x078644FA078644FAuLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000007F79454uLL},
{0x078644FA078644FAuLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x00000000078644FAuLL},
{0x078644FA078644FAuLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x00000000078644F9uLL},
{0x078644FA078644FAuLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x00000000078644F8uLL},
{0x078644FA078644FAuLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x00000000078644F7uLL},
{0x078644FA078644FAuLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x00000000078644EAuLL},
{0x078644FA078644FAuLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA078644FAuLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x00000000003E96E6uLL},
{0x078644FA078644FAuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E6078644FAuLL},
{0x078644FA078644FAuLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E6078644F9uLL},
{0x078644FA078644FAuLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E6078644F8uLL},
{0x078644FA078644FAuLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E6078644F7uLL},
{0x078644FA078644FAuLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E6078644EAuLL},
{0x078644FA078644FAuLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA078644FAuLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E6003E96E6uLL},
{0x078644FA078644FAuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E5878644FBuLL},
{0x078644FA078644FAuLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E5878644FAuLL},
{0x078644FA078644FAuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E5078644FDuLL},
{0x078644FA078644FAuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E5078644FCuLL},
{0x078644FA078644FAuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E5078644FBuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA078644FAuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA078644FAuLL},
{0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL},
{0x078644FA0747AE14uLL, 0x0000000000000002uLL, 0x03C3227D03A3D70AuLL, 0x0000000000000000uLL},
{0x078644FA0747AE14uLL, 0x0000000000000003uLL, 0x028216FE026D3A06uLL, 0x0000000000000002uLL},
{0x078644FA0747AE14uLL, 0x0000000000000010uLL, 0x0078644FA0747AE1uLL, 0x0000000000000004uLL},
{0x078644FA0747AE14uLL, 0x00000000078644FAuLL, 0x0000000100000000uLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x000000000747AE14uLL, 0x000000010898F296uLL, 0x0000000004D8C65CuLL},
{0x078644FA0747AE14uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F4uLL, 0x0000000016543808uLL},
{0x078644FA0747AE14uLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000001DDA7D02uLL},
{0x078644FA0747AE14uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000016543808uLL},
{0x078644FA0747AE14uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x000000000ECDF30EuLL},
{0x078644FA0747AE14uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x0000000100000001uLL, 0x00000000078644F9uLL, 0x00000000FFC1691BuLL},
{0x078644FA0747AE14uLL, 0x0000000100000002uLL, 0x00000000078644F9uLL, 0x00000000F83B2422uLL},
{0x078644FA0747AE14uLL, 0x0000000100000003uLL, 0x00000000078644F9uLL, 0x00000000F0B4DF29uLL},
{0x078644FA0747AE14uLL, 0x0000000100000010uLL, 0x00000000078644F9uLL, 0x000000008EE35E84uLL},
{0x078644FA0747AE14uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x0000000073515AE6uLL},
{0x078644FA0747AE14uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000089FDEC3CuLL},
{0x078644FA0747AE14uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x000000000C4BDC10uLL},
{0x078644FA0747AE14uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000001291158BuLL},
{0x078644FA0747AE14uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x000000000ECDF30EuLL},
{0x078644FA0747AE14uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x000000000B0AD091uLL},
{0x078644FA0747AE14uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x0000000003848B97uLL},
{0x078644FA0747AE14uLL, 0x0000000200000002uLL, 0x0000000003C3227CuLL, 0x00000001FFC1691CuLL},
{0x078644FA0747AE14uLL, 0x0000000200000003uLL, 0x0000000003C3227CuLL, 0x00000001FBFE46A0uLL},
{0x078644FA0747AE14uLL, 0x0000000200000010uLL, 0x0000000003C3227CuLL, 0x00000001CB158654uLL},
{0x078644FA0747AE14uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000188BCF396uLL},
{0x078644FA0747AE14uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D3uLL, 0x00000001A2949F98uLL},
{0x078644FA0747AE14uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x000000018A4A3011uLL},
{0x078644FA0747AE14uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x000000018747AE14uLL},
{0x078644FA0747AE14uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x000000000ECDF30EuLL},
{0x078644FA0747AE14uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x000000000C4BDC10uLL},
{0x078644FA0747AE14uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x0000000009C9C512uLL},
{0x078644FA0747AE14uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x0000000004C59716uLL},
{0x078644FA0747AE14uLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x0000000002438018uLL},
{0x078644FA0747AE14uLL, 0x0000000300000003uLL, 0x00000000028216FDuLL, 0x00000002FFC1691DuLL},
{0x078644FA0747AE14uLL, 0x0000000300000010uLL, 0x00000000028216FDuLL, 0x00000002DF263E44uLL},
{0x078644FA0747AE14uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x000000026DF498F8uLL},
{0x078644FA0747AE14uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000000D3AC3064uLL},
{0x078644FA0747AE14uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x00000002896E0AEDuLL},
{0x078644FA0747AE14uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x000000028747AE14uLL},
{0x078644FA0747AE14uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x000000020CEC61CEuLL},
{0x078644FA0747AE14uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x000000020B0AD090uLL},
{0x078644FA0747AE14uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000209293F52uLL},
{0x078644FA0747AE14uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A0747AE14uLL},
{0x078644FA0747AE14uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000A06CF49C5uLL},
{0x078644FA0747AE14uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000A0656E576uLL},
{0x078644FA0747AE14uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000A05DE8127uLL},
{0x078644FA0747AE14uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x00000009FFC16924uLL},
{0x078644FA0747AE14uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000002582B8DCuLL},
{0x078644FA0747AE14uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x000000044039AF80uLL},
{0x078644FA0747AE14uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C07BC6C70uLL},
{0x078644FA0747AE14uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C0747AE14uLL},
{0x078644FA0747AE14uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x00000000089B9C22uLL},
{0x078644FA0747AE14uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x00000000082A4CC8uLL},
{0x078644FA0747AE14uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000007B8FD6EuLL},
{0x078644FA0747AE14uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x000000000747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x000000000747AE13uLL},
{0x078644FA0747AE14uLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x000000000747AE12uLL},
{0x078644FA0747AE14uLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x000000000747AE11uLL},
{0x078644FA0747AE14uLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x000000000747AE04uLL},
{0x078644FA0747AE14uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA0747AE14uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E60747AE14uLL},
{0x078644FA0747AE14uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E60747AE13uLL},
{0x078644FA0747AE14uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E60747AE12uLL},
{0x078644FA0747AE14uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E60747AE11uLL},
{0x078644FA0747AE14uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E60747AE04uLL},
{0x078644FA0747AE14uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E5FFC1691AuLL},
{0x078644FA0747AE14uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA0747AE14uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E58747AE15uLL},
{0x078644FA0747AE14uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E58747AE14uLL},
{0x078644FA0747AE14uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E50747AE17uLL},
{0x078644FA0747AE14uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E50747AE16uLL},
{0x078644FA0747AE14uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E50747AE15uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA0747AE14uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA0747AE14uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000000000002uLL, 0x03C3227D3FFFFFFFuLL, 0x0000000000000001uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000000000003uLL, 0x028216FE2AAAAAAAuLL, 0x0000000000000001uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000000000010uLL, 0x0078644FA7FFFFFFuLL, 0x000000000000000FuLL},
{0x078644FA7FFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000100000011uLL, 0x0000000000156B65uLL},
{0x078644FA7FFFFFFFuLL, 0x000000000747AE14uLL, 0x000000010898F2A7uLL, 0x0000000001CE88F3uLL},
{0x078644FA7FFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F5uLL, 0x000000000F0C89F4uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000080000000uLL, 0x000000000F0C89F4uLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000009692CEEDuLL},
{0x078644FA7FFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000008F0C89F3uLL},
{0x078644FA7FFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000878644F9uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x000000007879BB05uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000100000002uLL, 0x00000000078644FAuLL, 0x0000000070F3760BuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000100000003uLL, 0x00000000078644FAuLL, 0x00000000696D3111uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000100000010uLL, 0x00000000078644FAuLL, 0x00000000079BB05FuLL},
{0x078644FA7FFFFFFFuLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x00000000EC09ACD1uLL},
{0x078644FA7FFFFFFFuLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000102B63E27uLL},
{0x078644FA7FFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000085042DFBuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000008B496776uLL},
{0x078644FA7FFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000878644F9uLL},
{0x078644FA7FFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000083C3227CuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x000000007C3CDD82uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x000000007879BB05uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000200000003uLL, 0x0000000003C3227DuLL, 0x0000000074B69888uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000200000010uLL, 0x0000000003C3227DuLL, 0x0000000043CDD82FuLL},
{0x078644FA7FFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000201754581uLL},
{0x078644FA7FFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000003B5A1D4uLL, 0x000000001405436FuLL},
{0x078644FA7FFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x00000002030281FCuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x00000001FFFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000878644F9uLL},
{0x078644FA7FFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000085042DFBuLL},
{0x078644FA7FFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000000828216FDuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x000000007D7DE901uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x000000007AFBD203uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x000000007879BB05uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000300000010uLL, 0x00000000028216FEuLL, 0x0000000057DE901FuLL},
{0x078644FA7FFFFFFFuLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x00000002E6ACEAE3uLL},
{0x078644FA7FFFFFFFuLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x000000014C64824FuLL},
{0x078644FA7FFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000302265CD8uLL},
{0x078644FA7FFFFFFFuLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x00000002FFFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000285A4B3B9uLL},
{0x078644FA7FFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000283C3227BuLL},
{0x078644FA7FFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000281E1913DuLL},
{0x078644FA7FFFFFFFuLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000A7F879BB0uLL},
{0x078644FA7FFFFFFFuLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000A7F0F3761uLL},
{0x078644FA7FFFFFFFuLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000A7E96D312uLL},
{0x078644FA7FFFFFFFuLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000A7879BB0FuLL},
{0x078644FA7FFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000009E3B0AC7uLL},
{0x078644FA7FFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x00000004B8F2016BuLL},
{0x078644FA7FFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C8074BE5BuLL},
{0x078644FA7FFFFFFFuLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000008153EE0DuLL},
{0x078644FA7FFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000080E29EB3uLL},
{0x078644FA7FFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000080714F59uLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFCuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFEFuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB05uLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851EBuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA7FFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFEuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFDuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFCuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFEFuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E67879BB05uLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E678B851EBuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E580000002uLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E580000001uLL},
{0x078644FA7FFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000000uLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA7FFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA7FFFFFFFuLL},
{0x078644FA80000000uLL, 0x0000000000000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL},
{0x078644FA80000000uLL, 0x0000000000000002uLL, 0x03C3227D40000000uLL, 0x0000000000000000uLL},
{0x078644FA80000000uLL, 0x0000000000000003uLL, 0x028216FE2AAAAAAAuLL, 0x0000000000000002uLL},
{0x078644FA80000000uLL, 0x0000000000000010uLL, 0x0078644FA8000000uLL, 0x0000000000000000uLL},
{0x078644FA80000000uLL, 0x00000000078644FAuLL, 0x0000000100000011uLL, 0x0000000000156B66uLL},
{0x078644FA80000000uLL, 0x000000000747AE14uLL, 0x000000010898F2A7uLL, 0x0000000001CE88F4uLL},
{0x078644FA80000000uLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F5uLL, 0x000000000F0C89F5uLL},
{0x078644FA80000000uLL, 0x0000000080000000uLL, 0x000000000F0C89F5uLL, 0x0000000000000000uLL},
{0x078644FA80000000uLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FAuLL, 0x000000009692CEEEuLL},
{0x078644FA80000000uLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FAuLL, 0x000000008F0C89F4uLL},
{0x078644FA80000000uLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FAuLL, 0x00000000878644FAuLL},
{0x078644FA80000000uLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x0000000080000000uLL},
{0x078644FA80000000uLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x000000007879BB06uLL},
{0x078644FA80000000uLL, 0x0000000100000002uLL, 0x00000000078644FAuLL, 0x0000000070F3760CuLL},
{0x078644FA80000000uLL, 0x0000000100000003uLL, 0x00000000078644FAuLL, 0x00000000696D3112uLL},
{0x078644FA80000000uLL, 0x0000000100000010uLL, 0x00000000078644FAuLL, 0x00000000079BB060uLL},
{0x078644FA80000000uLL, 0x00000001078644FAuLL, 0x00000000074F44A3uLL, 0x00000000EC09ACD2uLL},
{0x078644FA80000000uLL, 0x000000010747AE14uLL, 0x000000000751017EuLL, 0x0000000102B63E28uLL},
{0x078644FA80000000uLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000085042DFCuLL},
{0x078644FA80000000uLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x0000000080000000uLL},
{0x078644FA80000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000008B496777uLL},
{0x078644FA80000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000000878644FAuLL},
{0x078644FA80000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000083C3227DuLL},
{0x078644FA80000000uLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x0000000080000000uLL},
{0x078644FA80000000uLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x000000007C3CDD83uLL},
{0x078644FA80000000uLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x000000007879BB06uLL},
{0x078644FA80000000uLL, 0x0000000200000003uLL, 0x0000000003C3227DuLL, 0x0000000074B69889uLL},
{0x078644FA80000000uLL, 0x0000000200000010uLL, 0x0000000003C3227DuLL, 0x0000000043CDD830uLL},
{0x078644FA80000000uLL, 0x00000002078644FAuLL, 0x0000000003B52F6BuLL, 0x0000000201754582uLL},
{0x078644FA80000000uLL, 0x000000020747AE14uLL, 0x0000000003B5A1D4uLL, 0x0000000014054370uLL},
{0x078644FA80000000uLL, 0x000000027FFFFFFFuLL, 0x00000000030281FDuLL, 0x00000002030281FDuLL},
{0x078644FA80000000uLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x0000000200000000uLL},
{0x078644FA80000000uLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000000878644FAuLL},
{0x078644FA80000000uLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000085042DFCuLL},
{0x078644FA80000000uLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000000828216FEuLL},
{0x078644FA80000000uLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x0000000080000000uLL},
{0x078644FA80000000uLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x000000007D7DE902uLL},
{0x078644FA80000000uLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x000000007AFBD204uLL},
{0x078644FA80000000uLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x000000007879BB06uLL},
{0x078644FA80000000uLL, 0x0000000300000010uLL, 0x00000000028216FEuLL, 0x0000000057DE9020uLL},
{0x078644FA80000000uLL, 0x00000003078644FAuLL, 0x00000000027BDC26uLL, 0x00000002E6ACEAE4uLL},
{0x078644FA80000000uLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x000000014C648250uLL},
{0x078644FA80000000uLL, 0x000000037FFFFFFFuLL, 0x0000000002265CD9uLL, 0x0000000302265CD9uLL},
{0x078644FA80000000uLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x0000000300000000uLL},
{0x078644FA80000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000285A4B3BAuLL},
{0x078644FA80000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000283C3227CuLL},
{0x078644FA80000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000281E1913EuLL},
{0x078644FA80000000uLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000A80000000uLL},
{0x078644FA80000000uLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000A7F879BB1uLL},
{0x078644FA80000000uLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000A7F0F3762uLL},
{0x078644FA80000000uLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000A7E96D313uLL},
{0x078644FA80000000uLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000A7879BB10uLL},
{0x078644FA80000000uLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000009E3B0AC8uLL},
{0x078644FA80000000uLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x00000004B8F2016CuLL},
{0x078644FA80000000uLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000C8074BE5CuLL},
{0x078644FA80000000uLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000C80000000uLL},
{0x078644FA80000000uLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000008153EE0EuLL},
{0x078644FA80000000uLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000080E29EB4uLL},
{0x078644FA80000000uLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000080714F5AuLL},
{0x078644FA80000000uLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x078644FA80000000uLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x078644FA80000000uLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x078644FA80000000uLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x078644FA80000000uLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x000000007FFFFFF0uLL},
{0x078644FA80000000uLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x000000007879BB06uLL},
{0x078644FA80000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x0000000078B851ECuLL},
{0x078644FA80000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FA80000000uLL, 0x078644FA80000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FA80000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E680000000uLL},
{0x078644FA80000000uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFFuLL},
{0x078644FA80000000uLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFEuLL},
{0x078644FA80000000uLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFDuLL},
{0x078644FA80000000uLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFF0uLL},
{0x078644FA80000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E67879BB06uLL},
{0x078644FA80000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E678B851ECuLL},
{0x078644FA80000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FA80000000uLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FA80000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E580000003uLL},
{0x078644FA80000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E580000002uLL},
{0x078644FA80000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E580000001uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FA80000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FA80000000uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000000000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000000000002uLL, 0x03C3227D7FFFFFFEuLL, 0x0000000000000001uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000000000003uLL, 0x028216FE55555554uLL, 0x0000000000000001uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000000000010uLL, 0x0078644FAFFFFFFFuLL, 0x000000000000000DuLL},
{0x078644FAFFFFFFFDuLL, 0x00000000078644FAuLL, 0x0000000100000022uLL, 0x00000000002AD6C9uLL},
{0x078644FAFFFFFFFDuLL, 0x000000000747AE14uLL, 0x000000010898F2B8uLL, 0x00000000060BF99DuLL},
{0x078644FAFFFFFFFDuLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F6uLL, 0x000000000F0C89F3uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000080000000uLL, 0x000000000F0C89F5uLL, 0x000000007FFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FBuLL, 0x000000001692CEEEuLL},
{0x078644FAFFFFFFFDuLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FBuLL, 0x000000000F0C89F3uLL},
{0x078644FAFFFFFFFDuLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FBuLL, 0x00000000078644F8uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x00000000F879BB03uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000100000002uLL, 0x00000000078644FAuLL, 0x00000000F0F37609uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000100000003uLL, 0x00000000078644FAuLL, 0x00000000E96D310FuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000100000010uLL, 0x00000000078644FAuLL, 0x00000000879BB05DuLL},
{0x078644FAFFFFFFFDuLL, 0x00000001078644FAuLL, 0x00000000074F44A4uLL, 0x00000000648367D5uLL},
{0x078644FAFFFFFFFDuLL, 0x000000010747AE14uLL, 0x000000000751017FuLL, 0x000000007B6E9011uLL},
{0x078644FAFFFFFFFDuLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000105042DF9uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000010B496774uLL},
{0x078644FAFFFFFFFDuLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000001078644F7uLL},
{0x078644FAFFFFFFFDuLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000103C3227AuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x00000000FC3CDD80uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x00000000F879BB03uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000200000003uLL, 0x0000000003C3227DuLL, 0x00000000F4B69886uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000200000010uLL, 0x0000000003C3227DuLL, 0x00000000C3CDD82DuLL},
{0x078644FAFFFFFFFDuLL, 0x00000002078644FAuLL, 0x0000000003B52F6CuLL, 0x0000000079EF0085uLL},
{0x078644FAFFFFFFFDuLL, 0x000000020747AE14uLL, 0x0000000003B5A1D4uLL, 0x000000009405436DuLL},
{0x078644FAFFFFFFFDuLL, 0x000000027FFFFFFFuLL, 0x00000000030281FEuLL, 0x00000000030281FBuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x000000027FFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000001078644F7uLL},
{0x078644FAFFFFFFFDuLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000105042DF9uLL},
{0x078644FAFFFFFFFDuLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000001028216FBuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x00000000FD7DE8FFuLL},
{0x078644FAFFFFFFFDuLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x00000000FAFBD201uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x00000000F879BB03uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000300000010uLL, 0x00000000028216FEuLL, 0x00000000D7DE901DuLL},
{0x078644FAFFFFFFFDuLL, 0x00000003078644FAuLL, 0x00000000027BDC27uLL, 0x000000005F26A5E7uLL},
{0x078644FAFFFFFFFDuLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000001CC64824DuLL},
{0x078644FAFFFFFFFDuLL, 0x000000037FFFFFFFuLL, 0x0000000002265CDAuLL, 0x0000000002265CD7uLL},
{0x078644FAFFFFFFFDuLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x000000037FFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000305A4B3B7uLL},
{0x078644FAFFFFFFFDuLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000303C32279uLL},
{0x078644FAFFFFFFFDuLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000301E1913BuLL},
{0x078644FAFFFFFFFDuLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000AFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000AFF879BAEuLL},
{0x078644FAFFFFFFFDuLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000AFF0F375FuLL},
{0x078644FAFFFFFFFDuLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000AFE96D310uLL},
{0x078644FAFFFFFFFDuLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000AF879BB0DuLL},
{0x078644FAFFFFFFFDuLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000011E3B0AC5uLL},
{0x078644FAFFFFFFFDuLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000538F20169uLL},
{0x078644FAFFFFFFFDuLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000D0074BE59uLL},
{0x078644FAFFFFFFFDuLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000CFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000010153EE0BuLL},
{0x078644FAFFFFFFFDuLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000100E29EB1uLL},
{0x078644FAFFFFFFFDuLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000100714F57uLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFAuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEDuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB03uLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851E9uLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FA80000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFDuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFCuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFBuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFAuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFEDuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E6F879BB03uLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E6F8B851E9uLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFEuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FAFFFFFFFDuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFEuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFDuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFDuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000000000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000000000002uLL, 0x03C3227D7FFFFFFFuLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000000000003uLL, 0x028216FE55555554uLL, 0x0000000000000002uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000000000010uLL, 0x0078644FAFFFFFFFuLL, 0x000000000000000EuLL},
{0x078644FAFFFFFFFEuLL, 0x00000000078644FAuLL, 0x0000000100000022uLL, 0x00000000002AD6CAuLL},
{0x078644FAFFFFFFFEuLL, 0x000000000747AE14uLL, 0x000000010898F2B8uLL, 0x00000000060BF99EuLL},
{0x078644FAFFFFFFFEuLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F6uLL, 0x000000000F0C89F4uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000080000000uLL, 0x000000000F0C89F5uLL, 0x000000007FFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FBuLL, 0x000000001692CEEFuLL},
{0x078644FAFFFFFFFEuLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FBuLL, 0x000000000F0C89F4uLL},
{0x078644FAFFFFFFFEuLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FBuLL, 0x00000000078644F9uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x00000000F879BB04uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000100000002uLL, 0x00000000078644FAuLL, 0x00000000F0F3760AuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000100000003uLL, 0x00000000078644FAuLL, 0x00000000E96D3110uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000100000010uLL, 0x00000000078644FAuLL, 0x00000000879BB05EuLL},
{0x078644FAFFFFFFFEuLL, 0x00000001078644FAuLL, 0x00000000074F44A4uLL, 0x00000000648367D6uLL},
{0x078644FAFFFFFFFEuLL, 0x000000010747AE14uLL, 0x000000000751017FuLL, 0x000000007B6E9012uLL},
{0x078644FAFFFFFFFEuLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000105042DFAuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000010B496775uLL},
{0x078644FAFFFFFFFEuLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000001078644F8uLL},
{0x078644FAFFFFFFFEuLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000103C3227BuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x00000000FC3CDD81uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x00000000F879BB04uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000200000003uLL, 0x0000000003C3227DuLL, 0x00000000F4B69887uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000200000010uLL, 0x0000000003C3227DuLL, 0x00000000C3CDD82EuLL},
{0x078644FAFFFFFFFEuLL, 0x00000002078644FAuLL, 0x0000000003B52F6CuLL, 0x0000000079EF0086uLL},
{0x078644FAFFFFFFFEuLL, 0x000000020747AE14uLL, 0x0000000003B5A1D4uLL, 0x000000009405436EuLL},
{0x078644FAFFFFFFFEuLL, 0x000000027FFFFFFFuLL, 0x00000000030281FEuLL, 0x00000000030281FCuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x000000027FFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000001078644F8uLL},
{0x078644FAFFFFFFFEuLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000105042DFAuLL},
{0x078644FAFFFFFFFEuLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000001028216FCuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x00000000FD7DE900uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x00000000FAFBD202uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x00000000F879BB04uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000300000010uLL, 0x00000000028216FEuLL, 0x00000000D7DE901EuLL},
{0x078644FAFFFFFFFEuLL, 0x00000003078644FAuLL, 0x00000000027BDC27uLL, 0x000000005F26A5E8uLL},
{0x078644FAFFFFFFFEuLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000001CC64824EuLL},
{0x078644FAFFFFFFFEuLL, 0x000000037FFFFFFFuLL, 0x0000000002265CDAuLL, 0x0000000002265CD8uLL},
{0x078644FAFFFFFFFEuLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x000000037FFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000305A4B3B8uLL},
{0x078644FAFFFFFFFEuLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000303C3227AuLL},
{0x078644FAFFFFFFFEuLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000301E1913CuLL},
{0x078644FAFFFFFFFEuLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000AFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000AFF879BAFuLL},
{0x078644FAFFFFFFFEuLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000AFF0F3760uLL},
{0x078644FAFFFFFFFEuLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000AFE96D311uLL},
{0x078644FAFFFFFFFEuLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000AF879BB0EuLL},
{0x078644FAFFFFFFFEuLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000011E3B0AC6uLL},
{0x078644FAFFFFFFFEuLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000538F2016AuLL},
{0x078644FAFFFFFFFEuLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000D0074BE5AuLL},
{0x078644FAFFFFFFFEuLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000CFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000010153EE0CuLL},
{0x078644FAFFFFFFFEuLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000100E29EB2uLL},
{0x078644FAFFFFFFFEuLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000100714F58uLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFBuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEEuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB04uLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EAuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FA80000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FAFFFFFFFEuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFEuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFDuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFCuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFBuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFEEuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E6F879BB04uLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E6F8B851EAuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFFuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FAFFFFFFFEuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E5FFFFFFFFuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFEuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFEuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000000000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000000000002uLL, 0x03C3227D7FFFFFFFuLL, 0x0000000000000001uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000000000003uLL, 0x028216FE55555555uLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000000000010uLL, 0x0078644FAFFFFFFFuLL, 0x000000000000000FuLL},
{0x078644FAFFFFFFFFuLL, 0x00000000078644FAuLL, 0x0000000100000022uLL, 0x00000000002AD6CBuLL},
{0x078644FAFFFFFFFFuLL, 0x000000000747AE14uLL, 0x000000010898F2B8uLL, 0x00000000060BF99FuLL},
{0x078644FAFFFFFFFFuLL, 0x000000007FFFFFFFuLL, 0x000000000F0C89F6uLL, 0x000000000F0C89F5uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000080000000uLL, 0x000000000F0C89F5uLL, 0x000000007FFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x00000000FFFFFFFDuLL, 0x00000000078644FBuLL, 0x000000001692CEF0uLL},
{0x078644FAFFFFFFFFuLL, 0x00000000FFFFFFFEuLL, 0x00000000078644FBuLL, 0x000000000F0C89F5uLL},
{0x078644FAFFFFFFFFuLL, 0x00000000FFFFFFFFuLL, 0x00000000078644FBuLL, 0x00000000078644FAuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000100000000uLL, 0x00000000078644FAuLL, 0x00000000FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000100000001uLL, 0x00000000078644FAuLL, 0x00000000F879BB05uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000100000002uLL, 0x00000000078644FAuLL, 0x00000000F0F3760BuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000100000003uLL, 0x00000000078644FAuLL, 0x00000000E96D3111uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000100000010uLL, 0x00000000078644FAuLL, 0x00000000879BB05FuLL},
{0x078644FAFFFFFFFFuLL, 0x00000001078644FAuLL, 0x00000000074F44A4uLL, 0x00000000648367D7uLL},
{0x078644FAFFFFFFFFuLL, 0x000000010747AE14uLL, 0x000000000751017FuLL, 0x000000007B6E9013uLL},
{0x078644FAFFFFFFFFuLL, 0x000000017FFFFFFFuLL, 0x0000000005042DFCuLL, 0x0000000105042DFBuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000180000000uLL, 0x0000000005042DFCuLL, 0x00000000FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x00000001FFFFFFFDuLL, 0x0000000003C3227DuLL, 0x000000010B496776uLL},
{0x078644FAFFFFFFFFuLL, 0x00000001FFFFFFFEuLL, 0x0000000003C3227DuLL, 0x00000001078644F9uLL},
{0x078644FAFFFFFFFFuLL, 0x00000001FFFFFFFFuLL, 0x0000000003C3227DuLL, 0x0000000103C3227CuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000200000000uLL, 0x0000000003C3227DuLL, 0x00000000FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000200000001uLL, 0x0000000003C3227DuLL, 0x00000000FC3CDD82uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000200000002uLL, 0x0000000003C3227DuLL, 0x00000000F879BB05uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000200000003uLL, 0x0000000003C3227DuLL, 0x00000000F4B69888uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000200000010uLL, 0x0000000003C3227DuLL, 0x00000000C3CDD82FuLL},
{0x078644FAFFFFFFFFuLL, 0x00000002078644FAuLL, 0x0000000003B52F6CuLL, 0x0000000079EF0087uLL},
{0x078644FAFFFFFFFFuLL, 0x000000020747AE14uLL, 0x0000000003B5A1D4uLL, 0x000000009405436FuLL},
{0x078644FAFFFFFFFFuLL, 0x000000027FFFFFFFuLL, 0x00000000030281FEuLL, 0x00000000030281FDuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000280000000uLL, 0x00000000030281FDuLL, 0x000000027FFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x00000002FFFFFFFDuLL, 0x00000000028216FEuLL, 0x00000001078644F9uLL},
{0x078644FAFFFFFFFFuLL, 0x00000002FFFFFFFEuLL, 0x00000000028216FEuLL, 0x0000000105042DFBuLL},
{0x078644FAFFFFFFFFuLL, 0x00000002FFFFFFFFuLL, 0x00000000028216FEuLL, 0x00000001028216FDuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000300000000uLL, 0x00000000028216FEuLL, 0x00000000FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0000000300000001uLL, 0x00000000028216FEuLL, 0x00000000FD7DE901uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000300000002uLL, 0x00000000028216FEuLL, 0x00000000FAFBD203uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000300000003uLL, 0x00000000028216FEuLL, 0x00000000F879BB05uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000300000010uLL, 0x00000000028216FEuLL, 0x00000000D7DE901FuLL},
{0x078644FAFFFFFFFFuLL, 0x00000003078644FAuLL, 0x00000000027BDC27uLL, 0x000000005F26A5E9uLL},
{0x078644FAFFFFFFFFuLL, 0x000000030747AE14uLL, 0x00000000027C0F7CuLL, 0x00000001CC64824FuLL},
{0x078644FAFFFFFFFFuLL, 0x000000037FFFFFFFuLL, 0x0000000002265CDAuLL, 0x0000000002265CD9uLL},
{0x078644FAFFFFFFFFuLL, 0x0000000380000000uLL, 0x0000000002265CD9uLL, 0x000000037FFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x00000003FFFFFFFDuLL, 0x0000000001E1913EuLL, 0x0000000305A4B3B9uLL},
{0x078644FAFFFFFFFFuLL, 0x00000003FFFFFFFEuLL, 0x0000000001E1913EuLL, 0x0000000303C3227BuLL},
{0x078644FAFFFFFFFFuLL, 0x00000003FFFFFFFFuLL, 0x0000000001E1913EuLL, 0x0000000301E1913DuLL},
{0x078644FAFFFFFFFFuLL, 0x0000001000000000uLL, 0x000000000078644FuLL, 0x0000000AFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0000001000000001uLL, 0x000000000078644FuLL, 0x0000000AFF879BB0uLL},
{0x078644FAFFFFFFFFuLL, 0x0000001000000002uLL, 0x000000000078644FuLL, 0x0000000AFF0F3761uLL},
{0x078644FAFFFFFFFFuLL, 0x0000001000000003uLL, 0x000000000078644FuLL, 0x0000000AFE96D312uLL},
{0x078644FAFFFFFFFFuLL, 0x0000001000000010uLL, 0x000000000078644FuLL, 0x0000000AF879BB0FuLL},
{0x078644FAFFFFFFFFuLL, 0x00000010078644FAuLL, 0x0000000000782BCCuLL, 0x000000011E3B0AC7uLL},
{0x078644FAFFFFFFFFuLL, 0x000000100747AE14uLL, 0x0000000000782DA1uLL, 0x0000000538F2016BuLL},
{0x078644FAFFFFFFFFuLL, 0x000000107FFFFFFFuLL, 0x000000000074BE5CuLL, 0x0000000D0074BE5BuLL},
{0x078644FAFFFFFFFFuLL, 0x0000001080000000uLL, 0x000000000074BE5CuLL, 0x0000000CFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x00000010FFFFFFFDuLL, 0x0000000000714F5AuLL, 0x000000010153EE0DuLL},
{0x078644FAFFFFFFFFuLL, 0x00000010FFFFFFFEuLL, 0x0000000000714F5AuLL, 0x0000000100E29EB3uLL},
{0x078644FAFFFFFFFFuLL, 0x00000010FFFFFFFFuLL, 0x0000000000714F5AuLL, 0x0000000100714F59uLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA00000000uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA00000001uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFEuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA00000002uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFDuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA00000003uLL, 0x0000000000000001uLL, 0x00000000FFFFFFFCuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA00000010uLL, 0x0000000000000001uLL, 0x00000000FFFFFFEFuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA078644FAuLL, 0x0000000000000001uLL, 0x00000000F879BB05uLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA0747AE14uLL, 0x0000000000000001uLL, 0x00000000F8B851EBuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000001uLL, 0x0000000080000000uLL},
{0x078644FAFFFFFFFFuLL, 0x078644FA80000000uLL, 0x0000000000000001uLL, 0x000000007FFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000001uLL, 0x0000000000000002uLL},
{0x078644FAFFFFFFFFuLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x078644FAFFFFFFFFuLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFEuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFDuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1400000003uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFFCuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1400000010uLL, 0x0000000000000001uLL, 0x003E96E6FFFFFFEFuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE14078644FAuLL, 0x0000000000000001uLL, 0x003E96E6F879BB05uLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE140747AE14uLL, 0x0000000000000001uLL, 0x003E96E6F8B851EBuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E680000000uLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE1480000000uLL, 0x0000000000000001uLL, 0x003E96E67FFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000001uLL, 0x003E96E600000002uLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000001uLL, 0x003E96E600000001uLL},
{0x078644FAFFFFFFFFuLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000001uLL, 0x003E96E600000000uLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x078644FAFFFFFFFFuLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x078644FAFFFFFFFFuLL},
{0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x0747AE1400000000uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x0000000000000002uLL, 0x03A3D70A00000000uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x0000000000000003uLL, 0x026D3A06AAAAAAAAuLL, 0x0000000000000002uLL},
{0x0747AE1400000000uLL, 0x0000000000000010uLL, 0x00747AE140000000uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x00000000078644FAuLL, 0x00000000F7AE911DuLL, 0x0000000000CC95AEuLL},
{0x0747AE1400000000uLL, 0x000000000747AE14uLL, 0x0000000100000000uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x000000007FFFFFFFuLL, 0x000000000E8F5C28uLL, 0x000000000E8F5C28uLL},
{0x0747AE1400000000uLL, 0x0000000080000000uLL, 0x000000000E8F5C28uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x00000000FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000015D70A3CuLL},
{0x0747AE1400000000uLL, 0x00000000FFFFFFFEuLL, 0x000000000747AE14uLL, 0x000000000E8F5C28uLL},
{0x0747AE1400000000uLL, 0x00000000FFFFFFFFuLL, 0x000000000747AE14uLL, 0x000000000747AE14uLL},
{0x0747AE1400000000uLL, 0x0000000100000000uLL, 0x000000000747AE14uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x0000000100000001uLL, 0x000000000747AE13uLL, 0x00000000F8B851EDuLL},
{0x0747AE1400000000uLL, 0x0000000100000002uLL, 0x000000000747AE13uLL, 0x00000000F170A3DAuLL},
{0x0747AE1400000000uLL, 0x0000000100000003uLL, 0x000000000747AE13uLL, 0x00000000EA28F5C7uLL},
{0x0747AE1400000000uLL, 0x0000000100000010uLL, 0x000000000747AE13uLL, 0x000000008B851ED0uLL},
{0x0747AE1400000000uLL, 0x00000001078644FAuLL, 0x000000000712773FuLL, 0x000000000C50D07AuLL},
{0x0747AE1400000000uLL, 0x000000010747AE14uLL, 0x00000000071425A5uLL, 0x00000000CD13E91CuLL},
{0x0747AE1400000000uLL, 0x000000017FFFFFFFuLL, 0x0000000004DA740DuLL, 0x0000000084DA740DuLL},
{0x0747AE1400000000uLL, 0x0000000180000000uLL, 0x0000000004DA740DuLL, 0x0000000080000000uLL},
{0x0747AE1400000000uLL, 0x00000001FFFFFFFDuLL, 0x0000000003A3D70AuLL, 0x000000000AEB851EuLL},
{0x0747AE1400000000uLL, 0x00000001FFFFFFFEuLL, 0x0000000003A3D70AuLL, 0x000000000747AE14uLL},
{0x0747AE1400000000uLL, 0x00000001FFFFFFFFuLL, 0x0000000003A3D70AuLL, 0x0000000003A3D70AuLL},
{0x0747AE1400000000uLL, 0x0000000200000000uLL, 0x0000000003A3D70AuLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x0000000200000001uLL, 0x0000000003A3D709uLL, 0x00000001FC5C28F7uLL},
{0x0747AE1400000000uLL, 0x0000000200000002uLL, 0x0000000003A3D709uLL, 0x00000001F8B851EEuLL},
{0x0747AE1400000000uLL, 0x0000000200000003uLL, 0x0000000003A3D709uLL, 0x00000001F5147AE5uLL},
{0x0747AE1400000000uLL, 0x0000000200000010uLL, 0x0000000003A3D709uLL, 0x00000001C5C28F70uLL},
{0x0747AE1400000000uLL, 0x00000002078644FAuLL, 0x0000000003965801uLL, 0x000000007847CB06uLL},
{0x0747AE1400000000uLL, 0x000000020747AE14uLL, 0x000000000396C6B1uLL, 0x000000014B152C2CuLL},
{0x0747AE1400000000uLL, 0x000000027FFFFFFFuLL, 0x0000000002E978D4uLL, 0x0000000202E978D4uLL},
{0x0747AE1400000000uLL, 0x0000000280000000uLL, 0x0000000002E978D4uLL, 0x0000000200000000uLL},
{0x0747AE1400000000uLL, 0x00000002FFFFFFFDuLL, 0x00000000026D3A06uLL, 0x000000020747AE12uLL},
{0x0747AE1400000000uLL, 0x00000002FFFFFFFEuLL, 0x00000000026D3A06uLL, 0x0000000204DA740CuLL},
{0x0747AE1400000000uLL, 0x00000002FFFFFFFFuLL, 0x00000000026D3A06uLL, 0x00000002026D3A06uLL},
{0x0747AE1400000000uLL, 0x0000000300000000uLL, 0x00000000026D3A06uLL, 0x0000000200000000uLL},
{0x0747AE1400000000uLL, 0x0000000300000001uLL, 0x00000000026D3A06uLL, 0x00000001FD92C5FAuLL},
{0x0747AE1400000000uLL, 0x0000000300000002uLL, 0x00000000026D3A06uLL, 0x00000001FB258BF4uLL},
{0x0747AE1400000000uLL, 0x0000000300000003uLL, 0x00000000026D3A06uLL, 0x00000001F8B851EEuLL},
{0x0747AE1400000000uLL, 0x0000000300000010uLL, 0x00000000026D3A06uLL, 0x00000001D92C5FA0uLL},
{0x0747AE1400000000uLL, 0x00000003078644FAuLL, 0x0000000002673301uLL, 0x000000018425ED06uLL},
{0x0747AE1400000000uLL, 0x000000030747AE14uLL, 0x00000000026764ABuLL, 0x00000002F112E8A4uLL},
{0x0747AE1400000000uLL, 0x000000037FFFFFFFuLL, 0x0000000002147AE1uLL, 0x0000000082147AE1uLL},
{0x0747AE1400000000uLL, 0x0000000380000000uLL, 0x0000000002147AE1uLL, 0x0000000080000000uLL},
{0x0747AE1400000000uLL, 0x00000003FFFFFFFDuLL, 0x0000000001D1EB85uLL, 0x000000000575C28FuLL},
{0x0747AE1400000000uLL, 0x00000003FFFFFFFEuLL, 0x0000000001D1EB85uLL, 0x0000000003A3D70AuLL},
{0x0747AE1400000000uLL, 0x00000003FFFFFFFFuLL, 0x0000000001D1EB85uLL, 0x0000000001D1EB85uLL},
{0x0747AE1400000000uLL, 0x0000001000000000uLL, 0x0000000000747AE1uLL, 0x0000000400000000uLL},
{0x0747AE1400000000uLL, 0x0000001000000001uLL, 0x0000000000747AE1uLL, 0x00000003FF8B851FuLL},
{0x0747AE1400000000uLL, 0x0000001000000002uLL, 0x0000000000747AE1uLL, 0x00000003FF170A3EuLL},
{0x0747AE1400000000uLL, 0x0000001000000003uLL, 0x0000000000747AE1uLL, 0x00000003FEA28F5DuLL},
{0x0747AE1400000000uLL, 0x0000001000000010uLL, 0x0000000000747AE1uLL, 0x00000003F8B851F0uLL},
{0x0747AE1400000000uLL, 0x00000010078644FAuLL, 0x0000000000744433uLL, 0x0000000B94A5DA32uLL},
{0x0747AE1400000000uLL, 0x000000100747AE14uLL, 0x00000000007445F9uLL, 0x0000000BB84C4A8CuLL},
{0x0747AE1400000000uLL, 0x000000107FFFFFFFuLL, 0x000000000070F347uLL, 0x000000008070F347uLL},
{0x0747AE1400000000uLL, 0x0000001080000000uLL, 0x000000000070F347uLL, 0x0000000080000000uLL},
{0x0747AE1400000000uLL, 0x00000010FFFFFFFDuLL, 0x00000000006DA0D4uLL, 0x000000000148E27CuLL},
{0x0747AE1400000000uLL, 0x00000010FFFFFFFEuLL, 0x00000000006DA0D4uLL, 0x0000000000DB41A8uLL},
{0x0747AE1400000000uLL, 0x00000010FFFFFFFFuLL, 0x00000000006DA0D4uLL, 0x00000000006DA0D4uLL},
{0x0747AE1400000000uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000000uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000000uLL},
{0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x0747AE1400000001uLL, 0x0000000000000000uLL},
{0x0747AE1400000001uLL, 0x0000000000000002uLL, 0x03A3D70A00000000uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x0000000000000003uLL, 0x026D3A06AAAAAAABuLL, 0x0000000000000000uLL},
{0x0747AE1400000001uLL, 0x0000000000000010uLL, 0x00747AE140000000uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x00000000078644FAuLL, 0x00000000F7AE911DuLL, 0x0000000000CC95AFuLL},
{0x0747AE1400000001uLL, 0x000000000747AE14uLL, 0x0000000100000000uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x000000007FFFFFFFuLL, 0x000000000E8F5C28uLL, 0x000000000E8F5C29uLL},
{0x0747AE1400000001uLL, 0x0000000080000000uLL, 0x000000000E8F5C28uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x00000000FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000015D70A3DuLL},
{0x0747AE1400000001uLL, 0x00000000FFFFFFFEuLL, 0x000000000747AE14uLL, 0x000000000E8F5C29uLL},
{0x0747AE1400000001uLL, 0x00000000FFFFFFFFuLL, 0x000000000747AE14uLL, 0x000000000747AE15uLL},
{0x0747AE1400000001uLL, 0x0000000100000000uLL, 0x000000000747AE14uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x0000000100000001uLL, 0x000000000747AE13uLL, 0x00000000F8B851EEuLL},
{0x0747AE1400000001uLL, 0x0000000100000002uLL, 0x000000000747AE13uLL, 0x00000000F170A3DBuLL},
{0x0747AE1400000001uLL, 0x0000000100000003uLL, 0x000000000747AE13uLL, 0x00000000EA28F5C8uLL},
{0x0747AE1400000001uLL, 0x0000000100000010uLL, 0x000000000747AE13uLL, 0x000000008B851ED1uLL},
{0x0747AE1400000001uLL, 0x00000001078644FAuLL, 0x000000000712773FuLL, 0x000000000C50D07BuLL},
{0x0747AE1400000001uLL, 0x000000010747AE14uLL, 0x00000000071425A5uLL, 0x00000000CD13E91DuLL},
{0x0747AE1400000001uLL, 0x000000017FFFFFFFuLL, 0x0000000004DA740DuLL, 0x0000000084DA740EuLL},
{0x0747AE1400000001uLL, 0x0000000180000000uLL, 0x0000000004DA740DuLL, 0x0000000080000001uLL},
{0x0747AE1400000001uLL, 0x00000001FFFFFFFDuLL, 0x0000000003A3D70AuLL, 0x000000000AEB851FuLL},
{0x0747AE1400000001uLL, 0x00000001FFFFFFFEuLL, 0x0000000003A3D70AuLL, 0x000000000747AE15uLL},
{0x0747AE1400000001uLL, 0x00000001FFFFFFFFuLL, 0x0000000003A3D70AuLL, 0x0000000003A3D70BuLL},
{0x0747AE1400000001uLL, 0x0000000200000000uLL, 0x0000000003A3D70AuLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x0000000200000001uLL, 0x0000000003A3D709uLL, 0x00000001FC5C28F8uLL},
{0x0747AE1400000001uLL, 0x0000000200000002uLL, 0x0000000003A3D709uLL, 0x00000001F8B851EFuLL},
{0x0747AE1400000001uLL, 0x0000000200000003uLL, 0x0000000003A3D709uLL, 0x00000001F5147AE6uLL},
{0x0747AE1400000001uLL, 0x0000000200000010uLL, 0x0000000003A3D709uLL, 0x00000001C5C28F71uLL},
{0x0747AE1400000001uLL, 0x00000002078644FAuLL, 0x0000000003965801uLL, 0x000000007847CB07uLL},
{0x0747AE1400000001uLL, 0x000000020747AE14uLL, 0x000000000396C6B1uLL, 0x000000014B152C2DuLL},
{0x0747AE1400000001uLL, 0x000000027FFFFFFFuLL, 0x0000000002E978D4uLL, 0x0000000202E978D5uLL},
{0x0747AE1400000001uLL, 0x0000000280000000uLL, 0x0000000002E978D4uLL, 0x0000000200000001uLL},
{0x0747AE1400000001uLL, 0x00000002FFFFFFFDuLL, 0x00000000026D3A06uLL, 0x000000020747AE13uLL},
{0x0747AE1400000001uLL, 0x00000002FFFFFFFEuLL, 0x00000000026D3A06uLL, 0x0000000204DA740DuLL},
{0x0747AE1400000001uLL, 0x00000002FFFFFFFFuLL, 0x00000000026D3A06uLL, 0x00000002026D3A07uLL},
{0x0747AE1400000001uLL, 0x0000000300000000uLL, 0x00000000026D3A06uLL, 0x0000000200000001uLL},
{0x0747AE1400000001uLL, 0x0000000300000001uLL, 0x00000000026D3A06uLL, 0x00000001FD92C5FBuLL},
{0x0747AE1400000001uLL, 0x0000000300000002uLL, 0x00000000026D3A06uLL, 0x00000001FB258BF5uLL},
{0x0747AE1400000001uLL, 0x0000000300000003uLL, 0x00000000026D3A06uLL, 0x00000001F8B851EFuLL},
{0x0747AE1400000001uLL, 0x0000000300000010uLL, 0x00000000026D3A06uLL, 0x00000001D92C5FA1uLL},
{0x0747AE1400000001uLL, 0x00000003078644FAuLL, 0x0000000002673301uLL, 0x000000018425ED07uLL},
{0x0747AE1400000001uLL, 0x000000030747AE14uLL, 0x00000000026764ABuLL, 0x00000002F112E8A5uLL},
{0x0747AE1400000001uLL, 0x000000037FFFFFFFuLL, 0x0000000002147AE1uLL, 0x0000000082147AE2uLL},
{0x0747AE1400000001uLL, 0x0000000380000000uLL, 0x0000000002147AE1uLL, 0x0000000080000001uLL},
{0x0747AE1400000001uLL, 0x00000003FFFFFFFDuLL, 0x0000000001D1EB85uLL, 0x000000000575C290uLL},
{0x0747AE1400000001uLL, 0x00000003FFFFFFFEuLL, 0x0000000001D1EB85uLL, 0x0000000003A3D70BuLL},
{0x0747AE1400000001uLL, 0x00000003FFFFFFFFuLL, 0x0000000001D1EB85uLL, 0x0000000001D1EB86uLL},
{0x0747AE1400000001uLL, 0x0000001000000000uLL, 0x0000000000747AE1uLL, 0x0000000400000001uLL},
{0x0747AE1400000001uLL, 0x0000001000000001uLL, 0x0000000000747AE1uLL, 0x00000003FF8B8520uLL},
{0x0747AE1400000001uLL, 0x0000001000000002uLL, 0x0000000000747AE1uLL, 0x00000003FF170A3FuLL},
{0x0747AE1400000001uLL, 0x0000001000000003uLL, 0x0000000000747AE1uLL, 0x00000003FEA28F5EuLL},
{0x0747AE1400000001uLL, 0x0000001000000010uLL, 0x0000000000747AE1uLL, 0x00000003F8B851F1uLL},
{0x0747AE1400000001uLL, 0x00000010078644FAuLL, 0x0000000000744433uLL, 0x0000000B94A5DA33uLL},
{0x0747AE1400000001uLL, 0x000000100747AE14uLL, 0x00000000007445F9uLL, 0x0000000BB84C4A8DuLL},
{0x0747AE1400000001uLL, 0x000000107FFFFFFFuLL, 0x000000000070F347uLL, 0x000000008070F348uLL},
{0x0747AE1400000001uLL, 0x0000001080000000uLL, 0x000000000070F347uLL, 0x0000000080000001uLL},
{0x0747AE1400000001uLL, 0x00000010FFFFFFFDuLL, 0x00000000006DA0D4uLL, 0x000000000148E27DuLL},
{0x0747AE1400000001uLL, 0x00000010FFFFFFFEuLL, 0x00000000006DA0D4uLL, 0x0000000000DB41A9uLL},
{0x0747AE1400000001uLL, 0x00000010FFFFFFFFuLL, 0x00000000006DA0D4uLL, 0x00000000006DA0D5uLL},
{0x0747AE1400000001uLL, 0x078644FA00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FA80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FAFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FAFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x078644FAFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE1400000000uLL, 0x0000000000000001uLL, 0x0000000000000001uLL},
{0x0747AE1400000001uLL, 0x0747AE1400000001uLL, 0x0000000000000001uLL, 0x0000000000000000uLL},
{0x0747AE1400000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE1400000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE1400000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE14078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE140747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE147FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE1480000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE14FFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE14FFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x0747AE14FFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x7FFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000000000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000000000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000000000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000000000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000000000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x80000000078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x800000000747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x800000007FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x8000000080000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x80000000FFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x80000000FFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0x80000000FFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFD80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFDFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFDFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFDFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFE80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFEFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFEFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFEFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF00000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF00000001uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF00000002uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF00000003uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF00000010uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF078644FAuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF0747AE14uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF7FFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFF80000000uLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFFFFFFFFFDuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFFFFFFFFFEuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000001uLL, 0xFFFFFFFFFFFFFFFFuLL, 0x0000000000000000uLL, 0x0747AE1400000001uLL},
{0x0747AE1400000002uLL, 0x0000000000000001uLL, 0x0747AE1400000002uLL, 0x0000000000000000uLL},
{0x0747AE1400000002uLL, 0x0000000000000002uLL, 0x03A3D70A00000001uLL, 0x0000000000000000uLL},
{0x0747AE1400000002uLL, 0x0000000000000003uLL, 0x026D3A06AAAAAAABuLL, 0x0000000000000001uLL},
{0x0747AE1400000002uLL, 0x0000000000000010uLL, 0x00747AE140000000uLL, 0x0000000000000002uLL},
{0x0747AE1400000002uLL, 0x00000000078644FAuLL, 0x00000000F7AE911DuLL, 0x0000000000CC95B0uLL},
{0x0747AE1400000002uLL, 0x000000000747AE14uLL, 0x0000000100000000uLL, 0x0000000000000002uLL},
{0x0747AE1400000002uLL, 0x000000007FFFFFFFuLL, 0x000000000E8F5C28uLL, 0x000000000E8F5C2AuLL},
{0x0747AE1400000002uLL, 0x0000000080000000uLL, 0x000000000E8F5C28uLL, 0x0000000000000002uLL},
{0x0747AE1400000002uLL, 0x00000000FFFFFFFDuLL, 0x000000000747AE14uLL, 0x0000000015D70A3EuLL},
{0x0747AE1400000002uLL, 0x00000000FFFFFFFEuLL, 0x000000000747AE14uLL, 0x000000000E8F5C2AuLL},
{0x0747AE1400000002uLL, 0x00000000FFFFFFFFuLL, 0x000000000747AE14uLL, 0x000000000747AE16uLL},
{0x0747AE1400000002uLL, 0x0000000100000000uLL, 0x000000000747AE14uLL, 0x0000000000000002uLL},
{0x0747AE1400000002uLL, 0x0000000100000001uLL, 0x000000000747AE13uLL, 0x00000000F8B851EFuLL},
{0x0747AE1400000002uLL, 0x0000000100000002uLL, 0x000000000747AE13uLL, 0x00000000F170A3DCuLL},
{0x0747AE1400000002uLL, 0x0000000100000003uLL, 0x000000000747AE13uLL, 0x00000000EA28F5C9uLL},
{0x0747AE1400000002uLL, 0x0000000100000010uLL, 0x000000000747AE13uLL, 0x000000008B851ED2uLL},
{0x0747AE1400000002uLL, 0x00000001078644FAuLL, 0x000000000712773FuLL, 0x000000000C50D07CuLL},
{0x0747AE1400000002uLL, 0x000000010747AE14uLL, 0x00000000071425A5uLL, 0x00000000CD13E91EuLL},
{0x0747AE1400000002uLL, 0x000000017FFFFFFFuLL, 0x0000000004DA740DuLL, 0x0000000084DA740FuLL},
{0x0747AE1400000002uLL, 0x0000000180000000uLL, 0x0000000004DA740DuLL, 0x0000000080000002uLL},
{0x0747AE1400000002uLL, 0x00000001FFFFFFFDuLL, 0x0000000003A3D70AuLL, 0x000000000AEB8520uLL},
{0x0747AE1400000002uLL, 0x00000001FFFFFFFEuLL, 0x0000000003A3D70AuLL, 0x000000000747AE16uLL},
{0x0747AE1400000002uLL, 0x00000001FFFFFFFFuLL, 0x0000000003A3D70AuLL, 0x0000000003A3D70CuLL},
{0x0747AE1400000002uLL, 0x0000000200000000uLL, 0x0000000003A3D70AuLL, 0x0000000000000002uLL},
{0x0747AE1400000002uLL, 0x0000000200000001uLL, 0x0000000003A3D709uLL, 0x00000001FC5C28F9uLL},
{0x0747AE1400000002uLL, 0x0000000200000002uLL, 0x0000000003A3D709uLL, 0x00000001F8B851F0uLL},
{0x0747AE1400000002uLL, 0x0000000200000003uLL, 0x0000000003A3D709uLL, 0x00000001F5147AE7uLL},
{0x0747AE1400000002uLL, 0x0000000200000010uLL, 0x0000000003A3D709uLL, 0x00000001C5C28F72uLL},
{0x0747AE1400000002uLL, 0x00000002078644FAuLL, 0x0000000003965801uLL, 0x000000007847CB08uLL},
{0x0747AE1400000002uLL, 0x000000020747AE14uLL, 0x000000000396C6B1uLL, 0x000000014B152C2EuLL},
{0x0747AE1400000002uLL, 0x000000027FFFFFFFuLL, 0x0000000002E978D4uLL, 0x0000000202E978D6uLL},
{0x0747AE1400000002uLL, 0x0000000280000000uLL, 0x0000000002E978D4uLL, 0x0000000200000002uLL},
{0x0747AE1400000002uLL, 0x00000002FFFFFFFDuLL, 0x00000000026D3A06uLL, 0x000000020747AE14uLL},
{0x0747AE1400000002uLL, 0x00000002FFFFFFFEuLL, 0x00000000026D3A06uLL, 0x0000000204DA740EuLL},
{0x0747AE1400000002uLL, 0x00000002FFFFFFFFuLL, 0x00000000026D3A06uLL, 0x00000002026D3A08uLL},
{0x0747AE1400000002uLL, 0x0000000300000000uLL, 0x00000000026D3A06uLL, 0x0000000200000002uLL},
{0x0747AE1400000002uLL, 0x0000000300000001uLL, 0x00000000026D3A06uLL, 0x00000001FD92C5FCuLL},
{0x0747AE1400000002uLL, 0x0000000300000002uLL, 0x00000000026D3A06uLL, 0x00000001FB258BF6uLL},
{0x0747AE1400000002uLL, 0x0000000300000003uLL, 0x00000000026D3A06uLL, 0x00000001F8B851F0uLL},
{0x0747AE1400000002uLL, 0x0000000300000010uLL, 0x00000000026D3A06uLL, 0x00000001D92C5FA2uLL},
{0x0747AE1400000002uLL, 0x00000003078644FAuLL, 0x0000000002673301uLL, 0x000000018425