// widget-helper.h
// LiVES
// (c) G. Finch 2012 - 2013 <salsaman@gmail.com>
// released under the GNU GPL 3 or later
// see file ../COPYING or www.gnu.org for licensing details

#ifndef HAS_LIVES_WIDGET_HELPER_H
#define HAS_LIVES_WIDGET_HELPER_H

#ifndef M_PI
#define M_PI 3.1415926536
#endif

typedef enum {
  LIVES_DISPLAY_TYPE_UNKNOWN=0,
  LIVES_DISPLAY_TYPE_X11,
  LIVES_DISPLAY_TYPE_WIN32
} lives_display_t;


#define W_PACKING_WIDTH  10 // packing width for widgets with labels
#define W_PACKING_HEIGHT 10 // packing height for widgets
#define W_BORDER_WIDTH   10 // default border width


#ifdef PAINTER_CAIRO
typedef cairo_t lives_painter_t;
typedef cairo_surface_t lives_painter_surface_t;

typedef cairo_format_t lives_painter_format_t;

#define LIVES_PAINTER_FORMAT_A1   CAIRO_FORMAT_A1
#define LIVES_PAINTER_FORMAT_A8   CAIRO_FORMAT_A8
#define LIVES_PAINTER_FORMAT_ARGB32 CAIRO_FORMAT_ARGB32


typedef cairo_content_t lives_painter_content_t; // eg. color, alpha, color+alpha

#define LIVES_PAINTER_CONTENT_COLOR CAIRO_CONTENT_COLOR


typedef cairo_operator_t lives_painter_operator_t;

#define LIVES_PAINTER_OPERATOR_UNKNOWN CAIRO_OPERATOR_OVER
#define LIVES_PAINTER_OPERATOR_DEFAULT CAIRO_OPERATOR_OVER

#define LIVES_PAINTER_OPERATOR_DEST_OUT CAIRO_OPERATOR_DEST_OUT
#if CAIRO_VERSION < CAIRO_VERSION_ENCODE(1, 10, 0)
#define LIVES_PAINTER_OPERATOR_DIFFERENCE CAIRO_OPERATOR_OVER
#else
#define LIVES_PAINTER_OPERATOR_DIFFERENCE CAIRO_OPERATOR_DIFFERENCE
#endif

typedef cairo_fill_rule_t lives_painter_fill_rule_t;

#define LIVES_PAINTER_FILL_RULE_WINDING  CAIRO_FILL_RULE_WINDING 
#define LIVES_PAINTER_FILL_RULE_EVEN_ODD CAIRO_FILL_RULE_EVEN_ODD 


#endif


#ifdef GUI_GTK

#ifndef G_ENCODE_VERSION
#define G_ENCODE_VERSION(major,minor) ((major) << 16 | (minor) << 8)
#endif

#define return_true gtk_true

typedef GObject LiVESObject;

typedef GtkJustification LiVESJustification;

#define LIVES_JUSTIFY_LEFT   GTK_JUSTIFY_LEFT
#define LIVES_JUSTIFY_RIGHT  GTK_JUSTIFY_RIGHT
#define LIVES_JUSTIFY_CENTER GTK_JUSTIFY_CENTER
#define LIVES_JUSTIFY_FILL   GTK_JUSTIFY_RIGHT

typedef GtkOrientation LiVESOrientation;
#define LIVES_ORIENTATION_HORIZONTAL GTK_ORIENTATION_HORIZONTAL
#define LIVES_ORIENTATION_VERTICAL   GTK_ORIENTATION_VERTICAL

typedef GdkEvent                          LiVESXEvent;
typedef GdkEventButton                    LiVESXEventButton;
typedef GdkDisplay                        LiVESXDisplay;
typedef GdkScreen                         LiVESXScreen;
typedef GdkDevice                         LiVESXDevice;

#if GTK_CHECK_VERSION(3,0,0)
typedef GdkDeviceManager                  LiVESXDeviceManager;
#endif

#if GTK_CHECK_VERSION(3,0,0)
#define LIVES_WIDGET_EVENT_EXPOSE_EVENT "draw"
#define GTK_OBJECT(a)                     a
#else
#define LIVES_WIDGET_EVENT_EXPOSE_EVENT "expose_event"
#endif
typedef GtkWidget                         LiVESWidget;
typedef GtkWindow                         LiVESWindow;
typedef GtkContainer                      LiVESContainer;
typedef GtkBin                            LiVESBin;
typedef GtkDialog                         LiVESDialog;
typedef GtkBox                            LiVESBox;
typedef GtkComboBox                       LiVESCombo;
typedef GtkComboBox                       LiVESComboBox;
typedef GtkButton                         LiVESButton;
typedef GtkToggleButton                   LiVESToggleButton;
typedef GtkTextView                       LiVESTextView;
typedef GtkEntry                          LiVESEntry;
typedef GtkRadioButton                    LiVESRadioButton;
typedef GtkSpinButton                     LiVESSpinButton;
typedef GtkLabel                          LiVESLabel;
typedef GtkImage                          LiVESImage;
typedef GtkFileChooser                    LiVESFileChooser;
typedef GtkMenu                           LiVESMenu;
typedef GtkMenuItem                       LiVESMenuItem;
typedef GtkCheckMenuItem                  LiVESCheckMenuItem;
typedef GtkImageMenuItem                  LiVESImageMenuItem;
typedef GtkTreeView                       LiVESTreeView;
typedef GtkTreeModel                      LiVESTreeModel;
typedef GtkTreeIter                       LiVESTreeIter;
typedef GtkTable                          LiVESTable;

#if GTK_CHECK_VERSION(2,14,0)
typedef GtkScaleButton                    LiVESScaleButton;
#else
typedef GtkRange                          LiVESScaleButton;
#endif

typedef GClosure                          LiVESWidgetClosure;


#if GTK_CHECK_VERSION(3,0,0)
#define LIVES_WIDGET_COLOR_HAS_ALPHA (1)
#define LIVES_WIDGET_COLOR_SCALE(x) (x) ///< macro to get 0. to 1.
#define LIVES_WIDGET_COLOR_SCALE_255(x) ((double)x/255.) ///< macro to convert 0. - 255. to component
typedef GdkRGBA                           LiVESWidgetColor;
typedef GtkStateFlags LiVESWidgetState;

#define LIVES_WIDGET_STATE_NORMAL         GTK_STATE_FLAG_NORMAL
#define LIVES_WIDGET_STATE_ACTIVE         GTK_STATE_FLAG_ACTIVE
#define LIVES_WIDGET_STATE_PRELIGHT       GTK_STATE_FLAG_PRELIGHT
#define LIVES_WIDGET_STATE_SELECTED       GTK_STATE_FLAG_SELECTED
#define LIVES_WIDGET_STATE_INSENSITIVE    GTK_STATE_FLAG_INSENSITIVE
#define LIVES_WIDGET_STATE_INCONSISTENT   GTK_STATE_FLAG_INCONSISTENT
#define LIVES_WIDGET_STATE_FOCUSED        GTK_STATE_FLAG_FOCUSED
#define LIVES_WIDGET_STATE_BACKDROP       GTK_STATE_FLAG_BACKDROP

#else
#define LIVES_WIDGET_COLOR_HAS_ALPHA (0)
#define LIVES_WIDGET_COLOR_SCALE(x) ((double)x/65535.)     ///< macro to get 0. to 1.
#define LIVES_WIDGET_COLOR_SCALE_255(x) ((int)((double)x*256.+.5))     ///< macro to get 0 - 255
typedef GdkColor                          LiVESWidgetColor;
typedef GtkStateType LiVESWidgetState;

#define LIVES_WIDGET_STATE_NORMAL         GTK_STATE_NORMAL
#define LIVES_WIDGET_STATE_ACTIVE         GTK_STATE_ACTIVE
#define LIVES_WIDGET_STATE_PRELIGHT       GTK_STATE_PRELIGHT
#define LIVES_WIDGET_STATE_SELECTED       GTK_STATE_SELECTED
#define LIVES_WIDGET_STATE_INSENSITIVE    GTK_STATE_INSENSITIVE
#define LIVES_WIDGET_STATE_INCONSISTENT   GTK_STATE_INSENSITIVE+1
#define LIVES_WIDGET_STATE_FOCUSED        GTK_STATE_INSENSITIVE+2
#define LIVES_WIDGET_STATE_BACKDROP       GTK_STATE_INSENSITIVE+3
#endif

typedef enum {
  LIVES_EXPAND=GTK_EXPAND,
  LIVES_SHRINK=GTK_SHRINK,
  LIVES_FILL=GTK_FILL,
} lives_attach_options_t;


typedef enum {
  LIVES_FILE_CHOOSER_ACTION_OPEN=GTK_FILE_CHOOSER_ACTION_OPEN,
  LIVES_FILE_CHOOSER_ACTION_SAVE=GTK_FILE_CHOOSER_ACTION_SAVE,
  LIVES_FILE_CHOOSER_ACTION_SELECT_FOLDER=GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
  LIVES_FILE_CHOOSER_ACTION_CREATE_FOLDER=GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
  LIVES_FILE_CHOOSER_ACTION_SELECT_DEVICE
} lives_file_chooser_action_t;


typedef enum {
  LIVES_ICON_SIZE_INVALID=GTK_ICON_SIZE_INVALID,
  LIVES_ICON_SIZE_MENU=GTK_ICON_SIZE_MENU,
  LIVES_ICON_SIZE_SMALL_TOOLBAR=GTK_ICON_SIZE_SMALL_TOOLBAR,
  LIVES_ICON_SIZE_LARGE_TOOLBAR=GTK_ICON_SIZE_LARGE_TOOLBAR,
  LIVES_ICON_SIZE_BUTTON=GTK_ICON_SIZE_BUTTON,
  LIVES_ICON_SIZE_DND=GTK_ICON_SIZE_DND,
  LIVES_ICON_SIZE_DIALOG=GTK_ICON_SIZE_DIALOG
} lives_icon_size_t;


#define LIVES_ACCEL_VISIBLE GTK_ACCEL_VISIBLE
  


#if GTK_CHECK_VERSION(3,0,0)
typedef GtkScale                          LiVESRuler;
#else
typedef GtkRuler                          LiVESRuler;
#endif

typedef GtkRange                          LiVESRange;

typedef GtkAdjustment                     LiVESAdjustment;

typedef GdkPixbuf                         LiVESPixbuf;

typedef GdkWindow                         LiVESXWindow;

typedef GdkEventButton                    LiVESXEventButton;

typedef GdkCursor                         LiVESXCursor;

typedef GError                            LiVESError;

#ifndef IS_MINGW
typedef gboolean                          boolean;
#endif

typedef GList                             LiVESList;
typedef GSList                            LiVESSList;

typedef GtkAccelGroup                     LiVESAccelGroup;
typedef GtkAccelFlags                     LiVESAccelFlags;

typedef GdkPixbufDestroyNotify            LiVESPixbufDestroyNotify;

typedef GdkInterpType                     LiVESInterpType;

typedef gpointer                          LiVESObjectPtr;

#define LIVES_WIDGET(widget) GTK_WIDGET(widget)
#define LIVES_PIXBUF(widget) GDK_PIXBUF(widget)
#define LIVES_WINDOW(widget) GTK_WINDOW(widget)
#define LIVES_XWINDOW(widget) GDK_WINDOW(widget)
#define LIVES_BOX(widget) GTK_BOX(widget)
#define LIVES_ENTRY(widget) GTK_ENTRY(widget)
#define LIVES_CONTAINER(widget) GTK_CONTAINER(widget)
#define LIVES_BIN(widget) GTK_BIN(widget)
#define LIVES_ADJUSTMENT(widget) GTK_ADJUSTMENT(widget)
#define LIVES_DIALOG(widget) GTK_DIALOG(widget)
#define LIVES_COMBO(widget) GTK_COMBO_BOX(widget)
#define LIVES_COMBO_BOX(widget) GTK_COMBO_BOX(widget)
#define LIVES_BUTTON(widget) GTK_BUTTON(widget)
#define LIVES_LABEL(widget) GTK_LABEL(widget)
#define LIVES_FILES_CHOOSER(widget) GTK_FILE_CHOOSER(widget)
#define LIVES_RADIO_BUTTON(widget) GTK_RADIO_BUTTON(widget)
#define LIVES_SPIN_BUTTON(widget) GTK_SPIN_BUTTON(widget)
#define LIVES_MENU(widget) GTK_MENU(widget)
#define LIVES_IMAGE_MENU_ITEM(widget) GTK_IMAGE_MENU_ITEM(widget)
#define LIVES_FILE_CHOOSER(widget) GTK_FILE_CHOOSER(widget)

#if GTK_CHECK_VERSION(2,14,0)
#define LIVES_SCALE_BUTTON(widget) GTK_SCALE_BUTTON(widget)
#else
#define LIVES_SCALE_BUTTON(widget) GTK_RANGE(widget)
#endif

#define LIVES_TOGGLE_BUTTON(widget) GTK_TOGGLE_BUTTON(widget)
#define LIVES_TREE_VIEW(widget) GTK_TREE_VIEW(widget)
#define LIVES_TEXT_VIEW(widget) GTK_TEXT_VIEW(widget)

#if GTK_CHECK_VERSION(3,0,0)
#define LIVES_RULER(widget) GTK_SCALE(widget)
#define LIVES_ORIENTABLE(widget) GTK_ORIENTABLE(widget)
#else
#define LIVES_RULER(widget) GTK_RULER(widget)
#endif

#define LIVES_RANGE(widget) GTK_RANGE(widget)

#define LIVES_XEVENT(event) GDK_EVENT(event)

#define LIVES_IS_WIDGET(widget) GTK_IS_WIDGET(widget)
#if GTK_CHECK_VERSION(3,0,0)
#define LIVES_IS_HBOX(widget) (GTK_IS_BOX(widget)&&gtk_orientable_get_orientation(GTK_ORIENTABLE(widget))==GTK_ORIENTATION_HORIZONTAL)
#define LIVES_IS_VBOX(widget) (GTK_IS_BOX(widget)&&gtk_orientable_get_orientation(GTK_ORIENTABLE(widget))==GTK_ORIENTATION_HORIZONTAL)
#else
#define LIVES_IS_HBOX(widget) GTK_IS_HBOX(widget)
#define LIVES_IS_VBOX(widget) GTK_IS_VBOX(widget)
#endif
#define LIVES_IS_COMBO(widget) GTK_IS_COMBO_BOX(widget)
#define LIVES_IS_BUTTON(widget) GTK_IS_BUTTON(widget)

#define LIVES_INTERP_BEST   GDK_INTERP_HYPER
#define LIVES_INTERP_NORMAL GDK_INTERP_BILINEAR
#define LIVES_INTERP_FAST   GDK_INTERP_NEAREST

typedef GLogLevelFlags LiVESLogLevelFlags;

#define LIVES_LOG_LEVEL_WARNING G_LOG_LEVEL_WARNING
#define LIVES_LOG_LEVEL_MASK G_LOG_LEVEL_MASK
#define LIVES_LOG_LEVEL_CRITICAL G_LOG_LEVEL_CRITICAL
#define LIVES_LOG_FATAL_MASK G_LOG_FATAL_MASK

typedef GdkModifierType LiVESModifierType;

#define LIVES_CONTROL_MASK GDK_CONTROL_MASK
#define LIVES_ALT_MASK     GDK_MOD1_MASK
#define LIVES_SHIFT_MASK   GDK_SHIFT_MASK
#define LIVES_LOCK_MASK    GDK_LOCK_MASK

#ifdef GDK_KEY_a
#define LIVES_KEY_Left GDK_KEY_Left
#define LIVES_KEY_Right GDK_KEY_Right
#define LIVES_KEY_Up GDK_KEY_Up
#define LIVES_KEY_Down GDK_KEY_Down

#define LIVES_KEY_Space GDK_KEY_space
#define LIVES_KEY_BackSpace GDK_KEY_BackSpace
#define LIVES_KEY_Return GDK_KEY_Return
#define LIVES_KEY_Tab GDK_KEY_Tab
#define LIVES_KEY_Home GDK_KEY_Home
#define LIVES_KEY_End GDK_KEY_End
#define LIVES_KEY_Slash GDK_KEY_slash
#define LIVES_KEY_Space GDK_KEY_space
#define LIVES_KEY_Plus GDK_KEY_plus
#define LIVES_KEY_Minus GDK_KEY_minus
#define LIVES_KEY_Equal GDK_KEY_equal

#define LIVES_KEY_1 GDK_KEY_1
#define LIVES_KEY_2 GDK_KEY_2
#define LIVES_KEY_3 GDK_KEY_3
#define LIVES_KEY_4 GDK_KEY_4
#define LIVES_KEY_5 GDK_KEY_5
#define LIVES_KEY_6 GDK_KEY_6
#define LIVES_KEY_7 GDK_KEY_7
#define LIVES_KEY_8 GDK_KEY_8
#define LIVES_KEY_9 GDK_KEY_9
#define LIVES_KEY_0 GDK_KEY_0

#define LIVES_KEY_a GDK_KEY_a
#define LIVES_KEY_b GDK_KEY_b
#define LIVES_KEY_c GDK_KEY_c
#define LIVES_KEY_d GDK_KEY_d
#define LIVES_KEY_e GDK_KEY_e
#define LIVES_KEY_f GDK_KEY_f
#define LIVES_KEY_g GDK_KEY_g
#define LIVES_KEY_h GDK_KEY_h
#define LIVES_KEY_i GDK_KEY_i
#define LIVES_KEY_j GDK_KEY_j
#define LIVES_KEY_k GDK_KEY_k
#define LIVES_KEY_l GDK_KEY_l
#define LIVES_KEY_m GDK_KEY_m
#define LIVES_KEY_n GDK_KEY_n
#define LIVES_KEY_o GDK_KEY_o
#define LIVES_KEY_p GDK_KEY_p
#define LIVES_KEY_q GDK_KEY_q
#define LIVES_KEY_r GDK_KEY_r
#define LIVES_KEY_s GDK_KEY_s
#define LIVES_KEY_t GDK_KEY_t
#define LIVES_KEY_u GDK_KEY_u
#define LIVES_KEY_v GDK_KEY_v
#define LIVES_KEY_w GDK_KEY_w
#define LIVES_KEY_x GDK_KEY_x
#define LIVES_KEY_y GDK_KEY_y
#define LIVES_KEY_z GDK_KEY_z

#define LIVES_KEY_F1 GDK_KEY_F1
#define LIVES_KEY_F2 GDK_KEY_F2
#define LIVES_KEY_F3 GDK_KEY_F3
#define LIVES_KEY_F4 GDK_KEY_F4
#define LIVES_KEY_F5 GDK_KEY_F5
#define LIVES_KEY_F6 GDK_KEY_F6
#define LIVES_KEY_F7 GDK_KEY_F7
#define LIVES_KEY_F8 GDK_KEY_F8
#define LIVES_KEY_F9 GDK_KEY_F9
#define LIVES_KEY_F10 GDK_KEY_F10
#define LIVES_KEY_F11 GDK_KEY_F11
#define LIVES_KEY_F12 GDK_KEY_F12

#define LIVES_KEY_Page_Up GDK_KEY_Page_Up
#define LIVES_KEY_Page_Down GDK_KEY_Page_Down

#define LIVES_KEY_Escape GDK_KEY_Escape

#else
#define LIVES_KEY_Left GDK_Left
#define LIVES_KEY_Right GDK_Right
#define LIVES_KEY_Up GDK_Up
#define LIVES_KEY_Down GDK_Down

#define LIVES_KEY_Space GDK_space
#define LIVES_KEY_BackSpace GDK_BackSpace
#define LIVES_KEY_Return GDK_Return
#define LIVES_KEY_Tab GDK_Tab
#define LIVES_KEY_Home GDK_Home
#define LIVES_KEY_End GDK_End
#define LIVES_KEY_Slash GDK_slash
#define LIVES_KEY_Space GDK_space
#define LIVES_KEY_Plus GDK_plus
#define LIVES_KEY_Minus GDK_minus
#define LIVES_KEY_Equal GDK_equal

#define LIVES_KEY_1 GDK_1
#define LIVES_KEY_2 GDK_2
#define LIVES_KEY_3 GDK_3
#define LIVES_KEY_4 GDK_4
#define LIVES_KEY_5 GDK_5
#define LIVES_KEY_6 GDK_6
#define LIVES_KEY_7 GDK_7
#define LIVES_KEY_8 GDK_8
#define LIVES_KEY_9 GDK_9
#define LIVES_KEY_0 GDK_0

#define LIVES_KEY_a GDK_a
#define LIVES_KEY_b GDK_b
#define LIVES_KEY_c GDK_c
#define LIVES_KEY_d GDK_d
#define LIVES_KEY_e GDK_e
#define LIVES_KEY_f GDK_f
#define LIVES_KEY_g GDK_g
#define LIVES_KEY_h GDK_h
#define LIVES_KEY_i GDK_i
#define LIVES_KEY_j GDK_j
#define LIVES_KEY_k GDK_k
#define LIVES_KEY_l GDK_l
#define LIVES_KEY_m GDK_m
#define LIVES_KEY_n GDK_n
#define LIVES_KEY_o GDK_o
#define LIVES_KEY_p GDK_p
#define LIVES_KEY_q GDK_q
#define LIVES_KEY_r GDK_r
#define LIVES_KEY_s GDK_s
#define LIVES_KEY_t GDK_t
#define LIVES_KEY_u GDK_u
#define LIVES_KEY_v GDK_v
#define LIVES_KEY_w GDK_w
#define LIVES_KEY_x GDK_x
#define LIVES_KEY_y GDK_y
#define LIVES_KEY_z GDK_z

#define LIVES_KEY_F1 GDK_F1
#define LIVES_KEY_F2 GDK_F2
#define LIVES_KEY_F3 GDK_F3
#define LIVES_KEY_F4 GDK_F4
#define LIVES_KEY_F5 GDK_F5
#define LIVES_KEY_F6 GDK_F6
#define LIVES_KEY_F7 GDK_F7
#define LIVES_KEY_F8 GDK_F8
#define LIVES_KEY_F9 GDK_F9
#define LIVES_KEY_F10 GDK_F10
#define LIVES_KEY_F11 GDK_F11
#define LIVES_KEY_F12 GDK_F12

#define LIVES_KEY_Page_Up GDK_Page_Up
#define LIVES_KEY_Page_Down GDK_Page_Down

#define LIVES_KEY_Escape GDK_Escape

#endif

#endif


#ifdef GUI_QT
typedef QImage                            LiVESPixbuf;
typedef bool                              boolean;
typedef int                               gint;
typedef uchar                             guchar;
typedef (void *)                          gpointer;  
typedef (void *)(LiVESPixbufDestroyNotify(uchar *, gpointer));

// etc.



#define LIVES_INTERP_BEST   Qt::SmoothTransformation
#define LIVES_INTERP_NORMAL Qt::SmoothTransformation
#define LIVES_INTERP_FAST   Qt::FastTransformation


#endif














// basic functions (wrappers for Toolkit functions)

// lives_painter_functions

lives_painter_t *lives_painter_create(lives_painter_surface_t *target);
lives_painter_t *lives_painter_create_from_widget(LiVESWidget *);
void lives_painter_set_source_pixbuf (lives_painter_t *, const LiVESPixbuf *, double pixbuf_x, double pixbuf_y);
void lives_painter_set_source_surface (lives_painter_t *, lives_painter_surface_t *, double x, double y);
lives_painter_surface_t *lives_painter_image_surface_create(lives_painter_format_t format, int width, int height);
lives_painter_surface_t *lives_painter_image_surface_create_for_data(uint8_t *data, lives_painter_format_t, 
								     int width, int height, int stride);
lives_painter_surface_t *lives_painter_surface_create_from_widget(LiVESWidget *, lives_painter_content_t, 
								  int width, int height);
void lives_painter_surface_flush(lives_painter_surface_t *);

void lives_painter_destroy(lives_painter_t *);
void lives_painter_surface_destroy(lives_painter_surface_t *);

void lives_painter_new_path(lives_painter_t *);

void lives_painter_paint(lives_painter_t *);
void lives_painter_fill(lives_painter_t *);
void lives_painter_stroke(lives_painter_t *);
void lives_painter_clip(lives_painter_t *);

void lives_painter_set_source_rgb(lives_painter_t *, double red, double green, double blue);
void lives_painter_set_source_rgba(lives_painter_t *, double red, double green, double blue, double alpha);

void lives_painter_set_line_width(lives_painter_t *, double width);

void lives_painter_translate(lives_painter_t *, double x, double y);

void lives_painter_rectangle(lives_painter_t *, double x, double y, double width, double height);
void lives_painter_arc(lives_painter_t *, double xc, double yc, double radius, double angle1, double angle2);
void lives_painter_line_to(lives_painter_t *, double x, double y);
void lives_painter_move_to(lives_painter_t *, double x, double y);

boolean lives_painter_set_operator(lives_painter_t *, lives_painter_operator_t);

void lives_painter_set_fill_rule(lives_painter_t *, lives_painter_fill_rule_t);


lives_painter_surface_t *lives_painter_get_target(lives_painter_t *);
int lives_painter_format_stride_for_width(lives_painter_format_t, int width);

uint8_t *lives_painter_image_surface_get_data(lives_painter_surface_t *);
int lives_painter_image_surface_get_width(lives_painter_surface_t *);
int lives_painter_image_surface_get_height(lives_painter_surface_t *);
int lives_painter_image_surface_get_stride(lives_painter_surface_t *);
lives_painter_format_t lives_painter_image_surface_get_format(lives_painter_surface_t *);




// utils


// object funcs.

void lives_object_unref(LiVESObjectPtr);


// lives_pixbuf functions

int lives_pixbuf_get_width(const LiVESPixbuf *);
int lives_pixbuf_get_height(const LiVESPixbuf *);
boolean lives_pixbuf_get_has_alpha(const LiVESPixbuf *);
int lives_pixbuf_get_rowstride(const LiVESPixbuf *);
int lives_pixbuf_get_n_channels(const LiVESPixbuf *);
unsigned char *lives_pixbuf_get_pixels(const LiVESPixbuf *);
const unsigned char *lives_pixbuf_get_pixels_readonly(const LiVESPixbuf *);
LiVESPixbuf *lives_pixbuf_new(boolean has_alpha, int width, int height);
LiVESPixbuf *lives_pixbuf_new_from_data (const unsigned char *buf, boolean has_alpha, int width, int height, 
					 int rowstride, LiVESPixbufDestroyNotify lives_free_buffer_fn, 
					 gpointer destroy_fn_data);

LiVESPixbuf *lives_pixbuf_new_from_file(const char *filename, LiVESError **error);
LiVESWidget *lives_image_new_from_pixbuf(LiVESPixbuf *);
LiVESPixbuf *lives_pixbuf_new_from_file_at_scale(const char *filename, int width, int height, boolean preserve_aspect_ratio,
						 LiVESError **error);


LiVESPixbuf *lives_pixbuf_scale_simple(const LiVESPixbuf *src, int dest_width, int dest_height, 
				       LiVESInterpType interp_type);

// basic widget fns

void lives_widget_set_sensitive(LiVESWidget *, boolean state);
boolean lives_widget_get_sensitive(LiVESWidget *);

void lives_widget_show(LiVESWidget *);
void lives_widget_show_all(LiVESWidget *);
void lives_widget_hide(LiVESWidget *);
void lives_widget_destroy(LiVESWidget *);

void lives_widget_queue_draw(LiVESWidget *);
void lives_widget_queue_draw_area(LiVESWidget *, int x, int y, int width, int height);
void lives_widget_queue_resize(LiVESWidget *);
void lives_widget_set_size_request(LiVESWidget *, int width, int height);

int lives_dialog_run(LiVESDialog *);

void lives_widget_set_bg_color(LiVESWidget *, LiVESWidgetState state, const LiVESWidgetColor *);
void lives_widget_set_fg_color(LiVESWidget *, LiVESWidgetState state, const LiVESWidgetColor *);
void lives_widget_set_text_color(LiVESWidget *, LiVESWidgetState state, const LiVESWidgetColor *);
void lives_widget_set_base_color(LiVESWidget *, LiVESWidgetState state, const LiVESWidgetColor *);

void lives_widget_get_fg_state_color(LiVESWidget *, LiVESWidgetState state, LiVESWidgetColor *);
void lives_widget_get_bg_state_color(LiVESWidget *, LiVESWidgetState state, LiVESWidgetColor *);

boolean lives_color_parse(const char *spec, LiVESWidgetColor *);

LiVESWidgetColor *lives_widget_color_copy(LiVESWidgetColor *c1orNULL, const LiVESWidgetColor *c2);

LiVESWidget *lives_image_new_from_file(const char *filename);
LiVESWidget *lives_image_new_from_stock(const char *stock_id, lives_icon_size_t size);

void lives_image_set_from_pixbuf(LiVESImage *, LiVESPixbuf *);
LiVESPixbuf *lives_image_get_pixbuf(LiVESImage *);

LiVESWidget *lives_dialog_get_content_area(LiVESDialog *);
LiVESWidget *lives_dialog_get_action_area(LiVESDialog *);

void lives_dialog_add_action_widget(LiVESDialog *, LiVESWidget *, int response_id);

void lives_window_set_title(LiVESWindow *, const char *title);
void lives_window_move(LiVESWindow *, int x, int y);
void lives_window_resize(LiVESWindow *, int width, int height);
void lives_window_present(LiVESWindow *);
void lives_window_fullscreen(LiVESWindow *);
void lives_window_unfullscreen(LiVESWindow *);
void lives_window_maximize(LiVESWindow *);
void lives_window_unmaximize(LiVESWindow *);

LiVESAdjustment *lives_adjustment_new(double value, double lower, double upper, 
						   double step_increment, double page_increment, double page_size);

void lives_box_set_homogeneous(LiVESBox *, boolean homogeneous);
void lives_box_set_spacing(LiVESBox *, int spacing);

void lives_box_pack_start(LiVESBox *, LiVESWidget *child, boolean expand, boolean fill, uint32_t padding);
void lives_box_pack_end(LiVESBox *, LiVESWidget *child, boolean expand, boolean fill, uint32_t padding);

LiVESWidget *lives_hbox_new(boolean homogeneous, int spacing);
LiVESWidget *lives_vbox_new(boolean homogeneous, int spacing);

LiVESWidget *lives_hseparator_new(void);
LiVESWidget *lives_vseparator_new(void);

LiVESWidget *lives_hbutton_box_new(void);
LiVESWidget *lives_vbutton_box_new(void);

LiVESWidget *lives_hscale_new(LiVESAdjustment *);
LiVESWidget *lives_vscale_new(LiVESAdjustment *);

LiVESWidget *lives_hpaned_new(void);
LiVESWidget *lives_vpaned_new(void);

LiVESWidget *lives_hscrollbar_new(LiVESAdjustment *);
LiVESWidget *lives_vscrollbar_new(LiVESAdjustment *);

LiVESWidget *lives_combo_new(void);
LiVESWidget *lives_combo_new_with_model (LiVESTreeModel *);
LiVESTreeModel *lives_combo_get_model(LiVESCombo *);

void lives_combo_append_text(LiVESCombo *, const char *text);
void lives_combo_set_entry_text_column(LiVESCombo *, int column);

char *lives_combo_get_active_text(LiVESCombo *) WARN_UNUSED;
void lives_combo_set_active_text(LiVESCombo *, const char *text);
void lives_combo_set_active_index(LiVESCombo *, int index);
boolean lives_combo_get_active_iter(LiVESCombo *, LiVESTreeIter *);
void lives_combo_set_active_iter(LiVESCombo *, LiVESTreeIter *);
void lives_combo_set_active_string(LiVESCombo *, const char *active_str);

LiVESWidget *lives_combo_get_entry(LiVESCombo *);

void lives_combo_populate(LiVESCombo *, LiVESList *list);

boolean lives_toggle_button_get_active(LiVESToggleButton *);
void lives_toggle_button_set_active(LiVESToggleButton *, boolean active);


LiVESWidget *lives_button_new_with_mnemonic(const char *label);

void lives_button_set_label(LiVESButton *, const char *label);


void lives_tooltips_set (LiVESWidget *, const char *tip_text);

LiVESSList *lives_radio_button_get_group(LiVESRadioButton *);

LiVESWidget *lives_widget_get_parent(LiVESWidget *);
LiVESWidget *lives_widget_get_toplevel(LiVESWidget *);

LiVESXWindow *lives_widget_get_xwindow(LiVESWidget *);

void lives_widget_set_can_focus(LiVESWidget *, boolean state);
void lives_widget_set_can_default(LiVESWidget *, boolean state);

void lives_container_remove(LiVESContainer *, LiVESWidget *);
void lives_container_add(LiVESContainer *, LiVESWidget *);
void lives_container_set_border_width(LiVESContainer *container, uint32_t width);

double lives_spin_button_get_value(LiVESSpinButton *);
int lives_spin_button_get_value_as_int(LiVESSpinButton *);

void lives_spin_button_set_value(LiVESSpinButton *, double value);
void lives_spin_button_set_range(LiVESSpinButton *button, double min, double max);

double lives_ruler_get_value(LiVESRuler *);
double lives_ruler_set_value(LiVESRuler *, double value);

void lives_ruler_set_range(LiVESRuler *, double lower, double upper, double position, double max_size);
double lives_ruler_set_upper(LiVESRuler *, double value);
double lives_ruler_set_lower(LiVESRuler *, double value);

int lives_widget_get_allocation_x(LiVESWidget *);
int lives_widget_get_allocation_y(LiVESWidget *);
int lives_widget_get_allocation_width(LiVESWidget *);
int lives_widget_get_allocation_height(LiVESWidget *);

void lives_widget_set_state(LiVESWidget *, LiVESWidgetState state);
LiVESWidgetState lives_widget_get_state(LiVESWidget *widget);

LiVESWidget *lives_bin_get_child(LiVESBin *);

boolean lives_widget_is_sensitive(LiVESWidget *);
boolean lives_widget_is_visible(LiVESWidget *);
boolean lives_widget_is_realized(LiVESWidget *);


double lives_adjustment_get_upper(LiVESAdjustment *);
double lives_adjustment_get_lower(LiVESAdjustment *);
double lives_adjustment_get_page_size(LiVESAdjustment *);

void lives_adjustment_set_upper(LiVESAdjustment *, double upper);
void lives_adjustment_set_lower(LiVESAdjustment *, double lower);
void lives_adjustment_set_page_size(LiVESAdjustment *, double page_size);

LiVESAdjustment *lives_tree_view_get_hadjustment(LiVESTreeView *);

const char *lives_label_get_text(LiVESLabel *);
void lives_label_set_text(LiVESLabel *, const char *text);
void lives_label_set_text_with_mnemonic(LiVESLabel *, const char *text);

void lives_entry_set_editable(LiVESEntry *, boolean editable);
const char *lives_entry_get_text(LiVESEntry *);
void lives_entry_set_text(LiVESEntry *, const char *text);

double lives_scale_button_get_value(LiVESScaleButton *);

LiVESWidget *lives_menu_new(void);

void lives_menu_popup(LiVESMenu *, LiVESXEventButton *);

LiVESWidget *lives_menu_item_new(void);
LiVESWidget *lives_menu_item_new_with_mnemonic(const char *label);
LiVESWidget *lives_menu_item_new_with_label(const char *label);
LiVESWidget *lives_check_menu_item_new_with_mnemonic(const char *label);
LiVESWidget *lives_check_menu_item_new_with_label(const char *label);
LiVESWidget *lives_image_menu_item_new_with_label(const char *label);
LiVESWidget *lives_image_menu_item_new_with_mnemonic(const char *label);
LiVESWidget *lives_image_menu_item_new_from_stock(const char *stock_id, LiVESAccelGroup *accel_group);

void lives_image_menu_item_set_image(LiVESImageMenuItem *, LiVESWidget *image);

void lives_menu_item_set_submenu(LiVESMenuItem *, LiVESWidget *);

void lives_check_menu_item_set_active(LiVESCheckMenuItem *, boolean state);

void lives_menu_set_title(LiVESMenu *, const char *title);

char *lives_file_chooser_get_filename(LiVESFileChooser *);
LiVESSList *lives_file_chooser_get_filenames(LiVESFileChooser *);

void lives_widget_grab_focus(LiVESWidget *);

void lives_widget_set_tooltip_text(LiVESWidget *, const char *text);

void lives_table_attach(LiVESTable *, LiVESWidget *child, uint32_t left, uint32_t right, 
			uint32_t top, uint32_t bottom, lives_attach_options_t xoptions, lives_attach_options_t yoptions,
			uint32_t xpad, uint32_t ypad);


LiVESAccelGroup *lives_accel_group_new(void);
void lives_accel_group_connect(LiVESAccelGroup *, uint32_t key, LiVESModifierType mod, LiVESAccelFlags flags, LiVESWidgetClosure *closure);
void lives_accel_group_disconnect(LiVESAccelGroup *, LiVESWidgetClosure *closure);
void lives_accel_groups_activate(LiVESObject *object, uint32_t key, LiVESModifierType mod);

void lives_widget_add_accelerator(LiVESWidget *, const char *accel_signal, LiVESAccelGroup *accel_group,
				  uint32_t accel_key, LiVESModifierType accel_mods, LiVESAccelFlags accel_flags);

void lives_widget_get_pointer(LiVESXDevice *, LiVESWidget *, int *x, int *y);
LiVESXWindow *lives_display_get_window_at_pointer (LiVESXDevice *, LiVESXDisplay *, int *win_x, int *win_y);
void lives_display_get_pointer (LiVESXDevice *, LiVESXDisplay *, LiVESXScreen **, int *x, int *y, LiVESModifierType *mask);
void lives_display_warp_pointer (LiVESXDevice *, LiVESXDisplay *, LiVESXScreen *, int x, int y);

LiVESXDisplay *lives_widget_get_display(LiVESWidget *);
lives_display_t lives_widget_get_display_type(LiVESWidget *);

uint64_t lives_widget_get_xwinid(LiVESWidget *, const char *failure_msg);


// optional (return TRUE if implemented)

boolean lives_dialog_set_has_separator(LiVESDialog *, boolean has);
boolean lives_widget_set_hexpand(LiVESWidget *, boolean state);
boolean lives_widget_set_vexpand(LiVESWidget *, boolean state);
boolean lives_image_menu_item_set_always_show_image(LiVESImageMenuItem *, boolean show);
boolean lives_scale_button_set_orientation(LiVESScaleButton *, LiVESOrientation orientation);

// compound functions (composed of basic functions)

void lives_painter_set_source_to_bg(lives_painter_t *, LiVESWidget *);

LiVESWidget *lives_standard_label_new(const char *text);
LiVESWidget *lives_standard_label_new_with_mnemonic(const char *text, LiVESWidget *mnemonic_widget);

LiVESWidget *lives_standard_check_button_new(const char *labeltext, boolean use_mnemonic, LiVESBox *box, const char *tooltip);
LiVESWidget *lives_standard_radio_button_new(const char *labeltext, boolean use_mnemonic, LiVESSList *rbgroup, 
					     LiVESBox *, const char *tooltip);
LiVESWidget *lives_standard_spin_button_new(const char *labeltext, boolean use_mnemonic, double val, double min, 
					    double max, double step, double page, int dp, LiVESBox *, 
					    const char *tooltip);
LiVESWidget *lives_standard_combo_new (const char *labeltext, boolean use_mnemonic, LiVESList *list, LiVESBox *, 
				       const char *tooltip);

LiVESWidget *lives_standard_entry_new(const char *labeltext, boolean use_mnemonic, const char *txt, int dispwidth, int maxchars, LiVESBox *, 
				      const char *tooltip);

LiVESWidget *lives_standard_dialog_new(const char *title, boolean add_std_buttons);

LiVESWidget *lives_standard_hruler_new(void);

LiVESWidget *lives_standard_scrolled_window_new(int width, int height, LiVESWidget *child);

LiVESWidget *lives_standard_expander_new(const char *label, boolean use_mnemonic, LiVESBox *parent, LiVESWidget *child);

LiVESWidget *lives_volume_button_new(LiVESOrientation orientation, LiVESAdjustment *, double volume);

LiVESWidget *lives_standard_file_button_new(boolean is_dir, const char *def_dir);


// util functions

void lives_widget_apply_theme(LiVESWidget *, LiVESWidgetState state);

void lives_cursor_unref(LiVESXCursor *cursor);

void lives_widget_context_update(void);

void lives_widget_get_fg_color(LiVESWidget *, LiVESWidgetColor *);
void lives_widget_get_bg_color(LiVESWidget *, LiVESWidgetColor *);

LiVESWidget *lives_entry_new_with_max_length(int max);

void lives_widget_unparent(LiVESWidget *);

void lives_tooltips_copy(LiVESWidget *dest, LiVESWidget *source);

size_t calc_spin_button_width(double min, double max, int dp);

int get_box_child_index (LiVESBox *, LiVESWidget *child);

boolean label_act_toggle (LiVESWidget *, LiVESXEventButton *, LiVESToggleButton *);
boolean widget_act_toggle (LiVESWidget *, LiVESToggleButton *);

void lives_spin_button_configure(LiVESSpinButton *, double value, double lower, double upper, 
				 double step_increment, double page_increment);

void set_button_width(LiVESWidget *buttonbox, LiVESWidget *button, int width);

char *text_view_get_text(LiVESTextView *);
void text_view_set_text(LiVESTextView *, const char *text, int len);

void toggle_button_toggle (LiVESToggleButton *);

void set_child_colour(GtkWidget *widget, gpointer set_all);

void unhide_cursor(LiVESXWindow *);
void hide_cursor(LiVESXWindow *);

void get_border_size (LiVESWidget *win, int *bx, int *by);

void lives_widget_set_can_focus(LiVESWidget *, boolean state);
void lives_widget_set_can_focus_and_default(LiVESWidget *);

void lives_general_button_clicked (LiVESButton *, LiVESObjectPtr data_to_free);

LiVESWidget *add_hsep_to_box (LiVESBox *);

LiVESWidget *add_fill_to_box (LiVESBox *);


#define LIVES_JUSTIFY_DEFAULT (widget_opts.default_justify)

#define W_MAX_FILLER_LEN 65535


typedef enum {
  LIVES_CURSOR_NORMAL=0,  ///< must be zero
  LIVES_CURSOR_BUSY,
  LIVES_CURSOR_CENTER_PTR,
  LIVES_CURSOR_HAND2,
  LIVES_CURSOR_SB_H_DOUBLE_ARROW,
  LIVES_CURSOR_CROSSHAIR,
  LIVES_CURSOR_TOP_LEFT_CORNER,
  LIVES_CURSOR_BOTTOM_RIGHT_CORNER,

  /// non-standard cursors
  LIVES_CURSOR_BLOCK,
  LIVES_CURSOR_AUDIO_BLOCK,
  LIVES_CURSOR_FX_BLOCK
} lives_cursor_t;

void lives_set_cursor_style(lives_cursor_t cstyle, LiVESWidget *);

typedef enum {
  LIVES_EXPAND_DEFAULT,
  LIVES_EXPAND_NONE,
  LIVES_EXPAND_EXTRA
} lives_expand_t;


typedef struct {
  boolean no_gui; // show nothing !
  boolean swap_label; // swap label/widget position
  boolean pack_end;
  boolean line_wrap; // line wrapping for labels
  boolean non_modal; // non-modal for dialogs
  boolean expand; // whether spin,check,radio buttons should expand
  boolean apply_theme; // whether to apply theming to widget
  double scale; // scale factor for all sizes
  int packing_width; // default should be W_PACKING_WIDTH
  int packing_height; // default should be W_PACKING_HEIGHT
  int border_width; // default should be W_BORDER_WIDTH
  int filler_len; // length of extra "fill" between widgets
  LiVESWidget *last_label; // label widget of last standard widget (spin,radio,check,entry,combo) [readonly]
  LiVESJustification justify; // justify for labels
  LiVESJustification default_justify;
} widget_opts_t;


widget_opts_t widget_opts;

#ifdef NEED_DEF_WIDGET_OPTS

const widget_opts_t def_widget_opts = {
    FALSE, // no_gui
    FALSE, // swap_label
    FALSE, //pack_end
    FALSE, // line_wrap
    FALSE, // non_modal
    LIVES_EXPAND_DEFAULT, // default expand
    FALSE, // no themeing
    1.0, // default scale
    W_PACKING_WIDTH, // def packing width
    W_PACKING_HEIGHT, // def packing height
    W_BORDER_WIDTH, // def border width
    8, // def fill width (in chars)
    NULL, // last_label
    LIVES_JUSTIFY_LEFT, // justify
    LIVES_JUSTIFY_LEFT // default justify
};

#else

extern const widget_opts_t def_widget_opts;

#endif

#endif

