/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

 /* 
	Code in this file is ugly! Don't waste time to analise that.
	It waits for rewriting or (better) replace with ready made library
	or tool (maybe lex+yacc)
 */

#include <string>
#include "include/support.h"
#include "include/program.h"
#include "include/environment.h"
#include "include/paritycommand.h"

typedef GNode *GNodePtr;

static GtkWidget *copy_piece_widget (LWPiece *piece)
{
	LWPiece *copied = new LWPiece (piece, piece->row);
	return copied->getWidget();
}

static GtkWidget *widget_tree (GNode *node)
{
	GtkWidget *vbox = gtk_vbox_new (FALSE, 0);

	if (node == NULL) return vbox;
	
	if (node->data != NULL)
		gtk_box_pack_start (GTK_BOX (vbox), copy_piece_widget ((LWPiece *) node->data), FALSE, FALSE, 0);
	
	GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	
	for (GNode *n = node->children; n != NULL; n=n->next)
		gtk_box_pack_start (GTK_BOX(hbox), widget_tree (n), TRUE, TRUE, 0);

	return vbox;
}

static void match_with (LWParityCommand *pcmd, LWCommand *cmd, LWPiece *piece)
{
	try {
		pcmd->matchWith(cmd);
	} catch (LWMessage *msg) {
		msg->setPiece(piece);
		throw msg;
	}
}

/* False - kontynuuje pętle */
static void compare_parity (gboolean **r, 
							LWCommand *cmd_a, LWPiece *piece_a,
							LWCommand *cmd_b, LWPiece *piece_b)
{
	/* cmd_a = ")" , cmd_b = "("*/
	LWParityCommand *pcmd_a=NULL;
	LWParityCommand *pcmd_b=NULL;
	
	if (cmd_a != NULL)
		if (TRUE == cmd_a->isParity())
			pcmd_a = dynamic_cast <LWParityCommand *> (cmd_a);

	if (cmd_b != NULL)
		if (TRUE == cmd_b->isParity())
			pcmd_b = dynamic_cast <LWParityCommand *> (cmd_b);

	if ((pcmd_a != NULL) && (pcmd_b != NULL))
		if (FALSE == pcmd_b->isAlreadyMatched())
			if ((FALSE == pcmd_a->isOpen()) && (TRUE == pcmd_b->isOpen())) {
				
				if (FALSE == pcmd_b->matchAsLastOne ()) {
					match_with (pcmd_b, cmd_a, piece_b);
					match_with (pcmd_a, cmd_b, piece_a);
				} else {
					match_with (pcmd_a, cmd_b, piece_a);
					match_with (pcmd_b, cmd_a, piece_b);
				}
				
				**r = TRUE;
				return;
			}
			
	if (pcmd_b != NULL) 
		if (FALSE == pcmd_b->isAlreadyMatched())
			if (TRUE == pcmd_b->isOpen ()) {
				**r = TRUE;
				return;
			}
				
	if (pcmd_a != NULL)
		if (FALSE == pcmd_a->isOpen ()) {
			**r = FALSE;
			return;
		}
			
	*r = NULL;
}

static gboolean compare_priority (LWPiece *a, LWPiece *b) 
{
	const guint maxprio = 16;
//	if (b == NULL) return TRUE;
	
	LWSymbol *sym_a = a->getSymbol();
	LWSymbol *sym_b = b->getSymbol();

	LWCommand *cmd_a = NULL;
	LWCommand *cmd_b = NULL;
	
	guint prio_a = maxprio;
	guint prio_b = maxprio;
	
	LWLink link = LW_LINK_RIGHT;
	
	if (sym_a != NULL)
		if (TRUE == sym_a->isCommand()) {
			cmd_a = (LWCommand *) sym_a;
			prio_a = cmd_a->getPriority();
//			link = cmd_a->getLinkType();
		}

	if (sym_b != NULL)
		if (TRUE == sym_b->isCommand()) {
			cmd_b = (LWCommand *) sym_b;
			prio_b = cmd_b->getPriority();
			link = cmd_b->getLinkType();
		}

	gboolean bo;
	gboolean *r = &bo;
	
	compare_parity (&r, cmd_a, a, cmd_b, b);

	if (r != NULL) 
		return bo;
	
	switch (link) {
		case LW_LINK_LEFT:
			return (prio_a > prio_b) ? TRUE : FALSE;
		case LW_LINK_RIGHT:
			return (prio_a >= prio_b) ? TRUE : FALSE;
	}

	g_return_val_if_reached (FALSE);
}

/* \brief Zmiana symbolu skojarzonego z klockiem 

	Zmienia symbolu skojarzonego z klockiem

	\param piece Klocek z symbolem do zmiany
	\param symbolname Nazwa symbolu, który ma być skojarzony z klockiem 
*/
static void toggle_to_symbol (LWPiece *piece, const gchar *symbolname) 
{
	if (strcmp(piece->getSymbol()->getName(), symbolname)) {
		piece->setSymbol (symbolname);
	}
}

static void lex_scan (LWPiece *prev, LWPiece *piece)
{
	if (prev == NULL) return;
	g_return_if_fail (piece != NULL);
	
	LWSymbol *psymbol = prev->getSymbol();
	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol == NULL)
		return;
	
	if (TRUE == symbol->isValue())
		return;

	if (!strcmp (symbol->getName(), "opp") || !strcmp (symbol->getName(), "sub")) {

		if (psymbol == NULL) {
			toggle_to_symbol (piece, "sub");
			return;
		}

		if (TRUE == psymbol->isValue()) {
			toggle_to_symbol (piece, "sub");
			return;
		}
		
		if (TRUE == psymbol->isCommand())
			if (TRUE == ((LWCommand *) psymbol)->isParity())
				if (FALSE == ((LWParityCommand *) psymbol)->isOpen() &&
					strcmp (psymbol->getName(), "to") && strcmp (psymbol->getName(), "downto")) {
					toggle_to_symbol (piece, "sub");
					return;
				}
	}
	
	if (!strcmp (symbol->getName(), "sub")) {
		toggle_to_symbol (piece, "opp");
	}
}

/* \brief Pojedyncza iteraca tworzenia drzewa składniowego.
   
   Funkcja wstawia klocek piece do drzewa składniowego,
   znajdując odpowiednie miejsce na podstawia poprzednio
   wstawianego elementu prev. Funkcja realizuje produkcję
   drzewa przy użyciu tzw. metody operatorów.

   \param prev Poprzednio wstawiany element
   \param piece Wstawiany klecek, który może zawierać polecenie
*/
static GNode *parse_iteration (GNode *prev, LWPiece *piece)
{
GNode *node;

	if (TRUE == compare_priority (piece, (LWPiece *) prev->data)) 
		return g_node_append_data (prev, (gpointer) piece);

	g_return_val_if_fail (NULL != piece->getSymbol(), NULL);
	g_return_val_if_fail (TRUE == piece->getSymbol()->isCommand(), NULL);

	do {
	 	prev = prev->parent;
	} while (FALSE == compare_priority (piece, (LWPiece *) prev->data));

	//if (TRUE == ((LWCommand *) piece->getSymbol())->isParity())
	//	if (FALSE == ((LWParityCommand *) piece->getSymbol())->isOpen())
	//		return prev;	
	
	LWCommand *cmd = (LWCommand *) piece->getSymbol();
	
	if (TRUE == cmd->canBeSkiped()) {
//		g_print ("SKIP: %s (PREV: %s),", cmd->getName(), 
//			((LWCommand *) ((LWPiece *) prev->data)->getSymbol())->getName());
		cmd->reset();
		return prev;
	}

	/* Wstawienie zamykającej części operatora parzystego np. nawiasu */
	if (TRUE == cmd->isParity())
		if (FALSE == ((LWParityCommand *) cmd)->isOpen())
			return g_node_append_data (prev->parent, (gpointer) piece);
		
	/* Polecenia o niskim priorytecie, ustaw jedno za drugim */
	if (cmd->getPriority() <= 1)
		return g_node_append_data (prev, (gpointer) piece);
		
	/* Umieszczanie elementu piece pomiędzy prev, a ostatnim dzieckiem prev */
	node = g_node_last_child (prev); 
	g_node_unlink (node);
	GNode *t = g_node_append_data (prev, (gpointer) piece);
	g_node_append (t, node);
	return t;
}

static gboolean unset_command (GNode *node)
{
	LWPiece *piece = (LWPiece *) node->data;
	
	if (piece == NULL) 
		return FALSE;
	
	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol != NULL)
		if (TRUE == symbol->isCommand()) {
			LWCommand *cmd = (LWCommand *) symbol;
			cmd->reset();
		}
	
	return FALSE;
}

static gboolean unset_variable (LWVariable *variable)
{
	variable->clear();
	
	return FALSE;
}

static gboolean delete_cb (LWProgram *program)
{
	if (TRUE == program->isFinished())
		delete program;
	else
		program->stop ();
	
	return TRUE;
}

static gboolean keypress_cb (GtkWidget *widget,
GdkEventKey *event, GQueue **queue_keys)
{
	(void) widget;
	g_queue_push_tail (*queue_keys,	GUINT_TO_POINTER(event->keyval));
	
	return TRUE;
}

/*! \brief Konstruktor

	Tworzy program
*/
LWProgram::LWProgram () 
: world(NULL), list_vars(NULL), finished(FALSE), 
stoped(FALSE), tree_piece(NULL), begin_piece(NULL)
{
	context.wizard = NULL;
	context.stack = g_queue_new ();
	context.queue_keys = g_queue_new ();
	
	LWEnvironment::registerProgram (this);
}

/*! \brief Destruktor
	Niszczy zbiór ikon
*/	
LWProgram::~LWProgram ()
{
GtkWidget *window=NULL;

	g_queue_free (context.queue_keys);
	
	if (begin_piece != NULL)
		delete begin_piece->row;
	
	if (world != NULL)
		window = gtk_widget_get_ancestor(world->getWidget(), GTK_TYPE_WINDOW);

	if (context.wizard != NULL)
		delete context.wizard;
	
	if (world != NULL)
		delete world;
	
	if (window != NULL)
		gtk_widget_destroy (window);
	
	LWEnvironment::unregisterProgram ();
}

void LWProgram::registerVariable (LWVariable *variable)
{
	if (NULL == g_slist_find (list_vars, variable))
		list_vars = g_slist_prepend (list_vars, (gpointer) variable);
}

/*! \brief Przetwarzanie wartości

	Ze zbioru klocków reprezuntujących cyfry, zmienne, obrazki
	tworzy obiekt wartości, który może być przekazany do polecenia
	\param node Węzeł drzewa z klockami
	\return Stworzona wartość
*/
LWValue *LWProgram::computeValue (GNode *node)
{
	LWValue *value;
	
	LWPiece *piece = (LWPiece *) node->data;
	g_return_val_if_fail (piece != NULL, NULL);

	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol != NULL) {
		g_return_val_if_fail (TRUE == symbol->isValue(), NULL);
		
		value = dynamic_cast <LWValue *> (symbol);
		
		if (TRUE == value->isVariable()) {
			if (node->children == NULL) {
				registerVariable ((LWVariable *) value);
				return value;
			} else {
				LWMessage *m = new LWMessage (LW_ERROR_BadString);
				m->setPiece (piece);
				throw m;
			}
		} else
			value = new LWValue (value);
		
	} else {
		LWPixmap *pixmap = piece->getBackgroundPixmap();
		value = new LWValue (pixmap);
	}
			
	for (GNode *n=node->children; n != NULL; n = n->children) {
		piece = (LWPiece *) n->data;
		g_return_val_if_fail (piece != NULL, NULL);

		symbol = piece->getSymbol();
		
		if (symbol != NULL) {
			g_return_val_if_fail (TRUE == symbol->isValue(), NULL);
			
			LWValue *v = dynamic_cast <LWValue *> (symbol);
			
			if (TRUE == v->isVariable()) {
				delete value;
				LWMessage *m = new LWMessage (LW_ERROR_BadString);
				m->setPiece (piece);
				throw m;
			}
			
			value->concat (v);
			continue;
		}
		
		LWPixmap *pixmap = piece->getBackgroundPixmap();
		value->append (pixmap);
	}
	
	return value;		
}

/*! \brief Ustalenie argument dla polecenia

	Ustala argument dla polecenia
	\param cmd Polecenie dla którego będzie ustalony parametr
	\param node Węzeł, na podstawie którego zostanie obliczona wartość parametru
*/
void LWProgram::setArguments (LWCommand *cmd, GNode *node)
{
gint args=0;
GSList *destroy_list=NULL;
	
	for (GNode *n = node; n != NULL; n = n->next) {
		LWPiece *piece = (LWPiece *) n->data;
		g_return_if_fail (piece != NULL);
		
		LWSymbol *symbol = piece->getSymbol();
		LWCommand *cmd2 = NULL;
		
		if (symbol != NULL)
			if (TRUE == symbol->isCommand())
				cmd2 = (LWCommand *) symbol;
		
		if (cmd2 != NULL) {
			if (TRUE == cmd2->hasReturn())
				cmd->setArgument (args++, NULL);
		} else {
			cmd->setArgument (args++, computeValue (n));
			destroy_list = g_slist_prepend (destroy_list, n);
		}
	}

	cmd->setArgc (args);	
	
	g_slist_foreach (destroy_list, (GFunc) g_node_destroy, NULL);
	g_slist_free (destroy_list);
}

/*! \brief Sprawdzenie poprawności polecenia parzystego
	
	Jeśli dwu członowe polecenie jest nie zamknięte zwraca wyjątek
	\param cmd Sprawdzane polecenie
*/
static void checkParity (LWParityCommand *cmd)
{
	if (FALSE == cmd->isAlreadyMatched())
			 throw new LWMessage (LW_ERROR_UnmatchedBrace);
}

static LWCommand *get_node_command (GNode *node)
{
	g_return_val_if_fail (node != NULL, NULL);
	
	LWPiece *piece = (LWPiece *) node->data;
	g_return_val_if_fail (piece != NULL, NULL);
	
	LWSymbol *symbol = piece->getSymbol();
	g_return_val_if_fail (symbol != NULL, NULL);
	g_return_val_if_fail (TRUE == symbol->isCommand(), NULL);
	
	return (LWCommand *) symbol;
}

#define IS_PARITY(cmd) ((TRUE == (cmd)->isParity()) && (strcmp((cmd)->getName(), "begin")))

static LWParityCommand *get_node_parity_command (GNode *node)
{
	LWCommand *cmd = get_node_command (node);
	g_return_val_if_fail (node != NULL, NULL);
	
	if (IS_PARITY(cmd))
		return (LWParityCommand *) cmd;
	else
		return NULL;
}

static void moveParity (GNode *root, GNode *firstchild)
{
	gint lev=0;
	LWParityCommand *pcmd=NULL;
	GNode *next=NULL;
	GNode *n;
	
	for (n = firstchild; n != NULL; n = next) {
		next = n->next;
		
		g_node_unlink (n);
		g_node_append (root, n);
		
		pcmd = get_node_parity_command (n);
		
		if (pcmd != NULL)
			if (TRUE == pcmd->isOpen()) lev++;
			else if (--lev == 0) break;
	}

	g_return_if_fail (lev == 0);
	g_return_if_fail (n != NULL);
	
	if (next == NULL) return;
		
	LWCommand *cmd = get_node_command (next);
	
	if (TRUE == pcmd->isLoop()) {
		if (IS_PARITY(cmd)) {
				moveParity (root, next);
		} else {
				g_node_unlink (next);
				g_node_append (root, next);
		}
	}
}

/*! \breif Poprawianie gałęzi drzewa składniowego

	Poprawianie gałęzi drzewa składniowego zawierającej polecenie
	o niskim priorytecie (1), a będącym poleceniem dwuczłonowym.
	Poprawka polega na przeniesieniu dwóch członów polecenia oraz
	całej instrukcji stojącej za drugim członem polecenia (jeśli taka
	istnieje) do bloku logicznego, który staje w miejscu instrukcji
	dwuczłonowej.

	\param node Węzeł z poleceniem otwierającym polecenie dwuczłonowe
	\param block Klocek z poleceniem otwarcia bloku logicznego
*/
static void fixParity (GNodePtr &node, LWPiece *block)
{
	GNode *n = node;
	node = g_node_new (block);
	g_node_insert_before (n->parent, n, node);
	
	moveParity (node, n);
}

/*! \brief Sprawdzanie poprawności stworzonego węzła
	
	Sprawdza poprawność poleceń, na które wskazuje węzeł.
	W przypadku wystąpień błędów, będą żucane wyjątki.
	Dodatkowo poleceniu zawartemu w węźle, przydzialane są 
	argumenty (te, które mogą być ustalone na etapie analizy
	programu, czyli nie będące rezulatatem wykonania innych poleceń).

	\param node Sprawdzany węzeł
	\param hasMovedElement Przy wywołaniu rekurencyjnym flaga oznacza,
		że już instrukcja została przeniesiona, zapobiega
		nieskończonej rekurencyji
*/
void LWProgram::checkNode (GNode *node, gboolean hasMovedElement)
{
	for (GNode *n = node; n != NULL; n = n->next) {
		LWPiece *piece = (LWPiece *) n->data;
		g_return_if_fail (piece != NULL);
	
		LWSymbol *symbol = piece->getSymbol();

		if (symbol == NULL)
			continue;
		
		if (FALSE == symbol->isCommand())
			continue;
		
		LWCommand *cmd = (LWCommand *) symbol;
		
		if (TRUE == cmd->isParity()) {
			checkParity ((LWParityCommand *) (cmd));
		
			if (hasMovedElement == FALSE) {
				if (1 == cmd->getPriority())
					if (IS_PARITY(cmd))
						if (TRUE == ((LWParityCommand *) cmd)->isOpen()) {
							fixParity (n, begin_piece);

							checkNode (n->children, TRUE);
							continue;
							}
			} else hasMovedElement = FALSE;
		}

	try {
		setArguments (cmd, n->children);
	} catch (LWMessage *msg) {
		if (piece != begin_piece)
			msg->setPiece (piece);
		throw msg;			
	}
		
	checkNode (n->children);
	} 
	
}

/* \brief Deinicjuje program

	Niszczy drzewo składniowe oraz stos,
	czyści zmienne
*/
void LWProgram::uninit ()
{
	g_queue_free (context.stack);

	if (tree_piece != NULL) {
		g_node_traverse (tree_piece, G_POST_ORDER, G_TRAVERSE_ALL,
							-1, (GNodeTraverseFunc) unset_command, NULL);
	
		g_node_destroy (tree_piece);
	}
	
	if (list_vars != NULL) {
		g_slist_foreach (list_vars, (GFunc) unset_variable, NULL);
		
		g_slist_free (list_vars);
	}
}

/* \brief Wyświetla komunikat

	Wyświetla komunikat
	\msg Komunikat do wyświetlenia
*/
void LWProgram::showMessage (LWMessage *msg)
{
		uninit ();
	
		msg->setProgram (this);
		msg->show();
		delete msg;
}

/*! \brief Analiza programu

	Przeprowadza analizę planszy z programem,
	tworząc drzewo składniowe. W razie błędów
	metoda tworzy okno z błędem i zaznacza ikonę
	w związku z którą wystąpił

	\param program Plansza z programem
	\return Prawda jęśli analiza zakończyła się sukcesem
	w przeciwnym wypadku fałsz.
*/
gboolean LWProgram::parse (LWBoard *program)
{
	g_return_val_if_fail (tree_piece == NULL, FALSE);
	
	program->unmarkPiece ();

	/* Pobranie listy klocków z poleceniami*/
	GSList *list_piece = program->getPieceList();

	/* Jeśli lista poleceń jest pusta, analizy nie trzeba przeprowadzać */
	if (list_piece == NULL)
		return TRUE;
	
	/* Wstawianie na początku programu klocka otwierającego
		blok logiczny */
	LWRow *row = new LWRow (program);
	begin_piece = new LWPiece (row);
	begin_piece->setSymbol ("begin");
	GNode *node = g_node_new ((gpointer) begin_piece);

	LWPiece *prev = NULL;
	
	for (GSList *l = list_piece; l != NULL; l = l->next) {

		lex_scan (prev, (LWPiece *) l->data);
		prev = (LWPiece *) l->data;

		try {
			node = parse_iteration (node, (LWPiece *) l->data);
		} catch (LWMessage *msg) {
			msg->setPiece ( (LWPiece *) l->data);
			showMessage(msg);
			g_slist_free (list_piece);
			return FALSE;
		}
	
#if 0
		tree_piece = g_node_get_root (node); debugTree ();
#endif
		
	}
	
	LWPiece *end_piece = new LWPiece (row);
	end_piece->setSymbol ("end");
	
	try {
		node = parse_iteration (node, end_piece);
	} catch (LWMessage *msg) {
			showMessage(msg);
			g_slist_free (list_piece);
			return FALSE;
	}

	delete end_piece;		
	
	g_slist_free (list_piece);
	
	tree_piece = g_node_get_root (node);
	
	try {
		checkNode (tree_piece);
	} catch (LWMessage *msg) {
		showMessage(msg);
		return FALSE;
	}
	
return TRUE;
}

/*! \brief Przydzielenie świata

	Przydziela świat, na którym będzie operował program.
	Metoda tworzy kopię tego świata i umieszcza w lewym górnym
	rogu czarodzieja, który po nim będzie się poruszał.

	\param a_world Plansza ze światem
*/
void LWProgram::setWorld (LWBoard *a_world)
{
	g_return_if_fail (context.wizard == NULL);
	g_return_if_fail (world == NULL);
	g_return_if_fail (a_world != NULL);
	g_return_if_fail (a_world->getType() == LW_TYPE_WORLD);
	g_return_if_fail (a_world->getRowNth(0) != NULL);
	g_return_if_fail (a_world->getRowNth(0)->getPieceNth(0) != NULL);
	
	world = new LWBoard (a_world, LW_TYPE_PLAYGROUND);
	
	GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (window), _("Executing program..."));
	gtk_window_set_modal (GTK_WINDOW(window), TRUE);
	gtk_window_set_resizable (GTK_WINDOW(window), FALSE);
	
	GTK_WIDGET_SET_FLAGS (window, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (window);
	
	g_signal_connect (G_OBJECT (window), "key_press_event", 
		G_CALLBACK (keypress_cb), (gpointer) &context.queue_keys);
	
	if (NULL != LWEnvironment::getWidget()) {
		GtkWidget *parentwindow = gtk_widget_get_ancestor(
			LWEnvironment::getWidget(), GTK_TYPE_WINDOW);
		
		if (parentwindow != NULL)
			gtk_window_set_transient_for (GTK_WINDOW(window), GTK_WINDOW(parentwindow));
	}
	
    gtk_container_add (GTK_CONTAINER (window), world->getWidget());
	gtk_widget_show (window);

	g_signal_connect_swapped (G_OBJECT (window), "delete_event",
		G_CALLBACK (delete_cb), (gpointer) this);
	
	context.wizard = new LWWizard (world->getRowNth(0)->getPieceNth(0));
	context.wizard->restoreFromXML (LWEnvironment::getWizardXML());
}

LWBoard *LWProgram::getWorld ()
{
	return world;
}

/*! \brief Znajdź następne polecenie do wykonania
	
	Znajduje kolejne polecenie do wykonania w drzewie,
	na podstawie poprzednio wykonywanego polecenia,
	w większości przypadków będzie to node->next.
	Funkcja zmodyfikowana w ten sposób, że jeśli przedostatnim
	wykonywanym poleceniem była pętla, działanie będzie wracać
	do pętli.

	\param node Poprzednie polecenie w drzewie
	\return Kolejne polecenie do wykonania
*/
static GNode *get_next_node_to_execute (GNode *node)
{
	g_return_val_if_fail (node != NULL, NULL);
	
	if (node->prev != NULL) {
		LWPiece *piece = (LWPiece *) node->prev->data;
		g_return_val_if_fail (piece != NULL, NULL);
		
		LWSymbol *symbol = piece->getSymbol();
		g_return_val_if_fail (symbol != NULL, NULL);
		g_return_val_if_fail (TRUE == symbol->isCommand(), NULL);
		
		LWCommand *pcmd = (LWCommand *) symbol;
		
		if (pcmd != NULL)
			if (pcmd->isLoop())
				return node->prev;
		}
	
	return node->next;
}

/*! \brief Pojednycza iteracja wykonywania poleceń w drzewie

	Oblicza polecenia, które dostarczają dane do polecenia -
	polecenia znajdujące się w węźle, a następnie wykonuje polecenie.
	Jest to pojedyncza iteracja przechodzenia drzewa w głąb (w
	kolejności Post Order )

	\param cmd Polecenie do wykonania
	\param children Polecenia do wykonania przed poleceniem cmd
*/
void LWProgram::executeOrderPost (LWCommand *cmd, GNode *children)
{
GNodePtr &ip = context.instrPtr;

	/* Zachowaj wartość ip */
	GNodePtr tmpip = ip;
	
	ip = children;
	
	while (ip != NULL) {
		GNodePtr lastip = ip;
		
		/* Wykonaj polecenia związane z argumentami */
		executeNode (ip);

		/* Znajdź następne polecenie do wykonania,
		   niektóre polecenia modyfikuja wskaźnik ip,
		   w takim przpadku ip jest już ustalone,
		   nie modyfikuj go */
		
		if (lastip == ip)
			ip = get_next_node_to_execute (ip);
	}

	/* Przywróć wartość ip */
	ip = tmpip;
		
	/* Wykonaj polecenie */
	cmd->execute (&context);
}

/*! \brief Wykonanie polecenie w węźle

	Wykonuje polecenia w węźle.

	\param node Węzeł do wykonania 
*/
void LWProgram::executeNode (GNode *node)
{
	LWPiece *piece = (LWPiece *) node->data;
	g_return_if_fail (piece != NULL);
	
	LWSymbol *symbol = piece->getSymbol();
	g_return_if_fail (symbol != NULL);
	g_return_if_fail (TRUE == symbol->isCommand());
	
	LWCommand *cmd = (LWCommand *) symbol;
	
	try {
		executeOrderPost (cmd, node->children);
	} catch (LWMessage *msg) {
		msg->setPiece(piece);
		throw msg;
	}
	
	while (gtk_events_pending ())
		gtk_main_iteration ();

	if (stoped == TRUE)
		throw new LWMessage (LW_INFO_ProgramInterrupted);
	
}

/*! \brief Wykonanie programu

	Wykonuje program. Metoda może być wykonana dopiero po stworzeniu
	drzewa składniowego przez metodę parse.

	\param node Węzeł do wykonania 
*/
void LWProgram::execute ()
{
	g_return_if_fail (world != NULL);

#if 0
	debugTree();
	return;
#endif
	
	GNodePtr &ip = context.instrPtr;
	ip = tree_piece;
	
	if (ip != NULL) {
		/* Wykonaj polecenie (blok logiczny) */
		try {
			executeNode (ip);
		} catch (LWMessage *msg) {
			showMessage(msg);
			finished = TRUE;
			return;
		}
	}
	finished = TRUE;
	uninit();
	
	GtkWidget *window = gtk_widget_get_ancestor(world->getWidget(), GTK_TYPE_WINDOW);
	gtk_window_set_title (GTK_WINDOW (window), _("Program finished..."));
	gtk_window_set_modal (GTK_WINDOW(window), FALSE);
}

/*! \brief Sprawdzenie czy program już się zakończył

	Sprawdza czy wykonywany program już się zakończył

	\return Rezultat testu 
*/
gboolean LWProgram::isFinished ()
{
	return finished;
}

/*! \brief Zatrzymanie programu

	Zatrzymuje program. Program może nie zostać zatrzymany od
	razu, zakończenie programu nastąpi po zakończeniu wykonywania
	bierzącego polecenia.
*/
void LWProgram::stop ()
{
	/* Przyśpiesz czarodzieja, żeby szybciej zakończył bierzącą instrukcję */
	if (context.wizard != NULL)
		context.wizard->setSpeed (9);
	
	stoped = TRUE;
}

/*! \brief Odpluskwanie drzewa

	Rysuje drzewo składniowe w nowym oknie. Metoda jest przydatna
	wyłącznie przy analizie poprawności algorytmów tworzących drzewo
	składniowe.
*/
void LWProgram::debugTree ()
{
	GtkWidget *w = widget_tree (tree_piece);

	GtkWidget *sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw), w);
	
	GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_container_add (GTK_CONTAINER (window), sw);
	gtk_widget_show_all (window);
}
