# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import sys
import gobject
import threading
from pyinotify import ThreadedNotifier, WatchManager, ProcessEvent, EventsCodes
from xdg_support import get_xdg_music_dir

if hasattr(EventsCodes,"IN_MOVE_SELF"):
    # pynotify <= 0.8
    IN_MOVED_FROM = EventsCodes.IN_MOVED_FROM
    IN_CREATE = EventsCodes.IN_CREATE
    IN_MOVED_TO = EventsCodes.IN_MOVED_TO
    IN_DELETE = EventsCodes.IN_DELETE
    IN_ISDIR = EventsCodes.IN_ISDIR
    IN_DELETE_SELF = EventsCodes.IN_DELETE_SELF
    IN_CLOSE_WRITE = EventsCodes.IN_CLOSE_WRITE
    IN_MOVE_SELF = EventsCodes.IN_MOVE_SELF
else:
    import pyinotify
    IN_MOVED_FROM = pyinotify.IN_MOVED_FROM
    IN_CREATE = pyinotify.IN_CREATE
    IN_MOVED_TO = pyinotify.IN_MOVED_TO
    IN_DELETE = pyinotify.IN_DELETE
    IN_ISDIR = pyinotify.IN_ISDIR
    IN_DELETE_SELF = pyinotify.IN_DELETE_SELF
    IN_CLOSE_WRITE = pyinotify.IN_CLOSE_WRITE
    IN_MOVE_SELF = pyinotify.IN_MOVE_SELF

from logger import Logger
import vfs

from utils import threaded

class EventCatcher(ProcessEvent):
    def __init__(self,wm):
        self._wm = wm
        self.__cookie_move = {}

    def process_default(self, event):
        from song import file_is_supported
        from library import ListenDB
        if event.name and event.name[0] == "." : return 
        
        dir = False
        if hasattr(event,"is_dir"):
            dir = event.is_dir
        else:
            # pynotify >= 0.8
            dir = event.dir

        if dir:
            if event.name:
                fullpath = os.path.join(event.path,event.name)
                if event.mask & IN_MOVED_TO :
                    from source.local import ImportFolderJob
                    ImportFolderJob([fullpath])
                elif event.mask & IN_MOVED_FROM or \
                        event.mask & IN_CREATE:
                    songs = [ ListenDB.get_song(uri) for uri in ListenDB.get_all_uris() if uri.startswith("file://"+fullpath) ]
                    ListenDB.remove(songs)

        else:
            fullpath = os.path.join(event.path,event.name)
            if event.mask & IN_CLOSE_WRITE or \
                event.mask & IN_MOVED_TO :
                if file_is_supported(fullpath) :
                    uri = "file://"+fullpath
                    if ListenDB.has_uri(uri):
                        song =  ListenDB.get_song(uri)
                        if os.path.getctime(fullpath) > song.get("#ctime"):
                            ListenDB.get_or_create_song(song,song.get_type(),read_from_file=True)
                    else:
                        ListenDB.get_or_create_song({"uri":uri},"local",read_from_file=True)
            elif event.mask & IN_DELETE or \
                    event.mask & IN_MOVED_FROM :
                uri = "file://"+fullpath
                if ListenDB.has_uri(uri):
                    ListenDB.remove(ListenDB.get_song(uri))

class _FileMonitor(Logger):
    __mask = IN_MOVED_FROM | IN_CREATE | \
            IN_MOVED_TO | IN_DELETE | IN_ISDIR | \
            IN_DELETE_SELF | IN_CLOSE_WRITE | IN_MOVE_SELF
    def __init__(self):
        self.__wm = None
        self.__thread_notifier = None
        self._started = False
        self.__startup_id = None

        self.__cond = threading.Condition()

    def start(self):
        from config import config
        if config.getboolean("library","watcher"):
            self.__startup_id = gobject.timeout_add(10000,self.start_monitoring)

    @threaded
    def start_monitoring(self):
        if self.__startup_id:
            try: gobject.source_remove(self.__startup_id)
            except: pass
            self.__startup_id = None

        if self._started: 
            self.loginfo("file monitor already started")
            return 
        else:
            from config import config
            path = config.get("library","location", get_xdg_music_dir())
            path = os.path.expanduser(path)
            if not os.path.isdir(path): 
                self.logwarn("try to monitor a non exiting path: %s",path)
                return 
            self.__wm = WatchManager()
            self.__thread_notifier = ThreadedNotifier(self.__wm)
            self.__thread_notifier.start()
            self._started = True
            self.__wm.add_watch(path, self.__mask,  proc_fun=EventCatcher(self.__wm), rec=True , auto_add=True )
            self.loginfo("start monitoring")

    def stop_monitoring(self):
        if not self._started: return 
        self.__thread_notifier.stop()
        del self.__wm
        del self.__thread_notifier
        self.__thread_notifier = None
        self.__wm = None
        self._started = False
        self.loginfo("stop monitoring")

    def close(self):
        self.stop_monitoring()

FileMonitor = _FileMonitor()
