/* database.vala - Easy-to-use interface to Listaller's software database(s)
 *
 * Copyright (C) 2010-2012 Matthias Klumpp <matthias@tenstral.net>
 *
 * Licensed under the GNU Lesser General Public License Version 3
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 */

using GLib;
using Gee;
using Listaller;
using Listaller.Utils;

namespace Listaller {

public enum AppSource {
	ALL,
	EXTERN,
	NATIVEPKG,
	UNKNOWN;
}

private enum ForceDB {
	NONE,
	PRIVATE,
	SHARED;
}

private class SoftwareDB : MsgObject {
	private InternalDB? db_shared;
	private InternalDB? db_priv;
	private Settings conf;

	public ForceDB force_db { get; set; }

	public signal void application (AppItem appid);

	public SoftwareDB (Settings liconf, bool include_shared = true) {
		base ();

		db_shared = null;
		db_priv = null;
		force_db = ForceDB.NONE;

		// We just store the settings, so other objects can fetch them
		conf = liconf;

		if (include_shared) {
			db_shared = new InternalDB (true, conf.testmode);

			/* If we open a shared DB and have root-access, don't open the
			 * private database. It makes no sense someone is working as root
			 * and installing private stuff into root's home-dir */
			if (!is_root ()) {
				db_priv = new InternalDB (false, conf.testmode);
			}
		} else {
			// If we only want to open the personal DB, don't touch the shared one
			db_priv = new InternalDB (false, conf.testmode);
		}

		// Forward messages of the internal DBs to this meta-db
		if (db_priv != null)
			db_priv.message.connect ( (m) => { this.message (m); } );
		if (db_shared != null)
			db_shared.message.connect ( (m) => { this.message (m); } );
	}

	private void emit_dberror (string details) {
		// Emit a database error
		emit_error (ErrorEnum.DATABASE_FAILURE, details);
	}

	private bool shared_db_canbeused (bool error = false) {
		bool ret = true;
		if (db_shared == null) {
			ret = false;
		}

		if (force_db == ForceDB.PRIVATE)
			return false;

		/* If shared db does not exist AND we don't have root-access, opening the db will fail.
		 * (so we check for this case here, just to be sure) */
		if (ret)
			if ((!is_root ()) && (!FileUtils.test (db_shared.get_database_name (), FileTest.EXISTS))) {
				db_shared = null;
				ret = false;
			}

		if ((!ret) && (error))
			emit_dberror (_("Tried to perform action on shared software database, but the database is not opened! (maybe a permission problem?)"));

		return ret;
	}

	private bool private_db_canbeused (bool error = false) {
		if (force_db == ForceDB.SHARED)
			return false;

		if (db_priv == null) {
			if (error)
				emit_dberror (_("Tried to perform action on private software database, but the database is not opened! (this should never happen!)"));
			return false;
		}

		return true;
	}

	public Settings get_liconf () {
		return conf;
	}

	public bool open_read () {
		bool ret = false;
		try {
			if (private_db_canbeused ())
				ret = db_priv.open_r ();
			if (shared_db_canbeused ()) {
				if (!ret)
					ret = db_shared.open_r ();
				else
					db_shared.open_r ();
			}
		} catch (Error e) {
			emit_dberror (_("Unable to open database for read-only: %s").printf (e.message));
			ret = false;
		}
		return ret;
	}

	public bool open_write () {
		bool ret = false;
		try {
			if (private_db_canbeused ())
				ret = db_priv.open_rw ();
			if (shared_db_canbeused ()) {
				// We can only have write access to shared DB if we're root
				if (is_root ()) {
					if (!ret)
						ret = db_shared.open_rw ();
					else
						db_shared.open_rw ();
				} else {
					if (!ret)
						ret = db_shared.open_r ();
					else
						db_shared.open_r ();
				}
			}
		} catch (Error e) {
			emit_dberror (_("Unable to open database for writing: %s").printf (e.message));
			ret = false;
		}
		return ret;
	}

	public bool database_locked () {
		bool ret = false;
		if (shared_db_canbeused ())
			ret = db_shared.database_locked ();
		if (ret)
			return true;
		if (private_db_canbeused ())
			ret = db_priv.database_locked ();

		return ret;
	}

	public bool database_writeable () {
		bool ret = false;
		if (shared_db_canbeused ())
			ret = db_shared.database_writeable ();
		if (ret)
			return true;
		if (private_db_canbeused ())
			ret = db_priv.database_writeable ();

		return ret;
	}

	public bool add_application (AppItem app) {
		bool ret;
		try {
		if (app.shared) {
			if (shared_db_canbeused (true))
				ret = db_shared.add_application (app);
			else
				return false;
		} else {
			if (private_db_canbeused (true))
				ret = db_priv.add_application (app);
			else
				return false;
		}
		} catch (Error e) {
			emit_dberror (_("Unable to add application to database: %s").printf (e.message));
			ret = false;
		}
		return ret;
	}

	public bool add_application_filelist (AppItem app, Collection<IPK.FileEntry> flist) {
		bool ret;
		try {
			if (app.shared) {
				if (shared_db_canbeused (true))
					ret = db_shared.add_application_filelist (app, flist);
				else
					return false;
			} else {
				if (private_db_canbeused (true))
					ret = db_priv.add_application_filelist (app, flist);
				else
					return false;
			}
		} catch (Error e) {
			emit_dberror (_("Unable to add application file list to database: %s").printf (e.message));
			ret = false;
		}
		return ret;
	}

	public ArrayList<string>? get_application_filelist (AppItem app) {
		ArrayList<string>? res;
		try {
			if (app.shared) {
				if (shared_db_canbeused (true))
					res = db_shared.get_application_filelist (app);
				else
					return null;
			} else {
				if (private_db_canbeused (true))
					res = db_priv.get_application_filelist (app);
				else
					return null;
			}
		} catch (Error e) {
			emit_dberror (_("Unable to fetch application file list: %s").printf (e.message));
			res = null;
		}
		return res;
	}

	public int get_applications_count () {
		int cnt = 0;
		try {
			if (shared_db_canbeused ())
				cnt += db_shared.get_applications_count ();
			if (private_db_canbeused ())
				cnt += db_priv.get_applications_count ();
		} catch (Error e) {
			emit_dberror (_("Unable to count applications: %s").printf (e.message));
		}
		return cnt;
	}

	public AppItem? get_application_by_idname (string appIdName) {
		AppItem? app = null;

		try {
			if (shared_db_canbeused ())
				app = db_shared.get_application_by_idname (appIdName);

			if (app == null)
				if (private_db_canbeused ())
					app = db_priv.get_application_by_idname (appIdName);
		} catch (Error e) {
			emit_dberror (_("Unable to fetch application by id: %s").printf (e.message));
		}

		return app;
	}

	public AppItem? get_application_by_id (AppItem app) {
		AppItem? resApp;
		try  {
			if (app.shared) {
				if (shared_db_canbeused ())
					resApp = db_shared.get_application_by_idname (app.idname);
				else
					return null;
			} else {
				if (private_db_canbeused ())
					resApp = db_priv.get_application_by_idname (app.idname);
				else
					return null;
			}
		} catch (Error e) {
			emit_dberror (_("Unable to fetch application by id: %s").printf (e.message));
			resApp = null;
		}
		return resApp;
	}

	public AppItem? get_application_by_fullname (string appFullName) {
		AppItem? app = null;
		try {
			if (shared_db_canbeused ())
				app = db_shared.get_application_by_fullname (appFullName);

			if (app == null)
				if (private_db_canbeused ())
					app = db_priv.get_application_by_fullname (appFullName);
		} catch (Error e) {
			emit_dberror (_("Unable to fetch application by name: %s").printf (e.message));
		}

		return app;
	}

	public bool remove_application (AppItem app) {
		bool ret;
		try {
			if (app.shared) {
				if (shared_db_canbeused (true))
					ret = db_shared.remove_application (app);
				else
					return false;
			} else {
				if (private_db_canbeused (true))
					ret = db_priv.remove_application (app);
				else
					return false;
			}
		} catch (Error e) {
			emit_dberror (_("Unable to remove application from database: %s").printf (e.message));
			ret = false;
		}
		return ret;
	}

	public ArrayList<AppItem> find_applications ([CCode (array_null_terminated = true, array_length = false)] string[] values) {
		var list = new ArrayList<AppItem> ();

		try {
			if (shared_db_canbeused ()) {
				var tmp = db_shared.find_applications (values);
				list.add_all (tmp);
			}
			if (private_db_canbeused ()) {
				var tmp = db_priv.find_applications (values);
				list.add_all (tmp);
			}
		} catch (Error e) {
			emit_dberror (_("Unable to search application database: %s").printf (e.message));
		}

		return list;
	}

	public void _internal_process_dbapps (InternalDB db, double one, ref ArrayList<AppItem> appList) {
		uint i = 1;
		AppItem? capp = db.get_application_by_dbid (i);
		while (capp != null) {
			application (capp);
			appList.add (capp);
			change_main_progress ((int) Math.round (one * i));

			i++;
			capp = db.get_application_by_dbid (i);
		}
	}

	public bool find_all_applications (AppSource filter, out ArrayList<AppItem> appList = null) {
		ArrayList<AppItem> alist = new ArrayList<AppItem> ();

		double one = 100d / get_applications_count ();

		if (private_db_canbeused ()) {
			_internal_process_dbapps (db_priv, one, ref alist);
		}
		if (shared_db_canbeused ()) {
			_internal_process_dbapps (db_shared, one, ref alist);
		}
		appList = alist;

		return true;
	}

	public bool set_application_dependencies (string appName, ArrayList<IPK.Dependency> deps) {
		if (is_root ()) {
			if (shared_db_canbeused (true))
				return db_shared.set_application_dependencies (appName, deps);
			else
				return false;
		} else {
			if (private_db_canbeused (true))
				return db_priv.set_application_dependencies (appName, deps);
			else
				return false;
		}
	}

	/* Dependency stuff */

	public bool add_dependency (IPK.Dependency dep) {
		bool ret = false;
		try {
			if (is_root ()) {
				if (shared_db_canbeused (true))
					ret = db_shared.add_dependency (dep);
				else
					return false;
			} else {
				if (private_db_canbeused (true))
					ret = db_priv.add_dependency (dep);
				else
					return false;
			}
		} catch (Error e) {
			emit_dberror (_("Unable to add a dependency to database: %s").printf (e.message));
			return false;
		}
		return ret;
	}

	public IPK.Dependency? get_dependency_by_id (string depIdName) {
		IPK.Dependency? dep = null;
		if (shared_db_canbeused ())
			dep = db_shared.get_dependency_by_id (depIdName);

		if (dep == null)
			if (private_db_canbeused ())
				dep = db_priv.get_dependency_by_id (depIdName);
		return dep;
	}

	public bool set_dependency_environment (string depIdName, string env) {
		bool ret = false;
		if (shared_db_canbeused ())
			ret = db_shared.set_dependency_environment (depIdName, env);

		if (!ret)
			if (private_db_canbeused ())
				ret = db_priv.set_dependency_environment (depIdName, env);

		return ret;
	}
}

} // End of namespace

