package Lire::Count;

use strict;

use base qw/ Lire::Aggregate /;

use Carp;

use Lire::DataTypes qw/ format_numeric_type /;
use Lire::Utils qw/ ratio100 sql_quote_name check_object_param/;

=pod

=head1 NAME

Lire::Count

=head1 SYNOPSIS

FIXME

=head1 DESCRIPTION

Class that implements the count operator. This operator counts the
number of DLF records in the current group.

If the fields attribute is set, the count statistics will be equals to
the number of different values present in the fields specified by that
attribute.

Its also possible to express the count as a ratio of the total count
for the group or table.

=head1 METHODS

=head2 new( %params )

Creates a new Lire::Count object. In addition to the values
supported by its parents, the fields and ratio attributes will be
initialized to the values specified in the %params argument.

=cut

sub new {
    my $self  = bless { 'ratio' => 'none', }, shift;

    $self->init( @_, 'op' => 'count' );

    my %args = @_;

    $self->fields(  $args{'fields'} )
      if exists $args{'fields'};

    $self->ratio( $args{'ratio'} )
      if exists $args{'ratio'};

    return $self;
}

=pod

=head2 fields( [$new_fields] )

Returns the fields for which we are counting the different values.
This a reference to an array of DLF field names.

If the $new_fields parameter is set, it will be used as the new
$fields value. It must be an array reference and should only contains
valid field names for the current report specification's schema.

=cut

sub fields {
    my ( $self, $fields ) = @_;

    if ( @_ == 2 ) {
	if ( defined $fields ) {
            check_object_param( $fields, 'fields', 'ARRAY' );

	    foreach my $f ( @$fields ) {
		croak "'$f' isn't a defined field in the specification's schemas"
                  unless $self->report_spec()->has_field( $f );
	    }
	}
	$self->{'fields'} = $fields;
    }

    $self->{'fields'};
}

=pod

=head2 ratio([$new_ratio]) 

Returns how the count will be expressed. This can one of three
possible values:

=over

=item none

Default. The absolute count will be used.

=item group

The count will be expressed as a percentage of the group's count.

=item table

The count will be expressed as a percentage of the table's total
count.

=back

=cut

sub ratio {
    my ( $self, $ratio ) = @_;

    if ( @_ == 2 ) {
	croak "invalid value for ratio attribute: $ratio, should be one of none, group or table"
	  unless $ratio =~ /^(none|group|table)$/;

	$self->{'ratio'} = $ratio;
    }
    $self->{'ratio'};
}

#------------------------------------------------------------------------
# Method xml_attrs()
#
# Implementation needed by Lire::Aggregate
sub xml_attrs {
    my ($self) = @_;

    my $attr = "";
    if ( exists $self->{'fields'} ) {
	my $fields = join " ", @{$self->{'fields'}};
	$attr .= qq{fields="$fields"};
    }

    $attr .= qq{ ratio="$self->{'ratio'}"};

    $attr;
}

# Implementats Lire::Aggregate::create_numerical_info
sub create_numerical_info {
    my ( $self, $group_info ) = @_;

    $group_info->create_column_info( $self->name, 'numerical', 'int',
                                     $self->label );
}

# Overrides Lire::SimpleStat::sql_required_fields
sub sql_required_fields {
    my $self = $_[0];

    return $self->{'fields'} || [];
}

# Implements Lire::Aggregate::sql_aggr_expr
sub sql_aggr_expr {
    my $self = $_[0];

   if (defined $self->{'fields'}) {
        my @fields = map { sql_quote_name( $_ ) } @{ $self->{'fields'} };
        return 'lr_count_uniq(' . join (',', @fields) .')';
    } else {
        return 'count(*)';
    }
}

# Implements Lire::Aggregate::create_value()
sub create_value {
    my ( $self, $parent_group, $row ) = @_;

    my %value;
    my $name = $self->name();
    $value{'value'} = $row->{$name};

    $self->set_missing_cases_value( $row, \%value );

    if ( $self->{'ratio'} eq 'none' ) {
        $value{'content'} = format_numeric_type( $row->{$name}, 'int' );
    } else {
        my $group_sum = $self->_get_ratio_denominator( $parent_group );
        my $total = defined $group_sum->{'value'} ?
            $group_sum->{'value'} : $row->{$name};
        $value{'content'} = ratio100( $row->{$name}, $total );
    }

    return \%value;
}

sub _get_ratio_denominator {
    my ( $self, $parent_group ) = @_;

    return ( $self->{'ratio'} eq 'table' || ! $parent_group->parent_entry() )
      ? $parent_group->subreport()->get_summary_value( $self->name())
      : $parent_group->get_summary_value( $self->name() );
}

# Implements Lire::ReportOperator::init_group_data
sub init_group_data {
    my ( $self ) = @_;

    if ( $self->{'key_maker'} ) {
        return [ 0, {} ];
    } else {
	my $scalar = 0;
	return \$scalar;
    }
}

# Implements Lire::ReportOperator::merge_group_data
sub merge_group_data {
    my ( $self, $value, $data ) = @_;

    # FIXME: We should add a note when a key_maker was used. For
    # example, there is no guarentee that the sum of the number of
    # hosts that visited during two reports is the same than the
    # number of hosts calculated over the period spanned by the two
    # reports.

    # To merge two counts, we simply add them
    $$data += $value->{'value'};

    return;
}

# Implements Lire::ReportOperator::end_group_data
sub end_group_data {
    my ( $self, $data ) = @_;

    $data->[1] = undef
      if $self->{'key_maker'};

    return;
}

# Implements Lire::Aggregate::data2dlf()
sub data2dlf {
    my ($self, $data) = @_;

    my $name = $self->name();
    return { "$name" => $self->{'key_maker'} ? $data->[0] : $$data,
             "_lr_${name}_mc" => $self->missing_cases( $data ),
           };
}
# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::ReportOperator(3pm), Lire::Aggregator(3pm), Lire::Aggregate(3pm).

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Count.pm,v 1.15 2006/07/23 13:16:28 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html. 

=cut
