package Lire::DocBookParser;

use strict;

use base qw/Lire::XMLParser/;

=pod

=head1 NAME

Lire::XMLParser - Base DocBook parser.

=head1 SYNOPSIS

    package MyParser;

    use base qw/ Lire::XMLParser /;


=head1 DESCRIPTION

This package contains a Lire::XMLParser DocBook parser. It defines the
relevant C<_start>, C<_end> and C<_char> methods for the DocBook
subset used by Lire. All defined handlers have the form
C<dbk_I<name>_start>, C<dbk_I<name>_end> and C<dbk_I<name>_char> (when
the element accepts PCDATA content).

Its base functionality is to collect the DocBook markup for later use.
It will subclassed most of the time.

=pod

=head1 CLIENT METHODS

=head2 dbk_init()

Subclasses should call this method whenever the state of the DocBook
parse should start. This is called from the parse_start() method when
the parser is used standalone.

=cut

sub dbk_init {
    $_[0]->init_collector( 'docbook' );
    $_[0]->{'_dbk_processing'} = 1;
    return;
}

=pod

=head2 dbk_string()

Returns the DocBook XML accumulated by the parser. This is returned
from parse_end() method when used standalone.

=cut

sub dbk_string {
    $_[0]->{'_dbk_processing'} = 0;
    return $_[0]->get_collector( 'docbook' );
};

=pod

=head1 XML SPECIFICATIONS METHODS IMPLEMENTATION

=head2 namespaces()

DocBook doesn't use namespace, so this method returns an empty hash reference.

=cut

sub namespaces {
    return{};
}

=pod

=head2 elements_spec()

Returns the elements specification for the DocBook subset supported by
Lire. 

=cut

our @gen_char_mix = qw/ PCDATA abbrev acronym emphasis foreignphrase phrase
                       quote trademark wordasword
                       action application classname methodname
                       interfacename exceptionname ooclass oointerface
                       ooexception 
                       command computeroutput database email envar
                       errorcode errorname errortype errortext
                       filename function guibutton guiicon guilabel
                       guimenu guimenuitem guisubmenu 
                       hardware interface keycap keycode
                       keycombo keysym literal constantmarkup menuchoice
                       mousebutton option optional parameter prompt property
                       replaceable returnvalue sgmltag
                       structfield structname symbol
                       systemitem token type userinput varname nonterminal
                       productname productnumber subscript superscript
                       constant markup
                    /;
our @inline_mix = ( @gen_char_mix, 'ulink', 'replaceable' );
our  @lists_mix = qw/orderedlist itemizedlist variablelist/;
our @var_char_mix = qw/PCDATA replaceable/;
our @admonitions_mix = qw/caution warning note tip important /;

our @top_levels = ( 'para',
                    @Lire::DocBookParser::admonitions_mix,
                    @Lire::DocBookParser::lists_mix );

my %content_models =
  (
   # Inline elements
   'abbrev' => [ 'PCDATA' ],
   'acronym' => [ 'PCDATA' ],
   'emphasis' => [ @inline_mix ],
   'foreignphrase' => [ @inline_mix ],
   'phrase' => [ @inline_mix ],
   'quote' => [ @inline_mix ],
   'trademark' => [ 'PCDATA' ],
   'wordasword' => [ 'PCDATA' ],
   'action' => [ @var_char_mix ],
   'application' => [ @var_char_mix ],
   'classname' => [ @var_char_mix ],
   'constant' => [ @var_char_mix ],
   'markup' => [ @var_char_mix ],
   'methodname' => [ @var_char_mix ],
   'interfacename' => [ @var_char_mix ],
   'exceptionname' => [ @var_char_mix ],
   'ooclass' => [ @var_char_mix ],
   'oointerface' => [ @var_char_mix ],
   'ooexception' => [ @var_char_mix ],
   'command' => [ @var_char_mix ],
   'computeroutput' => [ @var_char_mix ],
   'database' => [ @var_char_mix ],
   'email' => [ @var_char_mix ],
   'envar' => [ @var_char_mix ],
   'errorcode' => [ @var_char_mix ],
   'errorname' => [ @var_char_mix ],
   'errortype' => [ @var_char_mix ],
   'errortext' => [ @var_char_mix ],
   'filename' => [ @var_char_mix ],
   'function' => [ @var_char_mix ],
   'guibutton' => [ @var_char_mix ],
   'guiicon' => [ @var_char_mix ],
   'guilabel' => [ @var_char_mix ],
   'guimenu' => [ @var_char_mix ],
   'guimenuitem' => [ @var_char_mix ],
   'guisubmenu' => [ @var_char_mix],
   'hardware' => [ @var_char_mix ],
   'interface' => [ @var_char_mix ],
   'keycap' => [ @var_char_mix ],
   'keycode' => [ @var_char_mix ],
   'keycombo' => [ @var_char_mix ],
   'keysym' => [ @var_char_mix ],
   'literal' => [ @var_char_mix ],
   'constantmarkup' => [ @var_char_mix ],
   'menuchoice' => [ @var_char_mix ],
   'mousebutton' => [ @var_char_mix ],
   'option' => [ @var_char_mix ],
   'optional' => [ @inline_mix ],
   'parameter' => [ @var_char_mix],
   'prompt' => [ @var_char_mix ],
   'property' => [ @var_char_mix ],
   'replaceable' => [ 'PCDATA' ],
   'returnvalue' => [ @var_char_mix ],
   'sgmltag' => [ @var_char_mix ],
   'structfield' => [ @var_char_mix ],
   'structname' => [ @var_char_mix ],
   'symbol' => [ @var_char_mix ],
   'systemitem' => [ @var_char_mix ],
   'token' => [ @var_char_mix ],
   'type' => [ @var_char_mix ],
   'userinput' => [ @var_char_mix ],
   'varname' => [ @var_char_mix ],
   'nonterminal' => [ @var_char_mix ],
   'productname' => [ @var_char_mix ],
   'productnumber' => [ @var_char_mix ],
   'subscript' => [ @var_char_mix ],
   'superscript' => [ @var_char_mix ],
   'trademark' => [ @var_char_mix ],

   # Para
   'para' => [ @inline_mix, @lists_mix, @admonitions_mix ],

   # Special
   'ulink' => [ @inline_mix ],
   'title' => [ @inline_mix ],
   'listitem' => [ 'para', @lists_mix, @admonitions_mix ],
   'varlistentry' => [ 'term', 'listitem' ],
   'term' => [ @inline_mix ],

   # Lists 
   'orderedlist' => [ 'title', @admonitions_mix, 'para', 'listitem' ],
   'itemizedlist' => [ 'title', @admonitions_mix, 'para', 'listitem' ],
   'variablelist' => [ 'title', @admonitions_mix, 'para', 'varlistentry' ],

   # Admonitions
   'caution' => [ 'title', 'para', @lists_mix, @admonitions_mix ],
   'important' => [ 'title', 'para', @lists_mix, @admonitions_mix ],
   'note' => [ 'title', 'para', @lists_mix, @admonitions_mix ],
   'tip' => [ 'title', 'para', @lists_mix, @admonitions_mix ],
   'warning' => [ 'title', 'para', @lists_mix, @admonitions_mix ],

  );

my $spec = {};
while ( my ( $name, $content ) = each %content_models ) {
    my $s = { 'start' => "dbk_${name}_start",
              'end' => "dbk_${name}_end",
              'content' => $content };
    $s->{'char'} = "dbk_${name}_char"
      if ( grep { $_ eq 'PCDATA' } @$content );
    $spec->{$name} = $s;


    no strict 'refs';
    while ( my ($name, $content) = each %$spec ) {
        *{"dbk_${name}_start"} = \&dbk_element_start;
        *{"dbk_${name}_char"} = \&dbk_element_char
          if defined $s->{'char'};
        *{"dbk_${name}_end"} = \&dbk_element_end;
    }
}

sub elements_spec {
    return { %$spec };
}

sub dbk_element_start {
    my ( $self, $element, $attributes ) = @_;

    $self->collect( 'docbook', $self->recognized_string() );

    return;
}

sub dbk_element_end {
    my ( $self, $element ) = @_;

    $self->collect( 'docbook', $self->recognized_string() );

    return;
}

sub dbk_element_char {
    my ( $self, $text ) = @_;

    $self->collect( 'docbook', $self->recognized_string() );

    return;
}

sub ignorable_ws {
    my ( $self, $text ) = @_;

    $self->collect( 'docbook', $self->recognized_string() )
      if $self->{'_dbk_processing'};

    return;
}

sub parse_start {
    $_[0]->dbk_init();
    return;
}

sub parse_end {
    return $_[0]->dbk_string();
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::XMLParser(3pm), Lire::ReportParser::AsciiDocBookFormatter(3pm),
 Lire::ReportParser::HtmlDocBookFormatter(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: DocBookParser.pm,v 1.3 2004/07/20 16:42:02 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
