package Lire::Config::VariableIndex;

use strict;

use base qw/Lire::Config::Index/;

use Lire::Utils qw/check_param check_object_param/;
use Carp;

=pod

=head1 NAME

Lire::Config::VariableIndex - Index of the available variables in a TableInfo

=head1 SYNOPSIS

  <lrcsml:reference name="name" index="categorical_variables" />
  <lrcsml:reference name="name" index="numerical_variables" />
  <lrcsml:reference name="name" index="variables" />

=head1 DESCRIPTION

This object make it possible to select a variable defined in a
TableInfo (usually created from a Lire::ReportSpec or coming from a
Lire::Report object).

=head2 new( $table_info, [ $type ] )

Creates a new VariableIndex which wraps the $table_info
Lire::Report::TableInfo object. The $type parameter can be either
'both', 'categorical' or 'numerical' depending on the kind of
variables one wants to select. Default to 'both'.

=head2 table_info()

Returns the Lire::Report::TableInfo object wrapped by this Index.

=head2 type()

Returns the type attribute.

=cut

sub new {
    my ( $class, $info, $type ) = @_;

    check_object_param( $info, 'info', 'Lire::Report::TableInfo' );
    $type ||= 'both';
    check_param( $type, 'type', qr/^(both|categorical|numerical)$/,
                 "'type' parameter should be either 'both', 'categorical' or 'numerical'" );

    return bless { '_info' => $info,
                   '_type' => $type,
                 }, $class;
}

sub table_info {
    return $_[0]{'_info'};
}

sub type {
    return $_[0]{'_type'};
}

sub has_entry {
    my ( $self, $name ) = @_;

    check_param( $name, 'name' );

    foreach my $col ( $self->{'_info'}->column_infos() ) {
        next if $self->{'_type'} ne 'both' && $col->class() ne $self->{'_type'};
        return 1 if $name eq $col->name();
    }

    return 0;
}

sub entries {
    my $self = $_[0];

    my @entries = ();
    foreach my $col ( $self->{'_info'}->column_infos() ) {
        next if $self->{'_type'} ne 'both' && $col->class() ne $self->{'_type'};
        push @entries, $col->name();
    }
    return \@entries;
}

sub get_ref {
    my ( $self, $name ) = @_;

    check_param( $name, 'name' );
    foreach my $col ( $self->{'_info'}->column_infos() ) {
        next if $self->{'_type'} ne 'both' && $col->class() ne $self->{'_type'};
        return $col if $name eq $col->name();
    }

    croak "no variable \"$name\"";
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::ReportSpec(3pm) Lire::Report::TableInfo(3pm) Lire:Config::Index(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: VariableIndex.pm,v 1.2 2004/07/15 22:07:35 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
