

type 'a chunk = 'a * int * int
type 'a buffer = 'a chunk Queue.t

(** All positions and lengths are in ticks. *)
type 'a t = {
  mutable length  : int ;
  mutable offset  : int ;
  mutable buffers : 'a buffer
}

let create () =
  {
    length    = 0 ;
    offset    = 0 ;
    buffers   = Queue.create ()
  }

let clear g =
  g.length <- 0 ;
  g.offset <- 0 ;
  g.buffers <- Queue.create ()

let length b = b.length

(** Remove [len] ticks of data. *)
let rec remove g len =
  assert (g.length >= len) ;
  if len>0 then
  let b,_,b_len = Queue.peek g.buffers in
    (* Is it enough to advance in the first buffer?
     * Or do we need to consume it completely and go farther in the queue? *)
    if g.offset + len < b_len then begin
      g.length <- g.length - len ;
      g.offset <- g.offset + len ;
    end else
      let removed = b_len - g.offset in
        ignore (Queue.take g.buffers) ;
        g.length <- g.length - removed ;
        g.offset <- 0 ;
        remove g (len-removed)

(** Remove data at the end of the generator: this is inefficient, not
  * natural for a Generator. *)
let rec remove_end g remove_len =
  (* Remove length [l] at the beginning of the buffers,
   * should correspond exactly to some last [n] chunks. *)
  let rec remove l =
    if l>0 then
      let (_,ofs,len) = Queue.take g.buffers in
        assert (l>=len) ;
        remove (l-len)
  in
  (* Go through the beginning of length [l] of the queue,
   * possibly cut some element in half, remove the rest.
   * The parsed elements are put back at the end
   * of the queue. *)
  let rec cut l =
    let (c,ofs,len) = Queue.take g.buffers in
      if len<l then begin
        Queue.push (c,ofs,len) g.buffers ;
        cut (l-len)
      end else begin
        Queue.push (c,ofs,l) g.buffers ;
        remove (remove_len-(len-l))
      end
  in
  let new_len = g.length - remove_len in
    cut new_len ;
    g.length <- new_len

(** Feed an item into a generator.
  * The item is put as such, not copied. *)
let put g content ofs len =
  g.length <- g.length + len ;
  Queue.add (content,ofs,len) g.buffers

(** Get [size] amount of data from [g].
  * Returns a list where each element will typically be passed to a blit:
  * its elements are of the form [b,o,o',l] where [o] is the offset of data
  * in the block [b], [o'] is the position at which it should be written
  * (the first position [o'] will always be [0]), and [l] is the length
  * of data to be taken from that block. *)
let get g size =
  (* The main loop takes the current offset in the output buffer,
   * and iterates on input buffer chunks. *)
  let rec aux chunks offset =
    (* How much (more) data should be output? *)
    let needed = size - offset in
      assert (needed>0) ;
      let block,block_ofs,block_len = Queue.peek g.buffers in
      let block_len = block_len - g.offset in
      let copied = min needed block_len in
      let chunks = (block, block_ofs + g.offset, offset, copied)::chunks in
        (* Update buffer data -- did we consume a full block? *)
        if block_len <= needed then begin
          ignore (Queue.take g.buffers) ;
          g.length <- g.length - block_len ;
          g.offset <- 0
        end else begin
          g.length <- g.length - needed ;
          g.offset <- g.offset + needed
        end ;
        (* Add more data by recursing on the next block, or finish. *)
        if block_len < needed then
          aux chunks (offset+block_len)
        else
          List.rev chunks
  in
    if size = 0 then [] else aux [] 0

let x = create ()
let print () =
  Queue.iter (fun (c,o,l) -> Printf.printf "+ %s %d\n" c l) x.buffers

let get () =
  Printf.printf "get %d\n" x.length ;
  let l = get x x.length in
    List.iter (fun (c,o,_,l) -> Printf.printf ". %s %d\n" c l ; put x c o l) l ;
    ()

let () =
  put x "0" 0 10 ;
  put x "1" 0 2 ;
  put x "2" 0 10 ;
  remove_end x 10 ;
  print () ;
  get () ;
  remove_end x 1 ;
  print () ;
  get () ;
  remove_end x 1 ;
  print () ;
  get () ;
  remove_end x 9 ;
  print () ;
  get () ;
