type token =
  | VAR of (string)
  | VARLPAR of (string)
  | VARLBRA of (string)
  | STRING of (string)
  | INT of (int)
  | FLOAT of (float)
  | BOOL of (bool)
  | TIME of (int option list)
  | INTERVAL of (int option list * int option list)
  | OGG
  | VORBIS
  | VORBIS_CBR
  | VORBIS_ABR
  | THEORA
  | DIRAC
  | SPEEX
  | WAV
  | AACPLUS
  | MP3
  | EXTERNAL
  | EOF
  | BEGIN
  | END
  | GETS
  | TILD
  | DEF of (Doc.item * (string*string) list)
  | IF
  | THEN
  | ELSE
  | ELSIF
  | LPAR
  | RPAR
  | COMMA
  | SEQ
  | SEQSEQ
  | COLON
  | LBRA
  | RBRA
  | LCUR
  | RCUR
  | FUN
  | YIELDS
  | BIN0 of (string)
  | BIN1 of (string)
  | BIN2 of (string)
  | BIN3 of (string)
  | TIMES
  | MINUS
  | NOT
  | REF
  | GET
  | SET
  | PP_IFDEF
  | PP_ENDIF
  | PP_ENDL
  | PP_DEF
  | PP_INCLUDE of (string)
  | PP_COMMENT of (string list)

open Parsing;;
# 24 "lang/lang_parser.mly"

  open Source
  open Lang_values

  (** Parsing locations. *)
  let curpos ?pos () =
    match pos with
      | None -> Parsing.symbol_start_pos (), Parsing.symbol_end_pos ()
      | Some (i,j) -> Parsing.rhs_start_pos i, Parsing.rhs_end_pos j

  (** Create a new value with an unknown type. *)
  let mk ?pos e =
    let kind =
      T.fresh_evar ~level:(-1) ~pos:(Some (curpos ?pos ()))
    in
      if Lang_values.debug then
        Printf.eprintf "%s (%s): assigned type var %s\n"
          (T.print_pos (Utils.get_some kind.T.pos))
          (try Lang_values.print_term {t=kind;term=e} with _ -> "<?>")
          (T.print kind) ;
      { t = kind ; term = e }

  let mk_fun ?pos args body =
    let bound = List.map (fun (_,x,_,_) -> x) args in
    let fv = Lang_values.free_vars ~bound body in
      mk ?pos (Fun (fv,args,body))

  (** Time intervals *)

  let time_units = [| 7*24*60*60 ; 24*60*60 ; 60*60 ; 60 ; 1 |]

  let date =
    let to_int = function None -> 0 | Some i -> i in
    let rec aux = function
      | None::tl -> aux tl
      | [] -> failwith "Invalid time"
      | l ->
          let a = Array.of_list l in
          let n = Array.length a in
          let tu = time_units and tn = Array.length time_units in
            Array.fold_left (+) 0
              (Array.mapi (fun i s ->
                             let s =
                               if n=4 && i=0 then
                                 (to_int s) mod 7
                               else
                                 to_int s
                             in
                               tu.(tn-1 + i - n+1) * s) a)
    in
      aux

  let rec last_index e n = function
    | x::tl -> if x=e then last_index e (n+1) tl else n
    | [] -> n

  let precision d = time_units.(last_index None 0 d)
  let duration d = time_units.(Array.length time_units - 1 - 
                               last_index None 0 (List.rev d))

  let between d1 d2 =
    let p1 = precision d1 in
    let p2 = precision d2 in
    let t1 = date d1 in
    let t2 = date d2 in
      if p1<>p2 then failwith "Invalid time interval: precisions differ" ;
      (t1,t2,p1)

  let during d =
    let t,d,p = date d, duration d, precision d in
      (t,t+d,p)

  let mk_time_pred (a,b,c) =
    let args = List.map (fun x -> "", mk (Int x)) [a;b;c] in
      mk (App (mk (Var "time_in_mod"), args))

  let mk_ty name args =
    match name with
      | "_" -> Lang_types.fresh_evar ~level:(-1) ~pos:None
      | "unit" -> Lang_types.make (Lang_types.Ground Lang_types.Unit)
      | "bool" -> Lang_types.make (Lang_types.Ground Lang_types.Bool)
      | "int" -> Lang_types.make (Lang_types.Ground Lang_types.Int)
      | "float" -> Lang_types.make (Lang_types.Ground Lang_types.Float)
      | "string" -> Lang_types.make (Lang_types.Ground Lang_types.String)
      | "source" ->
          (* TODO less confusion in hiding the stream_kind constructed type *)
          let audio,video,midi =
            match args with
              | ["",a;"",v;"",m] -> a,v,m
              | _::_::_::_ -> failwith "invalid type parameters"
              | l ->
                  let assoc x =
                    try List.assoc x l with
                      | Not_found ->
                          Lang_types.fresh_evar ~level:(-1) ~pos:None
                  in
                    assoc "audio", assoc "video", assoc "midi"
          in
            Lang_values.source_t (Lang_values.frame_kind_t audio video midi)
      | _ -> failwith "unknown type constructor"

  open Lang_encoders

# 167 "lang/lang_parser.ml"
let yytransl_const = [|
  266 (* OGG *);
  267 (* VORBIS *);
  268 (* VORBIS_CBR *);
  269 (* VORBIS_ABR *);
  270 (* THEORA *);
  271 (* DIRAC *);
  272 (* SPEEX *);
  273 (* WAV *);
  274 (* AACPLUS *);
  275 (* MP3 *);
  276 (* EXTERNAL *);
    0 (* EOF *);
  277 (* BEGIN *);
  278 (* END *);
  279 (* GETS *);
  280 (* TILD *);
  282 (* IF *);
  283 (* THEN *);
  284 (* ELSE *);
  285 (* ELSIF *);
  286 (* LPAR *);
  287 (* RPAR *);
  288 (* COMMA *);
  289 (* SEQ *);
  290 (* SEQSEQ *);
  291 (* COLON *);
  292 (* LBRA *);
  293 (* RBRA *);
  294 (* LCUR *);
  295 (* RCUR *);
  296 (* FUN *);
  297 (* YIELDS *);
  302 (* TIMES *);
  303 (* MINUS *);
  304 (* NOT *);
  305 (* REF *);
  306 (* GET *);
  307 (* SET *);
  308 (* PP_IFDEF *);
  309 (* PP_ENDIF *);
  310 (* PP_ENDL *);
  311 (* PP_DEF *);
    0|]

let yytransl_block = [|
  257 (* VAR *);
  258 (* VARLPAR *);
  259 (* VARLBRA *);
  260 (* STRING *);
  261 (* INT *);
  262 (* FLOAT *);
  263 (* BOOL *);
  264 (* TIME *);
  265 (* INTERVAL *);
  281 (* DEF *);
  298 (* BIN0 *);
  299 (* BIN1 *);
  300 (* BIN2 *);
  301 (* BIN3 *);
  312 (* PP_INCLUDE *);
  313 (* PP_COMMENT *);
    0|]

let yylhs = "\255\255\
\001\000\001\000\002\000\002\000\004\000\004\000\005\000\005\000\
\003\000\003\000\003\000\003\000\008\000\008\000\008\000\008\000\
\008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
\008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
\008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
\008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
\008\000\009\000\009\000\009\000\009\000\009\000\017\000\017\000\
\017\000\018\000\018\000\006\000\006\000\006\000\006\000\006\000\
\006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
\006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
\006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
\006\000\006\000\006\000\006\000\006\000\010\000\019\000\019\000\
\019\000\020\000\020\000\014\000\014\000\014\000\007\000\007\000\
\007\000\015\000\015\000\015\000\021\000\021\000\022\000\022\000\
\016\000\016\000\016\000\011\000\011\000\012\000\012\000\013\000\
\013\000\013\000\013\000\013\000\013\000\000\000\000\000"

let yylen = "\002\000\
\001\000\002\000\002\000\001\000\000\000\001\000\000\000\001\000\
\002\000\003\000\002\000\003\000\005\000\002\000\002\000\002\000\
\003\000\001\000\002\000\001\000\001\000\001\000\001\000\002\000\
\002\000\003\000\002\000\002\000\002\000\002\000\004\000\001\000\
\002\000\005\000\001\000\003\000\003\000\003\000\006\000\003\000\
\006\000\003\000\003\000\003\000\003\000\003\000\003\000\001\000\
\001\000\001\000\003\000\003\000\005\000\001\000\000\000\001\000\
\003\000\001\000\003\000\003\000\005\000\001\000\002\000\001\000\
\001\000\001\000\001\000\002\000\002\000\003\000\002\000\002\000\
\002\000\002\000\004\000\001\000\002\000\005\000\001\000\003\000\
\003\000\003\000\006\000\003\000\006\000\003\000\003\000\003\000\
\003\000\003\000\003\000\001\000\001\000\003\000\003\000\001\000\
\000\000\003\000\001\000\000\000\001\000\003\000\003\000\005\000\
\007\000\000\000\001\000\003\000\003\000\002\000\002\000\000\000\
\005\000\002\000\000\000\000\000\003\000\001\000\003\000\002\000\
\002\000\002\000\002\000\002\000\002\000\002\000\002\000"

let yydefred = "\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\066\000\062\000\
\065\000\064\000\093\000\092\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\126\000\000\000\000\000\000\000\067\000\076\000\
\004\000\127\000\000\000\000\000\000\000\000\000\000\000\022\000\
\018\000\021\000\020\000\049\000\048\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\023\000\032\000\000\000\000\000\
\035\000\000\000\000\000\000\000\120\000\121\000\122\000\123\000\
\124\000\125\000\074\000\072\000\071\000\073\000\000\000\000\000\
\000\000\000\000\077\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\069\000\002\000\006\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\003\000\000\000\
\000\000\000\000\000\000\000\000\030\000\028\000\027\000\029\000\
\000\000\000\000\033\000\000\000\000\000\000\000\015\000\014\000\
\000\000\000\000\000\000\025\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\080\000\000\000\081\000\000\000\000\000\
\000\000\082\000\008\000\000\000\000\000\000\000\000\000\000\000\
\000\000\060\000\000\000\000\000\000\000\094\000\084\000\000\000\
\000\000\000\000\000\000\089\000\090\000\000\000\000\000\010\000\
\012\000\000\000\036\000\037\000\000\000\038\000\000\000\017\000\
\000\000\000\000\040\000\000\000\000\000\000\000\000\000\045\000\
\046\000\000\000\000\000\102\000\075\000\000\000\117\000\000\000\
\000\000\110\000\000\000\000\000\000\000\000\000\000\000\050\000\
\000\000\054\000\000\000\000\000\000\000\095\000\000\000\031\000\
\000\000\000\000\000\000\000\000\119\000\104\000\000\000\109\000\
\000\000\108\000\000\000\000\000\000\000\078\000\000\000\058\000\
\000\000\000\000\000\000\000\000\061\000\000\000\000\000\034\000\
\013\000\000\000\000\000\114\000\000\000\085\000\000\000\051\000\
\000\000\000\000\052\000\000\000\041\000\000\000\105\000\000\000\
\059\000\057\000\000\000\000\000\053\000\113\000"

let yydgoto = "\003\000\
\035\000\042\000\036\000\109\000\148\000\037\000\038\000\068\000\
\224\000\069\000\077\000\143\000\070\000\071\000\151\000\221\000\
\225\000\226\000\094\000\072\000\152\000\194\000"

let yysindex = "\029\000\
\170\004\220\004\000\000\249\254\007\006\057\006\000\000\000\000\
\000\000\000\000\000\000\000\000\009\255\015\255\015\255\015\255\
\015\255\015\255\015\255\015\255\015\255\015\255\015\255\000\000\
\107\006\032\255\107\006\122\255\057\006\107\006\033\255\057\006\
\057\006\057\006\000\000\043\000\170\003\043\255\000\000\000\000\
\000\000\000\000\031\255\057\006\051\255\007\006\057\006\000\000\
\000\000\000\000\000\000\000\000\000\000\050\255\015\255\015\255\
\015\255\015\255\107\006\107\006\213\005\107\006\056\255\157\006\
\057\006\057\006\057\006\231\254\000\000\000\000\066\255\072\255\
\000\000\098\002\152\255\007\006\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\065\255\079\255\
\013\255\078\255\000\000\024\255\087\002\073\255\105\255\013\255\
\075\255\164\255\000\000\000\000\000\000\057\006\057\006\057\006\
\057\006\057\006\057\006\057\006\107\006\107\006\000\000\231\254\
\057\006\114\255\235\004\152\255\000\000\000\000\000\000\000\000\
\087\255\120\255\000\000\047\255\110\255\013\255\000\000\000\000\
\252\254\075\255\164\255\000\000\057\006\057\006\057\006\057\006\
\057\006\057\006\057\006\000\000\007\006\000\000\125\255\127\255\
\126\255\000\000\000\000\107\006\138\255\173\255\144\255\146\255\
\107\006\000\000\057\006\008\255\057\006\000\000\000\000\145\255\
\070\255\075\255\252\254\000\000\000\000\252\254\231\254\000\000\
\000\000\231\254\000\000\000\000\148\255\000\000\107\006\000\000\
\057\006\008\255\000\000\149\255\070\255\075\255\252\254\000\000\
\000\000\252\254\231\254\000\000\000\000\152\255\000\000\159\255\
\057\006\000\000\138\255\079\255\013\255\019\255\151\255\000\000\
\010\255\000\000\008\255\008\255\168\255\000\000\160\255\000\000\
\019\255\024\001\169\255\162\255\000\000\000\000\231\254\000\000\
\107\006\000\000\107\006\107\006\182\255\000\000\189\255\000\000\
\183\255\181\255\170\255\192\255\000\000\057\006\198\255\000\000\
\000\000\057\006\208\255\000\000\204\255\000\000\008\255\000\000\
\010\255\008\255\000\000\231\254\000\000\231\254\000\000\107\006\
\000\000\000\000\201\255\019\255\000\000\000\000"

let yyrindex = "\000\000\
\000\000\000\000\000\000\205\000\202\255\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\001\000\001\000\001\000\
\001\000\001\000\001\000\126\001\126\001\126\001\126\001\000\000\
\000\000\000\000\000\000\000\000\197\255\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\120\004\120\004\000\000\000\000\
\000\000\000\000\000\000\000\000\196\255\202\255\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\001\000\001\000\
\001\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\022\255\000\000\000\000\000\000\205\255\
\000\000\000\000\000\000\202\255\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\207\006\
\206\255\000\000\000\000\000\000\207\255\000\000\000\000\206\255\
\149\002\046\003\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\025\001\172\003\000\000\223\003\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\206\255\000\000\000\000\
\052\000\000\001\228\001\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\202\255\000\000\000\000\214\255\
\000\000\000\000\000\000\000\000\041\255\000\000\000\000\217\255\
\000\000\000\000\000\000\000\000\197\255\000\000\000\000\000\000\
\251\002\200\002\023\002\000\000\000\000\074\002\018\004\000\000\
\000\000\052\255\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\177\001\075\001\103\000\000\000\
\000\000\154\000\097\003\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\041\255\207\006\206\255\213\255\000\000\000\000\
\218\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\213\255\000\000\000\000\000\000\000\000\000\000\064\255\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\068\255\000\000\
\000\000\220\255\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\218\255\000\000\000\000\069\004\000\000\147\003\000\000\000\000\
\000\000\000\000\000\000\213\255\000\000\000\000"

let yygindex = "\000\000\
\000\000\000\000\254\255\214\000\057\000\000\000\000\000\012\006\
\102\255\246\005\168\000\147\255\002\000\216\255\165\255\055\255\
\013\000\000\000\098\000\000\000\000\000\062\000"

let yytablesize = 2049
let yytable = "\043\000\
\116\000\205\000\040\000\040\000\160\000\114\000\173\000\231\000\
\200\000\201\000\223\000\201\000\202\000\149\000\202\000\044\000\
\133\000\134\000\135\000\136\000\137\000\138\000\087\000\211\000\
\090\000\139\000\040\000\095\000\040\000\001\000\002\000\040\000\
\088\000\089\000\180\000\145\000\150\000\203\000\075\000\203\000\
\136\000\137\000\100\000\204\000\076\000\204\000\219\000\220\000\
\227\000\228\000\254\000\016\000\099\000\099\000\154\000\155\000\
\121\000\122\000\156\000\125\000\040\000\040\000\096\000\040\000\
\111\000\133\000\134\000\135\000\136\000\137\000\138\000\112\000\
\112\000\113\000\139\000\101\000\144\000\176\000\177\000\116\000\
\213\000\178\000\098\000\098\000\249\000\126\000\146\000\251\000\
\133\000\134\000\135\000\136\000\137\000\138\000\111\000\111\000\
\140\000\139\000\050\000\050\000\188\000\147\000\044\000\141\000\
\153\000\218\000\168\000\169\000\174\000\158\000\040\000\040\000\
\134\000\135\000\136\000\137\000\138\000\144\000\135\000\136\000\
\137\000\138\000\073\000\046\000\047\000\048\000\049\000\050\000\
\051\000\052\000\053\000\054\000\014\000\015\000\016\000\017\000\
\018\000\019\000\055\000\056\000\057\000\058\000\059\000\159\000\
\171\000\192\000\175\000\060\000\179\000\040\000\198\000\061\000\
\091\000\047\000\040\000\189\000\191\000\029\000\190\000\062\000\
\193\000\063\000\014\000\015\000\016\000\017\000\018\000\019\000\
\064\000\065\000\066\000\067\000\209\000\195\000\196\000\207\000\
\040\000\197\000\208\000\212\000\214\000\222\000\078\000\079\000\
\080\000\081\000\082\000\083\000\084\000\085\000\086\000\144\000\
\133\000\134\000\135\000\136\000\137\000\138\000\229\000\233\000\
\230\000\139\000\234\000\238\000\079\000\133\000\134\000\135\000\
\136\000\137\000\138\000\239\000\241\000\240\000\235\000\242\000\
\236\000\237\000\040\000\245\000\040\000\040\000\117\000\118\000\
\119\000\120\000\035\000\035\000\243\000\247\000\248\000\253\000\
\100\000\097\000\115\000\101\000\106\000\035\000\035\000\035\000\
\035\000\035\000\035\000\096\000\118\000\252\000\035\000\107\000\
\055\000\040\000\056\000\110\000\217\000\250\000\206\000\019\000\
\216\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\000\000\
\009\000\116\000\116\000\116\000\116\000\116\000\000\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\000\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\016\000\016\000\016\000\016\000\
\016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
\016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
\016\000\016\000\043\000\000\000\016\000\016\000\016\000\016\000\
\016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
\016\000\016\000\016\000\016\000\000\000\016\000\016\000\016\000\
\000\000\000\000\016\000\016\000\016\000\016\000\016\000\044\000\
\044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
\044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
\044\000\044\000\044\000\044\000\044\000\116\000\000\000\044\000\
\044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
\044\000\044\000\044\000\044\000\044\000\044\000\044\000\000\000\
\044\000\044\000\044\000\000\000\000\000\044\000\044\000\044\000\
\044\000\044\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
\042\000\000\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
\047\000\047\000\000\000\047\000\047\000\047\000\000\000\000\000\
\047\000\047\000\047\000\047\000\047\000\079\000\079\000\079\000\
\079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
\079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
\079\000\079\000\079\000\024\000\000\000\079\000\079\000\079\000\
\079\000\079\000\079\000\000\000\000\000\079\000\079\000\000\000\
\079\000\000\000\079\000\079\000\079\000\000\000\079\000\079\000\
\079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
\019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
\019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
\019\000\019\000\019\000\019\000\019\000\019\000\088\000\000\000\
\019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
\019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
\000\000\019\000\019\000\000\000\000\000\000\000\009\000\019\000\
\019\000\019\000\019\000\009\000\009\000\009\000\232\000\000\000\
\000\000\000\000\009\000\000\000\000\000\000\000\000\000\009\000\
\000\000\133\000\134\000\135\000\136\000\137\000\138\000\000\000\
\000\000\091\000\139\000\043\000\043\000\043\000\043\000\043\000\
\043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
\043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
\043\000\000\000\000\000\043\000\043\000\043\000\043\000\043\000\
\043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
\043\000\043\000\043\000\000\000\043\000\043\000\000\000\000\000\
\000\000\000\000\043\000\043\000\043\000\043\000\116\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\116\000\116\000\116\000\063\000\000\000\116\000\116\000\
\116\000\116\000\116\000\000\000\000\000\000\000\116\000\116\000\
\000\000\116\000\000\000\116\000\116\000\116\000\000\000\116\000\
\116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
\116\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
\042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
\042\000\042\000\042\000\042\000\042\000\042\000\042\000\087\000\
\000\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
\042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
\042\000\000\000\042\000\000\000\000\000\000\000\000\000\000\000\
\042\000\042\000\042\000\042\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\086\000\000\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\024\000\024\000\024\000\024\000\088\000\
\088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
\088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
\088\000\088\000\088\000\088\000\088\000\068\000\000\000\088\000\
\088\000\088\000\088\000\088\000\088\000\000\000\000\000\088\000\
\088\000\000\000\088\000\000\000\088\000\088\000\088\000\000\000\
\088\000\088\000\088\000\000\000\000\000\088\000\088\000\088\000\
\088\000\088\000\091\000\091\000\091\000\091\000\091\000\091\000\
\091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
\091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
\026\000\000\000\091\000\091\000\091\000\091\000\091\000\091\000\
\000\000\000\000\091\000\091\000\000\000\091\000\000\000\091\000\
\091\000\091\000\000\000\091\000\091\000\091\000\157\000\000\000\
\091\000\091\000\091\000\091\000\091\000\000\000\000\000\000\000\
\133\000\134\000\135\000\136\000\137\000\138\000\142\000\000\000\
\000\000\139\000\000\000\133\000\134\000\135\000\136\000\137\000\
\138\000\000\000\039\000\000\000\139\000\063\000\063\000\063\000\
\063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
\063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
\063\000\063\000\063\000\011\000\000\000\063\000\063\000\063\000\
\063\000\063\000\063\000\000\000\000\000\063\000\063\000\000\000\
\063\000\000\000\063\000\063\000\063\000\000\000\063\000\063\000\
\000\000\000\000\000\000\000\000\063\000\063\000\063\000\063\000\
\087\000\087\000\087\000\087\000\087\000\087\000\087\000\087\000\
\087\000\087\000\087\000\087\000\087\000\087\000\087\000\087\000\
\087\000\087\000\087\000\087\000\087\000\087\000\103\000\000\000\
\087\000\087\000\087\000\087\000\087\000\087\000\000\000\000\000\
\087\000\087\000\000\000\087\000\000\000\087\000\087\000\087\000\
\000\000\087\000\087\000\000\000\000\000\000\000\000\000\087\000\
\087\000\087\000\087\000\086\000\086\000\086\000\086\000\086\000\
\086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
\086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
\086\000\070\000\000\000\086\000\086\000\086\000\086\000\086\000\
\086\000\000\000\000\000\086\000\086\000\000\000\086\000\000\000\
\086\000\086\000\086\000\000\000\086\000\000\000\000\000\000\000\
\000\000\000\000\086\000\086\000\086\000\086\000\068\000\068\000\
\068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
\068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
\068\000\068\000\068\000\068\000\083\000\000\000\068\000\068\000\
\068\000\068\000\068\000\068\000\000\000\000\000\068\000\068\000\
\000\000\068\000\000\000\068\000\068\000\068\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\068\000\068\000\068\000\
\068\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
\026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
\026\000\026\000\026\000\026\000\026\000\026\000\026\000\005\000\
\000\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
\026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
\026\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\026\000\026\000\026\000\039\000\039\000\039\000\039\000\039\000\
\039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
\039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
\039\000\024\000\000\000\039\000\039\000\039\000\039\000\039\000\
\039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
\039\000\039\000\039\000\000\000\000\000\000\000\000\000\000\000\
\000\000\011\000\039\000\039\000\039\000\000\000\011\000\011\000\
\011\000\000\000\101\000\000\000\000\000\011\000\000\000\000\000\
\000\000\000\000\011\000\102\000\103\000\104\000\105\000\106\000\
\107\000\000\000\000\000\041\000\108\000\000\000\000\000\103\000\
\103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
\103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
\103\000\103\000\103\000\103\000\103\000\000\000\000\000\103\000\
\103\000\103\000\103\000\103\000\103\000\000\000\000\000\103\000\
\103\000\000\000\103\000\000\000\103\000\103\000\103\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\103\000\103\000\
\103\000\000\000\070\000\070\000\070\000\070\000\070\000\070\000\
\070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
\070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
\000\000\000\000\070\000\070\000\070\000\070\000\070\000\070\000\
\000\000\000\000\070\000\070\000\000\000\070\000\000\000\070\000\
\070\000\070\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\070\000\070\000\070\000\000\000\083\000\083\000\083\000\
\083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
\083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
\083\000\083\000\083\000\000\000\000\000\083\000\083\000\083\000\
\083\000\083\000\083\000\000\000\000\000\083\000\083\000\000\000\
\083\000\000\000\083\000\083\000\083\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\083\000\083\000\083\000\000\000\
\005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
\005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
\005\000\005\000\005\000\005\000\005\000\005\000\000\000\000\000\
\005\000\005\000\005\000\005\000\005\000\005\000\000\000\000\000\
\000\000\005\000\000\000\005\000\000\000\005\000\005\000\005\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\005\000\
\005\000\005\000\004\000\005\000\006\000\007\000\008\000\009\000\
\010\000\011\000\012\000\013\000\014\000\015\000\016\000\017\000\
\018\000\019\000\020\000\021\000\022\000\023\000\025\000\000\000\
\000\000\000\000\026\000\027\000\000\000\000\000\000\000\028\000\
\000\000\000\000\000\000\000\000\000\000\029\000\000\000\030\000\
\000\000\031\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\032\000\033\000\034\000\004\000\005\000\006\000\007\000\
\008\000\009\000\010\000\011\000\012\000\013\000\014\000\015\000\
\016\000\017\000\018\000\019\000\020\000\021\000\022\000\023\000\
\025\000\000\000\000\000\000\000\026\000\027\000\039\000\039\000\
\000\000\028\000\000\000\000\000\000\000\000\000\000\000\029\000\
\000\000\030\000\000\000\031\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\032\000\033\000\034\000\039\000\172\000\
\039\000\074\000\000\000\039\000\133\000\134\000\135\000\136\000\
\137\000\138\000\000\000\000\000\000\000\139\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\092\000\
\093\000\000\000\000\000\097\000\098\000\099\000\000\000\000\000\
\039\000\039\000\000\000\039\000\000\000\000\000\000\000\112\000\
\000\000\000\000\115\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\124\000\000\000\000\000\129\000\130\000\131\000\132\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\039\000\039\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\161\000\162\000\163\000\164\000\165\000\166\000\167\000\
\000\000\000\000\000\000\000\000\170\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\039\000\000\000\000\000\000\000\000\000\039\000\000\000\
\181\000\182\000\183\000\184\000\185\000\186\000\187\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\039\000\000\000\199\000\000\000\
\093\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\210\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\215\000\000\000\039\000\000\000\
\039\000\039\000\000\000\000\000\000\000\073\000\046\000\047\000\
\048\000\049\000\050\000\051\000\052\000\053\000\054\000\014\000\
\015\000\016\000\017\000\018\000\019\000\055\000\056\000\057\000\
\058\000\059\000\000\000\000\000\000\000\039\000\060\000\000\000\
\000\000\244\000\061\000\123\000\000\000\246\000\000\000\000\000\
\029\000\000\000\062\000\000\000\063\000\000\000\000\000\000\000\
\000\000\000\000\000\000\064\000\065\000\066\000\067\000\045\000\
\046\000\047\000\048\000\049\000\050\000\051\000\052\000\053\000\
\054\000\014\000\015\000\016\000\017\000\018\000\019\000\055\000\
\056\000\057\000\058\000\059\000\000\000\000\000\000\000\000\000\
\060\000\000\000\000\000\000\000\061\000\000\000\000\000\000\000\
\000\000\000\000\029\000\000\000\062\000\000\000\063\000\000\000\
\000\000\000\000\000\000\000\000\000\000\064\000\065\000\066\000\
\067\000\073\000\046\000\047\000\048\000\049\000\050\000\051\000\
\052\000\053\000\054\000\014\000\015\000\016\000\017\000\018\000\
\019\000\055\000\056\000\057\000\058\000\059\000\000\000\000\000\
\000\000\000\000\060\000\000\000\000\000\000\000\061\000\000\000\
\000\000\000\000\000\000\000\000\029\000\000\000\062\000\000\000\
\063\000\000\000\000\000\000\000\000\000\000\000\000\000\064\000\
\065\000\066\000\067\000\004\000\005\000\006\000\007\000\008\000\
\009\000\010\000\011\000\012\000\013\000\014\000\015\000\016\000\
\017\000\018\000\019\000\020\000\021\000\022\000\023\000\025\000\
\000\000\000\000\000\000\026\000\027\000\000\000\000\000\000\000\
\028\000\000\000\000\000\000\000\000\000\000\000\029\000\000\000\
\030\000\000\000\031\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\032\000\033\000\034\000\073\000\046\000\047\000\
\048\000\127\000\128\000\051\000\052\000\053\000\054\000\014\000\
\015\000\016\000\017\000\018\000\019\000\055\000\056\000\057\000\
\058\000\059\000\000\000\000\000\000\000\000\000\060\000\000\000\
\000\000\000\000\061\000\000\000\000\000\000\000\000\000\000\000\
\029\000\000\000\062\000\000\000\063\000\000\000\000\000\000\000\
\000\000\000\000\000\000\064\000\065\000\066\000\067\000\007\000\
\007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
\007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
\007\000\007\000\007\000\007\000\000\000\000\000\000\000\007\000\
\007\000\000\000\000\000\000\000\007\000\000\000\000\000\000\000\
\000\000\000\000\007\000\000\000\007\000\000\000\007\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\007\000\007\000\
\007\000"

let yycheck = "\002\000\
\000\000\156\000\001\000\002\000\096\000\046\000\116\000\209\000\
\001\001\002\001\001\001\002\001\005\001\001\001\005\001\023\001\
\042\001\043\001\044\001\045\001\046\001\047\001\025\000\178\000\
\027\000\051\001\025\000\030\000\027\000\001\000\002\000\030\000\
\001\001\002\001\126\000\076\000\024\001\030\001\030\001\030\001\
\045\001\046\001\000\000\036\001\030\001\036\001\028\001\029\001\
\203\000\204\000\252\000\000\000\031\001\032\001\031\001\032\001\
\059\000\060\000\035\001\062\000\059\000\060\000\030\001\062\000\
\034\001\042\001\043\001\044\001\045\001\046\001\047\001\031\001\
\032\001\023\001\051\001\033\001\075\000\031\001\032\001\030\001\
\190\000\035\001\031\001\032\001\239\000\030\001\022\001\242\000\
\042\001\043\001\044\001\045\001\046\001\047\001\031\001\032\001\
\031\001\051\001\031\001\032\001\141\000\023\001\000\000\032\001\
\027\001\197\000\109\000\110\000\022\001\037\001\109\000\110\000\
\043\001\044\001\045\001\046\001\047\001\116\000\044\001\045\001\
\046\001\047\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\039\001\
\031\001\148\000\027\001\026\001\039\001\148\000\153\000\030\001\
\031\001\000\000\153\000\031\001\031\001\036\001\032\001\038\001\
\023\001\040\001\011\001\012\001\013\001\014\001\015\001\016\001\
\047\001\048\001\049\001\050\001\175\000\001\001\031\001\031\001\
\175\000\032\001\031\001\031\001\022\001\031\001\015\000\016\000\
\017\000\018\000\019\000\020\000\021\000\022\000\023\000\190\000\
\042\001\043\001\044\001\045\001\046\001\047\001\031\001\031\001\
\041\001\051\001\041\001\022\001\000\000\042\001\043\001\044\001\
\045\001\046\001\047\001\023\001\032\001\031\001\217\000\046\001\
\219\000\220\000\217\000\022\001\219\000\220\000\055\000\056\000\
\057\000\058\000\031\001\032\001\037\001\022\001\027\001\031\001\
\031\001\037\001\022\001\031\001\031\001\042\001\043\001\044\001\
\045\001\046\001\047\001\037\001\031\001\248\000\051\001\031\001\
\031\001\248\000\031\001\038\000\196\000\241\000\157\000\000\000\
\195\000\001\001\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\255\255\
\000\000\025\001\026\001\027\001\028\001\029\001\255\255\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\039\001\
\040\001\255\255\042\001\043\001\044\001\045\001\046\001\047\001\
\048\001\049\001\050\001\051\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\000\000\255\255\025\001\026\001\027\001\028\001\
\029\001\030\001\031\001\032\001\033\001\034\001\035\001\036\001\
\037\001\038\001\039\001\040\001\255\255\042\001\043\001\044\001\
\255\255\255\255\047\001\048\001\049\001\050\001\051\001\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\022\001\000\000\255\255\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\039\001\040\001\255\255\
\042\001\043\001\044\001\255\255\255\255\047\001\048\001\049\001\
\050\001\051\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\000\000\255\255\025\001\026\001\027\001\028\001\029\001\030\001\
\031\001\032\001\033\001\034\001\035\001\036\001\037\001\038\001\
\039\001\040\001\255\255\042\001\043\001\044\001\255\255\255\255\
\047\001\048\001\049\001\050\001\051\001\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\022\001\000\000\255\255\025\001\026\001\027\001\
\028\001\029\001\030\001\255\255\255\255\033\001\034\001\255\255\
\036\001\255\255\038\001\039\001\040\001\255\255\042\001\043\001\
\044\001\045\001\046\001\047\001\048\001\049\001\050\001\051\001\
\001\001\002\001\003\001\004\001\005\001\006\001\007\001\008\001\
\009\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\022\001\000\000\255\255\
\025\001\026\001\027\001\028\001\029\001\030\001\031\001\032\001\
\033\001\034\001\035\001\036\001\037\001\038\001\039\001\040\001\
\255\255\042\001\043\001\255\255\255\255\255\255\022\001\048\001\
\049\001\050\001\051\001\027\001\028\001\029\001\031\001\255\255\
\255\255\255\255\034\001\255\255\255\255\255\255\255\255\039\001\
\255\255\042\001\043\001\044\001\045\001\046\001\047\001\255\255\
\255\255\000\000\051\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\255\255\255\255\025\001\026\001\027\001\028\001\029\001\
\030\001\031\001\032\001\033\001\034\001\035\001\036\001\037\001\
\038\001\039\001\040\001\255\255\042\001\043\001\255\255\255\255\
\255\255\255\255\048\001\049\001\050\001\051\001\001\001\002\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\000\000\255\255\025\001\026\001\
\027\001\028\001\029\001\255\255\255\255\255\255\033\001\034\001\
\255\255\036\001\255\255\038\001\039\001\040\001\255\255\042\001\
\043\001\044\001\045\001\046\001\047\001\048\001\049\001\050\001\
\051\001\001\001\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\000\000\
\255\255\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\039\001\
\040\001\255\255\042\001\255\255\255\255\255\255\255\255\255\255\
\048\001\049\001\050\001\051\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\022\001\000\000\255\255\025\001\026\001\027\001\028\001\
\029\001\030\001\031\001\032\001\033\001\034\001\035\001\036\001\
\037\001\038\001\039\001\040\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\048\001\049\001\050\001\051\001\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\022\001\000\000\255\255\025\001\
\026\001\027\001\028\001\029\001\030\001\255\255\255\255\033\001\
\034\001\255\255\036\001\255\255\038\001\039\001\040\001\255\255\
\042\001\043\001\044\001\255\255\255\255\047\001\048\001\049\001\
\050\001\051\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\000\000\255\255\025\001\026\001\027\001\028\001\029\001\030\001\
\255\255\255\255\033\001\034\001\255\255\036\001\255\255\038\001\
\039\001\040\001\255\255\042\001\043\001\044\001\032\001\255\255\
\047\001\048\001\049\001\050\001\051\001\255\255\255\255\255\255\
\042\001\043\001\044\001\045\001\046\001\047\001\037\001\255\255\
\255\255\051\001\255\255\042\001\043\001\044\001\045\001\046\001\
\047\001\255\255\000\000\255\255\051\001\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\022\001\000\000\255\255\025\001\026\001\027\001\
\028\001\029\001\030\001\255\255\255\255\033\001\034\001\255\255\
\036\001\255\255\038\001\039\001\040\001\255\255\042\001\043\001\
\255\255\255\255\255\255\255\255\048\001\049\001\050\001\051\001\
\001\001\002\001\003\001\004\001\005\001\006\001\007\001\008\001\
\009\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\022\001\000\000\255\255\
\025\001\026\001\027\001\028\001\029\001\030\001\255\255\255\255\
\033\001\034\001\255\255\036\001\255\255\038\001\039\001\040\001\
\255\255\042\001\043\001\255\255\255\255\255\255\255\255\048\001\
\049\001\050\001\051\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\000\000\255\255\025\001\026\001\027\001\028\001\029\001\
\030\001\255\255\255\255\033\001\034\001\255\255\036\001\255\255\
\038\001\039\001\040\001\255\255\042\001\255\255\255\255\255\255\
\255\255\255\255\048\001\049\001\050\001\051\001\001\001\002\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\019\001\020\001\021\001\022\001\000\000\255\255\025\001\026\001\
\027\001\028\001\029\001\030\001\255\255\255\255\033\001\034\001\
\255\255\036\001\255\255\038\001\039\001\040\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\048\001\049\001\050\001\
\051\001\001\001\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\022\001\000\000\
\255\255\025\001\026\001\027\001\028\001\029\001\030\001\031\001\
\032\001\033\001\034\001\035\001\036\001\037\001\038\001\039\001\
\040\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\048\001\049\001\050\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\022\001\000\000\255\255\025\001\026\001\027\001\028\001\029\001\
\030\001\031\001\032\001\033\001\034\001\035\001\036\001\037\001\
\038\001\039\001\040\001\255\255\255\255\255\255\255\255\255\255\
\255\255\022\001\048\001\049\001\050\001\255\255\027\001\028\001\
\029\001\255\255\033\001\255\255\255\255\034\001\255\255\255\255\
\255\255\255\255\039\001\042\001\043\001\044\001\045\001\046\001\
\047\001\255\255\255\255\000\000\051\001\255\255\255\255\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\022\001\255\255\255\255\025\001\
\026\001\027\001\028\001\029\001\030\001\255\255\255\255\033\001\
\034\001\255\255\036\001\255\255\038\001\039\001\040\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\048\001\049\001\
\050\001\255\255\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\022\001\
\255\255\255\255\025\001\026\001\027\001\028\001\029\001\030\001\
\255\255\255\255\033\001\034\001\255\255\036\001\255\255\038\001\
\039\001\040\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\048\001\049\001\050\001\255\255\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\022\001\255\255\255\255\025\001\026\001\027\001\
\028\001\029\001\030\001\255\255\255\255\033\001\034\001\255\255\
\036\001\255\255\038\001\039\001\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\048\001\049\001\050\001\255\255\
\001\001\002\001\003\001\004\001\005\001\006\001\007\001\008\001\
\009\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\019\001\020\001\021\001\022\001\255\255\255\255\
\025\001\026\001\027\001\028\001\029\001\030\001\255\255\255\255\
\255\255\034\001\255\255\036\001\255\255\038\001\039\001\040\001\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\048\001\
\049\001\050\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\019\001\020\001\021\001\255\255\
\255\255\255\255\025\001\026\001\255\255\255\255\255\255\030\001\
\255\255\255\255\255\255\255\255\255\255\036\001\255\255\038\001\
\255\255\040\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\048\001\049\001\050\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\019\001\020\001\
\021\001\255\255\255\255\255\255\025\001\026\001\001\000\002\000\
\255\255\030\001\255\255\255\255\255\255\255\255\255\255\036\001\
\255\255\038\001\255\255\040\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\048\001\049\001\050\001\025\000\037\001\
\027\000\006\000\255\255\030\000\042\001\043\001\044\001\045\001\
\046\001\047\001\255\255\255\255\255\255\051\001\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\028\000\
\029\000\255\255\255\255\032\000\033\000\034\000\255\255\255\255\
\059\000\060\000\255\255\062\000\255\255\255\255\255\255\044\000\
\255\255\255\255\047\000\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\061\000\255\255\255\255\064\000\065\000\066\000\067\000\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\109\000\110\000\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\102\000\103\000\104\000\105\000\106\000\107\000\108\000\
\255\255\255\255\255\255\255\255\113\000\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\148\000\255\255\255\255\255\255\255\255\153\000\255\255\
\133\000\134\000\135\000\136\000\137\000\138\000\139\000\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\175\000\255\255\155\000\255\255\
\157\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\177\000\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\193\000\255\255\217\000\255\255\
\219\000\220\000\255\255\255\255\255\255\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\255\255\255\255\255\255\248\000\026\001\255\255\
\255\255\230\000\030\001\031\001\255\255\234\000\255\255\255\255\
\036\001\255\255\038\001\255\255\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\047\001\048\001\049\001\050\001\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\255\255\255\255\255\255\255\255\
\026\001\255\255\255\255\255\255\030\001\255\255\255\255\255\255\
\255\255\255\255\036\001\255\255\038\001\255\255\040\001\255\255\
\255\255\255\255\255\255\255\255\255\255\047\001\048\001\049\001\
\050\001\001\001\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\013\001\014\001\015\001\
\016\001\017\001\018\001\019\001\020\001\021\001\255\255\255\255\
\255\255\255\255\026\001\255\255\255\255\255\255\030\001\255\255\
\255\255\255\255\255\255\255\255\036\001\255\255\038\001\255\255\
\040\001\255\255\255\255\255\255\255\255\255\255\255\255\047\001\
\048\001\049\001\050\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\019\001\020\001\021\001\
\255\255\255\255\255\255\025\001\026\001\255\255\255\255\255\255\
\030\001\255\255\255\255\255\255\255\255\255\255\036\001\255\255\
\038\001\255\255\040\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\048\001\049\001\050\001\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\019\001\
\020\001\021\001\255\255\255\255\255\255\255\255\026\001\255\255\
\255\255\255\255\030\001\255\255\255\255\255\255\255\255\255\255\
\036\001\255\255\038\001\255\255\040\001\255\255\255\255\255\255\
\255\255\255\255\255\255\047\001\048\001\049\001\050\001\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\019\001\020\001\021\001\255\255\255\255\255\255\025\001\
\026\001\255\255\255\255\255\255\030\001\255\255\255\255\255\255\
\255\255\255\255\036\001\255\255\038\001\255\255\040\001\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\048\001\049\001\
\050\001"

let yynames_const = "\
  OGG\000\
  VORBIS\000\
  VORBIS_CBR\000\
  VORBIS_ABR\000\
  THEORA\000\
  DIRAC\000\
  SPEEX\000\
  WAV\000\
  AACPLUS\000\
  MP3\000\
  EXTERNAL\000\
  EOF\000\
  BEGIN\000\
  END\000\
  GETS\000\
  TILD\000\
  IF\000\
  THEN\000\
  ELSE\000\
  ELSIF\000\
  LPAR\000\
  RPAR\000\
  COMMA\000\
  SEQ\000\
  SEQSEQ\000\
  COLON\000\
  LBRA\000\
  RBRA\000\
  LCUR\000\
  RCUR\000\
  FUN\000\
  YIELDS\000\
  TIMES\000\
  MINUS\000\
  NOT\000\
  REF\000\
  GET\000\
  SET\000\
  PP_IFDEF\000\
  PP_ENDIF\000\
  PP_ENDL\000\
  PP_DEF\000\
  "

let yynames_block = "\
  VAR\000\
  VARLPAR\000\
  VARLBRA\000\
  STRING\000\
  INT\000\
  FLOAT\000\
  BOOL\000\
  TIME\000\
  INTERVAL\000\
  DEF\000\
  BIN0\000\
  BIN1\000\
  BIN2\000\
  BIN3\000\
  PP_INCLUDE\000\
  PP_COMMENT\000\
  "

let yyact = [|
  (fun _ -> failwith "parser")
; (fun __caml_parser_env ->
    Obj.repr(
# 184 "lang/lang_parser.mly"
        ( mk Unit )
# 969 "lang/lang_parser.ml"
               : Lang_values.term))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 185 "lang/lang_parser.mly"
              ( _1 )
# 976 "lang/lang_parser.ml"
               : Lang_values.term))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 187 "lang/lang_parser.mly"
                 ( _1 )
# 983 "lang/lang_parser.ml"
               : Lang_values.term))
; (fun __caml_parser_env ->
    Obj.repr(
# 188 "lang/lang_parser.mly"
        ( raise End_of_file )
# 989 "lang/lang_parser.ml"
               : Lang_values.term))
; (fun __caml_parser_env ->
    Obj.repr(
# 190 "lang/lang_parser.mly"
     ()
# 995 "lang/lang_parser.ml"
               : 's))
; (fun __caml_parser_env ->
    Obj.repr(
# 190 "lang/lang_parser.mly"
               ()
# 1001 "lang/lang_parser.ml"
               : 's))
; (fun __caml_parser_env ->
    Obj.repr(
# 191 "lang/lang_parser.mly"
     ()
# 1007 "lang/lang_parser.ml"
               : 'g))
; (fun __caml_parser_env ->
    Obj.repr(
# 191 "lang/lang_parser.mly"
               ()
# 1013 "lang/lang_parser.ml"
               : 'g))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 's) in
    Obj.repr(
# 196 "lang/lang_parser.mly"
                             ( _1 )
# 1021 "lang/lang_parser.ml"
               : 'exprs))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 's) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'exprs) in
    Obj.repr(
# 197 "lang/lang_parser.mly"
                             ( mk (Seq (_1,_3)) )
# 1030 "lang/lang_parser.ml"
               : 'exprs))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'binding) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 's) in
    Obj.repr(
# 198 "lang/lang_parser.mly"
                             ( let doc,name,def = _1 in
                                 mk (Let { doc=doc ; var=name ;
                                           gen = [] ; def=def ;
                                           body = mk Unit }) )
# 1041 "lang/lang_parser.ml"
               : 'exprs))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'binding) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 's) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'exprs) in
    Obj.repr(
# 202 "lang/lang_parser.mly"
                             ( let doc,name,def = _1 in
                                 mk (Let { doc=doc ; var=name ;
                                           gen = [] ; def=def ;
                                           body = _3 }) )
# 1053 "lang/lang_parser.ml"
               : 'exprs))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'expr) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'ty) in
    Obj.repr(
# 212 "lang/lang_parser.mly"
                                     ( Lang_types.(<:) _2.Lang_values.t _4 ;
                                       _2 )
# 1062 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : float) in
    Obj.repr(
# 217 "lang/lang_parser.mly"
                                     ( mk (Float (-. _2)) )
# 1069 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : int) in
    Obj.repr(
# 218 "lang/lang_parser.mly"
                                     ( mk (Int (- _2)) )
# 1076 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 219 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var "~-"),
                                                ["", _2])) )
# 1084 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 221 "lang/lang_parser.mly"
                                     ( _2 )
# 1091 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int) in
    Obj.repr(
# 222 "lang/lang_parser.mly"
                                     ( mk (Int _1) )
# 1098 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 223 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var "not"),
                                                ["", _2])) )
# 1106 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : bool) in
    Obj.repr(
# 225 "lang/lang_parser.mly"
                                     ( mk (Bool _1) )
# 1113 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : float) in
    Obj.repr(
# 226 "lang/lang_parser.mly"
                                     ( mk (Float  _1) )
# 1120 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 227 "lang/lang_parser.mly"
                                     ( mk (String _1) )
# 1127 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'list) in
    Obj.repr(
# 228 "lang/lang_parser.mly"
                                     ( mk (List _1) )
# 1134 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 229 "lang/lang_parser.mly"
                                     ( mk (Ref _2) )
# 1141 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 230 "lang/lang_parser.mly"
                                     ( mk (Get _2) )
# 1148 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 231 "lang/lang_parser.mly"
                                     ( mk (Set (_1,_3)) )
# 1156 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 232 "lang/lang_parser.mly"
                                     ( mk_mp3 _2 )
# 1163 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 233 "lang/lang_parser.mly"
                                     ( mk_aacplus _2 )
# 1170 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 234 "lang/lang_parser.mly"
                                     ( mk_external _2 )
# 1177 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 235 "lang/lang_parser.mly"
                                     ( mk_wav _2 )
# 1184 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'ogg_items) in
    Obj.repr(
# 236 "lang/lang_parser.mly"
                                     ( mk (Encoder (Encoder.Ogg _3)) )
# 1191 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'ogg_item) in
    Obj.repr(
# 237 "lang/lang_parser.mly"
                                     ( mk (Encoder (Encoder.Ogg [_1])) )
# 1198 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    Obj.repr(
# 238 "lang/lang_parser.mly"
                                     ( mk Unit )
# 1204 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'expr) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 239 "lang/lang_parser.mly"
                                     ( mk (Product (_2,_4)) )
# 1212 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 240 "lang/lang_parser.mly"
                                     ( mk (Var _1) )
# 1219 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'app_list) in
    Obj.repr(
# 241 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var _1),_2)) )
# 1227 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 242 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var "_[_]"),
                                           ["",_2;"",mk ~pos:(1,1) (Var _1)])) )
# 1236 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 244 "lang/lang_parser.mly"
                                     ( _2 )
# 1243 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'arglist) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 246 "lang/lang_parser.mly"
                                     ( mk_fun _3 _6 )
# 1251 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 247 "lang/lang_parser.mly"
                                     ( mk_fun [] _2 )
# 1258 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'exprs) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : 'exprs) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'if_elsif) in
    Obj.repr(
# 249 "lang/lang_parser.mly"
                                     ( let cond = _2 in
                                       let then_b =
                                         mk_fun ~pos:(3,4) [] _4
                                       in
                                       let else_b = _5 in
                                       let op = mk ~pos:(1,1) (Var "if") in
                                         mk (App (op,["",cond;
                                                      "else",else_b;
                                                      "then",then_b])) )
# 1275 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 258 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1285 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 260 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1295 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 262 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1305 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 264 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1315 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 266 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var "*"),
                                                ["",_1;"",_3])) )
# 1324 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 268 "lang/lang_parser.mly"
                                   ( mk (App (mk ~pos:(2,2) (Var "-"),
                                                ["",_1;"",_3])) )
# 1333 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int option list * int option list) in
    Obj.repr(
# 270 "lang/lang_parser.mly"
                                   ( mk_time_pred (between (fst _1) (snd _1)) )
# 1340 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int option list) in
    Obj.repr(
# 271 "lang/lang_parser.mly"
                                   ( mk_time_pred (during _1) )
# 1347 "lang/lang_parser.ml"
               : 'expr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 274 "lang/lang_parser.mly"
                              ( mk_ty _1 [] )
# 1354 "lang/lang_parser.ml"
               : 'ty))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'ty_args) in
    Obj.repr(
# 275 "lang/lang_parser.mly"
                              ( mk_ty _1 _2 )
# 1362 "lang/lang_parser.ml"
               : 'ty))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'ty) in
    Obj.repr(
# 276 "lang/lang_parser.mly"
                              ( Lang_types.make (Lang_types.List _2) )
# 1369 "lang/lang_parser.ml"
               : 'ty))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'ty) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'ty) in
    Obj.repr(
# 277 "lang/lang_parser.mly"
                              ( Lang_types.make (Lang_types.Product (_2,_4)) )
# 1377 "lang/lang_parser.ml"
               : 'ty))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int) in
    Obj.repr(
# 278 "lang/lang_parser.mly"
                              ( Lang_values.type_of_int _1 )
# 1384 "lang/lang_parser.ml"
               : 'ty))
; (fun __caml_parser_env ->
    Obj.repr(
# 281 "lang/lang_parser.mly"
                         ( [] )
# 1390 "lang/lang_parser.ml"
               : 'ty_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'ty_arg) in
    Obj.repr(
# 282 "lang/lang_parser.mly"
                         ( [_1] )
# 1397 "lang/lang_parser.ml"
               : 'ty_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'ty_arg) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'ty_args) in
    Obj.repr(
# 283 "lang/lang_parser.mly"
                         ( _1::_3 )
# 1405 "lang/lang_parser.ml"
               : 'ty_args))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'ty) in
    Obj.repr(
# 286 "lang/lang_parser.mly"
       ( "",_1 )
# 1412 "lang/lang_parser.ml"
               : 'ty_arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'ty) in
    Obj.repr(
# 287 "lang/lang_parser.mly"
                ( _1,_3 )
# 1420 "lang/lang_parser.ml"
               : 'ty_arg))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 292 "lang/lang_parser.mly"
                                     ( _2 )
# 1427 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'expr) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'ty) in
    Obj.repr(
# 293 "lang/lang_parser.mly"
                                     ( Lang_types.(<:) _2.Lang_values.t _4 ;
                                       _2 )
# 1436 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int) in
    Obj.repr(
# 295 "lang/lang_parser.mly"
                                     ( mk (Int _1) )
# 1443 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 296 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var "not"),
                                                ["", _2])) )
# 1451 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : bool) in
    Obj.repr(
# 298 "lang/lang_parser.mly"
                                     ( mk (Bool _1) )
# 1458 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : float) in
    Obj.repr(
# 299 "lang/lang_parser.mly"
                                     ( mk (Float  _1) )
# 1465 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 300 "lang/lang_parser.mly"
                                     ( mk (String _1) )
# 1472 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'list) in
    Obj.repr(
# 301 "lang/lang_parser.mly"
                                     ( mk (List _1) )
# 1479 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 302 "lang/lang_parser.mly"
                                     ( mk (Ref _2) )
# 1486 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 303 "lang/lang_parser.mly"
                                     ( mk (Get _2) )
# 1493 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 304 "lang/lang_parser.mly"
                                     ( mk (Set (_1,_3)) )
# 1501 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 305 "lang/lang_parser.mly"
                                     ( mk_mp3 _2 )
# 1508 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 306 "lang/lang_parser.mly"
                                         ( mk_aacplus _2 )
# 1515 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 307 "lang/lang_parser.mly"
                                     ( mk_external _2 )
# 1522 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 308 "lang/lang_parser.mly"
                                     ( mk_wav _2 )
# 1529 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'ogg_items) in
    Obj.repr(
# 309 "lang/lang_parser.mly"
                                     ( mk (Encoder (Encoder.Ogg _3)) )
# 1536 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'ogg_item) in
    Obj.repr(
# 310 "lang/lang_parser.mly"
                                     ( mk (Encoder (Encoder.Ogg [_1])) )
# 1543 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    Obj.repr(
# 311 "lang/lang_parser.mly"
                                     ( mk Unit )
# 1549 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'expr) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 312 "lang/lang_parser.mly"
                                     ( mk (Product (_2,_4)) )
# 1557 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 313 "lang/lang_parser.mly"
                                     ( mk (Var _1) )
# 1564 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'app_list) in
    Obj.repr(
# 314 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var _1),_2)) )
# 1572 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'expr) in
    Obj.repr(
# 315 "lang/lang_parser.mly"
                                     ( mk (App (mk ~pos:(1,1) (Var "_[_]"),
                                           ["",_2;"",mk ~pos:(1,1) (Var _1)])) )
# 1581 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 317 "lang/lang_parser.mly"
                                     ( _2 )
# 1588 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _3 = (Parsing.peek_val __caml_parser_env 3 : 'arglist) in
    let _6 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 319 "lang/lang_parser.mly"
                                     ( mk_fun _3 _6 )
# 1596 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 320 "lang/lang_parser.mly"
                                     ( mk_fun [] _2 )
# 1603 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 4 : 'exprs) in
    let _4 = (Parsing.peek_val __caml_parser_env 2 : 'exprs) in
    let _5 = (Parsing.peek_val __caml_parser_env 1 : 'if_elsif) in
    Obj.repr(
# 322 "lang/lang_parser.mly"
                                     ( let cond = _2 in
                                       let then_b =
                                         mk_fun ~pos:(3,4) [] _4
                                       in
                                       let else_b = _5 in
                                       let op = mk ~pos:(1,1) (Var "if") in
                                         mk (App (op,["",cond;
                                                      "else",else_b;
                                                      "then",then_b])) )
# 1620 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 331 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1630 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 333 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1640 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 335 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1650 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 337 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var _2),
                                                ["",_1;"",_3])) )
# 1660 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 339 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var "*"),
                                                ["",_1;"",_3])) )
# 1669 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'cexpr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 341 "lang/lang_parser.mly"
                                    ( mk (App (mk ~pos:(2,2) (Var "-"),
                                                ["",_1;"",_3])) )
# 1678 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int option list * int option list) in
    Obj.repr(
# 343 "lang/lang_parser.mly"
                                   ( mk_time_pred (between (fst _1) (snd _1)) )
# 1685 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : int option list) in
    Obj.repr(
# 344 "lang/lang_parser.mly"
                                   ( mk_time_pred (during _1) )
# 1692 "lang/lang_parser.ml"
               : 'cexpr))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'inner_list) in
    Obj.repr(
# 347 "lang/lang_parser.mly"
                         ( _2 )
# 1699 "lang/lang_parser.ml"
               : 'list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'expr) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'inner_list) in
    Obj.repr(
# 349 "lang/lang_parser.mly"
                           ( _1::_3 )
# 1707 "lang/lang_parser.ml"
               : 'inner_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 350 "lang/lang_parser.mly"
                           ( [_1] )
# 1714 "lang/lang_parser.ml"
               : 'inner_list))
; (fun __caml_parser_env ->
    Obj.repr(
# 351 "lang/lang_parser.mly"
                           ( [] )
# 1720 "lang/lang_parser.ml"
               : 'inner_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 354 "lang/lang_parser.mly"
                  ( _1,_3 )
# 1728 "lang/lang_parser.ml"
               : 'app_list_elem))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 355 "lang/lang_parser.mly"
                  ( "",_1 )
# 1735 "lang/lang_parser.ml"
               : 'app_list_elem))
; (fun __caml_parser_env ->
    Obj.repr(
# 359 "lang/lang_parser.mly"
                                 ( [] )
# 1741 "lang/lang_parser.ml"
               : 'app_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'app_list_elem) in
    Obj.repr(
# 360 "lang/lang_parser.mly"
                                 ( [_1] )
# 1748 "lang/lang_parser.ml"
               : 'app_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'app_list_elem) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'app_list) in
    Obj.repr(
# 361 "lang/lang_parser.mly"
                                 ( _1::_3 )
# 1756 "lang/lang_parser.ml"
               : 'app_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 364 "lang/lang_parser.mly"
                  (
       let body = _3 in
         (Doc.none (),[]),_1,body
    )
# 1767 "lang/lang_parser.ml"
               : 'binding))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : Doc.item * (string*string) list) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'g) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 368 "lang/lang_parser.mly"
                        (
      let body = _4 in
        _1,_2,body
    )
# 1780 "lang/lang_parser.ml"
               : 'binding))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 6 : Doc.item * (string*string) list) in
    let _2 = (Parsing.peek_val __caml_parser_env 5 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 4 : 'arglist) in
    let _5 = (Parsing.peek_val __caml_parser_env 2 : 'g) in
    let _6 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    Obj.repr(
# 372 "lang/lang_parser.mly"
                                         (
      let arglist = _3 in
      let body = mk_fun arglist _6 in
        _1,_2,body
    )
# 1795 "lang/lang_parser.ml"
               : 'binding))
; (fun __caml_parser_env ->
    Obj.repr(
# 379 "lang/lang_parser.mly"
                      ( [] )
# 1801 "lang/lang_parser.ml"
               : 'arglist))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'arg) in
    Obj.repr(
# 380 "lang/lang_parser.mly"
                      ( [_1] )
# 1808 "lang/lang_parser.ml"
               : 'arglist))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'arg) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'arglist) in
    Obj.repr(
# 381 "lang/lang_parser.mly"
                      ( _1::_3 )
# 1816 "lang/lang_parser.ml"
               : 'arglist))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'opt) in
    Obj.repr(
# 383 "lang/lang_parser.mly"
                 ( _2,_2,
                   T.fresh_evar ~level:(-1) ~pos:(Some (curpos ~pos:(2,2) ())),
                   _3 )
# 1826 "lang/lang_parser.ml"
               : 'arg))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'opt) in
    Obj.repr(
# 386 "lang/lang_parser.mly"
                 ( "",_1,
                   T.fresh_evar ~level:(-1) ~pos:(Some (curpos ~pos:(1,1) ())),
                   _2 )
# 1836 "lang/lang_parser.ml"
               : 'arg))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'expr) in
    Obj.repr(
# 390 "lang/lang_parser.mly"
              ( Some _2 )
# 1843 "lang/lang_parser.ml"
               : 'opt))
; (fun __caml_parser_env ->
    Obj.repr(
# 391 "lang/lang_parser.mly"
              ( None )
# 1849 "lang/lang_parser.ml"
               : 'opt))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'exprs) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'exprs) in
    let _5 = (Parsing.peek_val __caml_parser_env 0 : 'if_elsif) in
    Obj.repr(
# 394 "lang/lang_parser.mly"
                                    ( let cond = _2 in
                                      let then_b =
                                        mk_fun ~pos:(3,4) [] _4
                                      in
                                      let else_b = _5 in
                                      let op = mk ~pos:(1,1) (Var "if") in
                                        mk_fun []
                                          (mk (App (op,["",cond;
                                                        "else",else_b;
                                                        "then",then_b]))) )
# 1867 "lang/lang_parser.ml"
               : 'if_elsif))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'exprs) in
    Obj.repr(
# 404 "lang/lang_parser.mly"
                                    ( mk_fun ~pos:(1,2) [] _2 )
# 1874 "lang/lang_parser.ml"
               : 'if_elsif))
; (fun __caml_parser_env ->
    Obj.repr(
# 405 "lang/lang_parser.mly"
                                    ( mk_fun [] (mk Unit) )
# 1880 "lang/lang_parser.ml"
               : 'if_elsif))
; (fun __caml_parser_env ->
    Obj.repr(
# 408 "lang/lang_parser.mly"
                 ( [] )
# 1886 "lang/lang_parser.ml"
               : 'app_opt))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'app_list) in
    Obj.repr(
# 409 "lang/lang_parser.mly"
                       ( _2 )
# 1893 "lang/lang_parser.ml"
               : 'app_opt))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'ogg_item) in
    Obj.repr(
# 412 "lang/lang_parser.mly"
             ( [_1] )
# 1900 "lang/lang_parser.ml"
               : 'ogg_items))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'ogg_item) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'ogg_items) in
    Obj.repr(
# 413 "lang/lang_parser.mly"
                             ( _1::_3 )
# 1908 "lang/lang_parser.ml"
               : 'ogg_items))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 415 "lang/lang_parser.mly"
                       ( mk_vorbis _2 )
# 1915 "lang/lang_parser.ml"
               : 'ogg_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 416 "lang/lang_parser.mly"
                       ( mk_vorbis_cbr _2 )
# 1922 "lang/lang_parser.ml"
               : 'ogg_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 417 "lang/lang_parser.mly"
                       ( mk_vorbis_abr _2 )
# 1929 "lang/lang_parser.ml"
               : 'ogg_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 418 "lang/lang_parser.mly"
                       ( mk_theora _2 )
# 1936 "lang/lang_parser.ml"
               : 'ogg_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 419 "lang/lang_parser.mly"
                       ( mk_dirac _2 )
# 1943 "lang/lang_parser.ml"
               : 'ogg_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'app_opt) in
    Obj.repr(
# 420 "lang/lang_parser.mly"
                       ( mk_speex _2 )
# 1950 "lang/lang_parser.ml"
               : 'ogg_item))
(* Entry program *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
(* Entry interactive *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
|]
let yytables =
  { Parsing.actions=yyact;
    Parsing.transl_const=yytransl_const;
    Parsing.transl_block=yytransl_block;
    Parsing.lhs=yylhs;
    Parsing.len=yylen;
    Parsing.defred=yydefred;
    Parsing.dgoto=yydgoto;
    Parsing.sindex=yysindex;
    Parsing.rindex=yyrindex;
    Parsing.gindex=yygindex;
    Parsing.tablesize=yytablesize;
    Parsing.table=yytable;
    Parsing.check=yycheck;
    Parsing.error_function=parse_error;
    Parsing.names_const=yynames_const;
    Parsing.names_block=yynames_block }
let program (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 1 lexfun lexbuf : Lang_values.term)
let interactive (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 2 lexfun lexbuf : Lang_values.term)
