(*****************************************************************************

  Liquidsoap, a programmable audio stream generator.
  Copyright 2003-2010 Savonet team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details, fully stated in the COPYING
  file at the root of the liquidsoap distribution.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *****************************************************************************)

let check = Theora.Decoder.check

let decoder os =
  let meta    = ref None in
  let fps     = ref None in
  let decoder = Theora.Decoder.create () in
  let fill feed =
    let fps =
      match !fps with
        | None ->
           let p = Ogg.Stream.get_packet os in
           begin
            try
             let (info,vendor,m) = Theora.Decoder.headerin decoder p in
             let nfps = (float (info.Theora.fps_numerator)) /.
                       (float (info.Theora.fps_denominator))
             in
             meta := Some (vendor,m);
             fps := Some nfps ;
             nfps
            with
              | Theora.Not_enough_data -> raise Ogg.Not_enough_data
           end
        | Some fps -> fps
    in
    let ret = Theora.Decoder.get_yuv decoder os in
    let ret =
    {
      Ogg_demuxer.
        y_width   = ret.Theora.y_width;
        y_height  = ret.Theora.y_height;
        y_stride  = ret.Theora.y_stride;
        (** This is an approximation when format
          * is YUV420 *)
        uv_width  = ret.Theora.u_width;
        uv_height = ret.Theora.u_height;
        uv_stride = ret.Theora.u_stride;
        fps       = fps;
        y = ret.Theora.y;
        u = ret.Theora.u;
        v = ret.Theora.v
    }
    in
    let m = ! meta in
    meta := None;
    feed (ret,m)
  in
  Ogg_demuxer.Video fill

let () = Ogg_demuxer.ogg_decoders#register "theora" (check,decoder)

let create_encoder ~quality ~metadata () =
  let frame_x = Fmt.video_width () in
  let frame_y = Fmt.video_height () in
  let video_r = 0 in
  (* TODO: variable FPS *)
  let fps = Fmt.video_frames_of_seconds 1. in
  let version_major,version_minor,version_subminor = Theora.version_number in
  let info =
    {
     Theora.
      frame_width = frame_x;
      frame_height = frame_y;
      picture_width = frame_x;
      picture_height = frame_y;
      picture_x = 0;
      picture_y = 0;
      fps_numerator = fps;
      fps_denominator = 1;
      aspect_numerator = 1;
      aspect_denominator = 1;
      keyframe_granule_shift = Theora.default_granule_shift;
      colorspace = Theora.CS_unspecified;
      target_bitrate = video_r;
      quality = quality;
      version_major = version_major;
      version_minor = version_minor;
      version_subminor = version_subminor;
      pixel_fmt = Theora.PF_420
    }
  in
  let enc = Theora.Encoder.create info metadata in
  let started = ref false in
  let header_encoder os = 
    Theora.Encoder.encode_header enc os;
    Ogg.Stream.flush_page os
  in
  let fisbone_packet os = 
    let serialno = Ogg.Stream.serialno os in
    Some (Theora.Skeleton.fisbone ~serialno ~info ())
  in
  let stream_start os = 
    Ogg_encoder.flush_pages os
  in
  let ((y,y_stride), (u, v, uv_stride) as yuv) =
    RGB.create_yuv (Fmt.video_width ()) (Fmt.video_height ())
  in
  let theora_yuv =
  {
    Theora.y_width = Fmt.video_width ();
    Theora.y_height = Fmt.video_height ();
    Theora.y_stride = y_stride;
    Theora.u_width = Fmt.video_width () / 2;
    Theora.u_height = Fmt.video_height () / 2;
    Theora.u_stride = uv_stride;
    Theora.v_width = Fmt.video_width () / 2;
    Theora.v_height = Fmt.video_height () / 2;
    Theora.v_stride = uv_stride;
    Theora.y = y;
    Theora.u = u;
    Theora.v = v;
  }
  in
  let convert =
    Video_converter.find_converter
      (Video_converter.RGB Video_converter.Rgba_32)
      (Video_converter.YUV Video_converter.Yuvj_420)
  in
  let data_encoder data os _ = 
    if not !started then
      started := true;
    let b,ofs,len = data.Ogg_encoder.data,data.Ogg_encoder.offset,
                    data.Ogg_encoder.length 
    in
    for i = ofs to ofs+len-1 do
      let frame = Video_converter.frame_of_internal_rgb b.(0).(i) in
      convert
        frame (* TODO: multiple channels.. *)
        (Video_converter.frame_of_internal_yuv
        (Fmt.video_width ())
        (Fmt.video_height ())
             yuv); (* TODO: custom video size.. *);
      Theora.Encoder.encode_buffer enc os theora_yuv 
    done
  in
  let end_of_page p = 
    let granulepos = Ogg.Page.granulepos p in
    if granulepos < Int64.zero then
      Ogg_encoder.Unknown
    else
      if granulepos <> Int64.zero then
       let index = 
         Int64.succ (Theora.Encoder.frames_of_granulepos enc granulepos)
       in
       Ogg_encoder.Time (Int64.to_float index /. (float fps)) 
      else
       Ogg_encoder.Time 0.
  in
  let end_of_stream os =
    (* Encode at least some data.. *)
    if not !started then
     begin
      RGB.blank_yuv yuv;
      Theora.Encoder.encode_buffer enc os theora_yuv
     end;
    Theora.Encoder.eos enc os
  in
  {
   Ogg_encoder.
    header_encoder = header_encoder;
    fisbone_packet = fisbone_packet;
    stream_start   = stream_start;
    data_encoder   = (Ogg_encoder.Video_encoder data_encoder);
    end_of_page    = end_of_page;
    end_of_stream  = end_of_stream
  }
