/****************************************************************
 *
 * vdictov: swap.H
 *
 * Copyright (C) Max Planck Institute 
 * for Human Cognitive and Brain Sciences, Leipzig
 *
 * Author Thomas Arnold, 2002, <lipsia@cbs.mpg.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * $Id: swap.H 687 2004-02-12 09:29:58Z karstenm $
 *
 *****************************************************************/


#ifndef SWAP_H_INCLUDED
#define SWAP_H_INCLUDED

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <time.h>
#include <assert.h>

extern "C"
{
   #include <Vlib.h>
   #include <VImage.h>
   #include <option.h>
   #include <mu.h>
}


/*------------------------------------------------------------------------------

Swap
====

Src    source image with voxels of type T
Dir    direction of swap (XYSwap = 0, XZSwap = 1 and YZSwap = 2)
Dest   swapped image with voxels of type T (created)

------------------------------------------------------------------------------*/

const int XYSwap = 0;
const int XZSwap = 1;
const int YZSwap = 2;

template <class T> void Swap (VImage Src, int Dir, VImage& Dest)
{
   int Bands;     /* number of bands   */
   int Rows;      /* number of rows    */
   int Columns;   /* number of columns */

   T* src;    /* source data pointer      */
   T* dest;   /* destination data pointer */

   int dx = 0, dy = 0, dz = 0;   /* strides */

   int x, y, z;   /* indices */


   /* get source image size */
   Bands   = VImageNBands   (Src);
   Rows    = VImageNRows    (Src);
   Columns = VImageNColumns (Src);

   /* create swapped image */
   Dest = NULL;
   if (Dir == XYSwap) Dest = VCreateImage (Bands,   Columns, Rows,    VPixelRepn (Src));
   if (Dir == XZSwap) Dest = VCreateImage (Columns, Rows,    Bands,   VPixelRepn (Src));
   if (Dir == YZSwap) Dest = VCreateImage (Rows,    Bands,   Columns, VPixelRepn (Src));
   VImageAttrList (Dest) = VCopyAttrList (VImageAttrList (Src));


   /* set swap parameters */
   if (Dir == XYSwap) {dx = Columns;        dy = 1;              dz = Columns * Rows;}
   if (Dir == XZSwap) {dx = Columns * Rows; dy = Columns;        dz = 1;             }
   if (Dir == YZSwap) {dx = 1;              dy = Columns * Rows; dz = Columns;       }

   /* swap image */
   src  = (T*) VPixelPtr (Src,  0, 0, 0);
   dest = (T*) VPixelPtr (Dest, 0, 0, 0);
   for (z = 0; z < Bands; ++z)
   {
      for (y = 0; y < Rows; ++y)
      {
         for (x = 0; x < Columns; ++x)
         {
            *dest = *(src++);
            dest += dx;
         }
         dest -= Columns * dx;
         dest += dy;
      }
      dest -= Rows * dy;
      dest += dz;
   }

} /* Swap */

template <class T> void Swap (VImage& Src, int Dir)
{
   VImage Dest;   /* destination image */


   /* swap image */
   Swap<T> (Src, Dir, Dest);
   VDestroyImage (Src);
   Src = Dest;

} /* Swap */

/*----------------------------------------------------------------------------*/

#endif
