/****************************************************************
 *
 * dictov:
 *
 * Copyright (C) Max Planck Institute
 * for Human Cognitive and Brain Sciences, Leipzig
 *
 * Author Thomas Arnold, 2002, <lipsia@cbs.mpg.de>
 * Author Alfred Anwander, 2003-2004, <lipsia@cbs.mpg.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: dictov.C 3501 2009-01-21 12:50:58Z karstenm $
 *
 *****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <time.h>
#include <assert.h>
#include <list>
#include <vector>
#include <gsl/gsl_vector_float.h>
#include <gsl/gsl_sort_vector_float.h>
#include <gsl/gsl_permutation.h>

extern "C"
{
   #include <Vlib.h>
   #include <VImage.h>
   #include <option.h>
   #include <mu.h>
}

/* these two header files should be the first to be included
 * especially on 64Bit platforms. */
#include <config/cfunix.h>
#include <ofstd/oftypes.h>
#include <dcmdata/dcuid.h>

#if OFFIS_DCMTK_VERSION_NUMBER > 351

#include <dcmdata/dcistrmf.h>
#endif

#if OFFIS_DCMTK_VERSION_NUMBER > 352
#define DCM_PhaseEncodingDirection DCM_InPlanePhaseEncodingDirection
#endif

#include <dcmdata/dcdeftag.h>
#include <dcmdata/dcfilefo.h>
#include <dcmdata/dcdicdir.h>

/* #include "nr.h" */

#include "axis.H"
#include "flip.H"
#include "swap.H"
#include "scale.H"
#include "contrast.H"
#include "convert.H"
#include "reorder.H"

extern "C"
{
  void VNormVals(VAttrList,VLong);
  extern char * getLipsiaVersion();
}

char prg_name[50];

/*------------------------------------------------------------------------------

This program implements a DICOM to VISTA converter for data of type "Basic
Directory Storage SOP Class" (UID 1.2.840.10008.1.3.10).

It especially supports the Siemens Mosaic format!

Note: The program makes use of libraries from DCMTK (Dicom Toolkit).
      For further information see:

      http://dicom.offis.de

      The program assumes that the dcmdata library includes a build-in data
      dictionary. Otherwise the program may show strange behaviour.
      Please RTFM "dcmdata/doc/datadict.txt" for details.

      The DICOM standard can be downloaded at:

      http://medical.nema.org

------------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------

The following functions are from the file dcmtk-3.5.3/dcmdata/libsrc/dcitem.cc
which do not link form the lib libdcmdata.a

EDIT: on Mac this don't hold. So we can skip the declaration.

------------------------------------------------------------------------------*/

#ifndef __APPLE__
OFCondition DcmItem::findAndGetElement(const DcmTagKey &tagKey,
                                       DcmElement *&element,
                                       const OFBool searchIntoSub)
{
    DcmStack stack;
    /* find the element */
    OFCondition status = search(tagKey, stack, ESM_fromHere, searchIntoSub);
    if (status.good())
    {
        element = OFstatic_cast(DcmElement *, stack.top());
        /* should never happen but ... */
        if (element == NULL)
            status = EC_CorruptedData;
    } else {
        /* reset element pointer */
        element = NULL;
    }
    return status;
}

OFCondition DcmItem::findAndGetString(const DcmTagKey& tagKey,
                                      const char *&value,
                                      const OFBool searchIntoSub)
{
    DcmElement *elem;
    /* find the element */
    OFCondition status = findAndGetElement(tagKey, elem, searchIntoSub);
    if (status.good())
    {
        /* get the value */
        status = elem->getString(OFconst_cast(char *&, value));
    }
    /* reset value */
    if (status.bad())
        value = NULL;
    return status;
}

OFCondition DcmItem::findAndGetUint16(const DcmTagKey& tagKey,
                                      Uint16 &value,
                                      const unsigned long pos,
                                      const OFBool searchIntoSub)
{
    DcmElement *elem;
    /* find the element */
    OFCondition status = findAndGetElement(tagKey, elem, searchIntoSub);
    if (status.good())
    {
        /* get the value */
        status = elem->getUint16(value, pos);
    }
    /* reset value */
    if (status.bad())
        value = 0;
    return status;
}

OFCondition DcmItem::findAndGetUint16Array(const DcmTagKey& tagKey,
                                           const Uint16 *&value,
                                           unsigned long *count,
                                           const OFBool searchIntoSub)
{
    DcmElement *elem;
    /* find the element */
    OFCondition status = findAndGetElement(tagKey, elem, searchIntoSub);
    if (status.good())
    {
        /* get the value */
        Uint16 *array = NULL;
        status = elem->getUint16Array(array);
        value = array;
    }
    /* set optional count parameter */
    if (count != NULL)
    {
        if (status.good())
            *count = elem->getLength() / sizeof(Uint16);
        else
            *count = 0;
    }
    /* reset value */
    if (status.bad())
        value = NULL;
    return status;
}

OFCondition DcmItem::findAndGetOFString(const DcmTagKey& tagKey,
                                        OFString &value,
                                        const unsigned long pos,
                                        const OFBool searchIntoSub)
{
    DcmElement *elem;
    /* find the element */
    OFCondition status = findAndGetElement(tagKey, elem, searchIntoSub);
    if (status.good())
    {
        /* get the value */
        status = elem->getOFString(value, pos);
    }
    /* reset value */
    if (status.bad())
        value.clear();
    return status;
}

#endif /* ifndef __APPLE__ */

/*----------------------------------------------------------------------------*/

void Lowercase (char *Name)
{
   int n;   /* index */


   /* change to lowercase */
   for (n = (int) strlen (Name) - 1; n >= 0; n--)
      if ((Name[n] > 64) && (Name[n] < 91))
         Name[n] += 32;

} /* Lowercase */

/*----------------------------------------------------------------------------*/

void Uppercase (char *Name)
{
   int n;   /* index */


   /* change to uppercase */
   for (n = (int) strlen (Name) - 1; n >= 0; n--)
      if ((Name[n] > 96) && (Name[n] < 123))
         Name[n] -= 32;

} /* Uppercase */

/*----------------------------------------------------------------------------*/

void ExchangeSeparators (char* Name)
{
   int n;   /* index */


   /* change Win/DOS and Mac to Linux */
   for (n = (int) strlen (Name) - 1; n >= 0; n--)
      if ((Name[n] == '\\') || (Name[n] == ':'))
         Name[n] = '/';

} /* ExchangeSeparators */

/*----------------------------------------------------------------------------*/

void GetDirectory (char* Name, char* Directory)
{
   int n;   /* index */


   /* split name */
   for (n = (int) strlen (Name) - 1; (n >= 0) && (Name[n] != '/'); n--);
   Directory[n + 1] = '\0';
   for ( ; n >= 0; n--) Directory[n] = Name[n];

} /* GetDirectory */

/*----------------------------------------------------------------------------*/

void ReadImage (char* Name, DcmDicomDir*& Image)
{
   FILE* file;   /* input file */


   /* open file */
   file = fopen (Name, "r");
   if (!file)
   {
      VError ("Failed to open input file '%s'", Name);
      Image = NULL;
      return;
   }
   fclose (file);

   /* read file */
   Image = new DcmDicomDir (Name);
   if (!Image->error ().good ())
   {
      VError ("Failed to read input file '%s'", Name);
      delete Image;
      Image = NULL;
      return;
   }

} /* ReadImage */

/*----------------------------------------------------------------------------*/

void ReadImage (char* Directory, char* Name, DcmFileFormat*& Image)
{
#if OFFIS_DCMTK_VERSION_NUMBER < 352
   DcmStream* stream;       /* input stream   */
#else
   DcmInputFileStream* stream;       /* input stream   */
   OFCondition l_error = EC_IllegalParameter;
#endif
   char       name[513];    /* complete name  */
   char       lower[257];   /* lowercase name */
   char       upper[257];   /* uppercase name */


   /* open stream */
   ExchangeSeparators (Name);
   sprintf (name, "%s%s", Directory, Name);
#if OFFIS_DCMTK_VERSION_NUMBER < 352
   stream = new DcmFileStream (name, DCM_ReadMode);
   if (stream->Fail ()) {delete stream; stream = NULL;};
#else
   stream = new DcmInputFileStream(name, 0);
   l_error = stream->status();
   if (!l_error.good()) {delete stream; stream = NULL;};
#endif

   if (!stream)   /* Win/DOS workaround, try lowercase instead! */
   {
      strcpy (lower, Name); Lowercase (lower);
      sprintf (name, "%s%s", Directory, lower);
#if OFFIS_DCMTK_VERSION_NUMBER < 352
      stream = new DcmFileStream (name, DCM_ReadMode);
      if (stream->Fail ()) {delete stream; stream = NULL;};
#else
      stream = new DcmInputFileStream(name, 0);
      l_error = stream->status();
      if (!l_error.good()) {delete stream; stream = NULL;};
#endif
   }
   if (!stream)   /* Win/DOS workaround, try uppercase instead! */
   {
      strcpy (upper, Name); Uppercase (upper);
      sprintf (name, "%s%s", Directory, upper);
#if OFFIS_DCMTK_VERSION_NUMBER < 352
      stream = new DcmFileStream (name, DCM_ReadMode);
      if (stream->Fail ()) {delete stream; stream = NULL;};
#else
      stream = new DcmInputFileStream(name, 0);
      l_error = stream->status();
      if (!l_error.good()) {delete stream; stream = NULL;};
#endif
   }

   if (!stream)
   {
      VError ("Failed to open input file '%s'", Name);
      Image = NULL;
      return;
   }

   /* read stream */
   Image = new DcmFileFormat ();
   Image->read (*stream);
   Image->loadAllDataIntoMemory();
   if (!Image->error ().good ())
   {
      VError ("Failed to read input file '%s'", Name);
      delete stream;
      delete Image;
      Image = NULL;
      return;
   }

   /* clean-up */
   delete stream;

} /* ReadImage */

/*----------------------------------------------------------------------------*/

void SearchRecords (DcmDirectoryRecord* Start, const E_DirRecType& Type, DcmStack& Stack, int& Size)
{
   int n;   /* index */


   /* initialize size */
   if (Stack.empty ()) Size = 0;

   /* depth-first search */
   for (n = 0; n < (int) Start->cardSub (); n++)
      SearchRecords (Start->getSub (n), Type, Stack, Size);

   /* store record */
   if (Start->getRecordType () == Type)
   {
      Stack.push (Start);
      Size++;
   }

} /* SearchRecords */

/*----------------------------------------------------------------------------*/

void SortRecords (char* Directory, DcmDirectoryRecord** Records, int N, const DcmTagKey& Tag)
{
   const char* str;   /* string value */

   gsl_vector_float*    parameters;   /* vector of parameters */
   gsl_permutation*     indices;      /* the index permutation */
   DcmDirectoryRecord** sorted;       /* array of records    */
   DcmFileFormat*       image;        /* first image of the series*/
   char                 name[257];    /* image name       */
   char                 sTime[257];   /* series time string */

   int n;   /* index */

   /* store parameters */
   indices = gsl_permutation_alloc(N);
   parameters = gsl_vector_float_alloc(N);
   for (n = 0; n < N; n++)
   {
      // aa the record lacks of the required info... read images
      // Records[n-1]->findAndGetString (Tag, str);

      Records[n]->getSub(0)->findAndGetString (DCM_ReferencedFileID, str);
      strcpy (name, str);

      /* Error handling - no name */
      if (strcmp (name, "") == 0) {
				VError ("Image does not reference a file");
				gsl_vector_float_free(parameters);
				return;
      }

      /* read Dicom image */
      ReadImage (Directory, name, image);

      image->getDataset()->findAndGetString (DCM_SeriesTime, str);
      if (str) strcpy (sTime, str);
      else     strcpy (sTime, "");

      delete image;

      /* Error handling no sTime */
      if (strcmp (sTime, "") == 0)  {
				gsl_vector_float_free(parameters);
				return;
      }

      sscanf (sTime, "%f", parameters->data + n);
   }

   /* sort parameters */
   gsl_sort_vector_float_index(indices,parameters);

   /* reorder records */
   sorted = new DcmDirectoryRecord*[N];
   /* for (n = 1; n <= N; n++) */
   /*    sorted[n - 1] = Records[indices[n] - 1]; */
   for (n = 0; n < N;n++)
     sorted[n] = Records[indices->data[n]];
   for (n = 0; n < N; n++)
      Records[n] = sorted[n];

   /* clean-up */
   gsl_vector_float_free(parameters);
   gsl_permutation_free(indices);
   delete[] sorted;

} /* SortRecords */

/*----------------------------------------------------------------------------*/

void SortImages (DcmFileFormat** Images, int N, const DcmTag& Tag)
{
   DcmDataset* dataset;   /* image dataset */
   const char* str;       /* string value  */

   gsl_vector_float*    parameters;   /* array of parameters */
   gsl_permutation*     indices;      /* array of indices    */
   DcmFileFormat**      sorted;       /* array of images     */

   int n;   /* index */

   /* store parameters */
   parameters = gsl_vector_float_alloc(N);
	 indices = gsl_permutation_alloc(N);
   for (n = 0; n < N; n++)
   {
      dataset = Images[n]->getDataset ();
      dataset->findAndGetString (Tag, str);
      if (!str) {gsl_vector_float_free(parameters); return;};
      sscanf (str, "%f", parameters->data + n);
   }

   /* gsl index sort */
   gsl_sort_vector_float_index(indices,parameters);

   /* reorder images */
   sorted = new DcmFileFormat*[N];
   for (n = 0;n < N;n++)
     sorted[n] = Images[indices->data[n]];
   for (n = 0; n < N; n++)
     Images[n] = sorted[n];

   /* clean-up */
   gsl_permutation_free (indices);
	 gsl_vector_float_free (parameters);
   delete[] sorted;

} /* SortImages */

/*----------------------------------------------------------------------------*/

void ExtractShadowParameter (DcmFileFormat* Image, const char* Key, float& Parameter) /* for Siemens only ! */
{
   DcmDataset* dataset;   /* image dataset */
   DcmStack    groups;    /* group stack   */
   DcmElement* group;     /* shadow group  */

   int    length;   /* length of key */
   Uint8* data;     /* data pointer  */
   Uint8* end;      /* end of data   */

   int n;   /* index */


   /* set result */
   Parameter = 0;

   /* extract shadow group */
   dataset = Image->getDataset ();
   dataset->search (DcmTag (0x0029, 0x1010), groups);   /* voodoo */
   group = (DcmElement*) groups.pop ();
   if (!group) return;

   /* search shadow parameter */
   length = strlen (Key);
   group->getUint8Array (data);
   end = data + group->getLength () - length;
   for (; data < end; data++)
   {
      for (n = 0; n < length; n++)
         if (data[n] != Key[n]) break;
      if (n == length) break;
   }
   if (data == end) return;

   /* extract shadow parameter */
   sscanf ((char*) (data + 100), "%f", &Parameter);   /* voodoo */

} /* ExtractShadowParameter */

/*----------------------------------------------------------------------------*/

void ExtractASCCONVParameter(DcmFileFormat* Image, const char* Key, long int& Parameter) /* for Siemens only ! */
{
   DcmDataset* dataset;   /* image dataset */
   DcmStack    groups;    /* group stack   */
   DcmElement* group;     /* shadow group  */

   int    length;         /* length of key */
   int    n;              /* index */
   Uint8* data;           /* data pointer  */
   Uint8* end;            /* end of data   */
   Uint8  tag[]="### ASCCONV BEGIN";
   char*  Key_start;      /* pointer to the parameter line */
   char   name[1024];     /* help variable in the scanning line */

   /* set result */
   Parameter = 0;

   /* extract third shadow group */
   dataset = Image->getDataset ();
   dataset->search (DcmTag (0x0029, 0x1020), groups);   /* voodoo */
   group = (DcmElement*) groups.pop ();
   if (!group) return;

   /* search the ASCCONV group*/
   length = 16;
   group->getUint8Array (data);
   end = data + group->getLength ();
   for (; data < end; data++)
   {
      for (n = 0; n < length; n++)
         if (data[n] != tag[n]) break;
      if (n == length) break;
   }
   if (data == end) return;

   Key_start = strstr((char *)data, Key);
   if( Key_start == NULL ) return ;

   sscanf( Key_start, "%1022s =%ld" ,name , &Parameter);
   if (Parameter==0)
      sscanf( Key_start, "%1022s = 0x%ld" ,name , &Parameter);

   return;

} /* ExtractASCCONVParameter */

/*----------------------------------------------------------------------------*/

void ExtractASCCONVString(DcmFileFormat* Image, const char* Key, char* Parameter) /* for Siemens only ! */
{
   DcmDataset* dataset;   /* image dataset */
   DcmStack    groups;    /* group stack   */
   DcmElement* group;     /* shadow group  */

   int    length;         /* length of key */
   int    n;              /* index */
   Uint8* data;           /* data pointer  */
   Uint8* end;            /* end of data   */
   Uint8  tag[]="### ASCCONV BEGIN";
   char*  Key_start;      /* pointer to the parameter line */
   char   name[1024];     /* help variable in the scanning line */
   char  scanned_parameter[1024];

   /* set result */
   Parameter[0] = '\0';
   Parameter[19]= '\0';

   /* extract third shadow group */
   dataset = Image->getDataset ();
   dataset->search (DcmTag (0x0029, 0x1020), groups);   /* voodoo */
   group = (DcmElement*) groups.pop ();
   if (!group) return;

   /* search the ASCCONV group*/
   length = 16;
   group->getUint8Array (data);
   end = data + group->getLength ();
   for (; data < end; data++)
   {
      for (n = 0; n < length; n++)
         if (data[n] != tag[n]) break;
      if (n == length) break;
   }
   if (data == end) return;

   Key_start = strstr((char *)data, Key);
   if( Key_start == NULL ) return ;

   sscanf( Key_start, "%1022s = %s" ,name , scanned_parameter);

   strncpy(Parameter, scanned_parameter, 19);

   return;

} /* ExtractASCCONVString */

/*----------------------------------------------------------------------------*/

void ExtractGeometry (DcmFileFormat** Images, int N, float Size[], float Voxel[], float Grid[], float Location[])
{
   DcmDataset* dataset;   /* image dataset   */
   const char* str;       /* string value    */
   Uint16      num;       /* numerical value */

   float loc0, loc1;      /* image locations */

   int n;   /* index */


   /* extract image size */
   Size[0] = Size[1] = Size[2] = 0;
   dataset = Images[0]->getDataset ();
   dataset->findAndGetUint16 (DCM_Columns, num);
   if (num > 0) Size[0] = num;
   dataset->findAndGetUint16 (DCM_Rows, num);
   if (num > 0) Size[1] = num;
   Size[2] = N;

   /* extract voxel size */
   Voxel[0] = Voxel[1] = Voxel[2] = 0;
   dataset = Images[0]->getDataset ();
   dataset->findAndGetString (DCM_PixelSpacing, str);
   if (str) sscanf (str, "%f\\%f", Voxel + 1, Voxel);
   dataset->findAndGetString (DCM_SliceThickness, str);
   if (str) sscanf (str, "%f", Voxel + 2);

   /* extract grid size */
   Grid[0] = Voxel[0]; Grid[1] = Voxel[1]; Grid[2] = 0;
   dataset = Images[0]->getDataset ();
   dataset->findAndGetString (DCM_SpacingBetweenSlices, str);
   if (str) sscanf (str, "%f", Grid + 2);

   if ((Grid[2] == 0) && (N > 1))   /* Workaround, try DCM_SliceLocation instead! */
   {
      loc0 = loc1 = 0;
      dataset = Images[0]->getDataset ();
      dataset->findAndGetString (DCM_SliceLocation, str);
      if (str) sscanf (str, "%f", &loc0);
      dataset = Images[1]->getDataset ();
      dataset->findAndGetString (DCM_SliceLocation, str);
      if (str) sscanf (str, "%f", &loc1);
      Grid[2] = fabs (loc0 - loc1);
   }


   /**
    * TM invented multi-echo 3D datasets in early 2006. Subsequently Grid[2] is still
    * 0.0 at this point (the first two images have identical SliceLocation values) and
    * that asks for yet another 'U-Boot'(TM).
    */
   if(Grid[2] == 0) {
      cout << "[WARNING] Setting 'voxel' values to 'lattice' values bcause of parameter inconsistencies." << endl;
      // setting Grid[2] to Voxel[2] which might be wrong but is certainly better than having it set to 0.0
      Grid[2] = Voxel[2];
   }


   /* extract locations */
   for (n = 0; n < N; n++)
   {
      Location[n] = 0;
      dataset = Images[0]->getDataset ();
      dataset->findAndGetString (DCM_SliceLocation, str);
      if (str) sscanf (str, "%f", Location + n);
   }
   for (n = 0; n < N; n++) Location[n] -= Location[0];

} /* ExtractGeometry */

/*----------------------------------------------------------------------------*/

float TimeToSeconds (float Time)
{
   int   hours, minutes;   /* time pieces       */
   float seconds;          /* number of seconds */


   /* convert time */
   hours    = (int) (Time / 10000);
   minutes  = (int) ((Time - 10000 * hours) / 100);
   seconds  = Time - 10000 * hours - 100 * minutes;
   seconds += hours * 3600 + minutes * 60;

   return seconds;

} /* TimeToSeconds */

/*----------------------------------------------------------------------------*/

void ExtractDynamics (DcmFileFormat** Images, int N, int& diffusionDirections, int& Steps, float& Duration, float& Repetition, float Time[])
{
   DcmDataset* dataset;   /* image dataset   */
   const char* str;       /* string value    */
   Uint16      num;       /* numerical value */
   long int    number;    /* numerical value */
   long int    number_max;/* numerical value */
   int         flag;      /* flag */

   float time0, time1;   /* image times */
   float _repTm;

   int n;   /* index */


   /* extract timesteps */
   Steps = 1;
   diffusionDirections = 0;

   //aa DCM_NumberOfTemporalPositions not set in the dicom files
   // use max_AcquisitionNumber instead
   //dataset = Images[0]->getDataset ();
   //dataset->findAndGetUint16 (DCM_NumberOfTemporalPositions, num);
   //if (num > 0) Steps = (int) num;

   number_max = 0;
   flag = 0;
   for (n = 0; n < N; n++)
   {
      dataset = Images[n]->getDataset ();
      dataset->findAndGetString  (DCM_AcquisitionNumber, str);
      if (str) sscanf (str, "%ld", &number);

      /* check if there are different Acquisition Numbers in the scan otherwise
         the scan has only one timestep and the number_max is corrected to 1*/
      if ((n>0)&&(number_max != number))
         flag = 1;

      number_max = ( number>number_max ? number : number_max );
   }
   if (!flag) number_max = 1;
   if (number_max > 1) Steps = (int) number_max;

   /* extract slice duration */
   Duration = 0;
   if (N > 1)
   {
      time0 = time1 = 0;
      dataset = Images[0]->getDataset ();
      dataset->findAndGetString (DCM_AcquisitionTime, str);
      if (str) sscanf (str, "%f", &time0);
      time0 = TimeToSeconds (time0);
      dataset = Images[1]->getDataset ();
      dataset->findAndGetString (DCM_AcquisitionTime, str);
      if (str) sscanf (str, "%f", &time1);
      time1 = TimeToSeconds (time1);
      Duration = fabs (time0 - time1) * 1000;

   }

   /* extract repetition time */
   Repetition = 0;
   if ((N > 1) && (Steps > 1))
   {
      dataset = Images[0]->getDataset();
      dataset->findAndGetString(DCM_RepetitionTime, str);
      if(str) {
	sscanf(str, "%f", &_repTm);
	Repetition = _repTm;
      } else {
	time0 = time1 = 0;
        dataset = Images[0]->getDataset ();
        dataset->findAndGetString (DCM_AcquisitionTime, str);
        if (str) sscanf (str, "%f", &time0);
        time0 = TimeToSeconds (time0);
        dataset = Images[N / Steps]->getDataset ();
        dataset->findAndGetString (DCM_AcquisitionTime, str);
        if (str) sscanf (str, "%f", &time1);
        time1 = TimeToSeconds (time1);
        Repetition = fabs (time0 - time1) * 1000;
      }
   }

   /* extract times */
   for (n = 0; n < N; n++)
   {
      Time[n] = 0;
      dataset = Images[0]->getDataset ();
      dataset->findAndGetString (DCM_AcquisitionTime, str);

      // fprintf(stderr," extract times: %s\n",str);
      if (str) sscanf (str, "%f", Time + n);
   }
   for (n = 0; n < N; n++) Time[n] -= Time[0];


   if (Steps == 1) // check for dti only if not functional
   {
      /* extract attributes from diffusion weighted image */
      long int diffusionMode=0;
      ExtractASCCONVParameter (Images[0], "sDiffusion.ulMode", diffusionMode);
      if (diffusionMode>1)
      {
          /* extract number of diffusion directions in tensor image */
          long int sliceArraySize=0;
          ExtractASCCONVParameter (Images[0], "sSliceArray.lSize", sliceArraySize);

          if (sliceArraySize)
              diffusionDirections = N / sliceArraySize;
          else
              diffusionDirections = 0;
      }
   }
} /* ExtractDynamics */

/*----------------------------------------------------------------------------*/

void ExtractAttributes (DcmFileFormat* Image, VAttrList List)
{
   DcmDataset* dataset;   /* image dataset */
   const char* str;       /* string value  */

   char patient[257], birth[257], sex[257];         /* patient  attributes */
   char device[257];                                /* device   attributes */
   char modality[257];                              /* modality attributes */
   char protocol[257];                              /* protocol attributes */
   char sequence[257];                              /* sequence attributes */
   char date[257], year[5], month[3], day[3];       /* date     attributes */
   char time[257], hour[3], minute[3], second[3];   /* time     attributes */
   char seriesNumber[257];                          /* series   attributes */
   char seriesDescription[257];                     /* series Description attributes */
   char echoTime[257];                              /* Echo Time          attributes */
   char phaseEncoding[257];                         /* phase encoding     attributes */
   char flipAngle[257];                             /* flip Angle         attributes */
   long int readoutFOV;                             /* Field of view      attributes */
   long int sliceMode;                              /* sliceArray.ucMode  attributes */

   /* extract patient attributes */
   dataset = Image->getDataset ();

   strcpy (patient, "unknown");
   // dataset->findAndGetString (DCM_PatientID, str); (nicht anonym !!)

   //  /usr/include/dcmtk/dcmdata/dcdeftag.h
   dataset->findAndGetString (DCM_PatientsName, str); // neu, G.L. 9.7.2003
   if (str) strcpy (patient, str);


   strcpy (birth, "unknown");
   dataset->findAndGetString (DCM_PatientsBirthDate, str);
   if (str)
   {
      sscanf (str, "%4s%2s%2s", year, month, day);
      sprintf (birth, "%s.%s.%s", day, month, year);
   }
   strcpy (sex, "unknown");
   dataset->findAndGetString (DCM_PatientsSex, str);
   if (str) switch (str[0])
   {
      case 'M': strcpy (sex, "male"); break;
      case 'F': strcpy (sex, "female"); break;
      default : strcpy (sex, "unknown"); break;
   }

   /* store patient attributes */
   VAppendAttr (List, "patient", NULL, VStringRepn, patient);
   VAppendAttr (List, "birth", NULL, VStringRepn, birth);
   VAppendAttr (List, "sex", NULL, VStringRepn, sex);

   /* extract image attributes */
   strcpy (device, "unknown");
   dataset->findAndGetString (DCM_Manufacturer, str);
   if (str) strcpy (device, str);
   dataset->findAndGetString (DCM_ManufacturersModelName, str);
   if (str) {strcat (device, " "); strcat (device, str);}
   dataset->findAndGetString (DCM_StationName, str);
   if (str) {strcat (device, " "); strcat (device, str);}
   strcpy (modality, "unknown");
   dataset->findAndGetString (DCM_Modality, str);
   if (str) strcpy (modality, str);
   strcpy (protocol, "unknown");
   dataset->findAndGetString (DCM_ProtocolName, str);
   if (str) strcpy (protocol, str);
   strcpy (sequence, "unknown");
   dataset->findAndGetString (DCM_SequenceName, str);
   if (str) strcpy (sequence, str);
   strcpy (date, "unknown");
   dataset->findAndGetString (DCM_AcquisitionDate, str);
   if (str)
   {
      sscanf (str, "%4s%2s%2s", year, month, day);
      sprintf (date, "%s.%s.%s", day, month, year);
   }
   strcpy (time, "unknown");
   dataset->findAndGetString (DCM_AcquisitionTime, str);
   if (str)
   {
      sscanf (str, "%2s%2s%2s", hour, minute, second);
      sprintf (time, "%s:%s:%s", hour, minute, second);
   }

   strcpy (seriesNumber, "unknown");
   dataset->findAndGetString (DCM_SeriesNumber, str);
   if (str) strcpy (seriesNumber, str);
   strcpy (seriesDescription, "unknown");
   dataset->findAndGetString (DCM_SeriesDescription, str);
   if (str) strcpy (seriesDescription, str);
   strcpy (echoTime, "unknown");
   dataset->findAndGetString (DCM_EchoTime, str);
   if (str) strcpy (echoTime, str);
   strcpy (phaseEncoding, "unknown");
   dataset->findAndGetString (DCM_PhaseEncodingDirection, str);
   if (str) strcpy (phaseEncoding, str);
   strcpy (flipAngle, "unknown");
   dataset->findAndGetString (DCM_FlipAngle, str);
   if (str) strcpy (flipAngle, str);

   readoutFOV=0;
   ExtractASCCONVParameter (Image, "dReadoutFOV", readoutFOV);
   sliceMode=0;
   ExtractASCCONVParameter (Image, "sSliceArray.ucMode", sliceMode);

   /* store image attributes */
   VAppendAttr (List, "device", NULL, VStringRepn, device);
   VAppendAttr (List, "modality", NULL, VStringRepn, modality);
   VAppendAttr (List, "protocol", NULL, VStringRepn, protocol);
   VAppendAttr (List, "sequence", NULL, VStringRepn, sequence);
   VAppendAttr (List, "date", NULL, VStringRepn, date);
   VAppendAttr (List, "time", NULL, VStringRepn, time);
   VAppendAttr (List, "seriesDescription", NULL, VStringRepn, seriesDescription);
   // tag component_interp used by the vm software
   //   VAppendAttr (List, "component_interp", NULL, VStringRepn, seriesDescription);
   VAppendAttr (List, "seriesNumber", NULL, VStringRepn, seriesNumber);
   VAppendAttr (List, "echoTime", NULL, VStringRepn, echoTime);
   VAppendAttr (List, "phaseEncoding", NULL, VStringRepn, phaseEncoding);
   VAppendAttr (List, "flipAngle", NULL, VStringRepn, flipAngle);

   if (readoutFOV)
      VAppendAttr (List, "readoutFOV", NULL, VLongRepn, readoutFOV);
   else
      VAppendAttr (List, "readoutFOV", NULL, VStringRepn, "unknown");
   if (sliceMode)
      VAppendAttr (List, "sliceMode", NULL, VLongRepn, sliceMode);
   else
      VAppendAttr (List, "sliceMode", NULL, VStringRepn, "unknown");

   /* extract attributes from diffusion weighted image */
   long int diffusionMode=0;
   ExtractASCCONVParameter (Image, "sDiffusion.ulMode", diffusionMode);
   if (diffusionMode>1)
   {
      VAppendAttr (List, "diffusionMode", NULL, VLongRepn, diffusionMode);

      long int diffusionWeightings  =0;
      ExtractASCCONVParameter (Image, "sDiffusion.lDiffWeightings", diffusionWeightings );
      VAppendAttr (List, "diffusionWeightings", NULL, VLongRepn, diffusionWeightings);

      //Diffusion effect in s/mm*mm of the ICE program for the performed acquisition
      long int diffusionBValue_1        =0;
      ExtractASCCONVParameter (Image, "sDiffusion.alBValue[1]", diffusionBValue_1);
      VAppendAttr (List, "diffusionBValue_1", NULL, VLongRepn, diffusionBValue_1);

      long int diffusionBValue_2        =0;
      ExtractASCCONVParameter (Image, "sDiffusion.alBValue[2]",diffusionBValue_2 );
      VAppendAttr (List, "diffusionBValue_2", NULL, VLongRepn, diffusionBValue_2);

      long int diffusionNoiseLevel      =0;
      ExtractASCCONVParameter (Image, "sDiffusion.lNoiseLevel", diffusionNoiseLevel);
      VAppendAttr (List, "diffusionNoiseLevel", NULL, VLongRepn, diffusionNoiseLevel);

      long int diffusionDirections  =0;
      ExtractASCCONVParameter (Image, "sDiffusion.lDiffDirections",diffusionDirections );
      VAppendAttr (List, "diffusionDirections", NULL, VLongRepn, diffusionDirections);

      long int diffusionWeightedImage    =0;
      ExtractASCCONVParameter (Image, "sDiffusion.ucDiffWeightedImage", diffusionWeightedImage);
      VAppendAttr (List, "diffusionWeightedImage", NULL, VLongRepn, diffusionWeightedImage);
   }

   char CoilId[20], read_str[20];
   strcpy (CoilId, "unknown");
   ExtractASCCONVString (Image, "sCOIL_SELECT_MEAS.asList[0].sCoilElementID.tCoilID", read_str);
   if (read_str[0]) strcpy (CoilId, read_str);
   VAppendAttr (List, "CoilID", NULL, VStringRepn, CoilId);

   fprintf(stderr,"Series Number = %s, Protocol = %s, Description = %s \n",
      seriesNumber, protocol,  seriesDescription); fflush (stderr);

} /* ExtractAttributes */


void ExtractEchoTimes(DcmFileFormat** Images, int N, float echoTimes[], std::list<float>* echoTimesList) {

	DcmDataset* dataset;
	const char* _parameterString;
	float _echoTime;


    SortImages(Images, N, DCM_InstanceNumber);

	for(int _n = 0; _n < N; _n++) {
       dataset = Images[_n]->getDataset();

//	   dataset->findAndGetString(DCM_InstanceNumber, _parameterString);
//       std::cerr << "Image Instance Number: " << _parameterString << "   ";
//       dataset->findAndGetString(DCM_EchoTime, _parameterString);
//       std::cerr << "Echo Time: " << _parameterString << "   ";
//	   dataset->findAndGetString(DCM_AcquisitionTime, _parameterString);
//	   std::cerr << "Acquisition Time: " << _parameterString << endl;


       dataset->findAndGetString(DCM_EchoTime, _parameterString);
       if(_parameterString) {
          sscanf(_parameterString, "%f", &_echoTime);
       }

       echoTimes[_n] = _echoTime;
//      std::cerr << "echo time: " << _echoTime << endl;

       echoTimesList->push_back(_echoTime);
	}

	echoTimesList->sort();
	echoTimesList->unique();

}












/*----------------------------------------------------------------------------*/

VBoolean ConvertSeries (char* Directory, DcmDirectoryRecord* Series, int Steps,
   int Mosaic, int Tiles, VAttrList& Images, VBoolean& Functional, VBoolean& Dti,
   char* MoCoName, char* inFilename)
{
   DcmDataset* dataset;   /* image dataset    */
   const char* str;       /* string value     */
   float       param;     /* shadow parameter */

   DcmStack        stack;       /* images stack     */
   DcmFileFormat** images;      /* images array     */
   char            name[257];   /* image name       */
   int             N;           /* number of images */

   float  size[3];                /* image size (columns, rows, bands) */
   float  voxel[3];               /* voxel size (X, Y, Z)              */
   float  grid[3];                /* image grid (X, Y, Z)              */
   float* location;               /* slice locations                   */
   int    steps;                  /* number of timesteps               */
   int    diffusionDirections;    /* number of diffusion directions    */
   float  duration;               /* duration of slice                 */
   float  repetition;             /* time between repetitions          */
   float* time;                   /* slice times                       */
   int    tiles;                  /* number of tiles per image         */
   int    mosaic;                 /* size of mosaic matrix             */
   int    bands, rows, columns;   /* image size                        */

#if OFFIS_DCMTK_VERSION_NUMBER > 352
   const Uint16* src;    /* Dicom source pointer      */
#else
   Uint16* src;    /* Dicom source pointer      */
#endif

   VShort* data;   /* Vista source pointer      */
   VShort* dest;   /* Vista destination pointer */

   VImage        image;      /* raw image data   */
   VImage        tile;       /* tile image data  */
   VImage        step;       /* timestep data    */
   VAttrList     list;       /* image attributes */
   char          attr[65];   /* attribute value  */
   VAttrListPosn pos;        /* position in list */

   Axis     U, V, W;                       /* image axes */
   Axis     X, Y, Z;                       /* image axes */
   VBoolean xyswap, xflip, yflip, zflip;   /* indicators */
   Axis     swap;                          /* storage    */
   float    tmp;                           /* storage    */

   int n, t, x, y, z;   /* indices */
   float pauseBetweenFrames; /* time between measurements */
   long int lDelayTimeInTR ; /* time between measurements in ns*/
   char sequence[257];       /* name of the dti sequence */
   char seriesNumber[257];

   // TS (07/12/05):
   // this doesn't work properly because the Siemens software sets
   // the 'Directory Record Type' of spectroscopic data to 'IMAGE'
   // (see below)
   /* search images */
   SearchRecords (Series, ERT_Image, stack, N);
   if (N == 0) {VError ("Scan does not contain any images"); return FALSE;}

   // TS (07/12/05):
   // check all SOP Class UID's within the series
   // if the UID is not
   // 1.2.840.10008.5.1.4.1.1.4    ('MR Image Storage')
   // or
   // 1.2.840.10008.5.1.4.1.1.4.1  ('Enhanced MR Image Storage')
   // issue a warning and return false
   //
   // NOTE: Siemens uses a 'private' UID (1.3.12.2.1107.5.9.1  'CSA Non-Image Storage')
   // for spectroscopic data rather than the one specified in 'Suppl. 49'
   // (1.2.840.10008.5.1.4.1.1.4.2  'MR Spectroscopy Storage')
   long _seriesCardSub = Series->cardSub();
   OFString referencedSOPClassUIDInFile;
   for(long _element = 0; _element < _seriesCardSub; _element++) {
       Series->getSub(_element)->findAndGetOFString(DCM_ReferencedSOPClassUIDInFile, referencedSOPClassUIDInFile);
       if(referencedSOPClassUIDInFile.compare(UID_MRImageStorage) != 0 &&
          referencedSOPClassUIDInFile.compare(UID_EnhancedMRImageStorage) != 0) {
          	VWarning("Series contains invalid or non-image objects (SOPClassUID: '%s')", referencedSOPClassUIDInFile.c_str());
          	return FALSE;
       }
   }


   /* load images */
   images = new DcmFileFormat*[N];
   for (n = 0; n < N; n++)
   {
      /* extract filename */
      dataset = (DcmDataset*) stack.pop ();
      dataset->findAndGetString (DCM_ReferencedFileID, str);
      strcpy (name, str);
      if (strcmp (name, "") == 0)
      {
         VError ("Image does not reference a file");
         for (n--; n >= 0; n--)
            delete images[n];
         delete[] images;
         return FALSE;
      }

      /* read Dicom image */
      ReadImage (Directory, name, images[n]);
      if (!images[n])
      {
         for (n--; n >= 0; n--)
            delete images[n];
         delete[] images;
         return FALSE;
      }
   }

   /* extract image geometry and dynamics (can be overriden by external parameters!) */
   location = new float[N];
   SortImages (images, N, DCM_SliceLocation);
   ExtractGeometry (images, N, size, voxel, grid, location);
   time = new float[N];
   SortImages (images, N, DCM_AcquisitionTime);
   ExtractDynamics (images, N, diffusionDirections, steps, duration, repetition, time);

   /* extract Siemens Mosaic geometry (can be overriden by external parameters!) */
   if (Tiles > 0) tiles = Tiles;
   else
   {
      tiles = 0;
      if (steps == N) /*aa check for NumberOfImagesInMosaic only in Mosaic images*/
      {
         ExtractShadowParameter (images[0], "NumberOfImagesInMosaic", param);
         if (param > 0) tiles = (int) param;
      }
   }
   if (Mosaic > 0) mosaic = Mosaic;
   else
      for (mosaic = 0; mosaic * mosaic < tiles; mosaic++);

   // set external steps after checking for mosaic mode
   if (Steps > 0) steps = Steps;

   fprintf(stderr,"Nb of images = %d, Nb of tiles = %d, Nb of time steps = %d\n",N, tiles, steps); fflush (stderr);

   /* set scan type */
   Functional = (steps > 1);
   Dti = ((diffusionDirections > 1) && !Functional);

   if (Dti && mosaic)
   {
      fprintf(stderr,"Dti data in mosaic format not yet supported\n"); fflush (stderr);
      return false;
   }


   /* sort anatomical images */
   if ((tiles == 0) && (steps == 1))
   {
      if(!Dti) {

        // TS (07/12/05): use Instance Number (0020,0013) instead of Slice Location (0020,1041)
        SortImages (images, N, DCM_InstanceNumber);
        // SortImages (images, N, DCM_SliceLocation);

      }
      else   /* sort dti images (different seqences in one series)*/
      {
         SortImages (images, N, DCM_AcquisitionTime);

         for (n = 0; n < diffusionDirections; n++){
            SortImages (images + n * (N / diffusionDirections), (N / diffusionDirections), DCM_SliceLocation);}

      }
   }

   /* sort functional images (not tested !) */
   else if ((tiles == 0) && (steps > 1))
   {
      SortImages (images, N, DCM_AcquisitionTime);
      for (n = 0; n < steps; n++)
         SortImages (images + n * (N / steps), N / steps, DCM_SliceLocation);
   }

   /* sort anatomical mosaic images */
   else if ((tiles > 0) && (steps == 1))
      // TS (07/12/05): use Instance Number (0020,0013) instead of Slice Location (0020,1041)
      SortImages (images, N, DCM_InstanceNumber);
      // SortImages (images, N, DCM_SliceLocation);

   /* sort functional mosaic images */
   else if ((tiles > 0) && (steps > 1))
      SortImages (images, N, DCM_AcquisitionTime);

   /* create image */
   bands   = (int) size[2];
   rows    = (int) size[1];
   columns = (int) size[0];
   image   = VCreateImage (bands, rows, columns, VShortRepn);

   /* extract image data */
   for (z = 0; z < bands; z++)
   {
      dataset = images[z]->getDataset ();


   	  // TS (07/12/05):
   	  // - use the true image dimensions when extracting the actual image data
   	  // - this results in either zero-filled or clipped images in the case
   	  //   of different image sizes within one series
   	  // TODO: need to find a proper way of dealing with different image sizes
   	  //       within a single series like creating more than one Vista image
   	  //       or finding the maximum size within the series and centering the
   	  //       smaller images or ... don't know yet
   	  Uint16 trueRows;
   	  Uint16 trueColumns;
   	  dataset->findAndGetUint16(DCM_Rows,    trueRows);
   	  dataset->findAndGetUint16(DCM_Columns, trueColumns);

      dataset->findAndGetUint16Array (DCM_PixelData, src);
      dest = (VShort*) VPixelPtr (image, z, 0, 0);

      for(int _row = 0; _row < rows; _row++) {
	      for(int _column = 0; _column < columns; _column++) {
	      	if(_column < trueColumns && _row < trueRows) {
	      		*(dest++) = (VShort)(src[_row * trueColumns + _column]);
	      	} else {
	      		*(dest++) = (VShort)(0);
	      	}
    	  }
      }

//      for (n = 0; n < columns * rows; n++)
//         *(dest++) = (VShort) *(src++);
   }


   /* store anatomical images */
   Images = VCreateAttrList ();
   if ((tiles == 0) && (steps == 1))
      if (!Dti)
         VAppendAttr (Images, "image", NULL, VImageRepn, image);
      else   /* store dti images (different seqences in one series)*/
      {
           /* adjust image geometry */
           size[2] /= diffusionDirections;

           /* create bands */
           bands   = (int) size[2];
           rows    = (int) size[1];
           columns = (int) size[0];
           for (t = 0; t < diffusionDirections; t++)
           {
               step = VCreateImage (bands, rows, columns, VShortRepn);

               /* extract band data */
               dest = (VShort*) VPixelPtr (step, 0, 0, 0);
               data = (VShort*) VPixelPtr (image, t * bands, 0, 0);
               for (n = 0; n < bands * rows * columns; n++)
                  *(dest++) = *(data++);

               /* store dti component */
               VAppendAttr (Images, "image", NULL, VImageRepn, step);
           }

           /* clean-up */
           delete image;
      }

   /* store functional images (not tested !) */
   else if ((tiles == 0) && (steps > 1))
   {
      /* adjust image geometry */
      size[2] /= steps;

      lDelayTimeInTR = 0;
      ExtractASCCONVParameter (images[0], "lDelayTimeInTR", lDelayTimeInTR);
      pauseBetweenFrames = lDelayTimeInTR / 1000;

      //time between two frames = (TR-"Pause in TR")/(No.Slices)
      duration = ( repetition - pauseBetweenFrames ) / size[2];

      /* create timestep */
      bands   = (int) size[2];
      rows    = (int) size[1];
      columns = (int) size[0];
      for (t = 0; t < steps; t++)
      {
         step = VCreateImage (bands, rows, columns, VShortRepn);

         /* extract timestep data */
         dest = (VShort*) VPixelPtr (step, 0, 0, 0);
         data = (VShort*) VPixelPtr (image, t * bands, 0, 0);
         for (n = 0; n < bands * rows * columns; n++)
            *(dest++) = *(data++);

         /* store timestep */
         VAppendAttr (Images, "image", NULL, VImageRepn, step);
      }

      /* clean-up */
      delete image;
   }

   /* store anatomical mosaic images */
   else if ((tiles > 0) && (steps == 1))
   {
      /* adjust image geometry and dynamics */
      size[2]    = tiles;
      size[1]   /= mosaic;
      size[0]   /= mosaic;

      /* create tile */
      bands   = (int) size[2];
      rows    = (int) size[1];
      columns = (int) size[0];
      tile = VCreateImage (bands, rows, columns, VShortRepn);

      /* extract tile data */
      dest = (VShort*) VPixelPtr (tile, 0, 0, 0);
      for (z = 0; z < bands; z++)
      {
         data = (VShort*) VPixelPtr (image, 0, (z / mosaic) * rows, (z % mosaic) * columns);
         for (y = 0; y < rows; y++)
         {
            for (x = 0; x < columns; x++)
               *(dest++) = *(data++);
            data += columns * (mosaic - 1);
         }
      }

      /* store tile */
      VAppendAttr (Images, "image", NULL, VImageRepn, tile);

      /* clean-up */
      delete image;
   }

   /* store functional mosaic images */
   else if ((tiles > 0) && (steps > 1))
   {
      /* adjust image geometry and dynamics */
      size[2]    = tiles;
      size[1]   /= mosaic;
      size[0]   /= mosaic;

      //ExtractShadowParameter (images[0], "SliceMeasurementDuration", duration);
      // aa SliceMeasurementDuration ist die Auslesezeit der EPI Sequenz vom ersten ADC bis zum letzten
      // (Vorsicht: Zeit inkl. Phasenkorrekturscans).

      lDelayTimeInTR = 0;
      ExtractASCCONVParameter (images[0], "lDelayTimeInTR", lDelayTimeInTR);
      pauseBetweenFrames = lDelayTimeInTR / 1000;

      //time between two frames = (TR-"Pause in TR")/(No.Slices)
      duration = ( repetition - pauseBetweenFrames ) / tiles;

      /* create tile */
      bands   = (int) size[2];
      rows    = (int) size[1];
      columns = (int) size[0];
      for (t = 0; t < steps; t++)
      {
         tile = VCreateImage (bands, rows, columns, VShortRepn);

         /* extract tile data */
         dest = (VShort*) VPixelPtr (tile, 0, 0, 0);
         for (z = 0; z < bands; z++)
         {
            data = (VShort*) VPixelPtr (image, t, (z / mosaic) * rows, (z % mosaic) * columns);
            for (y = 0; y < rows; y++)
            {
               for (x = 0; x < columns; x++)
                  *(dest++) = *(data++);
               data += columns * (mosaic - 1);
            }
         }

         /* store tile */
         VAppendAttr (Images, "image", NULL, VImageRepn, tile);
      }

      /* clean-up */
      delete image;


      /* write motion correction string for MoCoSeries if MoCoName is set*/
      images[0]->getDataset ()->findAndGetString (DCM_SeriesDescription, str);
      if ( ((strcmp(str,"MoCoSeries")==0) || (strcmp(str,"ep2d_boldMoCoSeries")==0))  && (strcmp(MoCoName,"")!=0) )
      {
         float a1, a2, a3, a4, a5, a6;
         FILE* file = fopen (MoCoName, "w");
         if (file)
         {
            fprintf (stderr, "\nWrite motion parameters for MoCoSeries to file %s\n\n", MoCoName); fflush (stderr);

            images[0]->getDataset ()->findAndGetString (DCM_SeriesNumber, str);
            if (str) strcpy (seriesNumber, str);

            fprintf(file,"# Motion correction logfile for acquisition %s\n",inFilename);
			fprintf(file,"#\n");
            fprintf(file,"# Column 0: Timestep; \n");
			fprintf(file,"# Column 1 - 3: Translation x,y,z in mm; \n");
			fprintf(file,"# Column 4 - 6: Rotation x,y,z in degree\n");
            fprintf(file,"# The values are printed in the SIEMENS coordinate system which is different to the LIPSIA convention.\n");
            fprintf(file,"#\n");
            fprintf(file,"# Series Number = %s, Nb of timesteps = %d\n#\n", seriesNumber, steps);

			for (int t = 0; t < steps; t++)
            {
               images[t]->getDataset()->findAndGetString (DCM_ImageComments, str);

               a1=0.0;a2=0.0;a3=0.0;a4=0.0;a5=0.0;a6=0.0;
               if (str) sscanf(str,"Motion: %f,%f,%f,%f,%f,%f",&a1,&a2,&a3,&a4,&a5,&a6);
               fprintf(file,"%d\t%f\t%f\t%f\t%f\t%f\t%f\n", t,a1,a2,a3,a4,a5,a6);
            }
            fclose (file);

            // reset MoCoName to write only once if several MoCoSeries are present
            strcpy(MoCoName,"");
         }
         else VWarning ("Failed to open motion output file '%s'", MoCoName);
      }

   }


   /* extract image axes wrt Dicom (images are sorted in positive direction!) */
   U = NAxis; V = NAxis;
   dataset = images[0]->getDataset ();
   dataset->findAndGetString (DCM_ImageOrientationPatient, str);
   if (str) sscanf (str, "%f\\%f\\%f\\%f\\%f\\%f", &U[0], &U[1], &U[2], &V[0], &V[1], &V[2]);
   W = Abs (U * V);

   /* compute image axes wrt Lipsia (x-component and z-component inverted!) */
   U[0] = -U[0]; U[2] = -U[2];
   V[0] = -V[0]; V[2] = -V[2];
   W[0] = -W[0]; W[2] = -W[2];

   /* compute principal axes */
   X = Round (U);
   Y = Round (V);
   Z = Round (W);

   /* swap and flip image to natural orientation */
   xyswap = xflip = yflip = zflip = FALSE;
   if (Z != NAxis)
   {
      /* compute swap to natural orientation */
      if (((Z | XAxis) &&                    /* Dicom z is Lipsia x (ie. sagittal slices), for natural orientation */
          !(X | YAxis) && !(Y | ZAxis)) ||   /* Dicom x should be Lipsia y and Dicom y should be Lipsia z          */
          ((Z | YAxis) &&                    /* Dicom z is Lipsia y (ie. coronal slices), for natural orientation  */
          !(X | XAxis) && !(Y | ZAxis)) ||   /* Dicom x should be Lipsia x and Dicom y should be Lipsia z          */
          ((Z | ZAxis) &&                    /* Dicom z is Lipsia z (ie. axial slices), for natural orientation    */
          !(X | XAxis) && !(Y | YAxis)))     /* Dicom x should be Lipsia x and Dicom y should be Lipsia y          */
      {
         /* swap axes */
         xyswap = TRUE;
         swap = U; U = V; V = swap;
         swap = X; X = Y; Y = swap;
         tmp = voxel[0]; voxel[0] = voxel[1]; voxel[1] = tmp;
         tmp = grid [0]; grid [0] = grid [1]; grid [1] = tmp;
      }

      /* compute flip to natural orientation (all Dicom axes should be positive Lipsia axes) */
      if ((X == -XAxis) || (X == -YAxis) || (X == -ZAxis)) {xflip = TRUE; U = -U; X = -X;}
      if ((Y == -XAxis) || (Y == -YAxis) || (Y == -ZAxis)) {yflip = TRUE; V = -V; Y = -Y;}
      if ((Z == -XAxis) || (Z == -YAxis) || (Z == -ZAxis)) {zflip = TRUE; W = -W; Z = -Z;}


      /* swap and flip images */
      for (VFirstAttr (Images, &pos); VAttrExists (&pos); VNextAttr (&pos))
      {
         VGetAttrValue (&pos, NULL, VImageRepn, &image);
         if (xyswap) Swap<VShort> (image, XYSwap);
         if (xflip)  Flip<VShort> (image, XFlip);
         if (yflip)  Flip<VShort> (image, YFlip);
         if (zflip)  Flip<VShort> (image, ZFlip);
         VSetAttrValue (&pos, NULL, VImageRepn, image);
      }
   }


   /* extract attributes */
   list = VCreateAttrList ();
   ExtractAttributes (images[0], list);

   /* append geometry */
   sprintf (attr, "%g %g %g", voxel[0], voxel[1], voxel[2]);
   // VAppendAttr (list, "voxel", NULL, VStringRepn, attr);
   VAppendAttr (list, "lattice", NULL, VStringRepn, attr);

   sprintf (attr, "%g %g %g", grid[0], grid[1], grid[2]);
   // VAppendAttr (list, "grid", NULL, VStringRepn, attr);
   VAppendAttr (list, "voxel", NULL, VStringRepn, attr);

   //append bandtype attribute
   if(Functional) {
	      VAppendAttr (list, "bandtype", NULL, VStringRepn, "temporal");
   } else {
	      VAppendAttr (list, "bandtype", NULL, VStringRepn, "spatial");
   }

   /* append dynamics */
   if (Functional)
   {
      sprintf (attr, "%g", duration);
      VAppendAttr (list, "duration", NULL, VStringRepn, attr);
      sprintf (attr, "%g", repetition);
      VAppendAttr (list, "repetition", NULL, VStringRepn, attr);
   }

   /* append axes, angle, orientation and convention */
   if (Z != NAxis)
   {
      sprintf (attr, "%f %f %f", U[0], U[1], U[2]);
      VAppendAttr (list, "x-axis", NULL, VStringRepn, attr);
      sprintf (attr, "%f %f %f", V[0], V[1], V[2]);
      VAppendAttr (list, "y-axis", NULL, VStringRepn, attr);
      sprintf (attr, "%f %f %f", W[0], W[1], W[2]);
      VAppendAttr (list, "z-axis", NULL, VStringRepn, attr);
      sprintf (attr, "%f %f %f", Angle (XAxis, U), Angle (YAxis, V), Angle (ZAxis, W));
      VAppendAttr (list, "angle", NULL, VStringRepn, attr);
      /* orientation is overwritten, if the image is rotated */
      if (Z | XAxis) VAppendAttr (list, "acquisitionOrientation", NULL, VStringRepn, "sagittal");
      if (Z | YAxis) VAppendAttr (list, "acquisitionOrientation", NULL, VStringRepn, "coronal");
      if (Z | ZAxis) VAppendAttr (list, "acquisitionOrientation", NULL, VStringRepn, "axial");
      if (Z | XAxis) VAppendAttr (list, "orientation", NULL, VStringRepn, "sagittal");
      if (Z | YAxis) VAppendAttr (list, "orientation", NULL, VStringRepn, "coronal");
      if (Z | ZAxis) VAppendAttr (list, "orientation", NULL, VStringRepn, "axial");
      VAppendAttr (list, "convention", NULL, VStringRepn, "natural");
   }

   /* set attributes */
	if(!Dti) {

		// not dti and not functional - meaning anatomical - and this
		// means we only have one object in the list
		// so, we see if we've got multiple echo times and (if we have)
		// create one object for each echo time
		if(!Functional) {
			float echoTimes[N];
			std::list<float> echoTimesList;

			ExtractEchoTimes(images, N, echoTimes, &echoTimesList);

			if(echoTimesList.size() > 1) {
				std::cerr << "(MR xx) Found " << echoTimesList.size() << " different echo times" << endl;

				// we (should) only have one Vista image at this point
				VFirstAttr(Images, &pos);
				VGetAttrValue(&pos, NULL, VImageRepn, &image);


				// create a new list
				Images = VCreateAttrList();


				std::cerr << "(MR xx) Extracted image (" << VImageNColumns(image) << "x" << VImageNRows(image) << "x" << VImageNBands(image) << ")" << endl;

				// now it's going to get a wee bit dodgy - but it achieves what we want
				std::list<float>::iterator _eTime = echoTimesList.begin();
				VImage _etImages[echoTimesList.size()];
				int _eti = 0;
				while(_eTime != echoTimesList.end()) {
					std::cerr << "(MR xx) ... echo time found: " << *_eTime << endl;

					// find out the number of bands having the current echo time
					// (we need this number to create the new image)
					int _etBands = 0;
					for(int _b = 0; _b < N; _b++) {
						if(echoTimes[_b] == *_eTime) {
							_etBands++;
						}
					}

					// create a new image for the current echo time
					_etImages[_eti] = VCreateImage(_etBands, VImageNRows(image), VImageNColumns(image), VPixelRepn(image));

					std::cerr << "(MR xx) ... created new image (" << VImageNColumns(_etImages[_eti]) << "x" << VImageNRows(_etImages[_eti]) << "x" << VImageNBands(_etImages[_eti]) << ")" << endl;

					// copy the existing attributes to the newly created image and set the 'echoTime'
					// parameter accordingly
					std::cerr << "(MR xx) ... copying existing image attributes and setting echo time" << endl;
					VImageAttrList(_etImages[_eti]) = VCopyAttrList(list);
                    VSetAttr(VImageAttrList(_etImages[_eti]), "numberOfEchoTimes", NULL, VShortRepn, echoTimesList.size());
                    VSetAttr(VImageAttrList(_etImages[_eti]), "echoTime", NULL, VFloatRepn, *_eTime);

					// copy all bands with the current echo time to the new image
					std::cerr << "(MR xx) ... copying " << VImageNBands(_etImages[_eti]) << " bands to new image" << endl;
					int _ni = 0;
					for(int _b = 0; _b < N; _b++) {
						if(echoTimes[_b] == *_eTime) {
                            if(zflip == TRUE) {
                                // TS (06/04/06):
                                // if 'zflip' is true then the slice order is reversed at this point
                                // and we have to correct the index used to extract the echo time specific
                                // slices
                                // NOTE: this is not thoroughly tested but seems to work for the datasets
                                //       we've got to deal with
                                int _correctedIndex = _etBands * (echoTimesList.size() - _eti - 1) + (_b - (_etBands * _eti));
//                              std::cerr << "corrected index for " << _b << ": " << _correctedIndex << endl;
                                VCopyBand(image, _correctedIndex, _etImages[_eti], _ni);
                            } else {
   		       					VCopyBand(image, _b, _etImages[_eti], _ni);
                            }
							_ni++;
						}
					}

					++_eTime;
					++_eti;
				}

				// add the new images to the list
				for(int _i = 0; _i < echoTimesList.size(); _i++) {
					VAppendAttr(Images, "image", NULL, VImageRepn, _etImages[_i]);
				}
			} else {
				VFirstAttr(Images, &pos);
				VImageAttrList(image) = VCopyAttrList(list);
				VSetAttrValue (&pos, NULL, VImageRepn, image);
			}
		} else {
			for(VFirstAttr (Images, &pos); VAttrExists (&pos); VNextAttr (&pos)) {
				VGetAttrValue (&pos, NULL, VImageRepn, &image);
				VImageAttrList (image) = VCopyAttrList (list);
				VSetAttrValue (&pos, NULL, VImageRepn, image);
			}
		}
   } else {
      fprintf(stderr,"Nb of dti diffusion directions = %d\n",diffusionDirections); fflush (stderr);

      for (VFirstAttr (Images, &pos), t=0; VAttrExists (&pos); VNextAttr (&pos), t++)
      {
         VGetAttrValue (&pos, NULL, VImageRepn, &image);
         VImageAttrList (image) = VCopyAttrList (list);
         VSetAttrValue (&pos, NULL, VImageRepn, image);

         // overwrite sequence name for dti images (different seqences in one series)
         strcpy (sequence, "unknown");
         if (t * (int)size[2]< N) // check if index is not out of rage
            // get the sequence name
            images[t * (int)size[2]]->getDataset ()->findAndGetString (DCM_SequenceName, str);
         if (str) strcpy (sequence, str);
         fprintf(stderr,"Sequence = %s\n", sequence); fflush (stderr);

         VSetAttr(VImageAttrList(image),"sequence", NULL, VStringRepn, sequence);
      }
   }

   /* clean-up */
   for (n = 0; n < N; n++)
      delete images[n];
   delete[] images;
   delete[] location;
   delete[] time;
   VDestroyAttrList (list);

   return TRUE;

} /* ConvertSeries */

/*----------------------------------------------------------------------------*/

VBoolean WriteImages (VString Name, int noptions, VOptionDescRec *options, VAttrList Images)
{
   FILE*         file;      /* output file      */
   VAttrList     list;      /* attribute list   */
   VAttrListPosn pos;       /* position in list */
   VImage        image;     /* image in list    */
   VBoolean      success;   /* success flag     */


   /* open file */
   file = fopen (Name, "w");
   if (!file)
   {
      VError ("Failed to open output file '%s'", Name);
      return FALSE;
   }

   /* create attr list */
   list = VCreateAttrList();

   /* add history */
   VHistory(noptions,options,prg_name,&list,&list);

   /* insert images */
   for (VFirstAttr (Images, &pos); VAttrExists (&pos); VNextAttr (&pos))
   {
      VGetAttrValue (&pos, NULL, VImageRepn, &image);
      VAppendAttr (list, VGetAttrName (&pos), NULL, VImageRepn, image);
   }

   /* write file */
   success = VWriteFile (file, list);
   if (!success) VError ("Failed to write output file '%s'", Name);

   /* remove images */
   for (VFirstAttr (list, &pos); VAttrExists (&pos); VNextAttr (&pos))
      if (VGetAttrRepn (&pos) == VImageRepn)
         VSetAttrValue (&pos, NULL, VImageRepn, NULL);

   /* clean-up*/
   VDestroyAttrList (list);
   fclose (file);

   return success;

} /* WriteImages */

/*----------------------------------------------------------------------------*/

int main (int argc, char* argv[])
{
   VString        inname;           /* name of input image  */
   VString        outname;          /* name of output image */
   VArgVector     indices;          /* indices of scans     */
   VShort         precision = -1;   /* precision of voxels  */
   VShort         steps = 0;        /* functional timesteps */
   VShort         mosaic = 0;       /* size of mosaic       */
   VShort         tiles = 0;        /* tiles in mosaic      */
   VLong          fscale = 1;       /* scaling factor for functional data */
   VBoolean       verbose = FALSE;   /* verbose flag         */
   VBoolean       movlog = FALSE;   /* flag to log the MoCoSeries */
   VFloat         black = 0.1;      /* lower histogram cut-off for contrast scaling */
   VFloat         white = 0.1;      /* upper histogram cut-off for contrast scaling */
   VDictEntry     values[] =        /* precision values     */
   {
       { "auto", -1 },
       { "original", 0 },
       { "single", 1 },
       { "double", 2 },
       { NULL }
   };
   VOptionDescRec options[] =       /* options of program   */
   {
      {"in",        VStringRepn,  1, &inname,    VRequiredOpt, NULL,   "Input file in DicomDirectory format"},
      {"out",       VStringRepn,  1, &outname,   VRequiredOpt, NULL,   "Output image in Vista format"},
      {"exp",       VShortRepn,   0, &indices,   VOptionalOpt, NULL,   "Indices of scans to be converted. Optional"},
      {"precision", VShortRepn,   1, &precision, VOptionalOpt, values, "Bytes per voxel in anatomical scans. Optional"},
      {"movlog",    VBooleanRepn, 1, &movlog,    VOptionalOpt, NULL,   "Write motion correction log file for one MoCoSeries. (*.mot). Optional"},
      {"steps",     VShortRepn,   1, &steps,     VOptionalOpt, NULL,   "Number of functional timesteps. Use with caution! Optional"},
      {"mosaic",    VShortRepn,   1, &mosaic,    VOptionalOpt, NULL,   "Edge size of mosaic matrix. Use with caution! Optional"},
      {"tiles",     VShortRepn,   1, &tiles,     VOptionalOpt, NULL,   "Number of tiles in mosaic matrix. Use with caution! Optional"},
      {"black",     VFloatRepn,   1, &black,     VOptionalOpt, NULL,   "Lower histogram cut-off for contrast scaling in %"},
      {"white",     VFloatRepn,   1, &white,     VOptionalOpt, NULL,   "Upper histogram cut-off for contrast scaling in %"},
      {"fscale",    VLongRepn,    1, &fscale,    VOptionalOpt, NULL,   "Scaling factor for functional data"},
      {"verbose",   VBooleanRepn, 1, &verbose,   VOptionalOpt, NULL,   "Show status messages. Optional"}
   };

   DcmDicomDir*  In;       /* input images     */
   VAttrList     Out;      /* output images    */
   VAttrList     images;   /* list of images   */
   VAttrListPosn pos;      /* position in list */
   VImage        image;    /* image in list    */

   DcmStack patients;   /* patient stack  */
   DcmStack studies;    /* study stack    */
   DcmStack series;     /* series stack   */
   int      size;       /* size of stacks */

   DcmDirectoryRecord** scans = NULL;   /* list of scans */
   int                  index;          /* index of scan */
   int                  seriesNb;       /* series Number of scan */

   char directory[257];     /* base directory    */
   char description[257];   /* image description */
   char MoCoName[257];      /* Name for motion log file */

   VBoolean functional;  /* functional flag */
   VBoolean dti;         /* dti flag */
   VBoolean success;     /* success flag    */

   VString attr;      /* image attribute */
   char    str[65];   /* image attribute */
	 // float   x, y, z;   /* coordinates     */

   int n, i;   /* indices */
   VAttrList history_list=NULL;

   /* print information */
   sprintf(prg_name,"dictov V%s", getLipsiaVersion());

   fprintf (stderr, "%s\n", prg_name); fflush (stderr);

   /* parse command line */
   if (!VParseCommand (VNumber (options), options, &argc, argv))
   {
      if (argc > 1) VReportBadArgs (argc, argv);
      VReportUsage (argv[0], VNumber (options), options, NULL);
      exit (1);
   }

   /* set name for MoCo-Motion out_file if -movlog TRUE else set to "" */
   std::string MoCoString(outname);
   if (movlog) {
      MoCoString = MoCoString.substr(0,MoCoString.rfind("."))+".mot";
	  strncpy(MoCoName,MoCoString.c_str(),256);
   }
   else
      strcpy(MoCoName,"");

   /* read input image */
   if (verbose) {fprintf (stderr, "Reading input image '%s' ...\n", inname); fflush (stderr);}
   GetDirectory (inname, directory);
   ReadImage (inname, In);
   if (!In) exit (2);

   /* search patients */
   SearchRecords (&(In->getRootRecord ()), ERT_Patient, patients, size);
   while (!patients.empty ())
   {
      /* search studies */
      SearchRecords ((DcmDirectoryRecord*) patients.pop (), ERT_Study, studies, size);
      while (!studies.empty ())
      {
         /* search series */
         SearchRecords ((DcmDirectoryRecord*) studies.pop (), ERT_Series, series, size);
         scans = new DcmDirectoryRecord*[size];
         for (n = 0; n < size; n++)
            scans[n] = (DcmDirectoryRecord*) series.pop ();
      }
   }

   /* check scans */
   if (size == 0) {VError ("Input image does not contain any scans"); exit (3);};

   /* sort scans */
   SortRecords (directory, scans, size, DCM_SeriesTime);

   /* fake indices */
   if (indices.number == 0)
   {
      indices.number = size;
      indices.vector = (VPointer) malloc (size * sizeof (VShort*));
      for (n = 0; n < indices.number; n++)
         ((VShort*) indices.vector)[n] = n + 1;
   }

   /* check indices */
   for (n = 0; n < indices.number; n++)
   {
      index = ((VShort*) indices.vector)[n];
      if (index < 1)    {VError ("Index of scan must be greater than or equal to 1");     exit (4);};
      if (index > size) {VError ("Index of scan must be less than or equal to %d", size); exit (5);};
   }

   /* check mosaic geometry */
   if (tiles > mosaic * mosaic) {VError ("An %dx%d matrix cannot contain %d tiles", mosaic, mosaic, tiles); exit (6);};


   /* convert scans */
   Out = VCreateAttrList ();
   for (n = 0; n < indices.number; n++)
   {
      /* convert series */
      index = ((VShort*) indices.vector)[n];
      if (verbose) {fprintf (stderr, "Converting scan ...\n"); fflush (stderr);}
      success = ConvertSeries (directory, scans[index - 1], steps, mosaic, tiles, images,
	      functional, dti, MoCoName, inname);
	  // TS (07/12/05):
	  // just ignore the scan if we weren't able to convert it instead
	  // of aborting the whole thing
	  // TODO: check with 'someone more competent' if this is OK
      if(!success) {
		 fprintf(stderr, "[WARNING] (MR %02d) Scan ignored ...\n", index);
		 fflush(stderr);
		 continue;
      }
      // if (!success) exit (7);

      /* print status */
      if (verbose)
      {
         if (functional)
            fprintf (stderr, "(MR %02d) Marking scan as FUNCTIONAL ...\n", index);
         else if (dti)
            fprintf (stderr, "(MR %02d) Marking scan as DIFFUSION ...\n", index);
         else
            fprintf (stderr, "(MR %02d) Marking scan as ANATOMICAL ...\n", index);
         fflush (stderr);
      }

      /* for historical reasons, axial images have to be upside-down! */
      for (VFirstAttr (images, &pos); VAttrExists (&pos); VNextAttr (&pos))
      {
         VGetAttrValue (&pos, NULL, VImageRepn, &image);
         VGetAttr (VImageAttrList (image), "orientation", NULL, VStringRepn, &attr);
         if (attr && (strcmp (attr, "axial") == 0))
         {
            /* flip image (violates right-handiness of coordinate system !) */
            Flip<VShort>  (image, ZFlip);

            // Flip<VShort>  (image, XFlip);  /* flip to anatomical convention, G.Lohmann, 16.3.03 */

            VSetAttrValue (&pos, NULL, VImageRepn, image);

//          /* flip z-axis (violates right-handiness of coordinate system !) */
//          VGetAttr (VImageAttrList (image), "z-axis", NULL, VStringRepn, &attr);
//          sscanf (attr, "%f %f %f", &x, &y, &z);
//          sprintf (str, "%f %f %f", -x, -y, -z);
//          VSetAttr (VImageAttrList (image), "z-axis", NULL, VStringRepn, &str);
         }
      }

      /* post-process functional scan */
      if (functional)
      {
         /* reorder data */
         if (verbose) {fprintf (stderr, "(MR %02d) Reordering data ...\n", index); fflush (stderr);}
         MajorAnatomical<VShort> (images);

	 /* normalize functional data */
	 if (fscale>1)
	   VNormVals(images,fscale);
      }

      /* post-process diffusion scan */
      else if (dti)
      {
         if (precision == 1) {
         /* linear contrast */
            for (i = 1, VFirstAttr (images, &pos); VAttrExists (&pos); VNextAttr (&pos), i++)
            {
               /* get image */
               VGetAttrValue (&pos, NULL, VImageRepn, &image);

               if (verbose) {fprintf (stderr, "(MR %02d-%02d) Linear contrast ...\n", index, i); fflush (stderr);}
               LinearContrast<VShort> (image, 0, 255, black / 100.0, white / 100.0);
               /* convert type */
               if (verbose) {fprintf (stderr, "(MR %02d-%02d) Converting type ...\n", index, i); fflush (stderr);}
               ConvertType<VShort,VUByte> (image, VUByteRepn);

               /* set image */
               VSetAttrValue (&pos, NULL, VImageRepn, image);
            }
         }
      }

      /* post-process anatomical scan */
      else
      {
          /* get images */
          for (i = 1, VFirstAttr (images, &pos); VAttrExists (&pos); VNextAttr (&pos), i++) {
	          VGetAttrValue (&pos, NULL, VImageRepn, &image);

	          /* linear scaling */
	          if (verbose) {fprintf (stderr, "(MR %02d-%02d) Linear scaling ...\n", index, i); fflush (stderr);}
	          if (precision == -1) LinearContrast<VShort> (image, 0, 255, black / 100.0, white / 100.0);
	          if (precision ==  1) LinearScale<VShort>    (image, 255.0 / 32768, 0);

	          /* convert type */
	          if (verbose) {fprintf (stderr, "(MR %02d-%02d) Converting type ...\n", index, i); fflush (stderr);}
	          if ((precision == -1) || (precision == 1))
	        	 ConvertType<VShort,VUByte> (image, VUByteRepn);

	          /* set image */
	          VSetAttrValue (&pos, NULL, VImageRepn, image);
          }
      }

      /* store scan */
      for (i = 1, VFirstAttr (images, &pos); VAttrExists (&pos); VDeleteAttr (&pos), i++)
      {
         /* get image */
         VGetAttrValue (&pos, NULL, VImageRepn, &image);
         VSetAttrValue (&pos, NULL, VImageRepn, NULL);

         /* store image */
         if (functional)
            sprintf (description, "MR %02d-%02d FUNCTIONAL", index, i);
         else if (dti)
            sprintf (description, "MR %02d-%02d DIFFUSION", index, i);
         else
            sprintf (description, "MR %02d-%02d ANATOMICAL", index, i);
         VPrependAttr (VImageAttrList (image), "name", NULL, VStringRepn, description);
         VAppendAttr (Out, "image", NULL, VImageRepn, image);
      }

      /* clean-up*/
      VDestroyAttrList (images);
   }

   /* write output images */
   if (verbose) {fprintf (stderr, "Writing output image '%s' ...\n", outname); fflush (stderr);}
   success = WriteImages (outname, VNumber (options), options, Out);
   if (!success) exit (8);


   /* clean-up */
   delete In;
   VDestroyAttrList (Out);
   delete[] scans;
   delete[] (VShort*) indices.vector;

   /* exit */
   if (verbose) {fprintf (stderr, "Finished.\n"); fflush (stderr);}
   return 0;

} /* main */


/*******************************************************************************

  $Log$
  Revision 1.29  2007/03/05 10:24:53  schlumm
  - [TS] removed typos in option descriptions

  Revision 1.28  2007/03/05 10:21:47  schlumm
  - [TS] added -black and -white options for contrast scaling

  Revision 1.27  2007/03/05 10:11:42  schlumm
  - [TS] removed debug output and the (commented out) buggy line

  Revision 1.26  2007/03/05 09:43:04  schlumm
  - [TS] fixed repetition time issue (using RepetitionTime DICOM parameter
         if present - and removing this old, old line 'repetition = duration'
         which is undoubtedly a bug that didn't surface until now because of
         some fancy stuff Siemens did with the AcqusitionTime parameter in VB13)

  Revision 1.25  2007/02/28 12:16:40  karstenm
  *** empty log message ***

  Revision 1.24  2007/02/22 09:27:15  karstenm
  *** empty log message ***

  Revision 1.23  2007/02/21 16:19:14  karstenm
  *** empty log message ***

  Revision 1.22  2007/02/21 13:21:27  karstenm
  *** empty log message ***

  Revision 1.21  2006/04/06 13:49:31  schlumm
  - added 'numberOfEchoTimes' parameters for multi-echo 3D
    datasets to better support automatic data conversion/processing

  Revision 1.20  2006/04/06 13:13:04  schlumm
  - fixed slice order issue with multi-echo 3D datasets

  Revision 1.19  2006/04/05 13:19:42  schlumm
  - fixing issue with voxel parameter values for multi-echo 3D datasets

  Revision 1.18  2006/03/24 12:47:12  schlumm
  - TS (24/03/2006): added support for 'multi-echo' anatomical data
    (in the presence of multiple echo times within one scan one Vista
    image per echo time is being created)

  Revision 1.17  2005/09/02 07:53:03  karstenm
  *** empty log message ***


  Revision 1.15  2004/07/27 09:43:19  anwander
  added support for post-processed MoCoSeries

  Revision 1.14  2004/07/22 14:32:36  anwander
  added functions from dcitem.cc to link without errors

  Revision 1.13  2004/07/22 12:51:07  anwander
  added support for dcmtk3.5.3

  Revision 1.12  2004/07/12 08:37:01  anwander
  added coil identification

  Revision 1.9  2004/01/28 07:51:36  anwander
  added optional write out of motion parameters

  Revision 1.8  2003/11/05 17:29:22  anwander
  corrected slicetime for interleaved mode with odd number of slices

  Revision 1.7  2003/10/23 12:26:17  anwander
  Changed nb. steps in anat. images from 0 to 1 for better support of mosaic images

  Revision 1.6  2003/10/23 11:04:55  anwander
  Adapted for dcmtk 3.5.2 and Acquisition Numbers > 1 in non funkt. scan

  Revision 1.5  2003/09/17 06:46:37  anwander
  Added support for dti images

  Revision 1.4  2003/09/11 12:22:25  anwander
  correct bug while selecting steps in fmri

  Revision 1.3  2003/09/09 09:26:53  anwander

  corrected slice time and ordering for fmri data

 *******************************************************************************/
