/*
 *  load.c: Load related functions, list management etc.
 *
 *  Copyright (C) 1997-2005 John Coppens (john@jcoppens.com)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <libxml/tree.h>
#include <gnome.h>
#include <unistd.h>

#include "support.h"
#include "callbacks.h"
#include "global.h"
#include "main.h"
#include "load.h"
#include "chart.h"
#include "aux.h"

char *ld_rbtns[] = {
  "load_rx_rbtn", "load_rc_rbtn", "load_rl_rbtn",
  "load_bg_rbtn", "load_rcpar_rbtn", "load_rlpar_rbtn"
};

typedef struct {
  char *hdr;
  int w;
} loaddef_entry;

loaddef_entry loaddef[] = {
  {N_("F (MHz)"), 80}, 
  {N_("Real"),    60}, 
  {N_("Imag"),    50}
};
#define ldef_len sizeof(loaddef)/sizeof(loaddef_entry)

GtkTreeIter	active_ld_iter;
load_definition	active_load;
GtkWidget	*ld_view;


//---------------------------------------------------------------------
//	Loads
//---------------------------------------------------------------------

gint
compare_frequencies(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
			gpointer user_data)
{
  double f1, f2;

  gtk_tree_model_get(model, a, LD_FREQ_VAL, &f1, -1);
  gtk_tree_model_get(model, b, LD_FREQ_VAL, &f2, -1);

  if (f1 < f2)
    return -1;
  else
    if (f1 == f2)
      return 0;
    else
      return 1;
}


void
initialize_loadtable(void)
{
  GtkListStore *loads;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *col, *sortcol;
  int def;
  
  loads = gtk_list_store_new(LD_COLS, 
                             G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                             G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,
                             G_TYPE_POINTER, G_TYPE_POINTER);
  
  ld_view = lookup_widget(MainWindow, "load_treeview");
  gtk_tree_view_set_model(GTK_TREE_VIEW(ld_view), GTK_TREE_MODEL(loads));
  g_object_unref(G_OBJECT(loads));

  for (def = 0; def < ldef_len; def++) {
    renderer = gtk_cell_renderer_text_new();
    gtk_object_set(GTK_OBJECT(renderer), 
		"height", 14, "ypad", 0, 
    		"xalign", 1.0, NULL);
    col = gtk_tree_view_column_new_with_attributes(
		_(loaddef[def].hdr),
		renderer, "text", def,
		NULL);
    if (def == 0) sortcol = col;
    gtk_tree_view_append_column(GTK_TREE_VIEW(ld_view), col);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(col),
                GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(col),
                loaddef[def].w);
  }

  gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(loads), LD_FREQ_VAL,
		compare_frequencies, NULL, NULL);
  gtk_tree_view_column_set_sort_column_id(sortcol, LD_FREQ_VAL);
}


void
load_buttons_sensitive(gboolean yes)
{
  GtkWidget *w;

  w = lookup_widget(MainWindow, "load_remove_btn");
  gtk_widget_set_sensitive(w, yes);
  w = lookup_widget(MainWindow, "load_edit_btn");
  gtk_widget_set_sensitive(w, yes);
//  w = lookup_widget(MainWindow, "load_setref_btn");
//  gtk_widget_set_sensitive(w, yes);
}


void
on_load_selected(GtkTreeView *treeview)
{
  GtkWidget *w;
  GtkTreeModel *store;
  GtkTreeSelection *sel;
  char bff[20];

  sel = gtk_tree_view_get_selection(treeview);
  if (gtk_tree_selection_get_selected(sel, &store, &active_ld_iter)) {
    gtk_tree_model_get(store, &active_ld_iter, 
                   LD_FREQ_VAL,  &active_load.f, 
                   LD_REAL_VAL,  &active_load.r, 
                   LD_REACT_VAL, &active_load.x,
                   LD_POINT,     &active_load.pt,
		   -1);

    sprintf(bff, "%.*f", pref.prec_mhz, active_load.f);
    w = lookup_widget(MainWindow, "load_freq_entry");
    gtk_entry_set_text(GTK_ENTRY(w), bff);

    sprintf(bff, "%.2f", active_load.r);
    w = lookup_widget(MainWindow, "load_r_entry");
    gtk_entry_set_text(GTK_ENTRY(w), bff);

    sprintf(bff, "%.2f", active_load.x);
    w = lookup_widget(MainWindow, "load_x_entry");
    gtk_entry_set_text(GTK_ENTRY(w), bff);

    load_buttons_sensitive(TRUE);
  }
}


int
load_valid(void)
{
  GtkWidget *w;
  char *fin = NULL;

  w = lookup_widget(MainWindow, "load_freq_entry");
  active_load.f = atof(gtk_entry_get_text(GTK_ENTRY(w)));

  w = lookup_widget(MainWindow, "load_r_entry");
  active_load.r = atof(gtk_entry_get_text(GTK_ENTRY(w)));

  w = lookup_widget(MainWindow, "load_x_entry");
  active_load.x = atof(gtk_entry_get_text(GTK_ENTRY(w)));

  return TRUE;
}


void
load_update(void)
{
  GtkListStore *store;
  char r[20], x[20], f[20];
  
  if (active_ld_iter.stamp == 0) return;
  store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(ld_view)));

  sprintf(f, "%.*f", pref.prec_mhz, active_load.f);
  sprintf(r, "%.1f", active_load.r);
  sprintf(x, "%.1f", active_load.x);
  set_point(active_load.pt, active_load.r + active_load.x*I, ZPT_LOAD);
  gtk_list_store_set(store, &active_ld_iter,
                      LD_FREQ, f,
                      LD_REAL, r,
                      LD_REACT, x,
                      LD_FREQ_VAL, active_load.f,
                      LD_REAL_VAL, active_load.r,
                      LD_REACT_VAL, active_load.x,
                      LD_POINT, active_load.pt,
                      -1);
}


void
active_load_update(void)
{
  if (active_ld_iter.stamp == 0) return;
  load_update();
}


void
load_append()
{
  GtkTreeIter iter;
  GtkTreeModel *ldstore;
  
  ldstore = gtk_tree_view_get_model(GTK_TREE_VIEW(ld_view));
  gtk_list_store_append(GTK_LIST_STORE(ldstore), &active_ld_iter);

  active_load.pt = create_load_point();
  load_update();
}


void
active_load_remove(void)
{
  GtkListStore *store;
  chart_point *pt;

  if (active_ld_iter.stamp == 0) return;

  store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(ld_view)));
  gtk_tree_model_get(GTK_TREE_MODEL(store), &active_ld_iter,
                          LD_POINT, &pt, -1);

  if (pt) {
    gtk_object_destroy(GTK_OBJECT(pt->point));
    g_free(pt);
  }
                              
  gtk_list_store_remove(store, &active_ld_iter);

  if (gtk_tree_model_iter_n_children(GTK_TREE_MODEL(store), NULL) == 0)
    load_buttons_sensitive(FALSE);
}


gboolean
save_each_load(GtkTreeModel *model,
               GtkTreePath  *path,
               GtkTreeIter  *iter,
               gpointer      user_data)
{
  load_definition load;
  FILE *xmlf = user_data;
                                         
  gtk_tree_model_get(model, iter,
                   LD_FREQ_VAL, &load.f,
                   LD_REAL_VAL, &load.r,
                   LD_REACT_VAL, &load.x,
		   -1);

  fprintf(xmlf, " <load f=\"%f\" r=\"%f\" x=\"%f\" />\n",
                load.f, load.r, load.x);

  return FALSE;
}


void
save_loads(char *fn)
{
  GtkWidget *loads = lookup_widget(MainWindow, "load_treeview");
  GtkTreeModel *store = gtk_tree_view_get_model(GTK_TREE_VIEW(loads));
  FILE *xmlf;
    
  if ((xmlf = fopen(fn, "r"))) {
    fclose(xmlf);
    if (ok_cancel_dialog(_("Confirmation"), 
			 _("File exists - want to overwrite?"))
        == 1) {
      return;
    }
  }
  xmlf = fopen(fn, "w");
  if (!strcmp(fn, "")) return;
  fprintf(xmlf, "<?xml version=\"1.0\"?>\n");
  fprintf(xmlf, "<loads>\n");
  gtk_tree_model_foreach(store, save_each_load, xmlf);
  fprintf(xmlf, "</loads>\n");
  fclose(xmlf);
}


void
load_loads(char *fn)
{
  load_definition load;
  xmlChar *dbl;
  xmlDocPtr doc;
  xmlNodePtr cur;

  if (!xml_file_loaded(&doc, fn, "loads")) return;

  cur = xmlDocGetRootElement(doc);
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if ((!xmlStrcmp(cur->name, (const xmlChar *) "load"))) {
      active_load.f = atof(dbl = xmlGetProp(cur, "f")); xmlFree(dbl);
      active_load.r = atof(dbl = xmlGetProp(cur, "r")); xmlFree(dbl);
      active_load.x = atof(dbl = xmlGetProp(cur, "x")); xmlFree(dbl);
      load_append();
    }
    cur = cur->next;
  }
	
  xmlFreeDoc(doc);
}


int
number_loads(void)
{
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(ld_view));

  return gtk_tree_model_iter_n_children(model, NULL);
}
