///  @file		main.cpp
///  @brief 	main source file
///  @author	mls programer
///  @date		2004-

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <ncurses.h>
#include <fcntl.h>

#include <iomanip>
#include <vector>
#include <deque>

#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <getopt.h>

#include "define.h"
#include "mainframe.h"

#include "configure.h"
#include "mlslocale.h"
#include "colorset.h"

#include "editor.h"

using namespace MLS;
using namespace std;

#ifdef _DEBUG
bool g_Log = false;
#endif

// define 에 extern 으로 선언
int	g_nLINES = 0;
int	g_nCOLS = 0;

namespace { // 처음 기본 세팅

bool		_bMcdExe  = false;	    ///< MCD실행 여부
ENCODING	_nLangSet = AUTO;       ///< 언어 설정
LINECODE	_nLineSet = ACSLINE;	///< LINE MODE

vector<string> _CfgFile, _ColFile, _KeyFile, _EditorKeyFile;

}

namespace MLS {

extern deque<string> g_CmdHistory;

bool Initialize();
void Destroy();

/// @brief 시그널 처리를 하는 함수
/// @param sig 시그널 번호
void signal_action(int sig)
{
# if (__GNUC__ * 1000 + __GNUC_MINOR__) >= 3000
	LOG("Signal :: [%d] [%s]", sig, strsignal(sig));
# else
	LOG("Signal :: [%d]", sig);
# endif
	static int nSigCount = 0;

	switch(sig)
	{
		case SIGINT:
				// Copy & paste를 위해서 Ctrl-C를 사용합니다.
			ungetch(3);
			return;
	}

	g_MainFrame->Destroy();
	delete g_MainFrame;

	Destroy();

# if (__GNUC__ * 1000 + __GNUC_MINOR__) >= 3000
	printf("Signal [%d] [%s]\n", sig, strsignal(sig));
# else
	printf("Signal [%d]\n", sig);
# endif
	exit(0);
}

/// @brief 시그널블럭 처리를 한다.
/// @return 성공할 경우 SUCCESS
int signal_blocking()
{
	struct sigaction act;

	// 다음은 시그널이 들어왔을 때 실행 함수
	act.sa_handler = signal_action;

	// 다음 주석을 해제 하면 시그널 무시
	//act.sa_handler = SIG_IGN;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;

	sigaction(SIGINT, &act, 0);		/* 2 Interrupt (ANSI).  */
	
#ifndef _DEBUG
	sigaction(SIGHUP, &act, 0);		/* 1 Hangup (POSIX).  */
	sigaction(SIGQUIT, &act, 0);	/* 3 Quit (POSIX).  */
	sigaction(SIGILL, &act, 0);		/* 4 Illegal instruction (ANSI).  */
	sigaction(SIGTRAP, &act, 0);	/* 5 Trace trap (POSIX).  */
	sigaction(SIGABRT, &act, 0);	/* 6 Abort (ANSI).  */
	sigaction(SIGBUS, &act, 0);		/* 7 BUS error (4.2 BSD).  */
	sigaction(SIGFPE, &act, 0);		/* 8 Floating-point exception (ANSI).  */
//	sigaction(SIGKILL, &act, 0);	/* 9 Kill, unblockable (POSIX).  */
	sigaction(SIGUSR1, &act, 0);	/* 10 User-defined signal 1 (POSIX).  */
	sigaction(SIGSEGV, &act, 0);	/* 11 Segmentation violation (ANSI).  */
	sigaction(SIGUSR2, &act, 0);	/* 12 User-defined signal 2 (POSIX).  */
	sigaction(SIGPIPE, &act, 0);	/* 13 Broken pipe (POSIX).  */
	sigaction(SIGALRM, &act, 0);	/* 14 Alarm clock (POSIX).  */
	sigaction(SIGTERM, &act, 0);	/* 15 Termination (ANSI).  */
//	sigaction(SIGCHLD, &act, 0);	/* 17 Child status has changed (POSIX).  */
	sigaction(SIGCONT, &act, 0);	/* 18 Continue (POSIX).  */
	sigaction(SIGSTOP, &act, 0);	/* 19 Stop, unblockable (POSIX).  */
	sigaction(SIGTSTP, &act, 0);	/* 20 Keyboard stop (POSIX).  */
	sigaction(SIGTTIN, &act, 0);	/* 21 Background read from tty (POSIX).  */
	sigaction(SIGTTOU, &act, 0);	/* 22 Background write to tty (POSIX).  */
	sigaction(SIGURG, &act, 0);		/* 23 Urgent condition on socket (4.2 BSD).  */
	sigaction(SIGXFSZ, &act, 0);	/* 24 CPU limit exceeded (4.2 BSD).  */
	sigaction(SIGXCPU, &act, 0);	/* 25 File size limit exceeded (4.2 BSD).  */
	sigaction(SIGVTALRM, &act, 0);	/* 26 File size limit exceeded (4.2 BSD).  */
	sigaction(SIGPROF, &act, 0);	/* 27 Profiling alarm clock (4.2 BSD).  */
//	sigaction(SIGWINCH, &act, 0);	/* 28 Window size change (4.3 BSD, Sun).  */
	sigaction(SIGIO, &act, 0);		/* 29 I/O now possible (4.2 BSD). Pollable event occurred (System V).*/
	sigaction(SIGSYS, &act, 0);		/* 31 Bad system call. (Unused) */

	#ifdef LINUX
       sigaction(SIGSTKFLT, &act, 0);  	/* 16 Stack fault.  */
       sigaction(SIGPWR, &act, 0);    	/* 30 Power failure restart (System V).  */
	#endif
#endif
	return SUCCESS;
}

/// @brief	mls 시작시 초기화 함수
///
/// @return	성공여부 0일때 성공
bool Initialize()
{
	int t;
	const char *succMsg = "[\033[01;36m SUCCESS \033[0m]";
	const char *failMsg = "[\033[01;31m  FAIL   \033[0m]";
	const char *errMsg  = "[\033[01;31m  ERROR  \033[0m]";
	
	signal_blocking();	    // 시그널 블럭 처리

	char*	cwd = getcwd(NULL, 0);
	if (cwd == NULL)
	{
		cout << strerror(errno) << setw(60);
		cout << errMsg << endl;
		return false;
	}
	if (access(cwd, R_OK | X_OK) == -1)
	{
		cout << strerror(errno) << setw(60);
		cout << errMsg << endl;
		return false;
	}
	free(cwd);

	Set_Locale(_nLangSet); 	// Locale  설정

	{ // Config 준비..
		struct passwd *pw = getpwuid(getuid());
		{
			std::string home  = pw->pw_dir;
			home += '/';

			g_Config.SetStaticValue("home", home);
		}

		// . config dir 지정
		g_Config.SetStaticValue("cfghome", g_Config.GetValue("home") + ".mls/");

		if (g_Color.Init())
		{
			cout << "Initialize Configure" << setw(60);
			cout << succMsg << endl;
		}
		else
		{
			cout << "Initialize Configure" << setw(60);
			cout << failMsg << endl;
			return false;
		}

		// 홈에 .mls를 만든다. mcd treeinfo를 저장하기 위해서도 필요
		mkdir((g_Config.GetValue("home") + ".mls").c_str(), 0755);
	}

	{ // 설정 파일 읽기
		_CfgFile.push_back( g_Config.GetValue("cfghome") + "mls.cfg" );
		_CfgFile.push_back( "/etc/mls/mls.cfg");

		for (t=0; t<_CfgFile.size(); t++)
		{
			string &cfgfile = _CfgFile[t];
			if (g_Config.Load(cfgfile.c_str()))
			{
				g_Config.SetStaticValue("cfgfile", cfgfile);
#ifdef _DEBUG
				cout << "Load configuration " + cfgfile <<	setw(60);
				cout << succMsg << endl;
#endif
				if (cfgfile == "/etc/mls/mls.cfg")
				{
					string sCmd = "cp /etc/mls/mls.cfg " + g_Config.GetValue("home") + ".mls";
					system(sCmd.c_str());
				}
				break;
			}
			else
			{
#ifdef _DEBUG
				cout << "Load configuration " + cfgfile <<	setw(60);
				cout << failMsg << endl;
#endif
			}
		}

		if (t == _CfgFile.size())
		{
#ifdef _DEBUG
			cout <<	"Load configuration " << setw(60);
			cout << failMsg << endl;
#endif
			return false;
		}
	}

	{ // 컬러셋 읽기
		_ColFile.push_back( g_Config.GetValue("cfghome") + g_Config.GetValue("colorset") );
		_ColFile.push_back( "/etc/mls/mls.col" );

		for (t=0; t<_ColFile.size(); t++)
		{
			string &colfile = _ColFile[t];
			
			if (g_Color.Load(colfile.c_str()))
			{
				g_Config.SetStaticValue("colfile", colfile);
#ifdef _DEBUG
				cout << "Load colorset " + colfile << setw(60);
				cout << succMsg << endl;
#endif
				if (colfile == "/etc/mls/mls.col")
				{
					string sCmd = "cp /etc/mls/mls.col " + g_Config.GetValue("home") + ".mls";
					system(sCmd.c_str());
				}
				break;
			}
			else
			{
#ifdef _DEBUG
				cout << "Load colorset " << setw(60);
				cout << failMsg << endl;
#endif
			}
		}

		if (t == _ColFile.size())
		{
#ifndef _DEBUG
			cout << "Load colorset " << setw(60);
			cout << failMsg << endl;
#endif
			return false;
		}
	}

	// history를 읽어온다 없어도 상관없다
	ifstream in((g_Config.GetValue("cfghome") + "history").c_str());
	if (in)
	{
		string line;
		while(!getline(in, line).eof())
		{
			if (line.empty()) continue;
			g_CmdHistory.push_back(line);
		}
	}
	return true;
}

// MainFrame 안에 KeyBind가 있기 때문에 따로 만듬.
bool	Load_KeyFile()
{
	int t;
	const char *succMsg = "[\033[01;36m SUCCESS \033[0m]";
	const char *failMsg = "[\033[01;31m  FAIL   \033[0m]";

	{// Key Binding file을 읽는다.
		_KeyFile.push_back( g_Config.GetValue("cfghome") + "mls.key" );
		_KeyFile.push_back( "/etc/mls/mls.key" );

		for (t=0; t<_KeyFile.size(); t++)
		{
			string &keyfile = _KeyFile[t];
			if (g_MainFrame->_tKeyBind.Load(keyfile))
			{
				g_Config.SetStaticValue("keyfile", keyfile);
#ifdef _DEBUG
				cout << "Load key settings... " + keyfile << setw(60);
				cout << succMsg << endl;
#endif
				if (keyfile == "/etc/mls/mls.key")
				{
					string sCmd = "cp /etc/mls/mls.key " + g_Config.GetValue("home") + ".mls";
					system(sCmd.c_str());
				}
				break;
			}
			else
			{
#ifdef _DEBUG
				cout << "Load key settings... " + keyfile << setw(60);
				cout << failMsg << endl;
#endif
			}
		}

		if (t==_KeyFile.size())
		{
#ifndef _DEBUG
			cout << "Load key settings... " << setw(60);
			cout << failMsg << endl;
#endif
			return false;
		}
	}

	{// Key Binding file을 읽는다.
		_EditorKeyFile.push_back( g_Config.GetValue("cfghome") + g_Config.GetValue("EditorKeyFile") );
		_EditorKeyFile.push_back( "/etc/mls/" + g_Config.GetValue("EditorKeyFile"));

		for (t=0; t<_EditorKeyFile.size(); t++)
		{
			string &keyfile = _EditorKeyFile[t];

			if (g_pEditor->KeyLoad(keyfile))
			{
				g_Config.SetStaticValue("editorkeyfile", keyfile);
#ifdef _DEBUG
				cout <<"Load editor key settings... " + keyfile << setw(60);
				cout << succMsg << endl;
#endif
				if (keyfile == "/etc/mls/"+g_Config.GetValue("EditorKeyFile"))
				{
					string sCmd = "cp /etc/mls/" + g_Config.GetValue("EditorKeyFile") + " " + g_Config.GetValue("home") + ".mls";
					system(sCmd.c_str());
				}
				break;
			}
			else
			{
#ifdef _DEBUG
				cout << "Load editor key settings... " + keyfile << setw(60);
				cout << failMsg << endl;
#endif
			}
		}

		if (t==_EditorKeyFile.size())
		{
#ifndef _DEBUG
			cout << "Load editor key settings... " << setw(60);
			cout << failMsg << endl;
#endif
		}
	}
	return true;
}

/// @brief	mls가 종료될때 호출되는 함수
void Destroy()
{
	// configuration 파일을 저장.
	g_Config.Save();

	// history를 저장한다 없어도 상관없다
	ofstream out((g_Config.GetValue("cfghome") + "history").c_str());
	if (out)
		for (int t = 0; t<g_CmdHistory.size(); t++)
			out << g_CmdHistory[t] << endl;

}

/// @brief	print help
void PrintHelp(void)
{
	const char *sStr_Ko =
		" Mls는 도스용 파일관리 툴 Mdir의 리눅스 클론입니다.\n"
		" 프로그램의 기능 버그, 추가할 사항, 기타 문의는 "
		"프로젝트 홈페이지나, 개발자 이메일을 통해서 연락주십시오.\n\n"
		"  * 프로젝트 홈페이지 : http://mls.kldp.net/\n"
		"  * 옵션 설명\n"
		"\t --help       : 도움말\n"
		"\t --lang=CODE  : 출력언어 설정, 사용가능한 CODE는 \n"
		"\t              : us(영어), ko(한글, utf-8), ko_euckr(한글, euc-kr)입니다.\n"
#ifdef _DEBUG
		"\t --debug=FILE : 디버그 메시지를 지정된 파일로 출력\n"
#endif
		"\t --cfg=FILE   : 설정파일 지정\n"
		"\t --col=FILE   : 컬러셋 파일 지정\n"
		"\t --key=FILE   : 키 파일 지정\n"
		"\t --noline     : 선형태를 -,|,+ 로 바꿈\n"
		"\t --mcd        : 바로 MCD 실행 \n";

	const char *sStr_En =
		"Mls is a clone of Mdir, the famous file manager from the MS-DOS age. \n"
		"Mls is rich full-screen text mode shell application that assist you copy, move, delete"
		"files and directories, search for files and run commands in the subshell.\n\n"
		" * Project homepage : http://mls.kldp.net/\n"
		" * Option\n"
		"\t --help       : print this page\n"
		"\t --lang=CODE  : set language, following codes are available\n"
		"\t              : us(english), ko(korean, utf-8), ko_euckr(korean, euc-kr)\n"
#ifdef _DEBUG
		"\t --debug=FILE : redirect debug message to FILE\n"
#endif
		"\t --cfg=FILE   : load config file\n"
		"\t --col=FILE   : load colorset file\n"
		"\t --key=FILE   : load keybind file\n"
		"\t --noline     : change box code to ascii character(-,|,+)\n"
		"\t --mcd        : excute Mcd \n";

	cout << ChgEngKor(sStr_En, sStr_Ko) << endl;
}

/// @brief	Option 처리함수
/// @param	argc	///< 프로그램 시작인자 개수
/// @param	argv	///< 프로그램 시작 인자
void OptionProc(int	argc, char * const	argv[])
{
	char opt = -1;
	struct option longopts[] = {
		{ "help",	no_argument,       NULL,   'h' },
		{ "noline", no_argument,       NULL,   'n' },
		{ "mcd",    no_argument,       NULL,   'm' },
		{ "lang",   required_argument, NULL,   'l' },
		{ "debug",  required_argument, NULL,   'd' },
		{ "cfg",    required_argument, NULL,   'c' },
		{ "col",    required_argument, NULL,   's' },
		{ "key",    required_argument, NULL,   'k' },
		{ NULL,		0,				   NULL,	0  }
	};

	if ( getenv("TERM") )
	{
		string sTerm = getenv("TERM");
		
		// 일반 콘솔 화면에서는 한글 처리가 미흡하기 때문에
		// TERM 이 linux 이면 영문으로 나오게 한다.
		// 옵션으로 한글을 바꿀수 있다.
		if (sTerm == "linux")
		{
			_nLangSet = US;
			e_nBoxLineCode = CHARLINE;
		}
	}

	while((opt = getopt_long(argc, argv, "hnmldcsk:", longopts, NULL)) != -1)
	{
		switch(opt)
		{
		case 'n':		// noline
			e_nBoxLineCode = CHARLINE;
			break;

		case 'l':		// lang
			if (!optarg) break;
			if (!strcmp(optarg, "us"))
				_nLangSet = US;
			else if (!strcmp(optarg, "ko_euckr"))
				_nLangSet = KO_EUCKR;
			else if (!strcmp(optarg, "ko"))
				_nLangSet = KO_UTF8;
			break;

#ifdef _DEBUG
		case 'd':       // debug

			if (optarg)
			{
				int fd;

				if ((fd = open(optarg, O_WRONLY | O_APPEND | O_CREAT, S_IRUSR | S_IWUSR)) == -1)
				{
					printf("Log file open error : %s ", optarg);
					exit(0);
				}
				else
				{
					dup2(fd, 2); // stderr에 리다이렉션한다..
					g_Log = true;
				}
			}

			break;
#endif

		case 'm':		// mcd
			_bMcdExe = true;
			break;

		case 'c':       // 컨피그 파일
			if (optarg) _CfgFile.push_back(optarg);
			break;

		case 's':       // 컬러셋 파일
			if (optarg) _ColFile.push_back(optarg);
			break;

		case 'k':       // keybind 파일
			if (optarg) _KeyFile.push_back(optarg);
			break;

		case '?':       // 오류..
		case 'h':		// help
		default:
			Set_Locale(_nLangSet);
			PrintHelp();
			exit(0);
	    }
	}
}

};

/// @brief	진입부
/// @return	성공여부
int main(int argc, char *argv[])
{
	cout << "Mls " VERSION ", user-friendly graphic shell, 2005\n" << endl;

	OptionProc(argc, argv);

	if (Initialize())  // 시스템 초기화
	{
		g_MainFrame = new MainFrame;
		g_pEditor = new Editor;
		g_pEditor->Init();

		if (Load_KeyFile())
		{
			if (g_MainFrame->Init())
			{
				// 현재 LINES, COLS 를 바꿈.
				g_nLINES = LINES;
				g_nCOLS = COLS;
				
				g_MainFrame->Do(_bMcdExe);
				g_MainFrame->Destroy();
			}
		}
		delete g_pEditor;
		delete g_MainFrame;
		Destroy();
	}
	return SUCCESS;
}

