/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_DIARYDATA_HEADER
#define LIFEOGRAPH_DIARYDATA_HEADER


#include "helpers.hpp"	// i18n headers

#include <gtkmm/treemodel.h>
#include <set>


namespace LIFEO
{

namespace STRING
{
const char SYSTEM_THEME[] = "[ - 0 - ]";
}

using namespace HELPERS;

typedef char SortingCriteria;
static const SortingCriteria SC_DATE    = 'd';
static const SortingCriteria SC_SIZE    = 's';
static const SortingCriteria SC_CHANGE  = 'c';  // last change date

typedef unsigned long DEID; // unique diary element id
static const DEID DEID_MIN = 10000;         // reserved for Diary itself
static const DEID DEID_UNSET = 404;         // :)
static const DEID HOME_CURRENT_ELEM = 1;    // element shown at startup
static const DEID HOME_LAST_ELEM = 2;       // element shown at startup
static const DEID HOME_FIXED_ELEM = 3;      // element shown at startup (defines boundary)

class DiaryElement;	// forward declaration


// DIARYBASE (FOR COMMUNICATION)
class DiaryBase
{
    public:
        typedef std::map< DEID, DiaryElement* >
                                    PoolDEIDs;

                                    DiaryBase( void )
        :   m_current_id( DEID_MIN ), m_force_id( DEID_UNSET ), m_language( "" ) {}
        virtual                     ~DiaryBase() {}

        DEID                        create_new_id( DiaryElement *element )
        {
            DEID retval;
            if( m_force_id == DEID_UNSET )
                retval = m_current_id;
            else
            {
                retval = m_force_id;
                m_force_id = DEID_UNSET;
            }
            m_ids[ retval ] = element;

            while( m_ids.find( m_current_id ) != m_ids.end() )
                m_current_id++;

            return retval;
        }
        void                        erase_id( DEID id ) { m_ids.erase( id ); }
        bool                        set_force_id( DEID id )
        {
            if( m_ids.find( id ) != m_ids.end() || id <= DEID_MIN )
                return false;
            m_force_id = id;
            return true;
        }

        void                        clear( void )
        {
            m_force_id = DEID_UNSET;
            m_current_id = DEID_MIN;
            m_ids.clear();
        }

        virtual bool                make_free_entry_order( Date& ) = 0;

        std::string                 get_lang( void ) const { return m_language; }
        void                        set_lang( const std::string &lang ) { m_language = lang; }

    protected:
        DEID                        m_current_id;
        DEID                        m_force_id;
        PoolDEIDs                   m_ids;
        std::string                 m_language;
};


class ListData;	// forward declaration

// DIARYELEMENT ====================================================================================
class NamedElement
{
    public:
                                NamedElement( void ) : m_name( "" ) {}
                                NamedElement( const Glib::ustring &name ) : m_name( name ) {}
        virtual                 ~NamedElement() {}  //needed because of virtual methods

        Glib::ustring           get_name( void ) const
        { return m_name; }
        std::string             get_name_std( void ) const  // std::string version
        { return m_name; }
        void                    set_name( const Glib::ustring &name )
        { m_name = name; }

    protected:
        Glib::ustring           m_name;
};

class DiaryElement : public NamedElement
{
    public:
        static bool             FLAG_ALLOCATE_GUI_FOR_DIARY;
        static const Glib::RefPtr< Gdk::Pixbuf >
                                s_pixbuf_null;

        enum Type
        {   // CAUTION: order is significant and shouldn't be changed!
            ET_NONE, ET_TAG, ET_TAG_CTG, ET_THEME, ET_CHAPTER_CTG,
            // entry list elements:
            ET_DIARY, ET_CHAPTER, ET_ENTRY, ET_DATE,
            // special types for pseudo elements:
            ET_HEADER
        };

                                DiaryElement( DiaryBase * const,
                                              const Glib::ustring& = "" );
        virtual                 ~DiaryElement()
        {
            if( m_ptr2diary != NULL )
                m_ptr2diary->erase_id( m_id );
        }

        virtual Date            get_date( void ) const
        { return Date( 0 ); }
        virtual int             get_size( void ) const = 0;
        virtual Type            get_type( void ) const = 0;
        virtual const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon( void ) const
        { return( s_pixbuf_null ); }
        virtual const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon32( void ) const
        { return( s_pixbuf_null ); }

        virtual Glib::ustring   get_list_str( void ) const
        { return ""; }

        DEID                    get_id( void ) const
        { return m_id; }

        virtual void            show( void ) = 0;
        virtual void            prepare_for_hiding( void ) {}
        virtual bool            get_filtered_out( void )
        { return false; }

        ListData                *m_list_data;

        DiaryBase * const       m_ptr2diary;

    protected:
        const DEID              m_id;
};

class ListData
{
    public:
        class Colrec : public Gtk::TreeModel::ColumnRecord
        {
            public:
                Colrec()
                {
                    add( ptr );
                    add( info );
                    add( icon );
                }
                // HIDDEN COLUMNS
                Gtk::TreeModelColumn< DiaryElement* >                   ptr;
                // COLUMNS
                Gtk::TreeModelColumn< Glib::ustring >                   info;
                Gtk::TreeModelColumn< Glib::RefPtr< Gdk::Pixbuf > >     icon;
        };
        static Colrec           *colrec;

        Gtk::TreePath           treepath;
};

bool compare_listitems( DiaryElement*, DiaryElement* );
bool compare_listitems_by_name( DiaryElement*, DiaryElement* );
bool compare_names( const Glib::ustring&, const Glib::ustring& );

typedef bool( *FuncCompareStrings )( const Glib::ustring&, const Glib::ustring& ) ;
typedef bool( *FuncCompareDiaryElem )( DiaryElement*, DiaryElement* ) ;
typedef bool( *FuncCompareDiaryElemNamed )( DiaryElement*, DiaryElement* ) ;
typedef std::set< DiaryElement*, FuncCompareDiaryElem > SetDiaryElements;

template< class T >
Glib::ustring
create_unique_name_for_map( const std::map< Glib::ustring, T*, FuncCompareStrings > &map,
                            const Glib::ustring &name0 )
{
    Glib::ustring name = name0;
    for( int i = 1;
         map.find( name ) != map.end();
         i++ )
    {
        name = Glib::ustring::compose( "%1 %2", name0, i );
    }

    return name;
}

// ElementShower class is for communication between Diary and...
// ...GUI classes. we may resort to signals if needed:
class ElementShowerProto
{
    public:
        ElementShowerProto( void ) : m_menu( NULL ) {}
        virtual             ~ElementShowerProto() {}

        virtual DiaryElement*   get_element( void ) = 0;

        Gtk::Widget*            get_widget( void )
        { return m_vbox; }
        virtual Menu2*          get_menu( void )
        { return m_menu; }
        virtual Gtk::Widget*    get_extra_tools( void )
        { return NULL; }

        virtual Glib::ustring   get_title_str( void ) const
        { return ""; }
        virtual Glib::ustring   get_info_str( void ) const
        { return ""; }
        virtual Glib::ustring   get_tip_str( void ) const
        { return ""; }

        virtual bool            is_title_editable( void ) const
        { return false; }
        virtual Glib::ustring   get_title_edit_str( void ) const
        { return get_title_str(); }
        virtual bool            check_title_applicable( const Glib::ustring& ) const
        { return false; }
        virtual bool            apply_title( const Glib::ustring& )
        { return false; }

    protected:
        Gtk::Box                *m_vbox;
        Menu2                   *m_menu;
};

template< class T >
class ElementShower : public ElementShowerProto
{
    public:
        ElementShower() : m_ptr2elem( NULL ) {}
        virtual         ~ElementShower() {}
        virtual void    show( T& ) = 0;
        virtual void    prepare_for_hiding( T& ) {}

        T*              get_element( void )
        { return m_ptr2elem; }

    protected:
        T               *m_ptr2elem;
};

class DiaryElementContainer : public DiaryElement
{
// NOTE: this class and its derivatives are not responsible for handling of...
// ...its children's allocation and deletion
	public:
								DiaryElementContainer( DiaryBase * const d )
								:	DiaryElement( d, "" ), m_items( compare_listitems ) {}
								DiaryElementContainer( DiaryBase * const d, const Glib::ustring &name )
								:	DiaryElement( d, name ), m_items( compare_listitems ) {}

		SetDiaryElements*		get_items( void )
		{ return &m_items; }
		virtual int				get_size( void ) const
		{ return m_items.size(); }

		void					clear( void )
		{ m_items.clear(); }


	protected:
		SetDiaryElements		m_items;
};

// TAG =============================================================================================
class Tag;	// forward declaration

// TAG CATEGORY
// TODO: should be a DiaryElementContainer as it is not responsible for memory handling
class CategoryTags : public DiaryElement, public std::set< Tag*, FuncCompareDiaryElemNamed >
{
    public:
        static ElementShower< CategoryTags > *shower;

                                CategoryTags( DiaryBase * const, const Glib::ustring& );

        void                    show( void );

        virtual int             get_size( void ) const
        { return size(); }
        Type                    get_type( void ) const
        { return ET_TAG_CTG; }
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon( void ) const;
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon32( void ) const;

        Glib::ustring           get_list_str( void ) const
        { return Glib::ustring::compose( "<b>%1</b>", Glib::Markup::escape_text( m_name ) ); }

        bool                    get_expanded( void ) const
        { return m_flag_expanded; }
        void                    set_expanded( bool flag_expanded )
        { m_flag_expanded = flag_expanded; }

    protected:
        bool                    m_flag_expanded;

    friend class CategoryTagsView;
    friend class PoolCategoriesTags;
    friend class PanelExtra;
};

// POOL OF DEFINED TAG CATEGORIES
class PoolCategoriesTags : public std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >
{
    public:
                                PoolCategoriesTags( void );
                                ~PoolCategoriesTags();

        bool                    rename_category( CategoryTags*, const Glib::ustring& );
        void                    clear( void );
};

// TAG
class Tag : public DiaryElementContainer
{
    public:
        static ElementShower< Tag > *shower;

                                Tag( DiaryBase * const d )
                                :   DiaryElementContainer( d ),
                                    m_ptr2category( NULL ) {}
        explicit                Tag( DiaryBase * const, const Glib::ustring&, CategoryTags* );
        virtual                 ~Tag( void );

        void                    show( void );

        CategoryTags*           get_category( void )
        { return m_ptr2category; }
        void                    set_category( CategoryTags *category );

        void                    set_name( const Glib::ustring& );
        //Date                  get_date( void ) const; // if earliest entry's date is needed
        Type                    get_type( void ) const
        { return ET_TAG; }
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon( void ) const;
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon32( void ) const;

        Glib::ustring           get_list_str( void ) const
        { return Glib::Markup::escape_text( m_name ); }

    protected:
        CategoryTags            *m_ptr2category;

    friend class PoolTags;
    friend class Tagset;
    friend class TagView;
};

// POOL OF DEFINED TAGS
class PoolTags : public std::map< Glib::ustring, Tag*, FuncCompareStrings >
{
    public:
                                PoolTags( void )
        :   std::map< Glib::ustring, Tag*, FuncCompareStrings >( compare_names ) {}
                                ~PoolTags();

        bool                    handle_tag_changed( );
        bool                    rename( Tag*, const Glib::ustring& );
        Tag*                    get_tag( unsigned int );
        Tag*                    get_tag( const Glib::ustring& );
        void                    clear( void );
};

// TAGS OF AN ENTRY
// not responsible for memory handling
class Tagset : public std::set< Tag*, FuncCompareDiaryElemNamed >
{
    public:
                                Tagset( void )
        :   std::set< Tag*, FuncCompareDiaryElemNamed >( compare_listitems_by_name ) {}

                                ~Tagset( void );
        bool                    add( Tag* );
        bool                    checkfor_member( const Tag* ) const;
        const Tag*              get_tag( unsigned int ) const;
        //bool                  remove_tag( const Glib::ustring& );

    protected:

};

// CHAPTER =========================================================================================
class Chapter : public DiaryElement
{
    public:
        static ElementShower< Chapter > *shower;

        explicit                Chapter( DiaryBase * const, const Glib::ustring& );
                                Chapter( DiaryBase * const, const Glib::ustring&, Date );
        // this class should have a virtual dtor for some reason because
        // if it does not, its derived classes cannot be deleted
        virtual                 ~Chapter() {}

        void                    show( void );

        void                    set_name( const Glib::ustring& );
        Date                    get_date( void ) const;
        Glib::ustring           get_date_str( void ) const;
        void                    set_date( Date::date_t );
        virtual int             get_size( void ) const  // TODO: stub!
        { return m_size; }
        Type                    get_type( void ) const
        { return ET_CHAPTER; }
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon( void ) const;
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon32( void ) const;

        Date                    get_free_order( void ) const;

        Glib::ustring           get_list_str( void ) const
        { return Glib::ustring::compose( "<b>%1 -  %2</b>", m_date_begin.format_string(),
                Glib::Markup::escape_text( m_name ) ); }

        bool                    get_expanded( void ) const
        { return m_flag_expanded; }
        void                    set_expanded( bool flag_expanded )
        { m_flag_expanded = flag_expanded; }

        bool                    is_initialized( void ) const
        { return( m_date_begin.m_date != Date::NOTSET ); }
        bool                    is_ordinal( void ) const
        { return m_date_begin.is_ordinal(); }

        void                    reset_size( void )
        { m_size = 0; }
        void                    increase_size()
        { m_size++; }
        void                    recalculate_span( const Chapter* );

    protected:
        Date                    m_date_begin;
        int                     m_time_span;
        int                     m_size;
        bool                    m_flag_expanded;

    friend class CategoryChapters;
    friend class ChapterView;
};

class CategoryChapters :
        public DiaryElement, public std::map< Date::date_t, Chapter*, FuncCompareDates >
{
    public:
        explicit                CategoryChapters( DiaryBase * const d, const Glib::ustring& );
        virtual                 ~CategoryChapters();

        virtual int             get_size( void ) const
        { return size(); }
        Type                    get_type( void ) const
        { return ET_CHAPTER_CTG; }

        void                    show( void ) {} // not used

        Chapter*                get_chapter( const Date::date_t ) const;

        Chapter*                create_chapter( const Glib::ustring&, const Date& );
        void                    dismiss_chapter( Chapter* );
        bool                    rename_chapter( Chapter*, const Glib::ustring& );
        bool                    set_chapter_date( Chapter*, Date::date_t );

        bool                    add( Chapter* );

        void                    clear( void );

        Date                    get_free_order_ordinal( void ) const;

    protected:

};

class PoolCategoriesChapters :
        public std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >
{
    public:
                                    PoolCategoriesChapters( void )
        :   std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >(
                compare_names ) {}
                                    ~PoolCategoriesChapters();
        void                        clear( void );
};

// THEMES ==========================================================================================
class Theme : public DiaryElement
{
    public:
        static ElementShower< Theme > *shower;

                                    Theme( DiaryBase * const, const Glib::ustring& );
                                    Theme( DiaryBase * const,
                                           const Glib::ustring&,
                                           const Glib::ustring&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string& );
        // duplicates an existing theme, works like a copy constructor
                                    Theme( const Theme* );

        void                        show( void );

        int                         get_size( void ) const
        { return 0; }   // redundant
        DiaryElement::Type          get_type( void ) const
        { return ET_THEME; }
        const Glib::RefPtr< Gdk::Pixbuf >&
                                    get_icon( void ) const;
        const Glib::RefPtr< Gdk::Pixbuf >&
                                    get_icon32( void ) const;

        Glib::ustring               get_list_str( void ) const
        { return Glib::Markup::escape_text( m_name ); }

        virtual bool                is_system( void ) const
        { return false; }

        Pango::FontDescription      font;
        Gdk::RGBA                   color_base;
        Gdk::RGBA                   color_text;
        Gdk::RGBA                   color_heading;
        Gdk::RGBA                   color_subheading;
        Gdk::RGBA                   color_highlight;

        // CONSTANT COLORS
        static const Gdk::RGBA      s_color_match;
        static const Gdk::RGBA      s_color_match2;
        static const Gdk::RGBA      s_color_link;
        static const Gdk::RGBA      s_color_link2;
        static const Gdk::RGBA      s_color_broken;
        static const Gdk::RGBA      s_color_broken2;

    friend class Diary;
};

class ThemeSystem : public Theme
{
    public:
        static ThemeSystem*         get( void );
        bool                        is_system( void ) const
        { return true; }

    protected:
                                    ThemeSystem( const Glib::ustring&,
                                                 const std::string&,
                                                 const std::string&,
                                                 const std::string&,
                                                 const std::string&,
                                                 const std::string& );
};

class PoolThemes : public std::map< Glib::ustring, Theme*, FuncCompareStrings >
{
    public:
                                    PoolThemes( void )
        :   std::map< Glib::ustring, Theme*, FuncCompareStrings >( compare_names ) {}
                                    ~PoolThemes();

        void                        clear( void );
        bool                        rename_theme( Theme*, const Glib::ustring& );
};

} // end of namespace LIFEO

#endif

