/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <cstdlib>
#include <cassert>

#include "lifeograph.hpp"
#include "app_window.hpp"
#include "dialog_preferences.hpp"


using namespace LIFEO;

AppWindow *AppWindow::p = NULL;

// CONSTRUCTOR
AppWindow::AppWindow( void )
:   m_view_login( NULL ), m_diary_view( NULL ), m_entry_view( NULL ), m_tag_view( NULL ),
    m_ctg_tags_view( NULL ), m_chapter_view( NULL ), m_theme_view( NULL ),
    m_about_dialog( NULL ), m_hpaned_main( NULL )
{
    p = this;

    Cipher::init();

    UndoManager::m = new UndoManager;
    Diary::d = new Diary;
    ListData::colrec = new ListData::Colrec;

    // CONFIGURATION
    set_default_size( Lifeograph::settings.width, Lifeograph::settings.height );
    if( Lifeograph::settings.position_x != POSITION_NOTSET &&
        Lifeograph::settings.position_y != POSITION_NOTSET )
        move( Lifeograph::settings.position_x, Lifeograph::settings.position_y );
    if( Lifeograph::settings.state_maximized )
        maximize();

    // GTKBUILDER
    Lifeograph::p->load_gui( UIDIR "/lifeograph.ui" );

    // DRAW UI
    if( Lifeograph::p->m_flag_first_time_user || Lifeograph::p->m_flag_force_welcome )
        draw_welcomescreen();
    else
        draw_loginscreen();
}

AppWindow::~AppWindow( void )
{
    remove();
    delete UndoManager::m;

    if( m_view_login )
        delete m_view_login;

    if( Diary::d )
        delete Diary::d;
}

bool
AppWindow::on_event( GdkEvent* )
{
    Lifeograph::p->handle_event();
    return false;
}

bool
AppWindow::on_delete_event( GdkEventAny* )
{
    PRINT_DEBUG( "AppWindow::on_delete_event()" );

    return( Lifeograph::quit() != true );
}

bool
AppWindow::write_backup( void )
{
    m_entry_view->sync();

    Result result( Diary::d->write( Diary::d->get_path() + LOCK_SUFFIX ) );

    return( result == SUCCESS );
}

void
AppWindow::show_about( void )
{
    if( m_about_dialog == NULL )
    {
        Lifeograph::builder->get_widget( "aboutdialog", m_about_dialog );
        m_about_dialog->set_name( PROGRAM_NAME );
        m_about_dialog->set_version( Lifeograph::PROGRAM_VERSION_STRING );
        // XXX if needed: m_about_dialog->signal_activate_link().connect( open_url );
        m_about_dialog->set_transient_for( *this );
    }

    m_about_dialog->run();
    m_about_dialog->hide();
}

void
AppWindow::handle_undo( void )
{
    UndoManager::m->undo();
}

void
AppWindow::handle_redo( void )
{
    UndoManager::m->redo();
}

bool
AppWindow::finish_editing( bool opt_save )
{
    // SAVING
    Lifeograph::settings.position_pane = m_hpaned_main->get_position();
    Lifeograph::settings.position_pane_tags = m_hpaned_entry->get_position();

    // files added to recent list here if not already there
    if( ! Diary::d->get_path().empty() )
        if( Lifeograph::stock_diaries.find( Diary::d->get_path() ) ==
                Lifeograph::stock_diaries.end() )
            Lifeograph::settings.recentfiles.insert( Diary::d->get_path() );

    if( ! Diary::d->is_read_only() )
    {
        m_entry_view->sync();
        Diary::d->set_last_elem( panel_main->get_cur_elem() );

        if( opt_save )
        {
            if( Diary::d->write() != SUCCESS )
            {
                Gtk::MessageDialog messagedialog( *this,
                                                  "",
                                                  false,
                                                  Gtk::MESSAGE_WARNING,
                                                  Gtk::BUTTONS_OK,
                                                  true );
                messagedialog.set_message( _( STRING::CANNOT_WRITE ) );
                messagedialog.set_secondary_text( _( STRING::CANNOT_WRITE_SUB ) );
                messagedialog.run();

                return false;
            }
        }
        else
        {
            Gtk::MessageDialog messagedialog( *this,
                                              "",
                                              false,
                                              Gtk::MESSAGE_WARNING,
                                              Gtk::BUTTONS_CANCEL,
                                              true );
            messagedialog.set_message(
                _( "Are you sure you want to log out without saving?" ) );
            messagedialog.set_secondary_text( Glib::ustring::compose(
                _( "Your changes will be backed up in %1. "
                   "If you exit normally, your diary is saved automatically." ),
                "<b>" + Diary::d->get_path() + ".~unsaved~</b>" ), true );
            messagedialog.add_button( _( "_Log out without Saving" ), Gtk::RESPONSE_ACCEPT );

            if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
                return false;
            // back up changes
            Diary::d->write( Diary::d->get_path() + ".~unsaved~" );
        }
    }

    // CLEARING
    // TODO: m_loginstatus = LOGGING_OUT_IN_PROGRESS;

    Lifeograph::m_actions_read->set_sensitive( false );

    Lifeograph::s_signal_logout.emit();	// only for DialogEvent

    Lifeograph::m_internaloperation++;
    panel_main->handle_logout();
    panel_diary->handle_logout();
    m_entry_view->get_buffer()->handle_logout();
    panel_extra->handle_logout();

    if( Lifeograph::loginstatus == Lifeograph::LOGGED_IN )
        Lifeograph::loginstatus = Lifeograph::LOGGED_OUT;
    Diary::d->clear();
    Lifeograph::p->m_connection_timeout.disconnect();
    m_connection_backup.disconnect();
    Lifeograph::m_internaloperation--;

    return true;
}

void
AppWindow::logout( bool opt_save )
{
    Lifeograph::p->m_flag_open_directly = false;   // should be reset to prevent logging in again
    if( finish_editing( opt_save ) )
        m_view_login->handle_logout();
}

bool
AppWindow::confirm_dismiss_element( const DiaryElement *elem, Gtk::Widget *extra_widget )
{
    Gtk::MessageDialog message( *this, "", false,
                                Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
    message.set_message(
            Glib::ustring::compose( _( "Are you sure, you want to dismiss %1?" ),
                                    Glib::ustring::compose( "\"%1\"", elem->get_name() ) ) );
    message.set_secondary_text( _( "This operation cannot be undone!" ) );
    message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
    message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

    if( extra_widget )
    {
        extra_widget->show();
        message.get_message_area()->add( *extra_widget );
    }

    if( message.run() != Gtk::RESPONSE_ACCEPT )
        return false;
    else
        return true;
}

Gtk::TreeRow
AppWindow::get_element_row( const DiaryElement *element )
{
    switch( element->get_type() )
    {
        case DiaryElement::ET_ENTRY:
        case DiaryElement::ET_CHAPTER:
            return( panel_diary->get_row( element->m_list_data->treepath ) );

        case DiaryElement::ET_TAG:
        case DiaryElement::ET_TAG_CTG:
        case DiaryElement::ET_THEME:
            return( panel_extra->get_row( element->m_list_data->treepath ) );
        default:
            return( Gtk::TreeRow() );
    }
}

void
AppWindow::update_title( void )
{
    Glib::ustring title( PROGRAM_NAME );

    if( Lifeograph::loginstatus == Lifeograph::LOGGED_IN )
    {
        title += " - ";
        title += Glib::filename_display_basename( Diary::d->get_path() );

        if( Diary::d->is_read_only() )
        {
            title += " <";
            title += "Read Only";
            title += ">";
        }
    }

    set_title( title );
}

// MODE DRAWING FUNCTIONS
void
AppWindow::draw_welcomescreen( void )
{
    Gtk::Alignment *alignment = Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.1, 0 ) );
    Gtk::Label *label_salutation =
            Gtk::manage( new Gtk::Label( _( STRING::SALUTATION ), 0.5, 0.5 ) );
    Gtk::Button *button_start = Gtk::manage( new Gtk::Button( _("_Continue"), true ) );
    Gtk::Image *icon_forward =
            Gtk::manage( new Gtk::Image( Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_MENU ) );
    Gtk::Box *vbox = Gtk::manage( new Gtk::Box( Gtk::ORIENTATION_VERTICAL, 50 ) );

    alignment->set_padding( 120, 120, 150, 150 );

    label_salutation->set_use_markup( true );

    button_start->set_image( *icon_forward );
    button_start->set_image_position( Gtk::POS_RIGHT );

    vbox->pack_start( *label_salutation );
    vbox->pack_start( *button_start );

    alignment->add( *vbox );
    alignment->show_all();

    button_start->signal_clicked().connect( sigc::mem_fun( *this, &AppWindow::draw_loginscreen ) );

    add( *alignment );
}

void
AppWindow::draw_loginscreen( void )
{
    // only to be called on start up
    assert( m_view_login == NULL );

    try
    {
        m_view_login = new ViewLogin;
        m_view_login->handle_start();
    }
    catch( ... )
    {
        throw LIFEO::Error( "creation of login view failed" );
    }
}

void
AppWindow::login( void )
{
    remove();

    if( m_hpaned_main == NULL )
    {
#ifdef LIFEO_NO_APP_MENU
        Gtk::MenuItem               *menuitem_preferences;
        Gtk::MenuItem               *menuitem_about;
#endif

        try
        {
            // GENERAL TOOLS
#ifdef LIFEO_NO_APP_MENU
            Lifeograph::builder->get_widget( "menuitem_preferences", menuitem_preferences );
            Lifeograph::builder->get_widget( "menuitem_about", menuitem_about );
#endif

            // PANES
            Lifeograph::builder->get_widget( "hpaned_main", m_hpaned_main );
            Lifeograph::builder->get_widget( "hpaned_second", m_hpaned_entry );

            // TAG PANEL
            Lifeograph::builder->get_widget_derived( "treeview_all_tags", panel_extra );

            // ICONS
            Glib::RefPtr< Gtk::IconTheme > theme = Gtk::IconTheme::get_default();
            Lifeograph::icons->diary_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/diary-16.png" );
            Lifeograph::icons->diary_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/diary-32.png" );
            Lifeograph::icons->entry_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/entry-16.png" );
            Lifeograph::icons->entry_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/entry-32.png" );
            Lifeograph::icons->entry_favorite_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/favorite-16.png" );
            Lifeograph::icons->entry_favorite_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/favorite-32.png" );
            Lifeograph::icons->tag_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/tag-16.png" );
            Lifeograph::icons->tag_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/tag-32.png" );
            Lifeograph::icons->tag_category_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/category_tags-16.png" );
            Lifeograph::icons->tag_category_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/category_tags-32.png" );
            Lifeograph::icons->chapter_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/chapter-16.png" );
            Lifeograph::icons->chapter_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/chapter-32.png" );
            // TODO: update icons on theme change
            Lifeograph::icons->theme_16 = theme->load_icon( "gtk-select-color", 16 );
            Lifeograph::icons->theme_default_16 = theme->load_icon( "gtk-apply", 16 );
            Lifeograph::icons->theme_32 = theme->load_icon( "gtk-select-color", 32 );
            Lifeograph::icons->month_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/month-16.png" );
            Lifeograph::icons->month_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/month-32.png" );
            Lifeograph::icons->filter_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/filter-16.png" );

            // PANELS & VIEWS
            panel_main = new PanelMain;
            panel_diary = new PanelDiary;
            //panel_extra = new PanelExtra;
            m_diary_view = new DiaryView;
            m_theme_view = new ThemeView;
            m_ctg_tags_view = new CategoryTagsView;
            m_chapter_view = new ChapterView;
            m_entry_view = new EntryView;
            m_tag_view = new TagView;
        }
        catch( std::exception &ex )
        {
            throw ex;
        }

        // GEOMETRY
        int width, dummy;
        get_size( width, dummy );
        // try to keep the same ratio between the panels before the resizing occurs:
        if( width != Lifeograph::settings.width )
        {
            Lifeograph::settings.position_pane *=  float( width ) / Lifeograph::settings.width;
        }

        // check the diary panel's size against the minimum:
        if( Lifeograph::settings.position_pane > ( width - PanelDiary::MIN_WIDTH ) )
        {
            PRINT_DEBUG( "Diary panel width set to minimum value" );
            Lifeograph::settings.position_pane = width - PanelDiary::MIN_WIDTH;
        }

        m_hpaned_main->set_position( Lifeograph::settings.position_pane );
        m_hpaned_entry->set_position( Lifeograph::settings.position_pane_tags );

        // ACTIONS
        Lifeograph::create_action( true, m_action_undo,"Undo", Gtk::Stock::UNDO,
                       _( "Undo" ), "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK ),
                       sigc::mem_fun( *this, &AppWindow::handle_undo ) );

        Lifeograph::create_action( true, m_action_redo, "Redo", Gtk::Stock::REDO,
                       _( "Redo" ), "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK | Gdk::SHIFT_MASK ),
                       sigc::mem_fun( *this, &AppWindow::handle_redo ) );

        // SIGNALS
#ifdef LIFEO_NO_APP_MENU
        menuitem_about->show();
        menuitem_preferences->show();
        menuitem_about->signal_activate().connect(
                sigc::mem_fun( this, &AppWindow::show_about ) );
        menuitem_preferences->signal_activate().connect(
                sigc::ptr_fun( &DialogPreferences::create ) );
#endif
    }

    m_connection_backup = Glib::signal_timeout().connect_seconds(
            sigc::mem_fun( this, &AppWindow::write_backup ), BACKUP_INTERVAL );

    Lifeograph::m_actions_read->set_sensitive( true );
    Lifeograph::m_actions_edit->set_visible( Diary::d->is_read_only() == false );
    Lifeograph::m_actions_edit->set_sensitive( Diary::d->is_read_only() == false );

    add( *m_hpaned_main );

    // LOGIN
    Lifeograph::m_internaloperation++;

    m_tag_view->handle_login();
    m_entry_view->handle_login();
    m_diary_view->handle_login();
    m_chapter_view->handle_login();
    panel_diary->handle_login();
    panel_extra->handle_login();

    Lifeograph::m_internaloperation--;

    Lifeograph::loginstatus = Lifeograph::LOGGED_IN;
    Lifeograph::p->m_flag_auto_logout_frozen = false;
    update_title();
}
