/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_VIEWS_HEADER
#define LIFEOGRAPH_VIEWS_HEADER


#include "diarydata.hpp"
#include "entry.hpp"
#include "widget_chart.hpp"
#include "dialog_export.hpp"
#include "dialog_password.hpp"


namespace LIFEO
{

// DIARY ELEMENT VIEW ==============================================================================
class GUIElement // BEWARE: subject to removal
{
	public:
		typedef sigc::signal< void, Gtk::Widget* >
								Signal_void_Widget;
		typedef sigc::signal< void, DiaryElement* >
								Signal_DiaryElement;
		typedef sigc::signal< void, const Date& >
								Signal_Date;
};

// TAG VIEW ========================================================================================
class TagView : public ElementShower< Tag >
{
	public:
								TagView( void );

		void					show( Tag &tag );

        DiaryElement*           get_element( void )
        { return m_ptr2tag; }

        Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon_32( void ) const;

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2tag->m_name; }
        Glib::ustring           get_info_str( void ) const
        { return Glib::ustring::compose( _( "Tag with %1 entries" ), m_ptr2tag->get_size() ); }

        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2tag->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

        void                    handle_login( void );
		void					handle_filter_toggled( void );
		void					handle_category_changed( void );
		bool					is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
												  const Gtk::TreeIter );

		void					update_combobox_categories( void );
        void                    update_button_filter_tag( void );

		void					dismiss_tag( void );

		Gtk::ToggleButton		*m_button_filter;
		Gtk::Button				*m_button_dismiss;
		Gtk::ComboBox			*m_combobox_category;
		WidgetChart				*m_widgetchart;

	protected:
		enum TagComboType { TCT_NONE, TCT_CATEGORY, TCT_NEW, TCT_SEPARATOR };

		class ColrecCategory : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecCategory() { add( name ); add( ptr ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >		name;
				Gtk::TreeModelColumn< CategoryTags* >		ptr;
				Gtk::TreeModelColumn< TagComboType >		type;
		};

		Tag						*m_ptr2tag;	//current tag

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_categories;
		ColrecCategory			colrec_category;

};

// TAG CATEGORY VIEW ===============================================================================
class CategoryTagsView : public ElementShower< CategoryTags >
{
	public:
								CategoryTagsView( void );

		void					show( CategoryTags& );

        DiaryElement*           get_element( void )
        { return m_ptr2category; }

        Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon_32( void ) const;

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2category->m_name; }
        Glib::ustring           get_info_str( void ) const
        { return Glib::ustring::compose( _( "Tag category with %1 tags" ),
                                         m_ptr2category->size() ); }

        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2category->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

	protected:
		void					dismiss_category( void );

		Gtk::Button				*m_button_dismiss;

		CategoryTags			*m_ptr2category;
};

// CHAPTER VIEW ====================================================================================
class ChapterView : public ElementShower< Chapter >
{
	public:
								ChapterView( void );

		void					show( Chapter& );

        DiaryElement*           get_element( void )
        { return m_ptr2chapter; }

	protected:
		void					update_labels( void );
		void					dismiss_chapter( void );

        Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon_32( void ) const;

        Glib::ustring           get_title_str( void ) const
        {
            if( m_ptr2chapter->m_date_begin.is_ordinal() )
                return Glib::ustring::compose( "%1: %2",
                                               m_ptr2chapter->m_date_begin.format_string(),
                                               m_ptr2chapter->m_name );
            else
                return m_ptr2chapter->m_name;
        }
        Glib::ustring           get_info_str( void ) const
        {
            if( m_ptr2chapter->m_date_begin.is_ordinal() )
                return Glib::ustring::compose( _( "Topic with %1 entries" ),
                                               m_ptr2chapter->m_size );
            else
                return Glib::ustring::compose(
                        m_ptr2chapter->m_time_span ?
                                _( "Chapter with %1 entries in %2 days" ) :
                                _( "Chapter with %1 entries" ),
                        m_ptr2chapter->get_size(), m_ptr2chapter->m_time_span );
        }

        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2chapter->m_name; }
		bool					check_title_applicable( const Glib::ustring& ) const;
		bool					apply_title( const Glib::ustring& );

		void					handle_date_changed( void );
		void					handle_date_applied( void );

		void					add_new_entry( void );

		Gtk::Button				*m_button_dismiss;

//		Gtk::Label				*m_label_period;
//		Gtk::HBox				*m_hbox_period;
		Gtk::Entry				*m_entry_begins;
		Gtk::Button				*m_button_date_apply;
		Gtk::Label				*m_label_ends;

		Gtk::Frame				*m_frame_edit;
        Gtk::Frame              *m_frame_actions;

		Chapter					*m_ptr2chapter;
};

// THEME VIEW ======================================================================================
class ThemeView : public ElementShower< Theme >
{
	public:
								ThemeView( void );

		void					show( Theme& );

        DiaryElement*           get_element( void )
        { return m_ptr2theme; }

        Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon_32( void ) const;

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2theme->get_name(); }
        virtual Glib::ustring   get_info_str( void ) const
        { return( m_ptr2theme->is_system() ? _( "System theme" ) : _( "Custom theme" ) ); }

        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2theme->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

	protected:
		Gtk::FontButton			*m_fontbutton;
		Gtk::ColorButton		*m_colorbutton_bground;
		Gtk::ColorButton		*m_colorbutton_text;
		Gtk::ColorButton		*m_colorbutton_heading;
		Gtk::ColorButton		*m_colorbutton_subheading;
		Gtk::ColorButton		*m_colorbutton_highlight;
		Gtk::EventBox           *m_eventbox_preview;
		Gtk::Label              *m_label_preview;

		Gtk::Button				*m_button_duplicate;
		Gtk::Button				*m_button_dismiss;
		Gtk::Button				*m_button_make_default;

		Theme					*m_ptr2theme;

		void					handle_font_changed( void );
		void					handle_color_changed( void );
		void					update_preview( void );

		void					duplicate( void );
		void					make_default( void );
		void					dismiss_theme( void );
};

// DIARY VIEW ======================================================================================
class DiaryView : public ElementShower< Diary >
{
	public:
								DiaryView( void );
								~DiaryView( void );

		void					handle_login( void );

		void					show( Diary& );

		DiaryElement*           get_element( void )
		{ return m_ptr2diary; }

        Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon_32( void ) const;

        Glib::ustring           get_list_str( void ) const
        { return Glib::ustring::compose( "<b>%1</b>",
                Glib::Markup::escape_text(
                        Glib::filename_display_basename( m_ptr2diary->m_path ) ) ); }
        Glib::ustring           get_title_str( void ) const
        { return m_ptr2diary->m_name; }
        Glib::ustring           get_info_str( void ) const
        {
            int timespan( 0 );
            // FIXME: does not work for some obscure reason..
            for( Entryciter iter_entry = m_ptr2diary->m_entries.begin();
                    iter_entry != m_ptr2diary->m_entries.end();
                    ++iter_entry )
            {
                if( ! iter_entry->second->get_date().is_ordinal() )
                {
                    timespan = m_ptr2diary->m_entries.rbegin()->second->get_date().
                            calculate_days_between( iter_entry->second->get_date() );
                    break;
                }
            }
            return Glib::ustring::compose( _( "Diary with %1 entries spanning %2 days" ),
                                           m_ptr2diary->m_entries.size(),
                                           timespan );
        }
		// DIALOGS
		void					start_dialog_password( void );
		void					start_dialog_import( void );
		void					start_dialog_export( void );
		void					open_diary_folder( void );

	protected:
		void					handle_startup_type_changed( void );
		void					handle_startup_elem_dropped(
										const Glib::RefPtr< Gdk::DragContext >&,
										int, int,
										const Gtk::SelectionData&,
										uint, guint );
		void					go_to_startup_elem( void );

		void					update_combobox_chapter_ctg( void );
		void					handle_cur_chapter_ctg_changed( void );
		void					rename_cur_chapter_ctg( void );
		void					dismiss_cur_chapter_ctg( void );
		bool					is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
												  const Gtk::TreeIter );

        Gtk::Frame              *m_frame_startup;
        Gtk::Frame              *m_frame_chapters;

		Gtk::Label				*m_label_path;
		Gtk::Label				*m_label_encryption;
		Gtk::Button				*m_button_logoutwosaving;
		Gtk::CheckButton		*m_checkbutton_autologout;
		Gtk::Button				*m_button_password;
        WidgetChart             *m_widgetchart;

		// STARTUP
		Gtk::ComboBox			*m_combobox_startup_type;
		Gtk::Label				*m_label_startup_elem;
		Gtk::Label				*m_label_startup_elem_drop;
		Gtk::Button				*m_button_startup_elem_go;

		class ColrecStartupType : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecStartupType() { add( name ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >	name;
				Gtk::TreeModelColumn< DEID >			type;
		};

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_startup_type;
		ColrecStartupType		colrec_startup_type;

		// CHAPTER CATEGORIES & TOPICS
		Gtk::ComboBox           *m_combobox_chapter_ctg;
		Gtk::Button				*m_button_chapter_ctg_dismiss;
		Gtk::Button				*m_button_chapter_ctg_rename;

		enum ChapterCtgComboType { CCCT_CATEGORY, CCCT_NEW, CCCT_SEPARATOR };

		class ColrecChapterCtg : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecChapterCtg() { add( name ); add( ptr ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >		name;
				Gtk::TreeModelColumn< CategoryChapters* >	ptr;
				Gtk::TreeModelColumn< ChapterCtgComboType >	type;
		};

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_chapter_ctg;
		ColrecChapterCtg		colrec_chapter_ctg;

		// DIALOGS
		DialogPassword			*m_dialog_password;
		DialogImport			*m_dialog_import;
		DialogExport			*m_dialog_export;

		Diary					*m_ptr2diary;

	friend class Lifeograph;
};

} // end of namespace LIFEO

#endif

