/***********************************************************************************

    Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <sys/stat.h>

#include "helpers.hpp"
#include "lifeobase.hpp"
#include "view_login.hpp"
#include "dialog_export.hpp" // DialogPasswordPrompt


using namespace LIFEO;
using namespace HELPERS;


ViewLogin::Colrec   *ViewLogin::colrec;
std::string         ViewLogin::m_path_cur;


ViewLogin::ViewLogin( void )
:   m_flag_info_is_visible( false )
{
    Gtk::Button *button_new_diary, *button_browse_diary;
    Gtk::TreeView::Column *col_diary, *col_date;

    try
    {
        colrec = new Colrec;

        Lifeobase::builder->get_widget( "grid_login", m_grid_login );
        Lifeobase::builder->get_widget( "treeview_diaries", m_treeview_diaries );
        Lifeobase::builder->get_widget( "button_new_diary", button_new_diary );
        Lifeobase::builder->get_widget( "button_open_diary", button_browse_diary );
        Lifeobase::builder->get_widget( "button_login_edit", m_button_edit_diary );
        Lifeobase::builder->get_widget( "button_login_read", m_button_read_diary );
        Lifeobase::builder->get_widget( "button_login_remove_diary", m_button_remove_diary );

        col_diary = Gtk::manage( new Gtk::TreeView::Column( _( "Name" ), colrec->name ) );
        col_date = Gtk::manage( new Gtk::TreeView::Column( _( "Date Last Saved" ),
                                                           colrec->date ) );

        m_infobar = Gtk::manage( new Gtk::InfoBar() );
        m_label_info = Gtk::manage( new Gtk::Label() );
        m_button_info = Gtk::manage( new Gtk::Button() );

        m_treestore_diaries = Gtk::ListStore::create( *colrec );
        m_treesel_diaries = m_treeview_diaries->get_selection();
    }
    catch( ... )
    {
        throw LIFEO::Error( "creation of login view failed" );
    }

    // INFOBAR
    m_label_info->set_line_wrap( true );
    Gtk::Container *infobar_container(
            dynamic_cast< Gtk::Container* >( m_infobar->get_content_area() ) );
    if( infobar_container )
        infobar_container->add( *m_label_info );
    m_grid_login->attach( *m_infobar, 0, 0, 1, 1 );
    m_infobar->add_action_widget( *m_button_info, 0 );
    m_infobar->show_all_children( true );

    // LIST OF DIARIES
    m_treeview_diaries->set_model( m_treestore_diaries );
    col_diary->set_expand( true );
    col_diary->set_clickable( true );
    col_date->set_clickable( true );
    m_treeview_diaries->append_column( *col_diary );
    m_treeview_diaries->append_column( *col_date );
    m_treestore_diaries->set_sort_column( 1, Gtk::SORT_DESCENDING );
    m_treeview_diaries->set_has_tooltip( true );

    Lifeobase::base->set_default( *m_button_edit_diary );

    m_button_edit_diary->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &ViewLogin::handle_button_open ), false ) );
    m_button_read_diary->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &ViewLogin::handle_button_open ), true ) );

    m_button_remove_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::remove_selected_diary ) );

    m_treesel_diaries->signal_changed().connect(
            sigc::mem_fun( this, &ViewLogin::handle_diary_selection_changed ) );

    m_treeview_diaries->signal_row_activated().connect(
            sigc::mem_fun( this, &ViewLogin::handle_diary_activated ) );

    m_treeview_diaries->signal_query_tooltip().connect(
            sigc::mem_fun( this, &ViewLogin::handle_diary_tooltip ) );

    button_new_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::create_new_diary ) );
    button_browse_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::add_existing_diary ) );

    col_diary->signal_clicked().connect( sigc::mem_fun( this, &ViewLogin::sort_by_name ) );
    col_date->signal_clicked().connect( sigc::mem_fun( this, &ViewLogin::sort_by_date ) );

    m_infobar->signal_response().connect(
            sigc::mem_fun( this, &ViewLogin::handle_infobar_response ) );
}

void
ViewLogin::handle_start( void )
{
    Lifeobase::base->remove();
    Lifeobase::base->add( *m_grid_login );
    Lifeobase::base->set_title( PROGRAM_NAME );
    if( Lifeobase::base->m_flag_open_directly )
        if( open_selected_diary( false ) )
            return;
    populate_diaries();
}

void
ViewLogin::handle_logout( void )
{
    handle_start();
    if( Lifeobase::loginstatus == Lifeobase::LOGGED_TIME_OUT )
        show_info( Gtk::MESSAGE_INFO, _( STRING::ENTER_PASSWORD_TIMEOUT ) );
    else if( m_flag_info_is_visible )
        hide_infobar();
}

void
ViewLogin::populate_diaries( void )
{
    Gtk::TreeModel::Row row;

    struct stat fst;    // file date stat

    m_treestore_diaries->clear();
    for( ListPaths::reverse_iterator iter = Lifeobase::settings.recentfiles.rbegin();
         iter != Lifeobase::settings.recentfiles.rend();
         ++iter )
    {
        row = *( m_treestore_diaries->append() );
        row[ colrec->name ] = Glib::filename_display_basename( *iter );
        row[ colrec->path ] = *iter;

        stat( iter->c_str(), &fst );
        row[ colrec->date ] = Date::format_string( ( time_t ) fst.st_mtime );
    }

    // STOCK DIARIES
    for( std::set< std::string >::iterator itr_diary = Lifeobase::stock_diaries.begin();
         itr_diary != Lifeobase::stock_diaries.end();
         ++itr_diary )
    {
        std::string path( *itr_diary );
        row = *( m_treestore_diaries->append() );
        row[ colrec->name ] = "[*] " + Glib::filename_display_basename( path );
        row[ colrec->path ] = path;

        stat( path.c_str(), &fst );
        row[ colrec->date ] = Date::format_string( ( time_t ) fst.st_mtime );
    }

    if( Lifeobase::settings.recentfiles.size() > 0 )
        m_treesel_diaries->select( m_treestore_diaries->get_iter( "0" ) );
}

void
ViewLogin::sort_by_date( void )
{
    m_treestore_diaries->set_sort_column( 1, Gtk::SORT_DESCENDING );
}

void
ViewLogin::sort_by_name( void )
{
    m_treestore_diaries->set_sort_column( 0, Gtk::SORT_ASCENDING );
}

bool
ViewLogin::open_selected_diary( bool read_only )
{
    if( ! Diary::d->set_path( m_path_cur, false, read_only ) )
    {
        show_info( Gtk::MESSAGE_INFO, _( STRING::DIARY_LOCKED ) );
        return false;
    }

    switch( Diary::d->read_header() )
    {
        case SUCCESS:
            break;
        case INCOMPATIBLE_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::INCOMPATIBLE_DB ) );
            return false;
        case CORRUPT_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::CORRUPT_DB ) );
            return false;
        default:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::FAILED_TO_OPEN_DB ) );
            return false;
    }

    if( Diary::d->is_old() )
    {
        Gtk::MessageDialog *messagedialog
                = new Gtk::MessageDialog( *Lifeobase::base,
                                          "",
                                          false,
                                          Gtk::MESSAGE_WARNING,
                                          Gtk::BUTTONS_CANCEL,
                                          true );
        messagedialog->set_message( _( "Are You Sure You Want to Upgrade The Diary?" ) );
        messagedialog->set_secondary_text( _( STRING::UPGRADE_DIARY_CONFIRM ) );
        messagedialog->add_button( _( "Upgrade The Diary" ), Gtk::RESPONSE_ACCEPT );

        int result( messagedialog->run() );

        delete messagedialog;

        if( result !=  Gtk::RESPONSE_ACCEPT )
            return false;
    }
    if( Diary::d->is_encrypted() )
    {
        static DialogPasswordPrompt *dialog_password = NULL;
        if( dialog_password == NULL )
            Lifeobase::builder->get_widget_derived(
                    "dialog_import_password", dialog_password );
        dialog_password->set_transient_for( *Lifeobase::base );

        if( dialog_password->run() == RESPONSE_GO )
        {
            Diary::d->set_passphrase( dialog_password->get_password() );
            dialog_password->hide();
        }
        else
        {
            dialog_password->hide();
            return false;
        }
    }

    switch( Diary::d->read_body() )
    {
        case EMPTY_DATABASE:    // no special treatment for now:
        case SUCCESS:
            Lifeobase::base->draw_editscreen();
            break;
        case WRONG_PASSWORD:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::WRONG_PASSWORD ) );
            sleep( 1 );
            return open_selected_diary( read_only );
        case CORRUPT_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::CORRUPT_DB ) );
            Diary::d->clear();  // clear partially read content if any
            return false;
        default:
            return false;
    }

    return true;
}

void
ViewLogin::remove_selected_diary( void )
{
    m_path_removed = m_path_cur;
    Lifeobase::settings.recentfiles.erase( m_path_removed );
    populate_diaries();
    show_info( Gtk::MESSAGE_INFO,
               Glib::ustring::compose( _( "Removed diary: %1" ), m_path_removed ),
               _( "Undo" ), RESP_UNDO );
}

void
ViewLogin::create_new_diary( void )
{
    DialogSaveDiary *dialogsavediary = new DialogSaveDiary;
    dialogsavediary->set_current_folder( Glib::get_home_dir() );

    if( dialogsavediary->run() == Gtk::RESPONSE_ACCEPT )
    {
        Diary::d->init_new(
                dialogsavediary->get_filename_default() );
        delete dialogsavediary;
        Lifeobase::base->draw_editscreen();
    }
    else
        delete dialogsavediary;
}

void
ViewLogin::add_existing_diary( void )
{
    DialogOpenDiary *filechooserdialog = new DialogOpenDiary;

    if( m_treesel_diaries->count_selected_rows() > 0 )
        filechooserdialog->set_current_folder( Glib::path_get_dirname( m_path_cur ) );
    else
        filechooserdialog->set_current_folder( Glib::get_home_dir() );

    if( filechooserdialog->run() == Gtk::RESPONSE_ACCEPT )
    {
        m_path_cur = filechooserdialog->get_filename();
        delete filechooserdialog;
        if( ! open_selected_diary( false ) )
            m_treesel_diaries->unselect_all(); // mostly to clear m_path_cur
    }
    else
        delete filechooserdialog;
}

void
ViewLogin::show_info( Gtk::MessageType type, const Glib::ustring &text,
                      const Glib::ustring &button, InfoResponse response )
{
    m_label_info->set_text( text );
    m_button_info->set_label( button );
    m_resp_cur = response;
    m_infobar->set_message_type( type );
    m_infobar->show();
    m_flag_info_is_visible = true;
}

void
ViewLogin::handle_infobar_response( int )
{
    if( m_resp_cur == RESP_UNDO )
    {
        Lifeobase::settings.recentfiles.insert( m_path_removed );
        populate_diaries();
    }

    hide_infobar();
}

inline void
ViewLogin::hide_infobar( void )
{
    //m_label_info->set_text( "" );
    m_infobar->hide();
    m_flag_info_is_visible = false;
}

void
ViewLogin::handle_diary_selection_changed( void )
{
    if( Lifeobase::m_internaloperation )
        return;

    bool flag_selection( m_treesel_diaries->count_selected_rows() > 0 );

    if( flag_selection )
        m_path_cur = ( *m_treesel_diaries->get_selected() )[ colrec->path ];
    else
        m_path_cur = "";

    bool flag_not_stock( Lifeobase::stock_diaries.find( m_path_cur ) ==
            Lifeobase::stock_diaries.end() );
    m_button_remove_diary->set_visible( flag_selection && flag_not_stock );
    m_button_edit_diary->set_visible( flag_selection && flag_not_stock );
    m_button_read_diary->set_visible( flag_selection );
}

void
ViewLogin::handle_diary_activated( const Gtk::TreePath &path, Gtk::TreeView::Column *col )
{
    open_selected_diary( Lifeobase::stock_diaries.find( m_path_cur ) !=
            Lifeobase::stock_diaries.end() ); // read only if stock diary
}

bool
ViewLogin::handle_diary_tooltip( int x, int y, bool, const Glib::RefPtr<Gtk::Tooltip> &tooltip )
{
    Gtk::TreeModel::Path path;
    Gtk::TreeView::Column *column;
    int cell_x, cell_y;
    int bx, by;
    m_treeview_diaries->convert_widget_to_bin_window_coords( x, y, bx, by );
    if( ! m_treeview_diaries->get_path_at_pos( bx, by, path, column, cell_x, cell_y ) )
        return false;
    Gtk::TreeIter iter( m_treeview_diaries->get_model()->get_iter( path ) );
    if( !iter )
        return false;
    Gtk::TreeRow row = *( iter );
    Glib::ustring tooltip_string( row[ colrec->path ] );
    tooltip->set_text( tooltip_string );
    m_treeview_diaries->set_tooltip_row( tooltip, path );
    return true;
}
