/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeobase.hpp"
#include "view_entry.hpp"
#include "printing.hpp"


using namespace LIFEO;


EntryView *Lifeobase::view_entry = NULL;


EntryView::EntryView( void )
:	m_menu_tag( NULL ), m_tag_operation_cur( TO_NONE ), m_ptr2entry( NULL ), m_entry_sync( NULL ),
	m_flag_entrychanged( false )
{
	Entry::shower = this;
	Menubutton *mnbutton_filter( NULL );
	Gtk::MenuItem *mnitem_hide( NULL );
	Gtk::MenuItem *mnitem_hide_before( NULL );
	Gtk::MenuItem *mnitem_hide_after( NULL );
    Gtk::MenuItem *mnitem_tag_remove( NULL );

	try
	{
		Lifeobase::builder->get_widget( "vbox_entry", m_vbox );
        Menubutton::s_builder_name = "menubutton_entry_filter";
        Lifeobase::builder->get_widget_derived( "menubutton_entry_filter", mnbutton_filter );
        Lifeobase::builder->get_widget( "menuitem_entry_filter", mnitem_hide );
        Lifeobase::builder->get_widget( "menuitem_entry_filter_before", mnitem_hide_before );
        Lifeobase::builder->get_widget( "menuitem_entry_filter_after", mnitem_hide_after );
        Lifeobase::builder->get_widget( "menu_entry_tags", m_menu_tag );
        Lifeobase::builder->get_widget( "menuitem_entry_tags_remove", mnitem_tag_remove );
		Lifeobase::builder->get_widget( "button_entry_print", m_button_print );
		Lifeobase::builder->get_widget( "togglebutton_entry_favorite", m_button_toggle_favorite );
		Lifeobase::builder->get_widget( "button_entry_reset_theme", m_button_reset_theme );
		Lifeobase::builder->get_widget( "button_entry_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "button_entry_bold", m_button_bold );
		Lifeobase::builder->get_widget( "button_entry_italic", m_button_italic );
		Lifeobase::builder->get_widget( "button_entry_strikethrough", m_button_strikethrough );
		Lifeobase::builder->get_widget( "button_entry_highlight", m_button_highlight );
		Lifeobase::builder->get_widget( "label_entry_highlight", m_label_highlight );
		Lifeobase::builder->get_widget( "evntb_entry_highlight", m_eventbox_highlight );
		Lifeobase::builder->get_widget( "button_entry_add_bullet", m_button_add_bullet );
		Lifeobase::builder->get_widget( "button_entry_add_checkbox", m_button_add_checkbox );
        Lifeobase::builder->get_widget( "tgglbutton_entry_comments", m_button_toggle_comments );
		Lifeobase::builder->get_widget_derived( "textview_entry", m_textviewdiary );
		Lifeobase::builder->get_widget_derived( "entry_tag_filter", m_tag_widget );
		Lifeobase::builder->get_widget( "button_tag_operation", m_button_tag_operation );
		Lifeobase::builder->get_widget( "iconview_current_tags", m_iconview_tags );
		Lifeobase::builder->get_widget( "box_entry_toolbar", m_box_toolbar );
		Lifeobase::builder->get_widget( "hbox_entry_toolbar_text", m_hbox_editing_toolbar );
	}
	catch( ... ) {}

	// TEXT VIEW
	m_textviewdiary->drag_dest_get_target_list()->add( Lifeobase::base->drag_targets_tag );
	m_textviewdiary->drag_dest_get_target_list()->add( Lifeobase::base->drag_targets_theme );
	m_textviewdiary->drag_dest_get_target_list()->add( Lifeobase::base->drag_targets_entry );

	// TAG LIST
	m_liststore_tags = Gtk::ListStore::create( *TagPanel::colrec );
	m_iconview_tags->set_model( m_liststore_tags );
	m_iconview_tags->set_pixbuf_column( TagPanel::colrec->icon );
	m_iconview_tags->set_markup_column( TagPanel::colrec->name );
	//m_iconview_tags->set_events( Gdk::ENTER_NOTIFY_MASK );
	m_iconview_tags->drag_dest_set( Lifeobase::base->drag_targets_tag );
	Gdk::RGBA bgc;
	bgc.set_alpha( 0.0 );
	m_iconview_tags->override_background_color(
			bgc, //Lifeobase::base->get_style_context()->get_background_color( Gtk::STATE_FLAG_NORMAL ),
			Gtk::STATE_FLAG_NORMAL );
	m_iconview_tags->override_background_color(
	        Lifeobase::base->get_style_context()->get_background_color( Gtk::STATE_FLAG_SELECTED ),
	        Gtk::STATE_FLAG_SELECTED );

	// SIGNALS
	TextviewDiary::m_buffer->signal_changed().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::handle_textview_changed ) );

	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::dismiss_entry ) );
	m_button_reset_theme->signal_clicked().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::reset_theme ) );
	m_button_toggle_favorite->signal_toggled().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::toggle_favoredness ) );
	m_button_print->signal_clicked().connect(
	        sigc::mem_fun( this, &EntryView::print ) );
    mnitem_hide->signal_activate().connect( sigc::mem_fun( this, &EntryView::hide ) );
    mnitem_hide_before->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::hide_before ) );
    mnitem_hide_after->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::hide_after ) );

    mnitem_tag_remove->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::remove_selected_tags ) );

	m_button_bold->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::toggle_bold ) );
	m_button_italic->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::toggle_italic ) );
	m_button_strikethrough->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::toggle_strikethrough ) );
	m_button_highlight->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::toggle_highlight ) );

	m_button_add_bullet->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::add_bullet ) );
	m_button_add_checkbox->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer, &TextbufferDiary::add_checkbox ) );

    m_button_toggle_comments->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::handle_comments_toggled ) );

	m_textviewdiary->signal_drag_data_received().connect(
			sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );

	m_tag_widget->signal_changed().connect(
			sigc::mem_fun( this, &EntryView::handle_entry_tag_changed ) );
	m_tag_widget->signal_activate().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
	m_button_tag_operation->signal_clicked().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
	m_iconview_tags->signal_drag_data_received().connect(
			sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );
	m_iconview_tags->signal_key_release_event().connect_notify(
			sigc::mem_fun( this, &EntryView::handle_iconview_tags_key_release ) );
	m_iconview_tags->signal_button_press_event().connect_notify(
			sigc::mem_fun( this, &EntryView::handle_iconview_tags_button_press ) );
	// TODO: -MAY BE REMOVED
	//m_iconview_tags->signal_enter_notify_event().connect(
			//sigc::ptr_fun( &EntryView::handle_hboxtagtools_enternotify ) );
}

void
EntryView::populate_tags( void )
{
	Gtk::TreeRow row;
	const Tagset &ptr2tagset = m_ptr2entry->get_tags();

	m_liststore_tags->clear();

	if( ptr2tagset.size() < 1 )
	{
		row = * m_liststore_tags->append();
		row[ TagPanel::colrec->name ] = Glib::ustring::compose(
		        "<small><i>%1</i></small>", _( "Not tagged" ) );
		row[ TagPanel::colrec->ptr ] = NULL;
		return;
	}

	for( PoolTags::const_iterator iter = Diary::d->get_tags()->begin();
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		Tag *tag = iter->second;
		if( ptr2tagset.checkfor_member( tag ) )
		{
			row = * m_liststore_tags->append();

			row[ TagPanel::colrec->name ] =
					Glib::ustring::compose( "<small>%1</small>",
							Glib::Markup::escape_text( tag->get_name() ) );
			row[ TagPanel::colrec->icon ] = Lifeobase::icons->tag_16;
			row[ TagPanel::colrec->ptr ] = ( tag );
		}
	}
}

void
EntryView::sync( void )
{
	if( m_flag_entrychanged )
	{
		m_entry_sync->m_date_changed = time( NULL );
		m_entry_sync->m_text = TextviewDiary::m_buffer->get_text();
		m_flag_entrychanged = false;
		PRINT_DEBUG( "entry synced" );
	}
}

void
EntryView::focus_tag( void )
{
	//m_hbox_tagtools->show();
	m_tag_widget->grab_focus();
}

void
EntryView::remove_selected_tags( void )
{
	std::vector< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();
	for( std::vector< Gtk::TreeModel::Path >::iterator iter = selected_items.begin();
		 iter != selected_items.end();
		 ++iter )
	{
		Gtk::TreeRow row = *m_liststore_tags->get_iter( *iter );
		DiaryElement *element = row[ TagPanel::colrec->ptr ];
		Tag *tag = dynamic_cast< Tag* >( element );

		m_ptr2entry->remove_tag( tag );
		m_liststore_tags->erase( row );
	}
}

/*bool
EntryView::handle_hboxtagtools_enternotify( GdkEventCrossing *event )
{
	// TODO:
	if( event->mode == GDK_CROSSING_NORMAL )
		m_hbox_tagtools->show();
	return false;
}*/

void
EntryView::handle_login( void )
{
	m_textviewdiary->m_buffer->handle_login();

    // READ ONLY
    bool editable( ! Diary::d->is_read_only() );
    m_textviewdiary->set_editable( editable );
    m_tag_widget->set_visible( editable );
    update_formatting_toolbar();
}

void
EntryView::handle_textview_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	m_flag_entrychanged = true;

	PRINT_DEBUG( "entry changed" );

	// TODO:
	//if( m_hbox_tagtools->is_visible() )
		//m_hbox_tagtools->hide();
}

void
EntryView::handle_drag_data_received(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info,
				guint time )
{
    if( Diary::d->is_read_only() )
        return;

	context->drag_finish( true, false, time );

	Glib::ustring name = selection_data.get_data_as_string();

	if( name.empty() )	// should never happen
		return;

	switch( info )
	{
		case DRAG_TARGET_TAG_INFO:
		{
			Tag *tag = Diary::d->m_tags.find( name )->second;
			if( m_ptr2entry->add_tag( tag ) )	// if does not already have the tag
				populate_tags();
			break;
		}
		case DRAG_TARGET_THEME_INFO:
		{
            // system theme added back to m_themes:
//		    Theme *theme = ( name == STRING::SYSTEM_THEME ) ? ThemeSystem::get() :
//                Diary::d->m_themes.find( name )->second;
			Theme *theme = Diary::d->m_themes.find( name )->second;
			m_ptr2entry->set_theme( theme );
			TextviewDiary::m_buffer->set_theme( theme );
			m_button_reset_theme->set_visible( true );
			update_highlight_button();
			break;
		}
		case DRAG_TARGET_ENTRY_INFO:
		{
			DiaryElement *element = Lifeobase::panel_diary->get_dragged_elem();
			if( element != NULL )
                if( element->get_type() == DiaryElement::ET_ENTRY )
                    TextviewDiary::m_buffer->insert_link( element );
			break;
		}
	}
}

void
EntryView::toggle_favoredness( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	m_ptr2entry->m_option_favored = m_button_toggle_favorite->get_active();
	Gtk::TreeRow row = Lifeobase::base->get_element_row( m_ptr2entry );
	row[ ListData::colrec->icon ] = m_ptr2entry->get_icon();
}

void
EntryView::print( void )
{
    Lifeobase::view_entry->sync();

    Glib::RefPtr< PrintOpr > print = PrintOpr::create();

    try
    {
        print->set_show_comments( m_button_toggle_comments->get_active() );
        print->run( Gtk::PRINT_OPERATION_ACTION_PRINT_DIALOG, *Lifeobase::base );
    }
    catch( const Gtk::PrintError &ex )
    {
        // See documentation for exact Gtk::PrintError error codes:
        print_error( "An error occurred while trying to run a print operation:" +
                ex.what() );
    }
}

void
EntryView::hide( void )
{
    m_ptr2entry->m_flag_filteredout = true;
    Diary::d->activate_individual_filter();
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::hide_before( void )
{
    Diary::d->set_filter_date_begin( m_ptr2entry->get_date().m_date );
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::hide_after( void )
{
    Diary::d->set_filter_date_end( m_ptr2entry->get_date().m_date );
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::reset_theme( void )
{
	m_ptr2entry->unset_theme();
	TextviewDiary::m_buffer->set_theme( Diary::d->get_default_theme() );
	m_button_reset_theme->set_visible( false );
	update_highlight_button();
}

void
EntryView::dismiss_entry( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    // first make sure that entry to be deleted is completely out of business...
    // ...else dark forces may take action and cause SEGFAULT:
    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2entry );
	Diary::d->dismiss_entry( m_ptr2entry );

	Lifeobase::m_internaloperation++;

	Lifeobase::panel_diary->update_calendar();
	Lifeobase::panel_diary->update_entry_list();

	Lifeobase::m_internaloperation--;
}

void
EntryView::handle_comments_toggled( void )
{
    m_textviewdiary->m_buffer->show_comments( m_button_toggle_comments->get_active() );
}

void
EntryView::toggle_comments( void )
{
    m_button_toggle_comments->set_active( ! m_button_toggle_comments->get_active() );
}

bool
EntryView::check_title_applicable( const Glib::ustring &str ) const
{
    try
    {
        Date date( str );
    }
    catch( LIFEO::Error &er )
    {
        return false;
    }

    return true;
}

bool
EntryView::apply_title( const Glib::ustring &str )
{
    try
    {
        Date date( str );
        date.reset_order_1();
        Diary::d->set_entry_date( m_ptr2entry, date );
        Lifeobase::panel_diary->update_entry_list();
        Lifeobase::panel_diary->select_date_in_calendar( m_ptr2entry->get_date() );
    }
    catch( LIFEO::Error &er )
    {
        return false;
    }

    return true;
}

void
EntryView::handle_entry_tag_changed( void )
{
	Glib::ustring filter = m_tag_widget->get_text();

	if( filter.empty() )
	{
		m_button_tag_operation->hide();
		m_tag_operation_cur = TO_NONE;
	}
	else
	{
		m_button_tag_operation->show();
		Tag *tag = Diary::d->get_tags()->get_tag( filter );
		if( tag == NULL )
		{
			m_button_tag_operation->set_label( _( "Create Tag" ) );
			m_tag_operation_cur = TO_CREATE_AND_ADD;
		}
		else
		{
			if( m_ptr2entry->get_tags().checkfor_member( tag ) )
			{
				m_button_tag_operation->set_label( _( "Remove Tag" ) );
				m_tag_operation_cur = TO_REMOVE;
			}
			else
			{
				m_button_tag_operation->set_label( _( "Add Tag" ) );
				m_tag_operation_cur = TO_ADD;
			}
		}
	}
}

void
EntryView::handle_button_tag_clicked( void )
{
	Tag *tag;
	Glib::ustring filter = m_tag_widget->get_text();

	switch( m_tag_operation_cur )
	{
		case TO_NONE:
			return;	// don't even clear
		case TO_REMOVE:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2entry->remove_tag( tag );
			break;
		case TO_CREATE_AND_ADD:
			tag = Diary::d->create_tag( filter );
			m_ptr2entry->add_tag( tag );
			Lifeobase::panel_tag->populate_main();
            m_tag_widget->populate();
			break;
		case TO_ADD:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2entry->add_tag( tag );
			break;
	}

	populate_tags();
	m_tag_widget->set_text( "" );
}

void
EntryView::handle_iconview_tags_key_release( GdkEventKey *event )
{
	if( event->keyval != GDK_KEY_Delete || Diary::d->is_read_only() )	// the cheapest check first
		return;

	std::vector< Gtk::TreeModel::Path > selected_items = m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tags->get_iter( selected_items.front() )->get_value(
				TagPanel::colrec->ptr ) == NULL )
		return;

	remove_selected_tags();
}

void
EntryView::handle_iconview_tags_button_press( GdkEventButton *event )
{
	if( event->button != 3 || Diary::d->is_read_only() )
		return;

	std::vector< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tags->get_iter( selected_items.front() )->get_value(
				TagPanel::colrec->ptr ) == NULL )
		return;

	m_menu_tag->popup( event->button, event->time );
}

void
EntryView::show( Entry &entry )
{
	// do nothing if entry is already the current element:
	if( Lifeobase::panel_main->is_cur_elem( &entry ) )
		return;

	m_ptr2entry = &entry; // must be first thing

	Lifeobase::panel_main->show( this );

	// TOOLBAR
	update_highlight_button();

	// BODY
	Lifeobase::m_internaloperation++;

	m_textviewdiary->set_richtext( &entry );
	populate_tags();
	m_button_toggle_favorite->set_active( entry.m_option_favored );
	m_button_reset_theme->set_visible( entry.get_theme_is_set() );

	Lifeobase::m_internaloperation--;

    m_tag_widget->populate();

    m_entry_sync = m_ptr2entry;

	PRINT_DEBUG( "entry shown" );
}

void
EntryView::prepare_for_hiding( Entry &entry )
{
	sync();
	m_tag_widget->set_text( "" );
}

Glib::RefPtr< Gdk::Pixbuf >&
EntryView::get_icon_32( void ) const
{
    return( // TODO: m_option_favored ? Lifeobase::icons->entry_favorite_32 :
            Lifeobase::icons->entry_32 );
}

void
EntryView::update_formatting_toolbar( void )
{
	m_hbox_editing_toolbar->set_visible( Lifeobase::settings.show_formatting_toolbar &&
	                                     Diary::d->is_read_only() == false );
}

void
EntryView::update_highlight_button( void )
{
	const Theme *theme = m_ptr2entry->get_theme_is_set() ?
			m_ptr2entry->get_theme() : Diary::d->get_default_theme();
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_NORMAL );
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_PRELIGHT );
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_ACTIVE );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_NORMAL );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_PRELIGHT );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_ACTIVE );
}
