/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeobase.hpp"
#include "printing.hpp"
#include "diary.hpp"


using namespace LIFEO;


PrintOpr::PrintOpr( void )
{
    m_refSettings = Gtk::PrintSettings::create();
    m_refPageSetup = Gtk::PageSetup::create();
    set_print_settings( m_refSettings );
    set_default_page_setup( m_refPageSetup );
    set_embed_page_setup( true );
    set_track_print_status();
}

PrintOpr::~PrintOpr()
{
}

void
PrintOpr::set_entry( void )
{
    if( Lifeobase::panel_main->get_cur_elem_type() != DiaryElement::ET_ENTRY )
        throw LIFEO::Error( "Current element is not an entry. Printing aborted." );

    Entry *entry( dynamic_cast< Entry* >( Lifeobase::panel_main->get_cur_elem() ) );
    m_parser.parse( entry );
    m_marked_up_text = Glib::ustring::compose( "<b>%1</b>\n%2",
            entry->get_date().format_string(),
            m_parser.m_text_pango );
}

void
PrintOpr::set_diary( void )
{
    const CategoryChapters  *chapters( Diary::d->get_current_chapter_ctg() );
    CategoryChapters::const_reverse_iterator iter_chapter( chapters->rbegin() );
    Chapter                 *chapter( NULL );
    EntryIterReverse        iter_entry( Diary::d->get_entries().rbegin() );
    Entry                   *entry( NULL );
    bool                    flag_chapter_round( true );    // chapter or topic round

    // diary title:
    m_marked_up_text = "<span size='xx-large'>";
    m_marked_up_text += Diary::d->get_name();
    m_marked_up_text += "</span>";

    for( ; iter_entry != Diary::d->get_entries().rend(); ++iter_entry )
    {
        entry = iter_entry->second;

        while( true )
        {
            if( iter_chapter == chapters->rend() )
            {
                if( flag_chapter_round )
                {
                    chapters = Diary::d->get_topics();
                    iter_chapter = chapters->rbegin();
                    flag_chapter_round = false;
                    continue;
                }
                else
                    break;
            }

            chapter = iter_chapter->second;

            if( chapter->is_initialized() )
            {
                if( entry->get_date() >= chapter->get_date() )
                {
                    m_marked_up_text += "\n\n\n\n<span size='x-large'>";
                    m_marked_up_text += chapter->get_list_str();
                    m_marked_up_text += "</span>";
                }
                else
                    break;
            }

            ++iter_chapter;
        }

        if( m_flag_entire_diary || entry->get_filtered_out() == false )
        {
            m_parser.parse( entry );
            m_marked_up_text += Glib::ustring::compose( "\n\n\n<b>%1</b>\n%2",
                    entry->get_date().format_string(),
                    m_parser.m_text_pango );
        }
    }
}

void
PrintOpr::set_show_comments( bool flag_show )
{
    m_parser.option_show_comments = flag_show;
}

Gtk::Widget*
PrintOpr::on_create_custom_widget( void )
{
    Gtk::Box *hbox_main( new Gtk::Box );
    Gtk::Box *vbox_extent( Gtk::manage( new Gtk::Box( Gtk::ORIENTATION_VERTICAL, 5 ) ) );
    Gtk::RadioButtonGroup rgroup_extent;
    m_rbutton_entry = ( Gtk::manage(
            new Gtk::RadioButton( rgroup_extent, _( "Only Current Entry" ) ) ) );
    m_rbutton_filtered = ( Gtk::manage(
            new Gtk::RadioButton( rgroup_extent, _( "All Filtered Entries" ) ) ) );
    m_rbutton_diary = ( Gtk::manage(
            new Gtk::RadioButton( rgroup_extent, _( "Entire Diary" ) ) ) );
    Gtk::Frame *frame_extent( create_frame( _( "What to Print?" ), *vbox_extent ) );

    Gtk::Box *vbox_font( Gtk::manage( new Gtk::Box( Gtk::ORIENTATION_VERTICAL, 5 ) ) );
    m_fontbutton = Gtk::manage( new Gtk::FontButton( ThemeSystem::get()->font.to_string() ) );
    m_check_use_entry_font = Gtk::manage(
            new Gtk::CheckButton( _( "Use Theme Fonts in Entries" ) ) );
    Gtk::Frame *frame_font( create_frame( _( "Font" ), *vbox_font ) );

    hbox_main->set_border_width( 8 );
    hbox_main->set_homogeneous( true );

    vbox_extent->set_border_width( 5 );
    vbox_extent->pack_start( *m_rbutton_entry, Gtk::PACK_SHRINK );
    vbox_extent->pack_start( *m_rbutton_filtered, Gtk::PACK_SHRINK );
    vbox_extent->pack_start( *m_rbutton_diary, Gtk::PACK_SHRINK );
    hbox_main->pack_start( *frame_extent );

    vbox_font->set_border_width( 5 );
    vbox_font->pack_start( *m_fontbutton, Gtk::PACK_SHRINK );
    vbox_font->pack_start( *m_check_use_entry_font, Gtk::PACK_SHRINK );
    hbox_main->pack_start( *frame_font );

    // could not figure out how to use Gtk::Builder in conjunction with PrintDialog
    // using a dedicated ui file may be one way of doing it
//    Lifeobase::builder->get_widget( "vbox_printing", s_vbox_extras );
//    Lifeobase::builder->get_widget( "rbutton_print_entry", m_rbutton_entry );
//    Lifeobase::builder->get_widget( "rbutton_print_filtered", m_rbutton_filtered );
//    Lifeobase::builder->get_widget( "rbutton_print_diary", m_rbutton_diary );

    hbox_main->show_all_children();

    return hbox_main;
}

void
PrintOpr::on_custom_widget_apply( Gtk::Widget *widget )
{
    m_font = Pango::FontDescription( m_fontbutton->get_font_name() );
    m_parser.option_use_theme_font = m_check_use_entry_font->get_active();

    if( m_rbutton_entry->get_active() )
        set_entry();
    else
    {
        m_flag_entire_diary = m_rbutton_diary->get_active();
        set_diary();
    }
}

Glib::RefPtr< PrintOpr >
PrintOpr::create()
{
    return Glib::RefPtr< PrintOpr >( new PrintOpr() );
}

void
PrintOpr::on_done( Gtk::PrintOperationResult result )
{
    //Printing is "done" when the print data is spooled.

    if( result == Gtk::PRINT_OPERATION_RESULT_ERROR )
    {
        Gtk::MessageDialog err_dialog( *Lifeobase::base, "Printing failed", false,
                Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true );
        err_dialog.run();
    }
    else
    if( result == Gtk::PRINT_OPERATION_RESULT_APPLY )
    {
        // Update PrintSettings with the ones used in this PrintOperation:
        m_refSettings = get_print_settings();
    }

   /* if( ! is_finished())
    {
    // We will connect to the status-changed signal to track status
    // and update a status bar. In addition, you can, for example,
    // keep a list of active print operations, or provide a progress dialog.
        signal_status_changed().connect( sigc::bind( sigc::mem_fun( this,
                    &PrintOpr::handle_print_status_changed ),
                operation ) );
    }*/
}

void
PrintOpr::on_status_changed( void )
{
    if( is_finished() )
    {
        print_info( "Print job completed" );
    }
    else
    {
        print_info( get_status_string() );
    }
}

void
PrintOpr::show_page_setup( void )
{
    // Show the page setup dialog, asking it to start with the existing settings:
    Glib::RefPtr< Gtk::PageSetup > new_page_setup =
            Gtk::run_page_setup_dialog( *Lifeobase::base, m_refPageSetup, m_refSettings );

    //Save the chosen page setup dialog for use when printing, previewing, or
    //showing the page setup dialog again:
    m_refPageSetup = new_page_setup;
}

void
PrintOpr::on_begin_print( const Glib::RefPtr< Gtk::PrintContext > &print_context )
{
    //Create and set up a Pango layout for PrintData based on the passed
    //PrintContext: We then use this to calculate the number of pages needed, and
    //the lines that are on each page.
    m_refLayout = print_context->create_pango_layout();

    const double width = print_context->get_width();
    const double height = print_context->get_height();

    m_refLayout->set_width( static_cast< int >( width * Pango::SCALE ) );
    m_refLayout->set_font_description( m_font );
    m_refLayout->set_markup( m_marked_up_text );

    //Set the number of pages to print by determining the line numbers
    //where page breaks occur:
    const int line_count = m_refLayout->get_line_count();

    Glib::RefPtr< Pango::LayoutLine > layout_line;
    double page_height = 0;

    for( int line = 0; line < line_count; ++line )
    {
        Pango::Rectangle ink_rect, logical_rect;

        layout_line = m_refLayout->get_line( line );
        layout_line->get_extents( ink_rect, logical_rect );

        const double line_height = logical_rect.get_height() / 1024.0;

        if( page_height + line_height > height )
        {
            m_PageBreaks.push_back( line );
            page_height = 0;
        }

        page_height += line_height;
    }

    set_n_pages( m_PageBreaks.size() + 1 );
}

void
PrintOpr::on_draw_page(
        const Glib::RefPtr< Gtk::PrintContext >& print_context, int page_nr )
{
    // Decide which lines we need to print in order to print the specified page:
    int start_page_line = 0;
    int end_page_line = 0;

    if( page_nr == 0 )
    {
        start_page_line = 0;
    }
    else
    {
        start_page_line = m_PageBreaks[ page_nr - 1 ];
    }

    if( page_nr < static_cast< int >( m_PageBreaks.size() ) )
    {
        end_page_line = m_PageBreaks[ page_nr ];
    }
    else
    {
        end_page_line = m_refLayout->get_line_count();
    }

    // Get a Cairo Context, which is used as a drawing board:
    Cairo::RefPtr< Cairo::Context > cairo_ctx( print_context->get_cairo_context() );

    // We'll use black letters:
    cairo_ctx->set_source_rgb( 0, 0, 0 );

    // Render Pango LayoutLines over the Cairo context:
#if( ( PANGOMM_MAJOR_VERSION > 2 ) || \
     ( ( PANGOMM_MAJOR_VERSION == 2 ) && ( PANGOMM_MINOR_VERSION >= 28 ) ) )
    Pango::LayoutIter iter( m_refLayout->get_iter() );
#else
    Pango::LayoutIter iter;
    m_refLayout->get_iter( iter );
#endif

    double start_pos = 0;
    int line_index = 0;
    do
    {
        if( line_index >= start_page_line )
        {
            Glib::RefPtr< Pango::LayoutLine > layout_line = iter.get_line();
            Pango::Rectangle logical_rect = iter.get_line_logical_extents();
            int baseline = iter.get_baseline();

            if( line_index == start_page_line )
            {
                start_pos = logical_rect.get_y() / 1024.0;
            }

            cairo_ctx->move_to( logical_rect.get_x() / 1024.0,
                    baseline / 1024.0 - start_pos );

            layout_line->show_in_cairo_context( cairo_ctx );
        }

        line_index++;
    }
    while( line_index < end_page_line && iter.next_line() );
}

// PANGO ENTRY PARSER ==============================================================================
void
EntryParserPango::parse( const Entry *entry )
{
    m_text_pango.clear();
    m_ptr2entry = entry;
    if( option_use_theme_font && entry->get_theme_is_set() )
    {
        // TODO: health of the font should also be checked as it may crash the program
        m_text_pango = "<span font='";
        m_text_pango += entry->get_theme()->font.to_string();
        m_text_pango += "'>";
    }

    EntryParser::parse( 0, m_ptr2entry->get_text()->size() );

    if( option_use_theme_font && entry->get_theme_is_set() )
    {
        m_text_pango += "</span>";
    }

    PRINT_DEBUG( m_text_pango );
}

void
EntryParserPango::apply_regular( void )
{
    if( pos_regular < pos_start )
    {
        const Glib::ustring esc_text(
                m_ptr2entry->get_text()->substr( pos_regular, pos_start - pos_regular ) );
        m_text_pango.append( Glib::Markup::escape_text( esc_text ) );
    }
}

void
EntryParserPango::apply_markup( const Glib::ustring &tag_o, const Glib::ustring &tag_c,
                                int off_o, int off_c )
{
    // before adding any marked up text, last regular text chunk must be handled:
    begin_apply();  // in turn calls apply_regular()

    const Glib::ustring esc_text(
            m_ptr2entry->get_text()->substr( pos_start + off_o,
                                             pos_current - pos_start + off_c ) );
    m_text_pango += tag_o;
    m_text_pango += Glib::Markup::escape_text( esc_text );
    m_text_pango += tag_c;
}

void
EntryParserPango::apply_heading_end( void )
{
    const Glib::ustring esc_text( m_ptr2entry->get_text()->substr( 0, pos_current ) );
    m_text_pango += "<big><big><b>";
    m_text_pango += Glib::Markup::escape_text( esc_text );
    m_text_pango += "</b></big></big>";
    pos_regular = pos_current;
}

void
EntryParserPango::apply_subheading( void )
{
    // before adding any marked up text, last regular text chunk must be handled:
    begin_apply();  // in turn calls apply_regular()
    pos_regular = pos_start;

    m_text_pango += "<big><b>";
}

void
EntryParserPango::apply_subheading_end( void )
{
    if( pos_regular < pos_current )
    {
        const Glib::ustring esc_text(
                m_ptr2entry->get_text()->substr( pos_regular, pos_current - pos_regular + 1 ) );
        m_text_pango += Glib::Markup::escape_text( esc_text );
        pos_regular = pos_current + 1;
    }
    m_text_pango += "</b></big>";
}

void
EntryParserPango::apply_bold( void )
{
    apply_markup( "<b>", "</b>", 1, -1 );
}

void
EntryParserPango::apply_italic( void )
{
    apply_markup( "<i>", "</i>", 1, -1 );
}

void
EntryParserPango::apply_strikethrough( void )
{
    apply_markup( "<s>", "</s>", 1, -1 );
}

void
EntryParserPango::apply_highlight( void )
{
    apply_markup( "<span bgcolor='yellow'>", "</span>", 1, -1 );
}

void
EntryParserPango::apply_comment( void )
{
    if( option_show_comments )
        apply_markup( "<small><sup>", "</sup></small>", 1, -1 );
    else
        begin_apply();  // ignores comment
}

void
EntryParserPango::apply_check_ccl( void )
{
    const Glib::ustring esc_text(
            m_ptr2entry->get_text()->substr( pos_regular, pos_current - pos_regular ) );
    m_text_pango += Glib::Markup::escape_text( esc_text );
    pos_regular = pos_current;

    m_text_pango += "<s>";
}

void
EntryParserPango::apply_check_ccl_end( void )
{
    if( pos_regular < pos_current )
    {
        const Glib::ustring esc_text(
                m_ptr2entry->get_text()->substr( pos_regular, pos_current - pos_regular ) );
        m_text_pango += Glib::Markup::escape_text( esc_text );
        pos_regular = pos_current;
    }
    m_text_pango += "</s>";
}

void
EntryParserPango::apply_link( void )
{
    int offset1( 0 ), offset2( 1 );
    if( m_flag_hidden_link )
    {
        offset1 = pos_tab - pos_start;
        offset2 = -offset1;
    }
    apply_markup( "<span color='blue' underline='single'>", "</span>", offset1, offset2 );
}

gunichar
EntryParserPango::get_char_at( int i )
{
    return m_ptr2entry->get_text()->at( i );
}
