/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <cstdlib>
#include <cassert>

#include "lifeograph.hpp"
#include "dialog_preferences.hpp"


using namespace LIFEO;


// CONSTRUCTOR
Lifeograph::Lifeograph( int argc, char *argv[] )
:	m_about_dialog( NULL ), m_hpaned_main( NULL ), m_view_login( NULL ),
	m_theme_view( NULL ),
	m_programpath( argv[ 0 ] ),
	m_flag_force_welcome( false )
{
	Cipher::init();

	UndoManager::m = new UndoManager;	// init()??
	// FIXME:
	Diary::d = new Diary;
	ListData::colrec = new ListData::Colrec;

	// INTERFACE INITIALIZATION
	TextviewDiary::m_buffer = new TextbufferDiary; //EntryView::init();
	//m_diary_view->init();

	// CONFIGURATION
	m_flag_firsttimeuser = ! settings.read();
	set_default_size( settings.width, settings.height );
	if( settings.position_x != POSITION_NOTSET && settings.position_y != POSITION_NOTSET )
		move( settings.position_x, settings.position_y );

	// COMMANDLINE OPTIONS
	for( int i = 1; i < argc; i++ )
	{
		if( ! strcmp( argv[ i ], "--open" ) || ! strcmp( argv[ i ], "-o" ) )
		{
			if( i < argc )
			{
				if( Glib::file_test( argv[ ++i ], Glib::FILE_TEST_EXISTS ) )
				{
					if( ! Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) )
					{
						ViewLogin::m_path_cur = argv[ i ];
						m_flag_open_directly = true;
					}
				}
				if( ! m_flag_open_directly )
				    Error( "File cannot be opened" );
			}
			else
				Error( "No path provided" );
		}
		else
		if( ! strcmp( argv[ i ], "--force-welcome" ) )
		{
			m_flag_force_welcome = true;
		}
		else
		if( ! strcmp( argv[ i ], "--ignore-locks" ) )
		{
			Diary::d->m_flag_ignore_locks = true;
		}
		else
		if( Glib::file_test( argv[ i ], Glib::FILE_TEST_EXISTS ) )
		{
			if( ! Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) )
			{
			    ViewLogin::m_path_cur = argv[ i ];
				m_flag_open_directly = true;
			}
		}
	}

	// GTKBUILDER
	load_gui( UIDIR "/lifeograph.ui" );

    //Gtk::Main::signal_run().connect( sigc::mem_fun( this, &Lifeograph::on_run ) );
	// FIXME
	on_run();
}

Lifeograph::~Lifeograph( void )
{
	remove();
	delete UndoManager::m;
	if( m_view_login )
	    delete m_view_login;
	if( Diary::d ) delete Diary::d;
}

void
Lifeograph::on_run( void )
{
    // purpose of this function is calling draw functions after Gtk::Main::run()
    if( m_flag_firsttimeuser || m_flag_force_welcome )
        draw_welcomescreen();
    else
        draw_loginscreen();
}

bool
Lifeograph::on_event( GdkEvent* )
{
    if( ! settings.autologout || loginstatus != LOGGED_IN || ! Diary::d->is_encrypted() )
		return false;
	if( m_secondsremaining < LOGOUT_COUNTDOWN )
		update_title();
	// restart
	m_secondsremaining = LOGOUT_COUNTDOWN;
	m_connection_timeout.disconnect();
	m_connection_timeout = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( this, &Lifeograph::handle_idle ),
			settings.idletime - LOGOUT_COUNTDOWN );

	return false;
}

bool
Lifeograph::on_delete_event( GdkEventAny* )
{
	// BEWARE: handle_button_quit_clicked() calls this function

	PRINT_DEBUG( "on_delete_event()" );

	if( loginstatus == LOGGED_IN )
		if( ! finish_editing() )
			return true;

	// SAVE SETTINGS
	get_size( settings.width, settings.height );
	get_position( settings.position_x, settings.position_y );
	settings.write();

	return false;
}

bool
Lifeograph::handle_idle()
{
    if( ! settings.autologout )
        return false;

	if( m_secondsremaining > 0 )
	{
		set_title( Glib::ustring::compose(
				_( "<<<%1 SECONDS TO LOG OUT>>>" ), m_secondsremaining ) );
		m_connection_timeout = Glib::signal_timeout().connect_seconds(
				sigc::mem_fun( *this, &Lifeograph::handle_idle ), 1 );
		m_secondsremaining--;
		return false;
	}

	loginstatus = LOGGED_TIME_OUT;
	logout( true );
	return false;
}

bool
Lifeograph::write_backup( void )
{
	view_entry->sync();

	std::string path_backup( Diary::d->m_path + LOCK_SUFFIX );
	Result result( Diary::d->m_passphrase.empty() ?
			Diary::d->write_plain( path_backup ) : Diary::d->write_encrypted( path_backup ) );

	return( result == SUCCESS );
}

void
Lifeograph::show_about( void )
{
    if( m_about_dialog == NULL )
    {
        Lifeobase::builder->get_widget( "aboutdialog", m_about_dialog );
        m_about_dialog->set_name( PROGRAM_NAME );
        m_about_dialog->set_version( PROGRAM_VERSION_STRING );
        // XXX if needed: m_about_dialog->signal_activate_link().connect( open_url );
        m_about_dialog->set_transient_for( *this );
    }

    m_about_dialog->run();
    m_about_dialog->hide();
}

void
Lifeograph::handle_undo( void )
{
	UndoManager::m->undo();
}

void
Lifeograph::handle_redo( void )
{
	UndoManager::m->redo();
}

bool
Lifeograph::finish_editing( bool opt_save )
{
	// SAVING
	settings.position_pane = m_hpaned_main->get_position();
	settings.position_pane_tags = m_hpaned_entry->get_position();

    // files added to recent list here if not already there
    if( ! Diary::d->get_path().empty() )
        if( stock_diaries.find( Diary::d->get_path() ) == stock_diaries.end() )
            settings.recentfiles.insert( Diary::d->get_path() );

	if( ! Diary::d->is_read_only() )
	{
	    view_entry->sync();
        Diary::d->m_last_elem = panel_main->get_cur_elem()->get_id();

        if( opt_save )
        {
            if( Diary::d->write() != SUCCESS )
            {
                Gtk::MessageDialog messagedialog(	*this,
                                                    "",
                                                    false,
                                                    Gtk::MESSAGE_WARNING,
                                                    Gtk::BUTTONS_OK,
                                                    true );
                messagedialog.set_message( _( STRING::CANNOT_WRITE ) );
                messagedialog.set_secondary_text( _( STRING::CANNOT_WRITE_SUB ) );
                messagedialog.run();

                return false;
            }
        }
        else
        {
            Gtk::MessageDialog messagedialog(	*this,
                                                "",
                                                false,
                                                Gtk::MESSAGE_WARNING,
                                                Gtk::BUTTONS_CANCEL,
                                                true );
            messagedialog.set_message(
                _( "Are you sure you want to log out without saving?" ) );
            messagedialog.set_secondary_text( Glib::ustring::compose(
                _( "Your changes will be backed up in %1. "
                   "If you exit normally, your diary is saved automatically." ),
                "<b>" + Diary::d->get_path() + ".~unsaved~</b>" ), true );
            messagedialog.add_button( _( "_Log out without Saving" ), Gtk::RESPONSE_ACCEPT );

            if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
                return false;
            // back up changes
            Diary::d->write( Diary::d->get_path() + ".~unsaved~" );
        }
	}

	// CLEARING
	// TODO: m_loginstatus = LOGGING_OUT_IN_PROGRESS;

	m_signal_logout.emit();	// only for DialogEvent

	m_internaloperation++;
	panel_diary->handle_logout();
	view_entry->m_textviewdiary->m_buffer->handle_logout();
	panel_main->clear_history();

	if( loginstatus == LOGGED_IN )
		loginstatus = LOGGED_OUT;
	Diary::d->clear();
	m_connection_timeout.disconnect();
	m_connection_backup.disconnect();
	m_internaloperation--;

	return true;
}

void
Lifeograph::logout( bool opt_save )
{
    m_flag_open_directly = false;   // should be reset to prevent logging in again
    if( finish_editing( opt_save ) )
        m_view_login->handle_logout();
}

bool
Lifeograph::confirm_dismiss_element( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
							    Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return false;
	else
		return true;
}

Gtk::TreeRow
Lifeograph::get_element_row( DiaryElement *element )
{
	switch( element->get_type() )
	{
		case DiaryElement::ET_ENTRY:
		case DiaryElement::ET_CHAPTER:
			return( panel_diary->get_row( element->m_list_data->treepath ) );

		case DiaryElement::ET_TAG:
			return( panel_tag->get_row( element->m_list_data->treepath ) );
		default:
			return( Gtk::TreeRow() );
	}
}

void
Lifeograph::update_theme_list( void ) // TODO: to be removed along with theme list
{
	m_widget_panel_exp->populate_themes();
}

void
Lifeograph::update_title( void )
{
	Glib::ustring title( PROGRAM_NAME );

	if( loginstatus == LOGGED_IN )
	{
		title += " - ";
		title += Glib::filename_display_basename( Diary::d->get_path() );

		if( Diary::d->is_read_only() )
		{
		    title += " <";
		    title += "Read Only";
		    title += ">";
		}
	}

	set_title( title );
}

// MODE DRAWING FUNCTIONS
void
Lifeograph::draw_welcomescreen( void )
{
	Gtk::Alignment	*alignment =
			Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.1, 0 ) );
	Gtk::Label		*label_salutation =
			Gtk::manage( new Gtk::Label( _( STRING::SALUTATION ), 0.5, 0.5 ) );
	Gtk::Button		*button_start =
			Gtk::manage( new Gtk::Button( _("_Continue"), true ) );
	Gtk::Image		*icon_forward =
			Gtk::manage( new Gtk::Image( Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_MENU ) );
	Gtk::Box		*vbox =
			Gtk::manage( new Gtk::Box( Gtk::ORIENTATION_VERTICAL, 50 ) );

	alignment->set_padding( 120, 120, 150, 150 );
	add( *alignment );

	label_salutation->set_use_markup( true );

	button_start->set_image( *icon_forward );
	button_start->set_image_position( Gtk::POS_RIGHT );

	vbox->pack_start( *label_salutation );
	vbox->pack_start( *button_start );

	alignment->add( *vbox );

	button_start->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::draw_loginscreen ) );

	show_all();
}

void
Lifeograph::draw_loginscreen( void )
{
    // only to be called on start up
    assert( m_view_login == NULL );

    // DETECT STOCK DIARIES
    Gio::init();

    try
    {
        Glib::RefPtr< Gio::File > directory = Gio::File::create_for_path( DIARYDIR );
        if( directory )
        {
            Glib::RefPtr< Gio::FileEnumerator > enumerator = directory->enumerate_children();
            if( enumerator )
            {
                Glib::RefPtr< Gio::FileInfo > file_info = enumerator->next_file();
                while( file_info )
                {
                    std::string path( DIARYDIR );
                    path += "/";
                    path += file_info->get_name();
                    //std::cout << "file: " << file_info->get_name() << std::endl;
                    stock_diaries.insert( path );
                    file_info = enumerator->next_file();
                }
            }
        }
    }
    catch( const Glib::Exception& ex )
    {
        LIFEO::Error( ex.what() );
    }

    try
    {
        m_view_login = new ViewLogin;
        m_view_login->handle_start();
    }
    catch( ... )
    {
        throw LIFEO::Error( "creation of login view failed" );
    }
}

void
Lifeograph::draw_editscreen( void )
{
	remove();

	if( m_hpaned_main == NULL )
	{
        Gtk::MenuItem               *menuitem_preferences;
        Gtk::MenuItem               *menuitem_about;

		try
		{
			Lifeobase::builder->get_widget( "hpaned_main", m_hpaned_main );
			// TOOLBAR AND ITS ITEMS
			Lifeobase::builder->get_widget( "button_logout", m_button_logout );

            Lifeobase::builder->get_widget( "menuitem_preferences", menuitem_preferences );
            Lifeobase::builder->get_widget( "menuitem_about", menuitem_about );

            // TIME BASED NAVIGATION PANE
            Lifeobase::builder->get_widget( "hpaned_second", m_hpaned_entry );

			panel_main = new PanelMain;
			panel_diary = new PanelDiary;
			panel_tag = new TagPanel;
			m_widget_panel_exp = new WidgetPanelExp;
			m_diary_view = new DiaryView;
			m_theme_view = new ThemeView;
			m_category_tags_view = new CategoryTagsView;
			m_chapter_view = new ChapterView;
			view_entry = new EntryView;
			view_tag = new TagView;
		}
		catch( std::exception &ex )
		{
			throw ex;
		}

        // GEOMETRY
        m_hpaned_main->set_position( settings.position_pane );
        m_hpaned_entry->set_position( settings.position_pane_tags );

		m_button_logout->add_accelerator(
				"clicked",
				get_accel_group(),
				GDK_KEY_Escape,
				Gdk::CONTROL_MASK,
				Gtk::ACCEL_VISIBLE );

		// ACTIONS
        create_action( m_action_undo,
                       "Undo",
                       Gtk::Stock::UNDO,
                       _( "Undo" ),
                       "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK ),
                       sigc::mem_fun( *this, &Lifeograph::handle_undo ) );

        create_action( m_action_redo,
                       "Redo",
                       Gtk::Stock::REDO,
                       _( "Redo" ),
                       "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK | Gdk::SHIFT_MASK ),
                       sigc::mem_fun( *this, &Lifeograph::handle_redo ) );

		// SIGNALS
		menuitem_about->signal_activate().connect(
				sigc::mem_fun( this, &Lifeograph::show_about ) );
		menuitem_preferences->signal_activate().connect(
				sigc::ptr_fun( &DialogPreferences::create ) );

		m_button_logout->signal_clicked().connect(
				sigc::bind( sigc::mem_fun( *this, &Lifeograph::logout ), true ) );
	}
	else
	{
		m_internaloperation++;
		panel_diary->m_entry_filter->set_text( "" );
		panel_diary->m_entry_replace->set_text( "" );
		m_internaloperation--;
	}

	m_connection_backup = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( this, &Lifeograph::write_backup ),
			BACKUP_INTERVAL );

	add( *m_hpaned_main );

	// LOGIN
	m_internaloperation++;

	view_tag->handle_login();
	m_diary_view->handle_login();
	view_entry->handle_login();
	panel_diary->handle_login();
	panel_tag->populate_main();
	m_widget_panel_exp->populate_themes();

	m_internaloperation--;
	loginstatus = LOGGED_IN;
	update_title();
}
