/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeobase.hpp"
#include "view_entry.hpp"
#include "gui_tag.hpp"


using namespace LIFEO;


// TAG WIDGET ======================================================================================
/*Tagwidget::Tagwidget( OperationType type, PoolTags *pool )
	:	m_entry( _( "Tag name" ) ),
	m_entrycompletion( Gtk::EntryCompletion::create() ),
	m_flag_activatable( false ), m_flag_empty( true ), m_operation( type ),
	m_ptr2tagpool( pool )
{
	Gtk::Bin::remove();
	// new entry widget is added in on_remove() to prevent warnings

	m_entrycompletion->set_popup_completion( true );
	m_entrycompletion->set_popup_single_match( true );
	m_entrycompletion->set_match_func(
			sigc::mem_fun( *this, &Tagwidget::compare_text ) );

	m_entry.set_completion( m_entrycompletion );
	m_entry.set_icon_from_stock(	Gtk::Stock::NO,
									Gtk::ENTRY_ICON_SECONDARY );

	m_entry.signal_activate().connect(
			sigc::mem_fun( *this, &Tagwidget::on_activate ) );

	m_entry.signal_icon_release().connect(
			sigc::mem_fun( *this, &Tagwidget::handle_icon_release ) );
}

void
Tagwidget::on_remove( Gtk::Widget* widget )
{
	// this is the only solution I could find to prevent Gtk from...
	// ...complaining when a ComboBoxText's Entry is removed from it
	if( widget != &m_entry )
		Gtk::Container::add( m_entry );
	else
		Gtk::Container::on_remove( widget );
}

void
Tagwidget::set_model( const Glib::RefPtr< Gtk::TreeModel > &model )
{
	ComboBoxEntry::set_model( model );
	m_entrycompletion->set_model( model );
	m_model = model;
}

void
Tagwidget::set_text_column( const Gtk::TreeModelColumnBase& column )
{
	ComboBoxEntry::set_text_column( column );
	m_entrycompletion->set_text_column( column );
}

void
Tagwidget::on_activate( void )
{
	if( m_flag_activatable )
		m_signal_activate.emit();
}

void
Tagwidget::on_changed( void )
{
	if( ! m_model )
		return;

	// add a tag directly if it was selected from the combobox menu
	if( get_active_row_number() >= 0 )
	{
		m_signal_activate.emit();
		return;
	}

	Glib::ustring tagname = m_entry.get_text();

	m_flag_activatable = ( tagname.size() > 0 );
	m_flag_empty = ! m_flag_activatable;

	if( m_flag_empty )
	{
		m_entry.set_icon_from_stock(	Gtk::Stock::NO,
										Gtk::ENTRY_ICON_SECONDARY );
	}
	else
	{
		Tag *tag = m_ptr2tagpool->get_tag( tagname );
		if( tag == NULL )
		{
			m_entry.set_icon_from_stock(
					m_operation == OT_EDIT ? Gtk::Stock::OK : Gtk::Stock::NEW,
					Gtk::ENTRY_ICON_SECONDARY );
		}
		else
		if( m_operation == OT_EDIT )
		{
			m_entry.set_icon_from_stock(	Gtk::Stock::NO,
											Gtk::ENTRY_ICON_SECONDARY );
			m_flag_activatable = false;
		}
		else
		{
			bool has_tag = true;
			for(	Gtk::TreeIter iter_tags = get_model()->children().begin();
					iter_tags != get_model()->children().end();
					iter_tags++ )
			{
				if( ( *iter_tags )[ TagPanel::colrec->ptr ] == tag )
				{
					// if it is in the available tags list, entry does not have it
					has_tag = false;
					break;
				}
			}

			if( has_tag )
				m_entry.set_icon_from_stock(	Gtk::Stock::REMOVE,
												Gtk::ENTRY_ICON_SECONDARY );
			else
				m_entry.set_icon_from_stock(	Gtk::Stock::ADD,
												Gtk::ENTRY_ICON_SECONDARY );
		}
	}

	m_entry.set_icon_activatable( m_flag_activatable, Gtk::ENTRY_ICON_SECONDARY );

	Gtk::ComboBoxEntry::on_changed();
}

void
Tagwidget::handle_icon_release( Gtk::EntryIconPosition, const GdkEventButton* event )
{
	if( event->button == 1 )	// left click
		on_activate();
}
*/
// TAG WIDGET ==================================================================
TagWidget::TagWidget( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& b )
:	EntryIdletext( cobject, b ), m_completion( Gtk::EntryCompletion::create() )
{
	set_placeholder_text( _( "Tag name" ) );

	m_completion->set_popup_completion( true );
	m_completion->set_popup_single_match( true );
	m_completion->set_match_func(
			sigc::mem_fun( *this, &TagWidget::compare_text ) );

	m_liststore = Gtk::ListStore::create( *TagPanel::colrec );
	m_completion->set_model( m_liststore );
	m_completion->set_text_column( TagPanel::colrec->name );

	set_completion( m_completion );
}

bool
TagWidget::compare_text( const Glib::ustring &text,
						 const Gtk::TreeModel::const_iterator &iter )
{
	Glib::ustring tagname = iter->get_value( TagPanel::colrec->name );
	if( text.size() == 1 )
		return(	Glib::Unicode::tolower( tagname[ 0 ] ) ==
				Glib::Unicode::tolower( text[ 0 ] ) );
	else
		return( tagname.lowercase().find( text.lowercase() ) != std::string::npos );
}

void
TagWidget::populate( const Tagset *set )
{
	m_liststore->clear();

	Gtk::TreeRow row;

	for( Tagset::const_iterator iter = set->begin();
		 iter != set->end();
		 ++iter )
	{
		row = *( m_liststore->append() );
		row[ TagPanel::colrec->name ] = ( *iter )->get_name();
	}
}

void
TagWidget::populate( void )
{
	m_liststore->clear();

	Gtk::TreeRow row;

	for( PoolTags::const_iterator iter = Diary::d->get_tags()->begin();
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		row = *( m_liststore->append() );
		row[ TagPanel::colrec->name ] = ( *iter ).second->get_name();
	}
}

// TAG PANEL =======================================================================================
// STATIC MEMBERS
TagPanel                *Lifeobase::panel_tag = NULL;
TagPanel::Colrec		*TagPanel::colrec;

TagPanel::TagPanel( void )
:	m_treestore_all_tags( NULL ), m_tag_dragged( NULL )
{
	Gtk::CellRendererPixbuf *cellrenderer_icon( NULL );
	Gtk::CellRendererText *cellrenderer_name( NULL );
	Gtk::TreeViewColumn *column_name_all( NULL );

	try
	{
		colrec = new Colrec;

		Lifeobase::builder->get_widget( "treeview_all_tags", m_treeview_all_tags );
		Lifeobase::builder->get_widget( "hbox_tagbar", m_hbox_tagbar );

		cellrenderer_icon = Gtk::manage( new Gtk::CellRendererPixbuf );
		cellrenderer_name = Gtk::manage( new Gtk::CellRendererText );
		column_name_all = Gtk::manage( new Gtk::TreeViewColumn( "" ) );
	}
	catch( ... ) { }

	cellrenderer_name->property_ellipsize() = Pango::ELLIPSIZE_END;
	cellrenderer_name->property_scale() = .90;

	column_name_all->pack_start( *cellrenderer_icon, Gtk::PACK_SHRINK );
	column_name_all->pack_start( *cellrenderer_name );
	column_name_all->add_attribute( cellrenderer_icon->property_pixbuf(),
							TagPanel::colrec->icon );
	column_name_all->add_attribute( cellrenderer_name->property_markup(),
							TagPanel::colrec->name );

	m_treestore_all_tags = Gtk::TreeStore::create( *TagPanel::colrec );

	// ALL TAGS
	m_treeview_all_tags->set_model( m_treestore_all_tags );
	m_treeview_all_tags->append_column( *column_name_all );
	m_treeview_all_tags->enable_model_drag_source( Lifeobase::base->drag_targets_tag );
	m_treeview_all_tags->override_background_color(
            Lifeobase::base->get_style_context()->get_background_color( Gtk::STATE_FLAG_NORMAL ),
            Gtk::STATE_FLAG_NORMAL );

	// SIGNALS
	m_treeview_all_tags->signal_row_activated().connect(
			sigc::mem_fun( *this, &TagPanel::handle_treeview_row_activated ) );

	m_treeview_all_tags->signal_button_release_event().connect_notify(
			sigc::mem_fun( this, &TagPanel::handle_treeview_button_release ) );

	m_treeview_all_tags->signal_button_press_event().connect_notify(
			sigc::mem_fun( this, &TagPanel::handle_treeview_button_press ) );

	m_treeview_all_tags->signal_drag_begin().connect_notify(
			sigc::mem_fun( this, &TagPanel::handle_treeview_drag_begin ) );

	m_treeview_all_tags->signal_drag_data_get().connect_notify(
			sigc::mem_fun( this, &TagPanel::handle_drag_data_get ) );

	m_treeview_all_tags->signal_row_expanded().connect(
			sigc::mem_fun( this, &TagPanel::handle_treeview_row_expanded ) );
	m_treeview_all_tags->signal_row_collapsed().connect(
			sigc::mem_fun( this, &TagPanel::handle_treeview_row_expanded ) );
}

void
TagPanel::populate_main( void )
{
	Lifeobase::m_internaloperation++;

	m_treestore_all_tags->clear();

	Gtk::TreeRow row, row_category;

	if( Diary::d->get_tags()->size() < 1 )
	{
		row = * m_treestore_all_tags->append();
		row[ colrec->name ] = Glib::ustring::compose( "<i>%1</i>", _( "None" ) );
		row[ colrec->ptr ] = NULL;
		Lifeobase::m_internaloperation--;
		return;
	}

	// ROOT TAGS
	for( PoolTags::const_iterator iter( Diary::d->get_tags()->begin() );
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		Tag *tag( iter->second );
		if( tag->get_category() == NULL )
		{
			row = * m_treestore_all_tags->append();
			row[ colrec->name ] = Glib::Markup::escape_text( tag->get_name() );
			row[ colrec->icon ] = ( tag == Diary::d->get_filter_tag() ?
									Lifeobase::icons->filter_16 :
									Lifeobase::icons->tag_16 );
			row[ colrec->ptr ] = ( tag );

			tag->m_list_data->treepath = m_treestore_all_tags->get_path( row );
		}
	}

	// CATEGORIES
	for( PoolCategoriesTags::const_iterator iter_category( Diary::d->m_tag_categories.begin() );
		 iter_category != Diary::d->m_tag_categories.end();
		 ++iter_category )
	{
		CategoryTags *category = iter_category->second;
		row_category = * m_treestore_all_tags->append();
		row_category[ colrec->name ] = Glib::ustring::compose( "<b>%1</b>",
				Glib::Markup::escape_text( category->get_name() ) );
		row_category[ colrec->icon ] = category->get_icon();
		row_category[ colrec->ptr ] = category;

		for( CategoryTags::const_iterator iter_tag = category->begin();
			 iter_tag != category->end();
			 ++iter_tag )
		{
			Tag *tag = *iter_tag;

			row = * m_treestore_all_tags->append( row_category.children() );
			row[ colrec->name ] = Glib::Markup::escape_text( tag->get_name() );
			row[ colrec->icon ] =  ( tag == Diary::d->get_filter_tag() ?
									Lifeobase::icons->filter_16 :
									Lifeobase::icons->tag_16 );
			row[ colrec->ptr ] = ( tag );

			tag->m_list_data->treepath = m_treestore_all_tags->get_path( row );
		}

		if( category->m_flag_expanded )
			m_treeview_all_tags->expand_row(
					m_treestore_all_tags->get_path( row_category ), false );
	}

	Lifeobase::m_internaloperation--;
}

void
TagPanel::update_tag_icon( const Tag *tag )
{
	Gtk::TreeRow row( * m_treestore_all_tags->get_iter( tag->m_list_data->treepath ) );
	if( tag == Diary::d->get_filter_tag() )
		row[ colrec->icon ] = Lifeobase::icons->filter_16;
	else
		row[ colrec->icon ] = tag->get_icon();
}

Gtk::TreeRow
TagPanel::get_row( const Gtk::TreePath &path )
{
	return( * m_treestore_all_tags->get_iter( path ) );
	// TODO: check validity
}

void
TagPanel::handle_treeview_drag_begin( const Glib::RefPtr< Gdk::DragContext >& )
{
	m_flag_dragging = true;
	m_treeview_all_tags->get_selection()->unselect_all();
}

void
TagPanel::handle_drag_data_get( const Glib::RefPtr<Gdk::DragContext> &context,
								Gtk::SelectionData &data,
								guint info, guint time )
{
	if( m_tag_dragged )
		data.set( TARGET_ENTRY_TAG, m_tag_dragged->get_name() );
	else	// should never be the case
	{
		data.set( "N/A", "" );
		PRINT_DEBUG( "Nothing is selected at the drag source" );
	}
}

void
TagPanel::handle_treeview_button_press( GdkEventButton *event )
{
	m_flag_dragging = false;

	Gtk::TreePath path;
	if( m_treeview_all_tags->get_path_at_pos( event->x, event->y, path ) )
	{
		Gtk::TreeRow row = * m_treestore_all_tags->get_iter( path );
		DiaryElement *element = row[ colrec->ptr ];
		if( element == NULL )
			m_tag_dragged = NULL;
		else
		if( element->get_type() == DiaryElement::ET_TAG )
			m_tag_dragged = dynamic_cast< Tag* >( element );
		else
			m_tag_dragged = NULL;
	}
	else
		m_tag_dragged = NULL;
}

void
TagPanel::handle_treeview_button_release( GdkEventButton *event )
{
	if( m_flag_dragging || event->button != 1 )
		return;

	Glib::RefPtr< Gtk::TreeSelection > selection = m_treeview_all_tags->get_selection();
	if( selection->count_selected_rows() > 0 )
	{
		Gtk::TreeRow row = ( *selection->get_selected() );

		DiaryElement *tag = row[ colrec->ptr ];
		if( tag )	// if not None
			tag->show();
	}
	selection->unselect_all();
}

void
TagPanel::handle_treeview_row_expanded( const Gtk::TreeIter &iter,
										const Gtk::TreePath &path )
{
	if( Lifeobase::m_internaloperation )
		return;

	DiaryElement *element = ( * iter )[ colrec->ptr ];
	if( element->get_type() == DiaryElement::ET_TAG_CTG )
	{
		CategoryTags *category = dynamic_cast< CategoryTags* >( element );
		category->m_flag_expanded = m_treeview_all_tags->row_expanded( path );
	}
}

void
TagPanel::handle_treeview_row_activated( const Gtk::TreePath &path, Gtk::TreeViewColumn* )
{
	Gtk::TreeRow row( * m_treestore_all_tags->get_iter( path ) );
	DiaryElement *element( row[ colrec->ptr ] );

	if( element->get_type() == DiaryElement::ET_TAG )
	{
		Tag *tag( dynamic_cast< Tag* >( element ) );

		if( tag == Diary::d->get_filter_tag() )	// remove filter
		{
			Diary::d->set_filter_tag( NULL );
			row[ colrec->icon ] = Lifeobase::icons->tag_16;
		}
		else
		{
			if( Diary::d->get_filter_tag() != NULL )
			{
				Gtk::TreeRow row_old( * m_treestore_all_tags->get_iter(
						Diary::d->get_filter_tag()->m_list_data->treepath ) );
				row_old[ colrec->icon ] = Lifeobase::icons->tag_16;
			}
			Diary::d->set_filter_tag( tag );
			row[ colrec->icon ] = Lifeobase::icons->filter_16;
		}
        Lifeobase::view_tag->update_button_filter_tag();
        Lifeobase::panel_diary->update_button_remove_filters();
        Lifeobase::panel_diary->update_entry_list();
	}
}
