/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "widget_entrylist.hpp"
#include "view_entry.hpp"


using namespace LIFEO;


// WIDGET ENTRY LIST ===========================================================
WidgetEntryList::WidgetEntryList( BaseObjectType* cobject,
								  const Glib::RefPtr<Gtk::Builder>& )
:	Gtk::TreeView( cobject )
{
// CELL RENDERERS
	Gtk::CellRendererPixbuf *cellrenderer_pixbuf = Gtk::manage(
			new Gtk::CellRendererPixbuf );
	Gtk::CellRendererText *cellrenderer_text = Gtk::manage(
			new Gtk::CellRendererText );
	Gtk::TreeViewColumn *column = Gtk::manage(
			new Gtk::TreeViewColumn( /*_( "Date" )*/ ) );

	column->pack_start( *cellrenderer_pixbuf, false );
	column->pack_start( *cellrenderer_text );
	column->add_attribute( cellrenderer_pixbuf->property_pixbuf(),
								ListData::colrec->icon );
	column->add_attribute( cellrenderer_text->property_markup(),
								ListData::colrec->info );

	cellrenderer_text->property_ellipsize() = Pango::ELLIPSIZE_END;
	cellrenderer_text->property_scale() = .90;
	cellrenderer_text->set_fixed_height_from_font( 1 );

	m_treestore_entries = Gtk::TreeStore::create( *ListData::colrec );
	m_treemodelsort_entries = Gtk::TreeModelSort::create( m_treestore_entries );
	m_treemodelsort_entries->set_sort_column(
			Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING );

	set_model( m_treemodelsort_entries );
	append_column( *column );

	enable_model_drag_source( Lifeobase::base->drag_targets_entry );
	enable_model_drag_dest( Lifeobase::base->drag_targets_entry, Gdk::ACTION_MOVE );
}

WidgetEntryList::~WidgetEntryList( void )
{
}

void
WidgetEntryList::present_element( const DiaryElement *element )
{
	Gtk::TreePath path( m_treemodelsort_entries->convert_child_path_to_path( 
			element->m_list_data->treepath ) );
	//Gtk::TreePath path( find_element( element ) );
	expand_to_path( path );
	if( ! get_selection()->is_selected( path ) )
		get_selection()->select( path );
	scroll_to_row( path);	// does nothing if already visible
}

void
WidgetEntryList::expand_element( const DiaryElement *element )
{
	Gtk::TreePath path( m_treemodelsort_entries->convert_child_path_to_path(
			element->m_list_data->treepath ) );
			//find_element( element ) );
	expand_to_path( path );
}

void
WidgetEntryList::go_up( bool flag_entry_operation )
{
    DiaryElement *elem( Lifeobase::base->get_cur_elem() );

    if( elem == NULL )
        return;
    else
    if( elem->get_type() < DiaryElement::IT_DIARY ) // not list element
        return;

    Gtk::TreePath path( m_treemodelsort_entries->convert_child_path_to_path(
                elem->m_list_data->treepath ) );

    do
    {
        if( ! path.prev() )
        {
            if( path.size() > 1 )
                path.up();
            else    // diary
            if( make_path_deeper_last( path ) )
                make_path_deeper_last( path ); // go still deeper if possible:
        }
        else
            make_path_deeper_last( path );

        elem = ( * m_treemodelsort_entries->get_iter( path ) )[ ListData::colrec->ptr ];
    }
    // repeat until an entry is found if that is what we are looking for
    // BEWARE: do not set this flag when there is no entry in the list
    while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

    PRINT_DEBUG( "previous path: " + path.to_string() );

    Gtk::TreeRow row( * m_treemodelsort_entries->get_iter( path ) );
    DiaryElement *element( row[ ListData::colrec->ptr ] );
    element->show();
}

void
WidgetEntryList::go_down( bool flag_entry_operation )
{
    DiaryElement *elem( Lifeobase::base->get_cur_elem() );

    if( elem == NULL )
        return;
    else
    if( elem->get_type() < DiaryElement::IT_DIARY ) // not list element
        return;

    Gtk::TreePath path( m_treemodelsort_entries->convert_child_path_to_path(
                elem->m_list_data->treepath ) );

    do
    {
        if( ! m_treemodelsort_entries->get_iter( path )->children().empty() )
            path.down();
        else
        if( ! move_path_next( path ) )
        {
            while( path.size() > 1 )
            {
                path.up();
                if( move_path_next( path ) )
                    break;
            }
        }

        elem = ( * m_treemodelsort_entries->get_iter( path ) )[ ListData::colrec->ptr ];
    }
    // repeat until an entry is found if that is what we are looking for
    // BEWARE: do not set this flag when there is no entry in the list
    while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

    PRINT_DEBUG( "next path: " + path.to_string() );

    Gtk::TreeRow row( * m_treemodelsort_entries->get_iter( path ) );
    DiaryElement *listitem = row[ ListData::colrec->ptr ];
    listitem->show();
}

void
WidgetEntryList::handle_sorting_criteria_changed( SortingCriteria sc )
{
	if( sc == Diary::d->get_sorting_criteria() )
		return;

	Diary::d->set_sorting_criteria( sc );

	switch( sc )
	{
		//case ST_NOT_SET:
		case SC_DATE:
			m_treemodelsort_entries->set_default_sort_func(
					sigc::mem_fun( *this, &WidgetEntryList::sort_by_date ) );
			break;
		case SC_SIZE:
			m_treemodelsort_entries->set_default_sort_func(
					sigc::mem_fun( *this, &WidgetEntryList::sort_by_size ) );
			break;
	}

	Lifeobase::base->update_entry_list();
}

void
WidgetEntryList::set_sorting_criteria( void )
{
	switch( Diary::d->get_sorting_criteria() )
	{
		//case ST_NOT_SET:
		case SC_DATE:
			m_treemodelsort_entries->set_default_sort_func(
					sigc::mem_fun( *this, &WidgetEntryList::sort_by_date ) );
			break;
		case SC_SIZE:
			m_treemodelsort_entries->set_default_sort_func(
					sigc::mem_fun( *this, &WidgetEntryList::sort_by_size ) );
			break;
	}
}

// not used now but may be used in the future
Gtk::TreePath
WidgetEntryList::find_element( const DiaryElement *element ) const
{
	Gtk::TreeIter iter_diary( m_treestore_entries->children().begin() );
	if( element == ( *iter_diary )[ ListData::colrec->ptr ] )
		return m_treestore_entries->get_path( iter_diary );

	for( Gtk::TreeIter iter = ( *iter_diary )->children().begin();
		 iter != ( *iter_diary )->children().end();
		 ++iter )
	{
		const DiaryElement *list_elem( ( *iter )[ ListData::colrec->ptr ] );
		if( element->get_id() == list_elem->get_id() )
		{
			return m_treestore_entries->get_path( iter );
		}
		else
		{
			if( element->get_type() == DiaryElement::IT_ENTRY &&
				list_elem->get_type() != DiaryElement::IT_ENTRY )
				if( element->get_date() > list_elem->get_date() )
					for( Gtk::TreeIter itr2 = ( *iter )->children().begin();
						 itr2 != ( *iter )->children().end();
						 ++itr2 )
					{
						if( element == ( *itr2 )[ ListData::colrec->ptr ] )
							return m_treestore_entries->get_path( itr2 );
					}
		}
	}

	return m_treestore_entries->get_path( m_treemodelsort_entries->children().begin() );
}

bool
WidgetEntryList::on_button_press_event( GdkEventButton *event )
{
	m_flag_dragging = false;

	Gtk::TreePath path;
	if( get_path_at_pos( event->x, event->y, path ) )
	{
		Gtk::TreeRow row = * m_treemodelsort_entries->get_iter( path );
		m_element_dragged = row[ ListData::colrec->ptr ];
	}
	else
		m_element_dragged = NULL;

/*	if( event->button != 3 )
		return Gtk::TreeView::on_button_press_event( event );

	Glib::RefPtr< Gtk::TreeSelection > selection = get_selection();

	if( selection->count_selected_rows() <= 0 )
		return Gtk::TreeView::on_button_press_event( event );

	// treeview issues selection changed signal after button press signal.
	// this causes context menus to be linked to the previously selected item
	// when a row is selected with right click.
	// to get around this we use a somewhat dirty hack and select the row
	// from within the button press event handler
	// p.s.: i dunno but there might be a simpler way of doing this
	Gtk::TreeModel::Path path;
	Gtk::TreeViewColumn  *column;
	int i, j;
	if( ! get_path_at_pos( (int) event->x, (int) event->y, path, column, i, j ) )
		return Gtk::TreeView::on_button_press_event( event );

	set_cursor( path );
	// end of dirty hack

	Gtk::TreeRow row = *( selection->get_selected() );
	DiaryElement *item = row[ ListData::colrec->ptr ];

	if( ! (	item->get_type() == DiaryElement::IT_ENTRY ||
			item->get_type() == DiaryElement::IT_TAG ) )
		return Gtk::TreeView::on_button_press_event( event );

	if( m_menu )
	{
		delete m_menu;
		m_menu = NULL;
	}

	m_menu = new Gtk::Menu;
	Gtk::Image *imageDismiss = Gtk::manage(
			new Gtk::Image( Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU ) );

	if( item->get_type() == DiaryElement::IT_ENTRY )
	{
		Entry *entry = dynamic_cast< Entry* >( item );


		m_menu->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem(
				entry->is_favored() ?
						_( "Remove from _Favorites" ) : _( "Add to _Favorites" ),
				*image_favored,
				sigc::mem_fun( *entry, &Entry::toggle_favoredness ) ) );
	}
	m_menu->accelerate( *this );
	m_menu->popup( event->button, event->time );
*/
	return Gtk::TreeView::on_button_press_event( event );
}

bool
WidgetEntryList::on_button_release_event( GdkEventButton *event )
{
	if( ! m_flag_dragging && event->button == 1 )
	{
		if( m_element_dragged )
		{
				m_element_dragged->show();
		}
		else
			Diary::d->show();	// if nothing is selected show the diary view
	}

	return Gtk::TreeView::on_button_release_event( event );
}

bool
WidgetEntryList::on_key_press_event( GdkEventKey *event )
{
    if( event->state == 0 )
    {
        switch( event->keyval )
        {
            case GDK_Home:
                Diary::d->show();
                return true;
            case GDK_End:
                return true;
            case GDK_Up:
                go_up( false );
                return true;
            case GDK_Down:
                go_down( false );
                return true;
        }
    }
    return Gtk::TreeView::on_key_press_event( event );
}

void
WidgetEntryList::on_drag_begin( const Glib::RefPtr< Gdk::DragContext >& )
{
	m_flag_dragging = true;
}

void
WidgetEntryList::on_drag_data_get( const Glib::RefPtr< Gdk::DragContext > &context,
								   Gtk::SelectionData &data,
								   guint info, guint time )
{
	if( m_element_dragged )
		data.set( TARGET_ENTRY_ENTRY, "element" );
	else	// should never be the case
	{
		data.set( "N/A", "" );
		PRINT_DEBUG( "Nothing is selected at the drag source" );
	}
}

bool
WidgetEntryList::on_drag_motion( const Glib::RefPtr< Gdk::DragContext > &context,
								 int x, int y, guint time )
{
	if( ! m_flag_dragging )
		return false;

	bool retval( true );
	Gtk::TreePath path;

	// AUTOSCROLL
	Gdk::Rectangle rect;
	get_visible_rect( rect );
	int new_y = rect.get_y();
	if( y < ( rect.get_height() * 0.15 ) )
	{
		if( new_y > 0 )
		{
			new_y -= 5;
			if( new_y < 0 )
				new_y = 0;
			scroll_to_point( -1, new_y );
		}
	}
	else
	if( y > ( rect.get_height() * 0.85 ) )
	{
		if( get_vadjustment()->get_value() < get_vadjustment()->get_upper() )
		{
			new_y += 5;
			scroll_to_point( -1, new_y );
		}
	}

	if( get_path_at_pos( x, y, path ) )
	{
		Gtk::TreeRow row = * m_treemodelsort_entries->get_iter( path );
		DiaryElement *element( row[ ListData::colrec->ptr ] );
		if( element->get_type() == DiaryElement::IT_DIARY ||
			m_element_dragged->get_type() == DiaryElement::IT_DIARY )
			retval = false;
		// chapters cannot be dropped
		else
		if( m_element_dragged->get_type() == DiaryElement::IT_CHAPTER )/*&&
			! m_element_dragged->get_date().is_ordinal() )*/
			retval = false;
		// chapters can only be dropped on ordinal chapters
		/*else
		if( m_element_dragged->get_type() == DiaryElement::IT_CHAPTER &&
			( element->get_type() != DiaryElement::IT_CHAPTER ||
			  ! element->get_date().is_ordinal() ) )
			retval = false;*/
		// do not allow dropping on its own folder
		else
		if( element->get_type() == DiaryElement::IT_CHAPTER &&
			element->get_date().get_pure() == m_element_dragged->get_date().get_pure() )
			retval = false;
		// do not allow dropping on itself
		else
		if( element == m_element_dragged )
			retval = false;
		else
			m_element_drop_target = element;
	}
	else
		retval = false;

	if( retval )
	{
		context->drag_status( Gdk::ACTION_MOVE, time );
		get_selection()->select( path );
	}
	else
		get_selection()->unselect_all();

	//return Gtk::TreeView::on_drag_motion( context, x, y, time );
	return retval;
}

bool
WidgetEntryList::on_drag_drop( const Glib::RefPtr< Gdk::DragContext > &context,
							   int x, int y, guint time )
{
	if( m_flag_dragging )
	{
		if( m_element_dragged != NULL || m_element_drop_target != NULL )
		{
			if( m_element_dragged->get_type() == DiaryElement::IT_ENTRY )
			{
				Entry *entry( dynamic_cast< Entry* >( m_element_dragged ) );
				if( m_element_drop_target->get_type() == DiaryElement::IT_CHAPTER )
				{
					Chapter *chapter( dynamic_cast< Chapter* >( m_element_drop_target ) );
					Diary::d->set_entry_date( entry, chapter->get_free_order() );
				}
				else	// IT_ENTRY
					Diary::d->set_entry_date( entry, m_element_drop_target->get_date() );
			}
			else	// IT_CHAPTER
			{
				Chapter *chapter( dynamic_cast< Chapter* >( m_element_dragged ) );
				Diary::d->get_current_chapter_ctg()->set_chapter_date(
						chapter, m_element_drop_target->get_date() );
			}

			Lifeobase::base->update_entry_list();
			context->drag_finish( true, false, time );

			return true;
		}
	}

	context->drag_finish( false, false, time );
	return false;
}
/*
void
WidgetEntryList::on_drag_data_received(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info, guint time )
{
	//Glib::ustring name = selection_data.get_data_as_string();
				PRINT_DEBUG( "Entry moved1" );

	//if( name.empty() )	// should never happen
		//return;
				PRINT_DEBUG( "Entry moved2" );

	switch( info )
	{
		case DRAG_TARGET_ENTRY_INFO:
		{
			DiaryElement *element = Lifeobase::base->get_dragged_elem();
			if( element != NULL || m_ptr2target_chapter != NULL )
			{
				Entry *entry( dynamic_cast< Entry* >( element ) );
				entry->set_date( m_ptr2target_chapter->get_free_order() );
				PRINT_DEBUG( "Entry moved3" );

			}
			break;
		}
		default:
			break;
	}

	context->drag_finish( true, false, time );
}
*/
int
WidgetEntryList::sort_by_date( const Gtk::TreeModel::iterator &itr1,
							   const Gtk::TreeModel::iterator &itr2 )
{
	// SORT BY DATE (ONLY DESCENDINGLY FOR NOW)
	DiaryElement *item1 = ( *itr1 )[ ListData::colrec->ptr ];
	DiaryElement *item2 = ( *itr2 )[ ListData::colrec->ptr ];
	if( !( item1 && item2 ) )
		return 0;
	else
	if( item1->get_type() == DiaryElement::IT_DIARY )
	return -1;

	int direction( ( item1->get_date().is_ordinal() && item2->get_date().is_ordinal() ) ? -1 : 1 );

	if( item1->get_date() > item2->get_date() )
		return( -1 * direction );
	else
	if( item1->get_date() < item2->get_date() )
		return( 1 * direction );
	else
		return 0;
}

int
WidgetEntryList::sort_by_size( const Gtk::TreeModel::iterator &itr1,
							   const Gtk::TreeModel::iterator &itr2 )
{
	DiaryElement *item1 = ( *itr1 )[ ListData::colrec->ptr ];
	DiaryElement *item2 = ( *itr2 )[ ListData::colrec->ptr ];
	if( !( item1 && item2 ) )
		return 0;

	// group ordinal entries together:
	if( item1->get_date().is_ordinal() != item2->get_date().is_ordinal() )
		return( item1->get_date().is_ordinal() ? -1 : 1 );

	if( item1->get_size() > item2->get_size() )
		return -1;
	else
	if( item1->get_size() < item2->get_size() )
		return 1;
	else
		return 0;
}

inline bool
WidgetEntryList::move_path_next( Gtk::TreePath &path )
{
    if( unsigned( path.back() ) < ( m_treemodelsort_entries->
                get_iter( path )->parent()->children().size() - 1 ) )
    {
        path.next();
        return true;
    }
    else
        return false;
}

inline bool
WidgetEntryList::make_path_deeper_last( Gtk::TreePath &path )
{
    if( ! m_treemodelsort_entries->get_iter( path )->children().empty() )
    {
        path.push_back( m_treemodelsort_entries->get_iter( path )->children().size() - 1 );
        return true;
    }
    return false;
}
