/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_DIARYDATA_HEADER
#define LIFEOGRAPH_DIARYDATA_HEADER


#include "helpers.hpp"	// i18n headers

#include <gtkmm/treemodel.h>
#include <set>


namespace LIFEO
{

typedef char SortingCriteria;
static const SortingCriteria SC_DATE	= 'd';
static const SortingCriteria SC_SIZE	= 's';
//static const SortingCriteria SC_CHANGE_DATE	= 'c';

typedef unsigned long DEID;	// unique diary element id
static const DEID	DEID_MIN = 10000;	// reserved for Diary itself
static const DEID	DEID_UNSET = 404;		// :)
static const DEID	HOME_CURRENT_ELEM = 1;	// element shown at startup
static const DEID	HOME_LAST_ELEM = 2;		// element shown at startup
static const DEID	HOME_FIXED_ELEM = 3;	// element shown at startup (defines boundary)

class DiaryElement;	// forward declaration


// DIARYBASE (FOR COMMUNICATION)
class DiaryBase
{
	public:
		typedef std::map< DEID, DiaryElement* >
									PoolDEIDs;

                                    DiaryBase( void )
		:	m_current_id( DEID_MIN ), m_force_id( DEID_UNSET ) {}
		virtual                     ~DiaryBase() { }

		DEID						create_new_id( DiaryElement *element )
		{
			DEID retval;
			if( m_force_id == DEID_UNSET )
				retval = m_current_id;
			else
			{
				retval = m_force_id;
				m_force_id = DEID_UNSET;
			}
			m_ids[ retval ] = element;

			while( m_ids.find( m_current_id ) != m_ids.end() )
				m_current_id++;

			return retval;
		}

		bool						set_force_id( DEID id )
		{
			if( m_ids.find( id ) != m_ids.end() || id <= DEID_MIN )
				return false;
			m_force_id = id;
			return true;
		}

		void						clear( void )
		{
			m_force_id = DEID_UNSET;
			m_current_id = DEID_MIN;
			m_ids.clear();
		}

		virtual bool				make_free_entry_order( Date& ) = 0;

	protected:
		DEID						m_current_id;
		DEID						m_force_id;
		PoolDEIDs					m_ids;
};


class ListData;	// forward declaration

// DIARYELEMENT ====================================================================================
class NamedElement
{
	public:
								NamedElement( void )
								:	m_name( "" ) {}
								NamedElement( const Glib::ustring &name )
								:	m_name( name ) {}
		virtual                 ~NamedElement() {}  //needed because of virtual methods

		virtual Glib::ustring	get_name( void ) const
		{ return m_name; }
		void					set_name( const Glib::ustring &name )
		{ m_name = name; }

	protected:
		Glib::ustring			m_name;
};

class DiaryElement : public NamedElement
{
	public:
		static bool				FLAG_ALLOCATE_GUI_FOR_DIARY;

		enum Type
		{
			IT_NONE, IT_TAG, IT_TAG_CATEGORY, IT_THEME,
			// entry list elements:
			IT_DIARY, IT_CHAPTER, IT_ENTRY, IT_DATE
		};

								DiaryElement( DiaryBase * const,
											  const Glib::ustring& = "" );
		virtual					~DiaryElement() {}

		virtual Date			get_date( void ) const
		{ return Date( 0 ); }
		virtual int				get_size( void ) const = 0;
		virtual Type			get_type( void ) const = 0;
		virtual Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const = 0;

		virtual	Glib::ustring	get_list_str( void ) const
		{ return ""; }

		DEID					get_id( void ) const
		{ return m_id; }

		virtual void			show( void ) = 0;
		virtual void			prepare_for_hiding( void ) {}
		virtual bool			get_filtered_out( void )
		{ return false; }

		ListData				*m_list_data;

		DiaryBase * const		m_ptr2diary;

	protected:
		const DEID				m_id;
};

class ListData
{
	public:
		class Colrec : public Gtk::TreeModel::ColumnRecord
		{
			public:
				Colrec()
				{
					add( ptr );
					add( info );
					add( icon );
				}
				// HIDDEN COLUMNS
				Gtk::TreeModelColumn< DiaryElement* >					ptr;
				// COLUMNS
				Gtk::TreeModelColumn< Glib::ustring >					info;
				Gtk::TreeModelColumn< Glib::RefPtr< Gdk::Pixbuf > >		icon;
		};
		static Colrec			*colrec;

		Gtk::TreePath			treepath;
};

bool compare_listitems( DiaryElement*, DiaryElement* );
bool compare_listitems_by_name( DiaryElement*, DiaryElement* );
bool compare_names( const Glib::ustring&, const Glib::ustring& );

typedef bool( *FuncCompareStrings )( const Glib::ustring&, const Glib::ustring& ) ;
typedef bool( *FuncCompareDiaryElem )( DiaryElement*, DiaryElement* ) ;
typedef bool( *FuncCompareDiaryElemNamed )( DiaryElement*, DiaryElement* ) ;
typedef std::set< DiaryElement*, FuncCompareDiaryElem > SetDiaryElements;

template< class T >
Glib::ustring
create_unique_name_for_map( const std::map< Glib::ustring, T*, FuncCompareStrings > &map,
						 const Glib::ustring &name0 )
{
	Glib::ustring name = name0;
	for( int i = 1;
		 map.find( name ) != map.end();
		 i++ )
	{
		name = Glib::ustring::compose( "%1 %2", name0, i );
	}

	return name;
}

// ElementShower class is for communication between Diary and...
// ...GUI classes. we may resort to signals if needed:
template< class T >
struct ElementShower
{
        virtual         ~ElementShower() {}
        virtual void    show( T& ) = 0;
        virtual void    prepare_for_hiding( T& ) {}
};

class DiaryElementContainer : public DiaryElement
{
// NOTE: this class and its derivatives are not responsible for handling of...
// ...its children's allocation and deletion
	public:
								DiaryElementContainer( DiaryBase * const d )
								:	DiaryElement( d, "" ), m_items( compare_listitems ) {}
								DiaryElementContainer( DiaryBase * const d, const Glib::ustring &name )
								:	DiaryElement( d, name ), m_items( compare_listitems ) {}

		SetDiaryElements*		get_items( void )
		{ return &m_items; }
		virtual int				get_size( void ) const
		{ return m_items.size(); }

		void					clear( void )
		{ m_items.clear(); }


	protected:
		SetDiaryElements		m_items;
};

// TAG =============================================================================================
class Tag;	// forward declaration

// TAG CATEGORY
// TODO: should be a DiaryElementContainer as it is not responsible for memory handling
class CategoryTags : public DiaryElement, public std::set< Tag*, FuncCompareDiaryElemNamed >
{
	public:
		static ElementShower< CategoryTags > *shower;

								CategoryTags( DiaryBase * const, const Glib::ustring& );

		void					show( void );

		virtual int				get_size( void ) const
		{ return size(); }
		Type					get_type( void ) const
		{ return IT_TAG_CATEGORY; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_list_str( void ) const
		{ return Glib::Markup::escape_text( m_name ); }

		bool					get_expanded( void ) const
		{ return m_flag_expanded; }
		void					set_expanded( bool flag_expanded )
		{ m_flag_expanded = flag_expanded; }

	protected:
		bool					m_flag_expanded;

	friend class CategoryTagsView;
	friend class PoolCategoriesTags;
	friend class TagPanel;
};

// POOL OF DEFINED TAG CATEGORIES
class PoolCategoriesTags : public std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >
{
	public:
								PoolCategoriesTags( void );
								~PoolCategoriesTags();

		bool					rename_category( CategoryTags*, const Glib::ustring& );
		void					clear( void );
};

// TAG
class Tag : public DiaryElementContainer
{
	public:
		static ElementShower< Tag > *shower;

								Tag( DiaryBase * const d )
								:	DiaryElementContainer( d ),
									m_ptr2category( NULL ) {}
		explicit				Tag( DiaryBase * const, const Glib::ustring&, CategoryTags* );
		virtual					~Tag( void );

		void					show( void );

		CategoryTags*			get_category( void )
		{ return m_ptr2category; }
		void					set_category( CategoryTags *category );

		void					set_name( const Glib::ustring& );
		//Date					get_date( void ) const;	// if earliest entry's date is needed
		Type					get_type( void ) const
		{ return IT_TAG; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_2nd_str( void ) const
		{ return m_name; }

	protected:
		CategoryTags			*m_ptr2category;

	friend class PoolTags;
	friend class Tagset;
	friend class TagView;
};

// POOL OF DEFINED TAGS
class PoolTags : public std::map< Glib::ustring, Tag*, FuncCompareStrings >
{
	public:
								PoolTags( void )
		:	std::map< Glib::ustring, Tag*, FuncCompareStrings >( compare_names ) {}
								~PoolTags();

		bool					handle_tag_changed( );
		bool					rename( Tag*, const Glib::ustring& );
		Tag*					get_tag( unsigned int );
		Tag*					get_tag( const Glib::ustring& );
		void					clear( void );
};

// TAGS OF AN ENTRY
// not responsible for memory handling
class Tagset : public std::set< Tag*, FuncCompareDiaryElemNamed >
{
	public:
								Tagset( void )
		:	std::set< Tag*, FuncCompareDiaryElemNamed >( compare_listitems_by_name ) {}

								~Tagset( void );
		bool					add( Tag* );
		bool					checkfor_member( const Tag* ) const;
		const Tag*				get_tag( unsigned int ) const;
		//bool					remove_tag( const Glib::ustring& );

	protected:

};

// CHAPTER =========================================================================================
class Chapter : public DiaryElement
{
	public:
        static ElementShower< Chapter > *shower;

		explicit				Chapter( DiaryBase * const, const Glib::ustring& );
								Chapter( DiaryBase * const, const Glib::ustring&, Date );
		// this class should have a virtual dtor for some reason because
		// if it does not, its derived classes cannot be deleted
		virtual					~Chapter() {}

		void					show( void );

		void					set_name( const Glib::ustring& );
		Date					get_date( void ) const;
		Glib::ustring			get_date_str( void ) const;
		void					set_date( const Date& );
		Type					get_type( void ) const
		{ return IT_CHAPTER; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon_32( void ) const;
		virtual int				get_size( void ) const	// TODO: stub!
		{ return m_size; }

		Date					get_free_order( void ) const;

		Glib::ustring			get_list_str( void ) const
		{ return Glib::ustring::compose( "<b>%1 -  %2</b>", m_date_begin.format_string(),
				Glib::Markup::escape_text( m_name ) ); }

		bool					get_expanded( void ) const
		{ return m_flag_expanded; }
		void					set_expanded( bool flag_expanded )
		{ m_flag_expanded = flag_expanded; }

		bool					is_initialized( void ) const
		{ return( m_date_begin.m_date != Date::NOTSET ); }
		bool					is_ordinal( void ) const
		{ return m_date_begin.is_ordinal(); }

		void					reset_size( void )
		{ m_size = 0; }
		void					increase_size()
		{ m_size++; }

	protected:
		Date					m_date_begin;
		int						m_size;
		bool					m_flag_expanded;

	friend class CategoryChapters;
	friend class ChapterView;
};

class CategoryChapters :
		public DiaryElement, public std::map< Date::date_t, Chapter*, FuncCompareDates >
{
	public:
		explicit				CategoryChapters( DiaryBase * const d, const Glib::ustring& );
		virtual					~CategoryChapters();

		virtual int				get_size( void ) const
		{ return size(); }
		Type					get_type( void ) const
		{ return IT_TAG_CATEGORY; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;
		void					show( void ) {}	// not used

		Chapter*				get_chapter( const Date& );

		Chapter*				create_chapter( const Glib::ustring&, const Date& );
		void					dismiss_chapter( Chapter* );
		bool					rename_chapter( Chapter*, const Glib::ustring& );
		bool					set_chapter_date( Chapter*, const Date& );

		void					clear( void );

		Date					get_free_order_ordinal( void ) const;

	protected:

};

class PoolCategoriesChapters :
		public std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >
{
	public:
									PoolCategoriesChapters( void )
		:	std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >(
				compare_names ) {}
									~PoolCategoriesChapters();
		void						clear( void );
};

// THEMES ==========================================================================================
class Theme : public DiaryElement
{
	public:
        static ElementShower< Theme > *shower;

                                    Theme( DiaryBase * const, const Glib::ustring& );
                                    Theme( DiaryBase * const,
                                           const Glib::ustring&,
                                           const Glib::ustring&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string&,
                                           const std::string& );
        // duplicates an existing theme, works like a copy constructor
                                    Theme( const Theme* );

		void						show( void );

		int							get_size( void ) const
		{ return 0; }	// redundant
		DiaryElement::Type			get_type( void ) const
		{ return IT_THEME; }
		Glib::RefPtr< Gdk::Pixbuf >&
									get_icon( void ) const;

		Glib::ustring				get_list_str( void ) const
		{ return Glib::Markup::escape_text( m_name ); }

		virtual bool				is_system( void ) const
		{ return false; }

		Pango::FontDescription		font;
		Gdk::Color					color_base;
		Gdk::Color					color_text;
		Gdk::Color					color_heading;
		Gdk::Color					color_subheading;
		Gdk::Color					color_highlight;

	friend class Diary;
};

class ThemeSystem : public Theme
{
	public:
		static ThemeSystem*			get( void );
		bool						is_system( void ) const
		{ return true; }

	protected:
									ThemeSystem( const Glib::ustring&,
												 const std::string&,
												 const std::string&,
												 const std::string&,
												 const std::string&,
												 const std::string& );
};

class PoolThemes : public std::map< Glib::ustring, Theme*, FuncCompareStrings >
{
	public:
									PoolThemes( void )
		:	std::map< Glib::ustring, Theme*, FuncCompareStrings >( compare_names ) {}
									~PoolThemes();

		void						clear( void );
		bool						rename_theme( Theme*, const Glib::ustring& );
};

} // end of namespace LIFEO

#endif

