/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "view_entry.hpp"


using namespace LIFEO;


EntryView::EntryView( void )
:	m_menu_tag( NULL ),
	m_tag_operation_cur( TO_NONE ),
	m_ptr2entry( NULL ),
	m_flag_entrychanged( false )
{
	Entry::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_entry", m_vbox_entry );
		Lifeobase::builder->get_widget( "togglebutton_entry_details", m_button_toggle_details );
		Lifeobase::builder->get_widget( "togglebutton_entry_favorite", m_button_toggle_favorite );
		Lifeobase::builder->get_widget( "button_entry_reset_theme", m_button_reset_theme );
		Lifeobase::builder->get_widget( "button_entry_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "button_entry_add_bullet", m_button_add_bullet );
		Lifeobase::builder->get_widget( "button_entry_add_checkbox", m_button_add_checkbox );
		Lifeobase::builder->get_widget_derived( "textview_entry", m_textviewdiary );
		Lifeobase::builder->get_widget_derived( "entry_tag_filter", m_entry_tag );
		Lifeobase::builder->get_widget( "button_tag_operation", m_button_tag_operation );
		Lifeobase::builder->get_widget( "iconview_current_tags", m_iconview_tags );
		Lifeobase::builder->get_widget( "alignment_entry_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "hbox_entry_2nd_toolbar", m_hbox_editing_toolbar );
	}
	catch( ... ) {}

	// TEXT VIEW
	m_textviewdiary->drag_dest_set_target_list( Lifeobase::base->drag_target_list );

	// TAG LIST
	m_liststore_tags = Gtk::ListStore::create( *TagPanel::colrec );
	m_iconview_tags->set_pixbuf_column( TagPanel::colrec->icon );
	m_iconview_tags->set_markup_column( TagPanel::colrec->name );
	m_iconview_tags->set_model( m_liststore_tags );
	m_iconview_tags->set_orientation( Gtk::ORIENTATION_HORIZONTAL );
	m_iconview_tags->set_selection_mode( Gtk::SELECTION_MULTIPLE );
	m_iconview_tags->set_events( Gdk::ENTER_NOTIFY_MASK );
	m_iconview_tags->set_item_width( 150 );
	m_iconview_tags->drag_dest_set( Lifeobase::base->drag_targets_tag );
	m_iconview_tags->modify_base(
			Gtk::STATE_NORMAL, Lifeobase::base->get_style()->get_bg( Gtk::STATE_NORMAL ) );

	// TAG WIDGET
	m_entry_tag->set_idle_text( _( "Tag name" ) );

	// SIGNALS
	TextviewDiary::m_buffer->signal_changed().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::handle_textview_changed ) );

	m_button_toggle_details->signal_toggled().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::toggle_details ) );
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::dismiss_entry ) );
	m_button_reset_theme->signal_clicked().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::reset_theme ) );
	m_button_toggle_favorite->signal_toggled().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::toggle_favoredness ) );

	m_button_add_bullet->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer,
			&TextbufferDiary::add_bullet ) );
	m_button_add_checkbox->signal_clicked().connect(
			sigc::mem_fun( m_textviewdiary->m_buffer,
			&TextbufferDiary::add_checkbox ) );

	m_textviewdiary->signal_drag_data_received().connect(
			sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );

	m_entry_tag->signal_changed().connect(
			sigc::mem_fun( this, &EntryView::handle_entry_tag_changed ) );
	m_entry_tag->signal_activate().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
	m_button_tag_operation->signal_clicked().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
	m_iconview_tags->signal_drag_data_received().connect(
			sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );
	m_iconview_tags->signal_key_release_event().connect_notify(
			sigc::mem_fun( this, &EntryView::handle_iconview_tags_key_release ) );
	m_iconview_tags->signal_button_press_event().connect_notify(
			sigc::mem_fun( this, &EntryView::handle_iconview_tags_button_press ) );
	// TODO: -MAY BE REMOVED
	//m_iconview_tags->signal_enter_notify_event().connect(
			//sigc::ptr_fun( &EntryView::handle_hboxtagtools_enternotify ) );
}

void
EntryView::populate_tags( void )
{
	Gtk::TreeRow row;
	const Tagset &ptr2tagset = m_ptr2entry->get_tags();

	m_liststore_tags->clear();

	if( ptr2tagset.size() < 1 )
	{
		row = * m_liststore_tags->append();
		row[ TagPanel::colrec->name ] = Glib::ustring::compose( "<i>%1</i>", _( "None" ) );
		row[ TagPanel::colrec->ptr ] = NULL;
		return;
	}

	for( PoolTags::const_iterator iter = Diary::d->get_tags()->begin();
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		Tag *tag = iter->second;
		if( ptr2tagset.checkfor_member( tag ) )
		{
			row = * m_liststore_tags->append();

			row[ TagPanel::colrec->name ] =
					Glib::ustring::compose( "<small>%1</small>",
							Glib::Markup::escape_text( tag->get_name() ) );
			row[ TagPanel::colrec->icon ] = Lifeobase::icons->tag_16;
			row[ TagPanel::colrec->ptr ] = ( tag );
		}
	}
}

void
EntryView::sync( void )
{
	if( m_flag_entrychanged )
	{
		m_ptr2entry->m_date_changed = time( NULL );
		m_ptr2entry->m_text = TextviewDiary::m_buffer->get_text();
		m_flag_entrychanged = false;
		PRINT_DEBUG( "entry synced" );
	}
}

void
EntryView::focus_tag( void )
{
	//m_hbox_tagtools->show();
	m_entry_tag->grab_focus();
}

void
EntryView::remove_selected_tags( void )
{
	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();
	for( std::list< Gtk::TreeModel::Path >::iterator iter = selected_items.begin();
		 iter != selected_items.end();
		 ++iter )
	{
		Gtk::TreeRow row = *m_liststore_tags->get_iter( *iter );
		DiaryElement *element = row[ TagPanel::colrec->ptr ];
		Tag *tag = dynamic_cast< Tag* >( element );

		m_ptr2entry->remove_tag( tag );
		m_liststore_tags->erase( row );
	}
}

/*bool
EntryView::handle_hboxtagtools_enternotify( GdkEventCrossing *event )
{
	// TODO:
	if( event->mode == GDK_CROSSING_NORMAL )
		m_hbox_tagtools->show();
	return false;
}*/

void
EntryView::handle_textview_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	m_flag_entrychanged = true;

	PRINT_DEBUG( "entry changed" );

	// TODO:
	//if( m_hbox_tagtools->is_visible() )
		//m_hbox_tagtools->hide();
}

void
EntryView::handle_drag_data_received(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info,
				guint time )
{
	context->drag_finish( true, false, time );

	Glib::ustring name = selection_data.get_data_as_string();

	if( name.empty() )	// should never happen
		return;

	switch( info )
	{
		case DRAG_TARGET_TAG_INFO:
		{
			Tag *tag = Diary::d->m_tags.find( name )->second;
			if( m_ptr2entry->add_tag( tag ) )	// if does not already have the tag
				populate_tags();
			break;
		}
		case DRAG_TARGET_THEME_INFO:
		{
			Theme *theme = ( name == Theme::system_theme.get_name() ) ?
					&Theme::system_theme : Diary::d->m_themes.find( name )->second;
			m_ptr2entry->set_theme( theme );
			TextviewDiary::m_buffer->set_theme( theme );
			m_button_reset_theme->set_visible( true );
			break;
		}
	}
}

void
EntryView::toggle_favoredness( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	m_ptr2entry->m_option_favored = m_button_toggle_favorite->get_active();
	Gtk::TreeRow row = Lifeobase::base->get_element_row( m_ptr2entry );
	row[ ListData::colrec->icon ] = m_ptr2entry->get_icon();
}

void
EntryView::reset_theme( void )
{
	m_ptr2entry->set_theme( NULL );
	TextviewDiary::m_buffer->set_theme( NULL );
	m_button_reset_theme->set_visible( false );
}

void
EntryView::dismiss_entry( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
							    Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	Lifeobase::base->dismiss_entry( m_ptr2entry );
}

void
EntryView::handle_entry_tag_changed( void )
{
	Glib::ustring filter = m_entry_tag->get_text();

	if( filter.empty() )
	{
		m_button_tag_operation->hide();
		m_tag_operation_cur = TO_NONE;
	}
	else
	{
		m_button_tag_operation->show();
		Tag *tag = Diary::d->get_tags()->get_tag( filter );
		if( tag == NULL )
		{
			m_button_tag_operation->set_label( _( "Create Tag" ) );
			m_tag_operation_cur = TO_CREATE_AND_ADD;
		}
		else
		{
			if( m_ptr2entry->get_tags().checkfor_member( tag ) )
			{
				m_button_tag_operation->set_label( _( "Remove Tag" ) );
				m_tag_operation_cur = TO_REMOVE;
			}
			else
			{
				m_button_tag_operation->set_label( _( "Add Tag" ) );
				m_tag_operation_cur = TO_ADD;
			}
		}
	}

	Lifeobase::base->update_tag_list( filter.lowercase() );
}

void
EntryView::handle_button_tag_clicked( void )
{
	Tag *tag;
	Glib::ustring filter = m_entry_tag->get_text();

	switch( m_tag_operation_cur )
	{
		case TO_NONE:
			return;	// don't even clear
		case TO_REMOVE:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2entry->remove_tag( tag );
			break;
		case TO_CREATE_AND_ADD:
			tag = Diary::d->get_tags()->create( filter );
			m_ptr2entry->add_tag( tag );
			break;
		case TO_ADD:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2entry->add_tag( tag );
			break;
	}

	populate_tags();
	m_entry_tag->set_text( "" );
}

void
EntryView::handle_iconview_tags_key_release( GdkEventKey *event )
{
	if( event->keyval != GDK_Delete )	// the cheapest check first
		return;

	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tags->get_iter( selected_items.front() )->get_value(
				TagPanel::colrec->ptr ) == NULL )
		return;

	remove_selected_tags();
}

void
EntryView::handle_iconview_tags_button_press( GdkEventButton *event )
{
	if( event->button != 3 )
		return;

	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tags->get_iter( selected_items.front() )->get_value(
				TagPanel::colrec->ptr ) == NULL )
		return;

	if( m_menu_tag )
	{
		delete m_menu_tag;
		m_menu_tag = NULL;
	}

	m_menu_tag = new Gtk::Menu;
	Gtk::Image *image_remove = Gtk::manage(
		new Gtk::Image( Gtk::Stock::REMOVE, Gtk::ICON_SIZE_MENU ) );

	m_menu_tag->items().push_back(
			// TRANSLATORS: label of the remove-selected-tags-from-current-entry
			// menu item
			Gtk::Menu_Helpers::ImageMenuElem( _( "_Remove from Entry" ), *image_remove,
					sigc::mem_fun( *this, &EntryView::remove_selected_tags ) ) );

	m_menu_tag->accelerate( *Lifeobase::base );
	m_menu_tag->popup( event->button, event->time );
}

void
EntryView::show( Entry &entry )
{
	Lifeobase::m_internaloperation++;

	Lifeobase::base->change_current_element( &entry, m_vbox_entry );

	m_ptr2entry = &entry;

	// HEADER
	Lifeobase::m_image_gui_elem->set( ICONDIR "/entry-32.png" );
	Lifeobase::base->set_header_texts(
			entry.m_date.format_string(),
			entry.m_date.get_weekday_str(),
			Glib::ustring::compose( "%1: <b>%2</b>\n%3: <b>%4</b>",
// TRANSLATORS: label of creation date
									_( "Created"), entry.get_date_created_str(),
// TRANSLATORS: label of last change date
									_( "Changed"), entry.get_date_changed_str() ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	Lifeobase::base->select_date_in_calendar( entry.m_date );

	// BODY
	TextviewDiary::m_buffer->set_richtext( &entry );
	populate_tags();
	m_button_toggle_favorite->set_active( entry.m_option_favored );
	m_button_reset_theme->set_visible( entry.get_theme() != NULL );

	Lifeobase::m_internaloperation--;

	PRINT_DEBUG( "entry shown" );
}

void
EntryView::prepare_for_hiding( Entry &entry )
{
	sync();
	m_entry_tag->set_text( "" );
}

void
EntryView::toggle_details( void )
{
	if( m_button_toggle_details->get_active() )
		//m_vbox_entry->pack_start( *m_vbox_details, false, false );
		m_hbox_editing_toolbar->set_visible( true );
	else
		//m_vbox_entry->remove( *m_vbox_details );
		m_hbox_editing_toolbar->set_visible( false );
}
