/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "panel_diary.hpp"
#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;

PanelDiary::PanelDiary( void )
{
	try
	{
		//Lifeobase::builder->get_widget( "calendar_main", m_calendar );	// TODO
		Lifeobase::builder->get_widget_derived( "treeview_main", m_treeview_entries );
		Lifeobase::builder->get_widget_derived( "entry_filter", m_entry_filter );
		Lifeobase::builder->get_widget_derived( "entry_replace", m_entry_replace );
		Lifeobase::builder->get_widget( "tbutton_match_prev", m_button_match_prev );
		Lifeobase::builder->get_widget( "tbutton_match_next", m_button_match_next );
		Lifeobase::builder->get_widget( "tbutton_replace", m_toolbutton_replace );
		Lifeobase::builder->get_widget( "tbutton_replace_all", m_toolbutton_replace_all );
		Lifeobase::builder->get_widget( "toolbar_replace", m_toolbar_replace );

		Lifeobase::builder->get_widget( "menuitem_sort_by_date", m_menuitem_sort_by_date );
		Lifeobase::builder->get_widget( "menuitem_sort_by_size", m_menuitem_sort_by_size );
		Lifeobase::builder->get_widget( "menuitem_filter_favorites", m_menuitem_filter_favorites );
	}
	catch( ... ) { }

	// FILTER & REPLACE
	m_entry_filter->set_idle_text( _( "Filter" ) );
	m_entry_replace->set_idle_text( _( "Replace" ) );

	// SIGNALS
	m_entry_filter->signal_changed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_entry_filter_changed ) );
	m_entry_filter->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::go_next_match ) );

	m_entry_replace->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_toolbutton_replace->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_toolbutton_replace_all->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_all_matches ) );

	m_treeview_entries->signal_row_expanded().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );
	m_treeview_entries->signal_row_collapsed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );

	m_menuitem_sort_by_date->signal_activate().connect(
			sigc::bind(
					sigc::mem_fun( m_treeview_entries,
								   &WidgetEntryList::handle_sorting_criteria_changed ),
					SC_DATE ) );
	m_menuitem_sort_by_size->signal_activate().connect(
			sigc::bind(
					sigc::mem_fun( m_treeview_entries,
								   &WidgetEntryList::handle_sorting_criteria_changed ),
					SC_SIZE ) );
	m_menuitem_filter_favorites->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::handle_filter_favorites_toggled ) );
}

void
PanelDiary::update_entry_list( void )
{
	Lifeobase::m_internaloperation++;

	Gtk::TreeRow	row_entry;
	Entry			*entry;
	Entryiter		iter_entry = Diary::d->m_entries.begin();
	Chapter			*chapter = NULL;

	Lifeobase::base->sync_entry();

	m_treeview_entries->m_treestore_entries->clear();
	Lifeobase::m_entrycount = 0;

	// DIARY ITEM
	Gtk::TreeRow row_diary = * m_treeview_entries->m_treestore_entries->append();
	row_diary[ ListData::colrec->ptr ] = Diary::d; // ptr must be first
	row_diary[ ListData::colrec->id ] =  Glib::ustring::compose( "<b>%1</b>", _( "Diary" ) );
	row_diary[ ListData::colrec->info ] = Diary::d->get_name();
	row_diary[ ListData::colrec->icon ] = Lifeobase::icons->diary_16;

	Diary::d->m_list_data->treepath =
			m_treeview_entries->m_treestore_entries->get_path( row_diary );


	// CHAPTERS
	CategoryChapters *chapters = Diary::d->get_current_chapter_ctg();

	if( Diary::d->get_sorting_criteria() == SC_DATE && chapters )
	{
		DiaryElement *element( NULL ), *element_ret;

		for( CategoryChapters::const_iterator iter_chapter = chapters->begin();
			 iter_chapter != chapters->end();
			 ++iter_chapter )
		{
			chapter = * iter_chapter;
			element_ret = chapter->add_to_list(
					m_treeview_entries->m_treestore_entries, row_diary );
			if( element_ret )	// skip empty folders
			{
				element = element_ret;
				if( chapter->get_expanded() )
					m_treeview_entries->expand_element( chapter );
			}
		}
		if( element )
		{
			iter_entry = Diary::d->m_entries.find( element->get_date().m_date );
			if( iter_entry != Diary::d->m_entries.end() )
				++iter_entry;
		}
	}

	for( ;
		 iter_entry != Diary::d->m_entries.end();
		 ++iter_entry )
	{
		entry = ( *iter_entry ).second;
		if( entry->get_filtered_out() == false )
		{
			//if( entry->get_date() >= ( *iter_chapter )->get_date() )

			Lifeobase::m_entrycount++;

			row_entry = *m_treeview_entries->m_treestore_entries->append(
					row_diary.children() );
			row_entry[ ListData::colrec->ptr ] = entry; // ptr must be first
			row_entry[ ListData::colrec->id ] = entry->get_1st_str();
			row_entry[ ListData::colrec->info ] = entry->get_2nd_str();
			row_entry[ ListData::colrec->icon ] = entry->get_icon();

			entry->m_list_data->treepath =
					m_treeview_entries->m_treestore_entries->get_path( row_entry );
		}
	}

	// UPDATE DIARY ELEM IF LAST ONE IS NOT IN THE LIST ANYMORE
	DiaryElement *elem = Lifeobase::base->get_cur_elem();
	bool flag_show_elem = true;
	if( Lifeobase::m_entrycount <= 0 )
		elem = Diary::d;
	else
	if( elem != NULL )
	{
		if( elem->get_filtered_out() )
			elem = Diary::d->get_entry_first();
		else
			flag_show_elem = false;
	}
	else	// the case at the start-up
		elem = Diary::d->get_entry_first();

	if( flag_show_elem )
		elem->show();
	else
	if( elem->get_type() >= DiaryElement::IT_DIARY )	// only entry list elements
		m_treeview_entries->present_row( elem->m_list_data->treepath );

	Diary::d->set_filtering_status_applied();

	Lifeobase::m_internaloperation--;
}

void
PanelDiary::handle_login( void )
{
	if( Diary::d->get_sorting_criteria() == SC_DATE )
		m_menuitem_sort_by_date->set_active();
	else
		m_menuitem_sort_by_size->set_active();

	// not called automatically because of m_internaloperation:
	m_treeview_entries->set_sorting_criteria();
	update_entry_list();
}

void
PanelDiary::handle_logout( void )
{
	m_entry_filter->set_text( "" );
	m_entry_replace->set_text( "" );
	m_menuitem_filter_favorites->set_active( false );
	m_treeview_entries->m_treestore_entries->clear();
}

void
PanelDiary::present_current_row( void )
{
	Gtk::TreeIter iter = m_treeview_entries->get_selection()->get_selected();
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path = m_treeview_entries->m_treemodelsort_entries->get_path( iter );

	Lifeobase::m_internaloperation++;

	m_treeview_entries->present_row( path );

	Lifeobase::m_internaloperation--;
}

void
PanelDiary::handle_entry_filter_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;
	// current entry must be closed here or else it loses...
	// ...changes since it was selected:
	Lifeobase::base->sync_entry();

	const std::string filterstring( m_entry_filter->get_text().lowercase() );

	Diary::d->set_filter_text( filterstring );
	update_entry_list();

	if( filterstring.size() > 0 )
	{
		if( Lifeobase::m_entrycount > 0 )
		{
			TextviewDiary::m_buffer->set_searchstr( filterstring );
			if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
				TextviewDiary::m_buffer->select_searchstr_next();
		}

		m_entry_filter->set_tooltip_markup(
				Glib::ustring::compose(
						_( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
						Lifeobase::m_entrycount, Diary::d->get_size() ) );
	}
	else
	{
		TextviewDiary::m_buffer->set_searchstr( "" );
		m_entry_filter->set_tooltip_text( _( "Enter text to be filtered" ) );
	}

	// TODO:
	// update_calendar();

	if( Lifeobase::m_entrycount < 1 || filterstring.size() < 1 )
	{
		m_button_match_prev->hide();
		m_button_match_next->hide();
		m_toolbar_replace->hide();
	}
	else
	{
		m_button_match_prev->show();
		m_button_match_next->show();
		m_toolbar_replace->show();
	}
}

void
PanelDiary::handle_filter_favorites_toggled( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Diary::d->toggle_filter_favorites();	// set filter
	update_entry_list();					// apply the filter
}

inline bool
PanelDiary::make_path_deeper_last( Gtk::TreePath &path )
{
	if( ! m_treeview_entries->m_treemodelsort_entries->
					get_iter( path )->children().empty() )
	{
		path.push_back( m_treeview_entries->m_treemodelsort_entries->
				get_iter( path )->children().size() - 1 );
		return true;
	}
	return false;
}

inline bool
PanelDiary::move_path_next( Gtk::TreePath &path )
{
	if( unsigned( path.back() ) < ( m_treeview_entries->m_treemodelsort_entries->
				get_iter( path )->parent()->children().size() - 1 ) )
	{
		path.next();
		return true;
	}
	else
		return false;
}

void
PanelDiary::go_up( bool flag_entry_operation )
{
	// flag_entry: if true, goes to an entry item even when the current...
	// ...item is a folder
	Gtk::TreeIter iter( m_treeview_entries->get_selection()->get_selected() );
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path( m_treeview_entries->m_treemodelsort_entries->get_path( iter ) );
	DiaryElement *elem;

	do
	{
		if( ! path.prev() )
		{
			if( path.size() > 1 )
				path.up();
			else	// diary
			if( make_path_deeper_last( path ) )
				make_path_deeper_last( path ); // go still deeper if possible:
		}
		else
			make_path_deeper_last( path );

		elem = ( * m_treeview_entries->m_treemodelsort_entries->
						get_iter( path ) )[ ListData::colrec->ptr ];
	}
	// repeat until an entry is found if that is what we are looking for
	// BEWARE: do not set this flag when there is no entry in the list
	while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

	PRINT_DEBUG( "previous path: " + path.to_string() );

	Gtk::TreeRow row = * m_treeview_entries->m_treemodelsort_entries->get_iter( path );
	DiaryElement *element( row[ ListData::colrec->ptr ] );
	element->show();
}

void
PanelDiary::go_down( bool flag_entry_operation )
{
	// flag_entry: if true, goes to an entry item even when the current...
	// ...item is a folder
	Gtk::TreeIter iter( m_treeview_entries->get_selection()->get_selected() );
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path( m_treeview_entries->m_treemodelsort_entries->get_path( iter ) );
	DiaryElement *elem;

	do
	{
		if( ! m_treeview_entries->m_treemodelsort_entries->
					get_iter( path )->children().empty() )
			path.down();
		else
		if( ! move_path_next( path ) )
		{
			while( path.size() > 1 )
			{
				path.up();
				if( move_path_next( path ) )
					break;
			}
		}

		elem = ( * m_treeview_entries->m_treemodelsort_entries->
						get_iter( path ) )[ ListData::colrec->ptr ];
	}
	// repeat until an entry is found if that is what we are looking for
	// BEWARE: do not set this flag when there is no entry in the list
	while( flag_entry_operation && elem->get_type() != DiaryElement::IT_ENTRY );

	PRINT_DEBUG( "next path: " + path.to_string() );

	Gtk::TreeRow row = * m_treeview_entries->m_treemodelsort_entries->get_iter( path );
	DiaryElement *listitem = row[ ListData::colrec->ptr ];
	listitem->show();
}

void
PanelDiary::go_prev_match( void )
{
	if( Lifeobase::m_entrycount < 1 )
		return;

	// TODO: move to Entry
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
		if( TextviewDiary::m_buffer->select_searchstr_previous() )
			return;

	int i = 0;
	do
	{
		go_up( true );
		// put cursor to the end:
		TextviewDiary::m_buffer->select_range( TextviewDiary::m_buffer->end(),
											   TextviewDiary::m_buffer->end() );
		i++;
	}
	while( ! TextviewDiary::m_buffer->select_searchstr_previous() &&
		   i < Lifeobase::m_entrycount );
}

void
PanelDiary::go_next_match( void )
{
	if( Lifeobase::m_entrycount < 1 )
		return;

	// TODO: move to Entry
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
		if( TextviewDiary::m_buffer->select_searchstr_next() )
			return;

	int i = 0;
	do
	{
		go_down( true );
		i++;
	}
	while( ! TextviewDiary::m_buffer->select_searchstr_next() &&
		   i < Lifeobase::m_entrycount );
}

void
PanelDiary::replace_match( void )
{
	if( ! TextviewDiary::m_buffer->get_has_selection() )
		go_next_match();
	if( TextviewDiary::m_buffer->get_has_selection() )
	{
		TextviewDiary::m_buffer->erase_selection();
		TextviewDiary::m_buffer->insert_at_cursor( m_entry_replace->get_text() );
		go_next_match();
	}
}

void
PanelDiary::replace_all_matches( void )
{
	Lifeobase::base->sync_entry();

	Diary::d->replace_text( m_entry_replace->get_text() );
	// update current entry:
	if( Lifeobase::base->get_cur_elem_type() == DiaryElement::IT_ENTRY )
	{
		Entry *entry = dynamic_cast< Entry* >( Lifeobase::base->get_cur_elem() );
		TextviewDiary::m_buffer->set_richtext( entry );
	}
}

Gtk::TreeRow
PanelDiary::get_row( const Gtk::TreePath &path )
{
	return( * m_treeview_entries->m_treestore_entries->get_iter( path ) );
	// TODO: check validity
}

void
PanelDiary::handle_treeview_row_expanded( const Gtk::TreeIter &iter,
										  const Gtk::TreePath &path )
{
	//if( Lifeobase::m_internaloperation )
		//return;

	DiaryElement *element = ( * iter )[ ListData::colrec->ptr ];
	if( element->get_type() == DiaryElement::IT_CHAPTER )
	{
		Chapter *chapter = dynamic_cast< Chapter* >( element );
		chapter->set_expanded( m_treeview_entries->row_expanded( path ) );
	}
}
