/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cmath>

#include "textview.hpp"
#include "dialogtheme.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_heading;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_subheading;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_match;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_markup;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_bold;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_italic;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_strikethrough;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_highlight;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_link;
Glib::RefPtr<Gtk::TextTag>		TextbufferDiary::m_tag_link_broken;

Pango::FontDescription	TextbufferDiary::m_theme_font;
Gdk::Color				TextbufferDiary::m_theme_color_base( "white" );
Gdk::Color				TextbufferDiary::m_theme_color_text( "black" );
Gdk::Color				TextbufferDiary::m_theme_color_match( "green" );
Gdk::Color				TextbufferDiary::m_theme_color_markup( "gray" );
Gdk::Color				TextbufferDiary::m_theme_color_heading( "blue" );
Gdk::Color				TextbufferDiary::m_theme_color_subheading( "violet" );
Gdk::Color				TextbufferDiary::m_theme_color_highlight( "yellow" );
Gdk::Color				TextbufferDiary::m_theme_color_link( "navy" );
Gdk::Color				TextbufferDiary::m_theme_color_linkbroken( "red" );

Gtk::TextBuffer					*UndoEdit::m_ptr2buffer;

// LINK
Link::Link(	const Glib::RefPtr< Gtk::TextMark > &start,
			const Glib::RefPtr< Gtk::TextMark > &end )
	:	m_mark_start( start ), m_mark_end( end )
{

}

Link::~Link()
{
	// TODO: is this necessary?
	Glib::RefPtr< Gtk::TextBuffer > buffer = m_mark_start->get_buffer();
	if ( buffer )
	{
		buffer->delete_mark( m_mark_start );
		buffer->delete_mark( m_mark_end );
	}
}

// LINK TO ENTRY
// static variable:
LinkEntry::Signal_void_Date LinkEntry::m_signal_activated;

LinkEntry::LinkEntry(	const Glib::RefPtr< Gtk::TextMark > &start,
						const Glib::RefPtr< Gtk::TextMark > &end,
						Date date,
						unsigned int order )
	:	Link( start, end ), m_date( date ), m_order( order )
{
}

void
LinkEntry::go( void ) const
{
	m_signal_activated.emit( m_date );
}

// LINK TO URI
Gtk::TextBuffer *LinkUri::m_ptr2buffer;

LinkUri::LinkUri(	const Glib::RefPtr< Gtk::TextMark > &start,
					const Glib::RefPtr< Gtk::TextMark > &end,
					const std::string& url )
	:	Link( start, end ), m_url( url )
{
}

void
LinkUri::go( void ) const
{
	GError *err = NULL;
	gtk_show_uri (NULL, m_url.c_str(), GDK_CURRENT_TIME, &err);
}


// TEXTBUFFERDIARY
TextbufferDiary::TextbufferDiary( UndoManager *undomanager, Database *ptr2database  )
	:	Gtk::TextBuffer(), m_title( "" ), m_option_search( false ),
	m_flag_settextoperation( false ), m_flag_ongoingoperation( false ),
	m_flag_parsing( false ),
	m_ptr2undomanager( undomanager ),
	m_ptr2database( ptr2database ), m_ptr2entry( NULL ),
	m_ptr2textview( NULL ), m_ptr2spellobj( NULL )
{
	LinkUri::m_ptr2buffer = this;
	UndoEdit::m_ptr2buffer = this;
	Glib::RefPtr< TagTable > tag_table = get_tag_table();

	m_tag_heading = Tag::create( "heading" );
	m_tag_heading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_heading->property_scale() = 1.5;
	tag_table->add( m_tag_heading );

	m_tag_subheading = Tag::create( "subheading" );
	m_tag_subheading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_subheading->property_scale() = 1.2;
	tag_table->add( m_tag_subheading );

	m_tag_match = Tag::create( "match" );
	tag_table->add( m_tag_match );

	m_tag_markup = Tag::create( "markup" );
	tag_table->add( m_tag_markup );

	m_tag_bold = Tag::create( "bold" );
	m_tag_bold->property_weight() = Pango::WEIGHT_BOLD;
	tag_table->add( m_tag_bold );

	m_tag_italic = Tag::create( "italic" );
	m_tag_italic->property_style() = Pango::STYLE_ITALIC;
	tag_table->add( m_tag_italic );

	m_tag_strikethrough = Tag::create( "strikethrough" );
	m_tag_strikethrough->property_strikethrough() = true;
	tag_table->add( m_tag_strikethrough );

	m_tag_highlight = Tag::create( "highlight" );
	tag_table->add( m_tag_highlight );

	m_tag_link = Tag::create( "link" );
	m_tag_link->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link );

	m_tag_link_broken = Tag::create( "link.broken" );
	m_tag_link_broken->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link_broken );
}

void
TextbufferDiary::reset( void )
{
	set_searchstr( "" );
	m_title.clear();
	set_spellcheck( false );
}

Glib::ustring
TextbufferDiary::get_title( void ) const
{
	return m_title;
}

void
TextbufferDiary::set_search( bool option_search )
{
	m_option_search = option_search;
}

void
TextbufferDiary::set_searchstr( const Glib::ustring &searchstr )
{
	m_searchstr = searchstr;
	// disable search if searchstr is empty:
	m_option_search = static_cast< bool >( searchstr.size() );
}

bool
TextbufferDiary::select_searchstr_previous( void )
{
	if( ! m_option_search )
		return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_start.is_start() )
		return false;

	iter_bound_end = iter_bound_start;
	
	if( ! iter_bound_start.backward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_start.backward_to_tag_toggle( m_tag_match );
	iter_bound_start.backward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_end.backward_search(	m_searchstr,
											Gtk::TextSearchFlags( 0 ),
											iter_start,
											iter_end,
											iter_bound_start ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

bool
TextbufferDiary::select_searchstr_next( void )
{
	if( ! m_option_search )
		return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_end.is_end() )
		return false;

	iter_bound_start = iter_bound_end;

	if( ! iter_bound_end.forward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_end.forward_to_tag_toggle( m_tag_match );
	if( iter_bound_end.has_tag( m_tag_match ) )
		iter_bound_end.forward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_start.forward_search(	m_searchstr,
													Gtk::TextSearchFlags( 0 ),
													iter_start,
													iter_end,
													iter_bound_end ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

void
TextbufferDiary::parse( unsigned int pos_start, unsigned int pos_end )
{
	m_flag_parsing = true;
	// BEGINNING & END
	Gtk::TextIter	iter_start = get_iter_at_offset( pos_start );
	Gtk::TextIter	const iter_end = get_iter_at_offset( pos_end );
	Gtk::TextIter	iter_current = iter_start;

	// remove all tags:
	remove_all_tags( iter_start, iter_end );
	clear_links( pos_start, pos_end );

	LookingFor		lookingfor = pos_start == 0 ? LF_FIRSTNEWLINE : LF_SPACE;
	LookingFor		lookingfor2 = LF_NOTHING;
	gunichar		char_last = 0;
	gunichar		char_current;
	Glib::ustring	word_last = "";
	unsigned int	int_last = 0;
	Date			date_link = 0;

	// SEARCHING
	gunichar		char_search = 0;
	Gtk::TextIter	iter_search = iter_start;
	Gtk::TextIter	iter_word = iter_start;
	size_t			pos_search = 0;
	const size_t	pos_search_end = m_searchstr.size();

	for( ; ; ++iter_current )
	{
		char_current = iter_current.get_char();
		// SEARCH WORD MATCHING
		// TODO: eliminate partial duplication of effort with database filter functions
		if( m_option_search )
		{
			if( pos_search == pos_search_end )
			{
				apply_tag( m_tag_match, iter_search, iter_current );
				pos_search = 0;
			}
			char_search = m_searchstr[ pos_search ];
			if( char_search == Glib::Unicode::tolower( char_current ) )
			{
				if( pos_search == 0 )
					iter_search = iter_current;
				pos_search++;
			}
			else
			{
				pos_search = 0;
			}
		}

		// MARKUP PARSING
		switch( char_current )
		{
			case 0:		// treat end of the buffer like a line end
			case '\n':
			case '\r':
				switch( lookingfor )
				{
					case LF_FIRSTNEWLINE:
						apply_tag( m_tag_heading, iter_start, iter_current );
						if( char_current == 0 && iter_current.get_offset() == 0 )
							m_title = _( "<empty entry>" );
						else
							m_title = get_slice( iter_start, iter_current );
						if( ! m_flag_settextoperation )
							m_signal_title_changed.emit( m_title );
						lookingfor = LF_SPACE;
						break;
					case LF_NEWLINE:
						apply_tag( m_tag_subheading, iter_start, iter_current );
						lookingfor = LF_SPACE;
						break;
					case LF_URI:
						add_link( word_last, lookingfor, iter_start, iter_current );
						break;
					case LF_SPACE:
						// do nothing!
						break;
					default:
						// new line breaks bold, italic, etc.
						lookingfor = LF_SPACE;
						break;
				}

				word_last.clear();
				char_last = '\n';
			break;
			case '\t':
			case ' ':
				switch( lookingfor )
				{
					case LF_SPACE: // subheading
						lookingfor = LF_ORDINARYCHAR;
						lookingfor2 = LF_NEWLINE;
						iter_start = iter_current;
						break;
					case LF_URI:
						add_link( word_last, lookingfor, iter_start, iter_current );
						break;
					case LF_ORDINARYCHAR:
						lookingfor = LF_NOTHING;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				word_last.clear();
				char_last = ' ';
			break;
			case '*':
				process_markup( lookingfor, lookingfor2, '*',
						char_last, m_tag_bold, iter_start, iter_current );
			break;
			case '_':
				process_markup( lookingfor, lookingfor2, '_',
						char_last, m_tag_italic, iter_start, iter_current);
			break;
			case '=':
				process_markup( lookingfor, lookingfor2, '=',
						char_last, m_tag_strikethrough, iter_start, iter_current );
			break;
			case '#':
				process_markup( lookingfor, lookingfor2, '#',
						char_last, m_tag_highlight, iter_start, iter_current );
			break;
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				if( char_last == '1' )
				{
					int_last *= 10;
					int_last += ( char_current - '0' );
				}
				else
					int_last = ( char_current - '0' );
				switch ( lookingfor )
				{
					case LF_NOTHING:
					case LF_SPACE:
						lookingfor = LF_DIGITYEAR2;
						iter_start = iter_current;
						break;
					case LF_DIGITYEAR2:
					case LF_DIGITYEAR3:
					case LF_DIGITMONTH1:
					case LF_DIGITDAY1:
						lookingfor = LookingFor( lookingfor + 1 );
						break;
					case LF_DIGITYEAR4:
						if(	int_last > YEAR_MAX || int_last < YEAR_MIN )
							lookingfor = LF_NOTHING;
						else
						{
							date_link = make_year( int_last );
							lookingfor = LF_DOTYM;
						}
						break;
					case LF_DIGITMONTH2:
						if(	int_last > 12 || int_last < 1 )
							lookingfor = LF_NOTHING;
						else
						{
							date_link |= make_month( int_last );
							lookingfor = lookingfor2;
						}
						break;
					case LF_DIGITDAY2:
						date_link |= int_last;
						add_link_date(	date_link,
										lookingfor,
										iter_start,
										iter_current );
						break;
					case LF_ORDINARYCHAR:
						lookingfor = lookingfor2;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = '1';
			break;
			case '.':
				switch( lookingfor )
				{
					case LF_DOTYM:
						lookingfor = LF_DIGITMONTH1;
						lookingfor2 = LF_DOTMD;
						break;
					case LF_DOTMD:
						lookingfor = LF_DIGITDAY1;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = '.';
				break;
			case '/':
				switch( lookingfor )
				{
					case LF_SLASH1:
						lookingfor = LF_SLASH2;
						break;
					case LF_SLASH2:
						if( lookingfor2 == LF_SLASH3 )
						{
							lookingfor = LF_SLASH3;
							lookingfor2 = LF_URI;
						}
						else
							lookingfor = LF_ORDINARYCHAR;
						break;
					case LF_SLASH3:
						// link has to be at least one char long:
						lookingfor = LF_ORDINARYCHAR;
						break;
					case LF_DOTYM:
						lookingfor = LF_DIGITMONTH1;
						lookingfor2 = LF_SLASHMD;
						break;
					case LF_SLASHMD:
						lookingfor = LF_DIGITDAY1;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = '/';
				break;
			case ':':
				switch( lookingfor )
				{
					case LF_NOTHING:
						if(	word_last == "http" ||
							word_last == "https" ||
							word_last == "ftp" )
						{
							iter_start = iter_word;
							lookingfor = LF_SLASH1;
							lookingfor2 = LF_URI; // char to look for after SLASH2
						}
						else
						if(	word_last == "file" )
						{
							iter_start = iter_word;
							lookingfor = LF_SLASH1;
							lookingfor2 = LF_SLASH3; // char to look for after SLASH2
						}
						else
						if( word_last == "mailto" )
						{
							iter_start = iter_word;
							lookingfor = LF_ORDINARYCHAR;
							lookingfor2 = LF_AT;
						}
						else
							lookingfor = LF_NOTHING;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = ':';
				break;
			case '@':
				switch( lookingfor )
				{
					case LF_NOTHING:
						word_last.insert( 0, "mailto:" );
						iter_start = iter_word;
					case LF_AT:
						lookingfor = LF_ORDINARYCHAR;
						lookingfor2 = LF_URI;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = '@';
				break;
			default:
				switch ( lookingfor )
				{
					case LF_ORDINARYCHAR:
						lookingfor = lookingfor2;
						break;
					default:
						process_immediate( lookingfor );
						break;
				}

				add_char_toword( word_last, iter_word, iter_current, char_current );
				char_last = 'A';	// symbolizes any ordinary char
			break;
        }

		// this is here instead of being in the for() to...
		// ...continue for one more cycle after iter_end is reached:
		if( iter_current == iter_end )
			break;
    }

	m_flag_parsing = false;
}

// HELPER FUNCTIONS FOR PARSING
inline void
TextbufferDiary::add_char_toword(	Glib::ustring &word_last,
									Gtk::TextIter &iter_word,
									const Gtk::TextIter &iter_current,
									const gunichar char_current )
{
	if( word_last.empty() )
		iter_word = iter_current;
	word_last += char_current;
}

inline void
TextbufferDiary::process_markup(	LookingFor &lookingfor,
									LookingFor &lookingfor2,
									const char &char_current,
									gunichar &char_last,
									const Glib::RefPtr<Tag> &tag,
									Gtk::TextIter &iter_start,
									const Gtk::TextIter &iter_current )
{
	if( lookingfor == char_current )
	{
		if( char_last != ' ' )
		{
			Gtk::TextIter iter2 = iter_start;
			iter2++;
			apply_tag( m_tag_markup, iter_start, iter2 );
			apply_tag( tag, iter2, iter_current );
			iter2 = iter_current;
			iter2++;
			apply_tag( m_tag_markup, iter_current, iter2 );
			lookingfor = LF_NOTHING;
		}
	}
	else if(	lookingfor == LF_NOTHING ||
				lookingfor == LF_SPACE )
	{
		lookingfor = LF_ORDINARYCHAR;
		lookingfor2 = static_cast<LookingFor>( char_current );
		iter_start = iter_current;
	}
	else if( lookingfor == LF_ORDINARYCHAR )
		lookingfor = lookingfor2;
	else
		process_immediate( lookingfor );

	char_last = char_current;
}

inline void
TextbufferDiary::process_immediate( LookingFor &lookingfor )
{
	// process chars that must be found immediately when needed
	switch( lookingfor )
	{
		case LF_SPACE:
		case LF_ORDINARYCHAR:
		case LF_SLASH1:
		case LF_SLASH2:
		case LF_SLASH3:
		case LF_DIGITYEAR2:
		case LF_DIGITYEAR3:
		case LF_DIGITYEAR4:
		case LF_DOTYM:
		case LF_DIGITMONTH1:
		case LF_DIGITMONTH2:
		case LF_DOTMD:
		case LF_DIGITDAY1:
		case LF_DIGITDAY2:
			lookingfor = LF_NOTHING;
			break;
		default:
			break;
	}
}

inline void
TextbufferDiary::add_link(	const std::string& url,
							LookingFor& lookingfor,
							const Gtk::TextIter& iter_start,
							const Gtk::TextIter& iter_tagend )
{
	Gtk::TextIter iter_end = iter_tagend;
	iter_end--;
	m_list_links.push_back( new LinkUri(	create_mark( iter_start ),
											create_mark( iter_end ),
											url ) );

	apply_tag( m_tag_link, iter_start, iter_tagend );

	lookingfor = LF_NOTHING;
}

// TODO: this function might be combined with add_link()
inline void
TextbufferDiary::add_link_date(	const Date date,
								LookingFor& lookingfor,
								const Gtk::TextIter& iter_start,
								const Gtk::TextIter& iter_end )
{
	if( validate_day( date ) )
	{
		LinkStatus status = m_signal_link_needs_checking.emit( date );
		if( status < LS_INVALID )
		{
			Gtk::TextIter iter_tagend = iter_end;
			iter_tagend++;
			m_list_links.push_back(
					new LinkEntry(	create_mark( iter_start ),
									create_mark( iter_end ),
									date ) );

			if( status == LS_OK )
				apply_tag( m_tag_link, iter_start, iter_tagend );
			else
			if( status == LS_ENTRY_UNAVAILABLE )
				apply_tag(	m_tag_link_broken, iter_start, iter_tagend );
		}
	}
	lookingfor = LF_NOTHING;
}

void
TextbufferDiary::set_richtext( const Glib::ustring &text )
{
	m_flag_settextoperation = true;
	clear_links();
	m_title = _( "<empty entry>" );
	m_ptr2undomanager->freeze();
	Gtk::TextBuffer::set_text( text );
	place_cursor( begin() );
	m_ptr2undomanager->thaw();
	m_ptr2entry = NULL;
	m_flag_settextoperation = false;
}

void
TextbufferDiary::set_richtext( Entry *entry )
{
	set_richtext( entry->get_text() );
	m_ptr2entry = entry;
	// TODO: evaluate per entry spellcheck options here
}

void
TextbufferDiary::reparse( void )
{
	place_cursor( begin() );
	parse( 0, get_char_count() );
}

void
TextbufferDiary::on_insert(	const Gtk::TextIter& iterator,
							const Glib::ustring& text,
							int bytes )
{
	if( ! m_ptr2undomanager->is_freezed() )
	{
		UndoInsert * undo_insert = new UndoInsert( iterator.get_offset(), text );
		m_ptr2undomanager->add_action( undo_insert );
	}
	else
	{
		Gtk::TextIter iter_scroll( iterator );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_insert( iterator, text, bytes );

	if( m_flag_ongoingoperation )
		return;

	Glib::ustring::size_type pos_start = 0;
	Glib::ustring::size_type pos_end = 0;
	if( m_flag_settextoperation == false )
	{
		pos_start = get_text().find_last_of( '\n', iterator.get_offset() - text.size()  );
		if( pos_start != Glib::ustring::npos && pos_start > 0 )
			pos_start = get_text().find_last_of( '\n', pos_start - 1 );
		if( pos_start == Glib::ustring::npos )
			pos_start = 0;

		pos_end = get_text().find( '\n', iterator.get_offset() );
		if( pos_end == Glib::ustring::npos )
			pos_end = get_char_count();
	}
	else
	{
		pos_end = get_char_count();
	}

	parse( pos_start, pos_end );
}

void
TextbufferDiary::on_erase(	const Gtk::TextIter& iter_start,
							const Gtk::TextIter& iter_end )
{
	if( ! m_ptr2undomanager->is_freezed() )
	{
		UndoErase * undo_erase = new UndoErase(
				iter_start.get_offset(), get_slice( iter_start, iter_end ) );
		m_ptr2undomanager->add_action( undo_erase );
	}
	else
	{
		Gtk::TextIter iter_scroll( iter_start );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_erase( iter_start, iter_end );

	// set_text() calls on_erase too:
	if( m_flag_settextoperation == false && m_flag_ongoingoperation == false )
	{
		Glib::ustring::size_type pos_start = iter_start.get_offset();
		Glib::ustring::size_type pos_end = iter_end.get_offset();

		if( pos_start > 0 )
		{
			pos_start = get_text().rfind( '\n', pos_start - 1 );
			if( pos_start == Glib::ustring::npos )
				pos_start = 0;
			else
				pos_start++;
		}

		pos_end = get_text().find( '\n', pos_end );
		if( pos_end == Glib::ustring::npos )
			pos_end = get_char_count();

		parse( pos_start, pos_end );
	}
}

void
TextbufferDiary::on_remove_tag(	const Glib::RefPtr< TextBuffer::Tag >& tag,
								const Gtk::TextIter& iter_begin,
								const Gtk::TextIter& iter_end )
{
	// do not remove gtkspell tags while parsing:
	if( m_flag_parsing && tag->property_name() == "gtkspell-misspelled" )
		return;
	else
		Gtk::TextBuffer::on_remove_tag( tag, iter_begin, iter_end );
}

const Link*
TextbufferDiary::get_link( int offset ) const
{
	for (	ListLinks::const_iterator iter = m_list_links.begin();
			iter != m_list_links.end();
			iter++ )
	{
		if(	offset >= ( *iter )->m_mark_start->get_iter().get_offset() &&
			offset <= ( *iter )->m_mark_end->get_iter().get_offset() )
			return( *iter );
	}

	return NULL;
}

void
TextbufferDiary::clear_links( int pos_start, int pos_end )
{
	ListLinks::iterator iter_tmp;
	for(	ListLinks::iterator iter = m_list_links.begin();
			iter != m_list_links.end(); )
		if(	pos_start <= ( *iter )->m_mark_start->get_iter().get_offset() &&
			pos_end >= ( *iter )->m_mark_end->get_iter().get_offset() )
		{
			iter_tmp = iter;
			iter++;
			delete( *iter_tmp );
			m_list_links.erase( iter_tmp );
		}
		else
			iter++;
}

void
TextbufferDiary::clear_links( void )
{
	for(	ListLinks::iterator iter = m_list_links.begin();
			iter != m_list_links.end();
			iter++ )
	{
		delete( *iter );
	}
	m_list_links.clear();
}

void
TextbufferDiary::adjust_colors( void )
{
	Gdk::Color color_base = m_ptr2textview->get_style()->get_base( Gtk::STATE_NORMAL );

	m_tag_match->property_background_gdk() = contrast(
			color_base, m_theme_color_match );
	m_tag_markup->property_foreground_gdk() = midtone(
			color_base, m_ptr2textview->get_style()->get_text( Gtk::STATE_NORMAL ) );
	m_tag_link->property_foreground_gdk() = contrast(
			color_base, m_theme_color_link );
	m_tag_link_broken->property_foreground_gdk() = contrast(
			color_base, m_theme_color_linkbroken );
}

void
TextbufferDiary::handle_menu( Gtk::Menu *menu )
{
	Gtk::CheckMenuItem *menuitem_spell = Gtk::manage(
			new Gtk::CheckMenuItem( _("Check Spelling" ) ) );

	Gtk::MenuItem *menuitem_theme = Gtk::manage(
			new Gtk::MenuItem( _("Edit Theme..." ) ) );

	Gtk::SeparatorMenuItem *separator = Gtk::manage( new Gtk::SeparatorMenuItem );

	menuitem_spell->set_active( m_ptr2database->get_spellcheck() );

	menuitem_spell->signal_toggled().connect(
			sigc::bind(	sigc::mem_fun( *this, &TextbufferDiary::set_spellcheck ),
						! m_ptr2database->get_spellcheck() ) );

	menuitem_theme->signal_activate().connect(
			sigc::mem_fun( *this, &TextbufferDiary::edit_theme ) );

	menuitem_spell->show();
	menuitem_theme->show();
	separator->show();

	menu->prepend( *separator );
	menu->prepend( *menuitem_theme );
	menu->prepend( *menuitem_spell );
}

void
TextbufferDiary::toggle_format(	Glib::RefPtr< Tag > tag, const Glib::ustring &markup )
{
	Gtk::TextIter iter_start, iter_end;
	if( get_has_selection() )
	{
		get_selection_bounds( iter_start, iter_end );
		iter_end--;
		int pos_start = -1, pos_end = -1;

		for( ; ; iter_start++ )
		{
			if( iter_start.has_tag( tag ) )
				return;
			switch( iter_start.get_char() )
			{
				// do nothing if selection spreads over more than one line:
				case '\n':
					return;
				case ' ':
				case '\t':
					break;
				case '*':
				case '_':
				case '#':
				case '=':
					if( iter_start.get_char() == markup[ 0 ] )
						break;
				default:
					if( pos_start < 0 )
						pos_start = iter_start.get_offset();
					pos_end = iter_start.get_offset();
					break;
			}
			if( iter_start == iter_end )
				break;
		}
		// add markup chars to the beginning and end:
		if( pos_start >= 0 )
		{
			pos_end += 2;
			insert( get_iter_at_offset( pos_start ), markup );
			insert( get_iter_at_offset( pos_end ), markup );
			place_cursor( get_iter_at_offset( pos_end ) );
		}
	}
	else	// no selection case
	{
		Glib::RefPtr< Gtk::TextMark > mark = get_insert();
		iter_start = mark->get_iter();
		if( Glib::Unicode::isspace( iter_start.get_char() ) ||
			iter_start.has_tag( TextbufferDiary::m_tag_markup ) )
		{
			if( ! iter_start.starts_line() )
				iter_start--;
			else
				return;
		}
		if( iter_start.has_tag( tag ) )
		{
			m_flag_ongoingoperation = true;

			// find start and end points of formatting:
			if( iter_start.starts_word() )
				iter_start++;	// necessary when cursor is between a space char
								// and non-space char
			iter_start.backward_to_tag_toggle( tag );
			backspace( iter_start );

			iter_end = mark->get_iter();
			if( iter_end.has_tag( tag ) )
				iter_end.forward_to_tag_toggle( TextbufferDiary::m_tag_markup );

			m_flag_ongoingoperation = false;

			backspace( ++iter_end );
		}
		else
		if( iter_start.get_tags().empty() )		// nested tags are not supported atm
		{
			// find word boundaries:
			if( !( iter_start.starts_word() || iter_start.starts_line() ) )
				iter_start.backward_word_start();
			insert( iter_start, markup );

			iter_end = mark->get_iter();
			if( !( iter_end.ends_word() || iter_end.ends_line() ) )
			{
				iter_end.forward_word_end();
				insert( iter_end, markup );
			}
			else
			{
				int offset = iter_end.get_offset();
				insert( iter_end, markup );
				place_cursor( get_iter_at_offset( offset ) );

			}
		}
	}

}

void
TextbufferDiary::toggle_bold( void )
{
	toggle_format( m_tag_bold, "*" );
}

void
TextbufferDiary::toggle_italic( void )
{
	toggle_format( m_tag_italic, "_" );
}

void
TextbufferDiary::toggle_strikethrough( void )
{
	toggle_format( m_tag_strikethrough, "=" );
}

void
TextbufferDiary::toggle_highlight( void )
{
	toggle_format( m_tag_highlight, "#" );
}

void
TextbufferDiary::set_spellcheck( bool option_spell )
{
	if( option_spell )
	{
		if( ! m_ptr2spellobj )
			m_ptr2spellobj = gtkspell_new_attach( m_ptr2textview->gobj(), NULL, NULL );
	}
	else
	if( m_ptr2spellobj )
	{
		gtkspell_detach( m_ptr2spellobj );
		m_ptr2spellobj = NULL;
	}

	// FIXME: here is not the most right place to do this!
	m_ptr2database->set_spellcheck( option_spell );
}

void
TextbufferDiary::edit_theme( void )
{
	DialogTheme *dialogtheme = new DialogTheme( m_ptr2database, this );
	dialogtheme->present();	//redundant?
}

void
TextbufferDiary::reset_theme( void )
{
	m_ptr2textview->modify_font( m_theme_font );
	m_ptr2textview->modify_base( Gtk::STATE_NORMAL, m_theme_color_base  );
	m_ptr2textview->modify_text( Gtk::STATE_NORMAL, m_theme_color_text  );
	m_tag_heading->property_foreground_gdk() = m_theme_color_heading;
	m_tag_subheading->property_foreground_gdk() = m_theme_color_subheading;
	m_tag_match->property_background_gdk() = m_theme_color_match;
	m_tag_markup->property_foreground_gdk() = m_theme_color_markup;
	m_tag_highlight->property_background_gdk() = m_theme_color_highlight;
	m_tag_link->property_foreground_gdk() = m_theme_color_link;
	m_tag_link_broken->property_foreground_gdk() = m_theme_color_linkbroken;
}

void
TextbufferDiary::set_theme( Theme *theme )
{
	m_ptr2textview->modify_font( Pango::FontDescription( theme->font ) );
	m_ptr2textview->modify_base( Gtk::STATE_NORMAL, Gdk::Color( theme->color_base ) );
	m_ptr2textview->modify_text( Gtk::STATE_NORMAL, Gdk::Color( theme->color_text ) );
	m_tag_heading->property_foreground() = theme->color_heading;
	m_tag_subheading->property_foreground() = theme->color_subheading;
	m_tag_highlight->property_background() = theme->color_highlight;

	adjust_colors();
}

// TEXTVIEW
TextviewDiary::TextviewDiary( TextbufferDiary *buffer )
	:	Gtk::TextView( static_cast< Glib::RefPtr< TextbufferDiary > >( buffer ) ),
	m_ptr2buffer( buffer ),
	m_cursor_hand( Gdk::HAND2 ), m_cursor_xterm( Gdk::XTERM ),
	m_ptr2cursor_last( &m_cursor_xterm ), m_link_hovered( NULL )
{
	buffer->set_textview( this );
	set_wrap_mode( Gtk::WRAP_WORD );
	set_left_margin( 10 );

	signal_populate_popup().connect(
			sigc::mem_fun( *buffer, &TextbufferDiary::handle_menu ) );
}

inline void
TextviewDiary::update_link( void )
{
	Gtk::TextIter		iter;
	const Gdk::Cursor	*ptr2cursor = &m_cursor_xterm;
	int					pointer_x, pointer_y;
	int					buffer_x, buffer_y;
    Gdk::ModifierType	modifiers;

    Gtk::Widget::get_window()->get_pointer( pointer_x, pointer_y, modifiers );
	window_to_buffer_coords(	Gtk::TEXT_WINDOW_WIDGET,
								pointer_x, pointer_y,
								buffer_x, buffer_y );
	get_iter_at_location( iter, buffer_x, buffer_y );
	m_link_hovered = m_ptr2buffer->get_link( iter.get_offset() );

	if( m_link_hovered != NULL )
	{
		if( !( modifiers & Gdk::CONTROL_MASK ) )
			ptr2cursor = &m_cursor_hand;
	}

	if( ptr2cursor != m_ptr2cursor_last )
	{
		m_ptr2cursor_last = ptr2cursor;
		get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( *ptr2cursor );
	}
}

bool
TextviewDiary::on_motion_notify_event( GdkEventMotion *event )
{
	update_link();
	return Gtk::TextView::on_motion_notify_event( event );
}

bool
TextviewDiary::on_button_release_event( GdkEventButton *event )
{
	if( m_link_hovered != NULL )
		if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK )
			m_link_hovered->go();

	return Gtk::TextView::on_button_release_event( event );
}

bool
TextviewDiary::on_key_press_event( GdkEventKey *event )
{
	if( event->keyval == GDK_Control_L || event->keyval == GDK_Control_R )
	{
		if( m_link_hovered )
			update_link();
	}

	if( ( event->state & Gdk::CONTROL_MASK ) == Gdk::CONTROL_MASK )
	{
		switch( event->keyval )
		{
			case GDK_b:
				m_ptr2buffer->toggle_bold();
				break;
			case GDK_i:
				m_ptr2buffer->toggle_italic();
				break;
			case GDK_s:
				m_ptr2buffer->toggle_strikethrough();
				break;
			case GDK_h:
				m_ptr2buffer->toggle_highlight();
				break;
		}
	}

	return Gtk::TextView::on_key_press_event( event );
}

bool
TextviewDiary::on_key_release_event( GdkEventKey *event )
{
	if( event->keyval == GDK_Control_L || event->keyval == GDK_Control_R )
		if( m_link_hovered )
			update_link();
		
	return Gtk::TextView::on_key_release_event( event );
}

//void
//TextviewDiary::on_style_changed( const Glib::RefPtr< Gtk::Style > &style_prev )
//{
//	Gtk::TextView::on_style_changed( style_prev );
////	TextbufferDiary = get_pango_context()->get_font_description();
//	TextbufferDiary::m_theme_font = get_style()->get_font();
//}

