/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>
#include <cassert>

#include "lifeobase.hpp"
#include "view_entry.hpp"
#include "widget_textview.hpp"


#define GTKSPELL_MISSPELLED_TAG "gtkspell-misspelled"


using namespace LIFEO;


// STATIC MEMBERS
Glib::RefPtr<Gtk::TextTag>  TextbufferDiary::m_tag_strikethrough;
Glib::RefPtr<Gtk::TextTag>  TextbufferDiary::m_tag_highlight;
Glib::RefPtr<Gtk::TextTag>  TextbufferDiary::m_tag_checkbox;

Gdk::RGBA                   TextbufferDiary::m_theme_color_match( "green" );
Gdk::RGBA                   TextbufferDiary::m_theme_color_markup( "gray" );
Gdk::RGBA                   TextbufferDiary::m_theme_color_link( "navy" );
Gdk::RGBA                   TextbufferDiary::m_theme_color_linkbroken( "red" );

Gtk::TextBuffer             *UndoEdit::m_ptr2buffer;

PredicateNL                 TextbufferDiary::s_predicate_nl;

// LINK
Link::Link(	const Glib::RefPtr< Gtk::TextMark > &start,
			const Glib::RefPtr< Gtk::TextMark > &end )
	:	m_mark_start( start ), m_mark_end( end )
{

}

Link::~Link()
{
	// TODO: is this necessary?
	Glib::RefPtr< Gtk::TextBuffer > buffer = m_mark_start->get_buffer();
	if ( buffer )
	{
		buffer->delete_mark( m_mark_start );
		buffer->delete_mark( m_mark_end );
	}
}

// LINK TO ID
LinkID::LinkID( const Glib::RefPtr< Gtk::TextMark > &start,
				const Glib::RefPtr< Gtk::TextMark > &end,
				DEID id )
:	Link( start, end ), m_id( id )
{
}

void
LinkID::go( void )
{
	Diary::d->get_element( m_id )->show();
}

// LINK TO ENTRY
Gtk::Menu *LinkEntry::menu_link = NULL;
LinkEntry::LinkEntry( const Glib::RefPtr< Gtk::TextMark > &start,
					  const Glib::RefPtr< Gtk::TextMark > &end,
					  Date date )
:	Link( start, end ), m_date( date )
{
}

void
LinkEntry::go( void )
{
    Entryvector *entries = Diary::d->get_entries( m_date.get_pure() );

    // seek and destroy cyclic links
    for( Entryiterv iter = entries->begin(); iter != entries->end(); ++iter )
    {
        if( ( *iter ) == Lifeobase::panel_main->get_cur_elem() )
        {
            entries->erase( iter );
            break;
        }
    }

    if( entries->size() > 1 )
    {
        if( menu_link )
            delete menu_link;
        menu_link = new Gtk::Menu;

        for( Entryiterv iter = entries->begin(); iter != entries->end(); ++iter )
        {
            Entry *entry = ( *iter );
            Gtk::MenuItem *mitem = Gtk::manage( new Gtk::MenuItem( entry->get_name() ) );
            mitem->signal_activate().connect(
                    sigc::mem_fun( entry, &Entry::show ) );
            menu_link->append( *mitem );
        }
        menu_link->show_all_children();
        menu_link->popup( 0, gtk_get_current_event_time() );
    }
    else
    if( entries->size() == 1 )
        entries->at( 0 )->show();
    else
    {
        Entry *entry = Diary::d->create_entry( m_date.m_date );
        Lifeobase::panel_diary->update_entry_list();
        entry->show();
    }

    delete entries;
}

// LINK TO URI
Gtk::TextBuffer *LinkUri::m_ptr2buffer;

LinkUri::LinkUri(	const Glib::RefPtr< Gtk::TextMark > &start,
					const Glib::RefPtr< Gtk::TextMark > &end,
					const std::string& url )
	:	Link( start, end ), m_url( url )
{
}

void
LinkUri::go( void )
{
	GError *err = NULL;
	gtk_show_uri (NULL, m_url.c_str(), GDK_CURRENT_TIME, &err);
}

// LINK CHECKBOX
const Glib::ustring		LinkCheck::boxes = "☐☑☒";

LinkCheck::LinkCheck( const Glib::RefPtr< Gtk::TextMark > &start,
					  const Glib::RefPtr< Gtk::TextMark > &end,
					  unsigned int state_index )
	:	Link( start, end ), m_state_index( state_index )
{
}

void
LinkCheck::go( void )
{
    PRINT_DEBUG( "LinkCheck::go()" );
    m_state_index = ( m_state_index + 1 ) % 3;

    // link should be preserved
    Lifeobase::view_entry->get_buffer()->m_flag_ongoing_operation++;
    Lifeobase::view_entry->get_buffer()->insert(
            m_mark_start->get_iter(), boxes.substr( m_state_index, 1) );

    Gtk::TextIter iter_start = m_mark_start->get_iter();
    iter_start++;
    Lifeobase::view_entry->get_buffer()->erase( iter_start, m_mark_end->get_iter() );

    iter_start = m_mark_start->get_iter();
    Gtk::TextIter iter_end = m_mark_end->get_iter();
    Lifeobase::view_entry->get_buffer()->apply_tag(
                    TextbufferDiary::m_tag_checkbox, iter_start, iter_end );

    iter_start = m_mark_end->get_iter();
    iter_end.forward_line();
    switch( m_state_index )
    {
        case 1:
            Lifeobase::view_entry->get_buffer()->remove_tag(
                    TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
            Lifeobase::view_entry->get_buffer()->apply_tag(
                    TextbufferDiary::m_tag_highlight, iter_start, iter_end );
            break;
        case 2:
            Lifeobase::view_entry->get_buffer()->remove_tag(
                    TextbufferDiary::m_tag_highlight, iter_start, iter_end );
            Lifeobase::view_entry->get_buffer()->apply_tag(
                    TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
            break;
        case 0:
            Lifeobase::view_entry->get_buffer()->remove_tag(
                    TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
            Lifeobase::view_entry->get_buffer()->remove_tag(
                    TextbufferDiary::m_tag_highlight, iter_start, iter_end );
            break;
    }

    Lifeobase::view_entry->get_buffer()->m_flag_ongoing_operation--;
}

InlineImage::InlineImage( const Glib::RefPtr< Gtk::TextMark >& mark )
:   m_mark( mark )
{ }

InlineImage::~InlineImage()
{
    Glib::RefPtr< Gtk::TextBuffer > buffer = m_mark->get_buffer();
    if( buffer )
        buffer->delete_mark( m_mark );
}

// TEXTBUFFERDIARY =================================================================================
TextbufferDiary::TextbufferDiary( void )
:	Gtk::TextBuffer(),
	m_flag_settextoperation( false ), m_flag_ongoing_operation( 0 ), m_flag_parsing( false ),
	m_ptr2textview( NULL ), m_enchant_dict( NULL )
{
	LinkUri::m_ptr2buffer = this;
	UndoEdit::m_ptr2buffer = this;

    // TAGS
    Glib::RefPtr< TagTable > tag_table = get_tag_table();

	m_tag_heading = Tag::create( "heading" );
	m_tag_heading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_heading->property_scale() = 1.5;
	m_tags.push_back( m_tag_heading );

	m_tag_subheading = Tag::create( "subheading" );
	m_tag_subheading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_subheading->property_scale() = 1.2;
	m_tags.push_back( m_tag_subheading );

	m_tag_match = Tag::create( "match" );
	m_tags.push_back( m_tag_match );

	m_tag_markup = Tag::create( "markup" );
	m_tag_markup->property_scale() = 0.5;
	m_tags.push_back( m_tag_markup );

	m_tag_markup_link = Tag::create( "markup.link" );
	m_tags.push_back( m_tag_markup_link );

	m_tag_hidden = Tag::create( "hidden" );
	m_tag_hidden->property_invisible() = true;
	m_tags.push_back( m_tag_hidden );

	m_tag_bold = Tag::create( "bold" );
	m_tag_bold->property_weight() = Pango::WEIGHT_BOLD;
	m_tags.push_back( m_tag_bold );

	m_tag_italic = Tag::create( "italic" );
	m_tag_italic->property_style() = Pango::STYLE_ITALIC;
	m_tags.push_back( m_tag_italic );

	m_tag_strikethrough = Tag::create( "strikethrough" );
	m_tag_strikethrough->property_strikethrough() = true;
	m_tags.push_back( m_tag_strikethrough );

	m_tag_highlight = Tag::create( "highlight" );
	m_tags.push_back( m_tag_highlight );

    m_tag_comment = Tag::create( "comment" );
    m_tag_comment->property_scale() = 0.8;
    m_tag_comment->property_rise() = 5000;
    m_tag_comment->property_strikethrough() = false; // for comments in canceled check list items
    m_tags.push_back( m_tag_comment );

    m_tag_region = Tag::create( "region" );
    m_tags.push_back( m_tag_region );

	m_tag_link = Tag::create( "link" );
	m_tag_link->property_underline() = Pango::UNDERLINE_SINGLE;
	m_tags.push_back( m_tag_link );

	m_tag_link_broken = Tag::create( "link.broken" );
	m_tag_link_broken->property_underline() = Pango::UNDERLINE_SINGLE;
	m_tags.push_back( m_tag_link_broken );

	// this is just for keeping the boundaries:
	m_tag_link_hidden = Tag::create( "link.hidden" );
	m_tags.push_back( m_tag_link_hidden );

	m_tag_checkbox = Tag::create( "checkbox" );
	m_tag_checkbox->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_checkbox->property_scale() = 1.2;
	m_tags.push_back( m_tag_checkbox );

    m_tag_image = Tag::create( "image" );
    m_tag_image->property_justification() = Gtk::JUSTIFY_CENTER;
    m_tags.push_back( m_tag_image );

    m_tag_misspelled = Tag::create( GTKSPELL_MISSPELLED_TAG );
    m_tag_misspelled->property_underline() = Pango::UNDERLINE_ERROR;
    m_tags.push_back( m_tag_misspelled );

    for( VecTags::iterator iter_tag = m_tags.begin(); iter_tag != m_tags.end(); ++iter_tag )
        tag_table->add( *iter_tag );
}

void
TextbufferDiary::handle_login( void )
{
	// SPELL CHECKING
    if( Diary::d->is_read_only() )
        m_flag_handle_word = false;
}

void
TextbufferDiary::handle_logout( void )
{
    EntryParser::set_search_str( "" );

    if( m_enchant_dict )
    {
        enchant_broker_free_dict( Lifeobase::s_enchant_broker, m_enchant_dict );
        m_enchant_dict = NULL;
    }
}

void
TextbufferDiary::set_search_str( const Glib::ustring &str )
{
    EntryParser::set_search_str( str );
    reparse();
}

bool
TextbufferDiary::select_searchstr_previous( void )
{
    if( m_search_str.size() <= 0 )
        return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_start.is_start() )
		return false;

	iter_bound_end = iter_bound_start;
	
	if( ! iter_bound_start.backward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_start.backward_to_tag_toggle( m_tag_match );
	iter_bound_start.backward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_end.backward_search(	m_search_str,
											Gtk::TextSearchFlags( 0 ),
											iter_start,
											iter_end,
											iter_bound_start ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

bool
TextbufferDiary::select_searchstr_next( void )
{
    if( m_search_str.size() <= 0 )
        return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_end.is_end() )
		return false;

	iter_bound_start = iter_bound_end;

	if( ! iter_bound_end.forward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_end.forward_to_tag_toggle( m_tag_match );
	if( iter_bound_end.has_tag( m_tag_match ) )
		iter_bound_end.forward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_start.forward_search(	m_search_str,
													Gtk::TextSearchFlags( 0 ),
													iter_start,
													iter_end,
													iter_bound_end ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

void
TextbufferDiary::parse( const Gtk::TextIter &start, const Gtk::TextIter &end )
{
    m_flag_parsing = true;

    // COMPLETELY CLEAR THE PARSING REGION
    remove_all_tags( start, end );

    int i_start( start.get_offset() ), i_end( end.get_offset() );
    clear_images( i_start, i_end ); // alters the buffer
    clear_links( i_start, i_end );

    // DO THE PARSING
    EntryParser::parse( i_start, i_end );

    m_flag_parsing = false;
}

void
TextbufferDiary::reparse( void )
{
    place_cursor( begin() );
    m_flag_settextoperation = true;
    parse( begin(), end() );
    m_flag_settextoperation = false;
}

gunichar
TextbufferDiary::get_char_at( int i )
{
    return get_iter_at_offset( i ).get_char();
}

// PARSING APPLIERS
void
TextbufferDiary::apply_heading( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( 0 ) );
    Gtk::TextIter iter_end( iter_start );
    if( ! iter_start.ends_line() )
        iter_end.forward_to_line_end();
    apply_tag( m_tag_heading, iter_start, iter_end );

    if( ! m_flag_settextoperation )
    {
        m_ptr2entry->set_name( iter_end.get_offset() > 0 ?
                get_text( iter_start, iter_end ) :
                get_char_count() < 1 ? STRING::EMPTY_ENTRY_TITLE : "" );

        Lifeobase::panel_diary->handle_entry_title_changed( m_ptr2entry );
        Lifeobase::panel_main->refresh_title();
    }
}

void
TextbufferDiary::apply_subheading( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_current( iter_start );
    iter_current.forward_to_line_end();
    apply_tag( m_tag_subheading, iter_start, iter_current );
}

void
TextbufferDiary::apply_bold( void )
{
	apply_markup( m_tag_bold );
}

void
TextbufferDiary::apply_italic( void )
{
	apply_markup( m_tag_italic );
}

void
TextbufferDiary::apply_strikethrough( void )
{
	apply_markup( m_tag_strikethrough );
}

void
TextbufferDiary::apply_highlight( void )
{
	apply_markup( m_tag_highlight );
}

void
TextbufferDiary::apply_comment( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current ) );
    iter_end++;
    apply_tag( m_tag_comment, iter_start, iter_end );
}

void
TextbufferDiary::apply_ignore( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    apply_tag( m_tag_region, iter_start, iter_end );
}

void
TextbufferDiary::apply_hidden_link_tags( Gtk::TextIter &iter_end,
                                         const Glib::RefPtr< Tag > &tag_link )
{
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );

	apply_tag( m_tag_markup_link, iter_start, iter_tab );
	apply_tag( m_tag_markup_link, iter_current, iter_end );
	remove_tag( m_tag_misspelled, iter_start, iter_tab );

	apply_tag( tag_link, iter_tab, iter_current );
	apply_tag( m_tag_link_hidden, iter_start, iter_end );

	// hide link markup if cursor is not in it:
	Gtk::TextIter iter_insert = get_iter_at_mark( get_insert() );
	if( iter_insert.get_offset() < (int) m_pos_start ||
		iter_insert.get_offset() > (int) pos_current )
	{
		apply_tag( m_tag_hidden, iter_start, iter_tab );
		apply_tag( m_tag_hidden, iter_current, iter_end );
		parser_open_tag_begin = std::string::npos;
		parser_open_tag_is_link = false;
	}
	else
	{
		parser_open_tag_begin = m_pos_start;
		parser_open_tag_end = iter_end.get_offset();
        parser_open_tag_is_link = true;
	}
}

void
TextbufferDiary::apply_link( void )
{
    // HANDLE RELATIVE PATHS
    if( word_last.find( "rel://" ) == 0 )
    {
        word_last.replace( 0, 5, "file://" + Glib::path_get_dirname( Diary::d->get_path() ) );
    }

    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );

	if( m_flag_hidden_link )
	{
        Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );
        Gtk::TextIter iter_url_start( get_iter_at_offset( m_pos_start + 1 ) );
        Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );
        m_list_links.push_back( new LinkUri( create_mark( iter_tab ),
                                             create_mark( iter_current ),
                                             get_slice( iter_url_start, iter_tab ) ) );

		apply_hidden_link_tags( iter_end, m_tag_link );
	}
	else
	{
		PRINT_DEBUG( "url: " + word_last );

        if( word_last.find( "file:///" ) == 0  &&
            iter_start.starts_line() && iter_current.ends_line() )
        {
            try
            {
                Glib::RefPtr< Gdk::Pixbuf > buf(
                        Lifeobase::get_thumbnail( Glib::filename_from_uri( word_last ) ) );

                apply_tag( m_tag_hidden, iter_start, iter_current );
                remove_tag( m_tag_misspelled, iter_start, iter_current );

                m_list_images.insert( new InlineImage( create_mark( iter_current ) ) );
                insert_pixbuf( iter_current, buf );

                pos_current++;
                m_pos_end++;

                iter_start = get_iter_at_offset( m_pos_start );
                iter_current = get_iter_at_offset( pos_current );

                apply_tag( m_tag_image, iter_start, iter_current );
            }
            catch( Glib::FileError &er )
            {
                print_error( "Link target not found" );
                apply_tag( m_tag_link, iter_start, iter_current );
                remove_tag( m_tag_misspelled, iter_start, iter_current );
            }
            catch( Gdk::PixbufError &er )
            {
                PRINT_DEBUG( "Link is not an image" );
                apply_tag( m_tag_link, iter_start, iter_current );
                remove_tag( m_tag_misspelled, iter_start, iter_current );
            }
        }
        else
        {
            apply_tag( m_tag_link, iter_start, iter_current );
            remove_tag( m_tag_misspelled, iter_start, iter_current );
        }

        m_list_links.push_back( new LinkUri( create_mark( iter_start ),
                                             create_mark( iter_current ),
                                             word_last ) );
	}
}

void
TextbufferDiary::apply_link_id( void )
{
	DiaryElement *element( Diary::d->get_element( id_last ) );

    if( element != NULL )
    {
        if( element->get_type() == DiaryElement::ET_ENTRY )
        {
            Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );
            Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
            Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );

            m_list_links.push_back( new LinkID( create_mark( iter_tab ),
                                                create_mark( iter_current ),
                                                id_last ) );

            apply_hidden_link_tags( iter_end, m_tag_link );
            return;
        }
    }
    // indicate dead links
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );
    apply_tag( m_tag_link_broken, iter_start, iter_end );
}

void
TextbufferDiary::apply_link_date( void )
{
    LinkStatus status( LS_OK );
    Entry *ptr2entry( Diary::d->get_entry( date_last.m_date + 1 ) ); // + 1 fixes order
    if( ptr2entry == NULL )
        status = LS_ENTRY_UNAVAILABLE;
    else
    if( date_last.get_pure() == m_ptr2entry->get_date().get_pure() )
        status = Diary::d->get_day_has_multiple_entries( date_last ) ? LS_OK : LS_CYCLIC;

    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );

    if( status < LS_INVALID )
    {
        Gtk::TextIter iter_end( iter_current );
        iter_end++;
        // if hidden link:
        if( char_current == '>' )
        {
            m_list_links.push_back( new LinkEntry( create_mark( iter_tab ),
                                                   create_mark( iter_current ),
                                                   date_last ) );
            apply_hidden_link_tags( iter_end,
                    status == LS_OK ? m_tag_link : m_tag_link_broken );
        }
        else
        {
            m_list_links.push_back(
                    new LinkEntry( create_mark( iter_start ),
                                   create_mark( iter_end ),
                                   date_last ) );

            apply_tag( status == LS_OK ? m_tag_link :
                            m_tag_link_broken, iter_start, iter_end );
        }
    }
}

void
TextbufferDiary::apply_check_unf( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current - 1 ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current ) );
    if( ! Diary::d->is_read_only() )
        m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                               create_mark( iter_end ),
                                               0 ) );
    apply_tag( m_tag_checkbox, iter_start, iter_end );
}

void
TextbufferDiary::apply_check_fin( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current -1 ) );
    Gtk::TextIter iter_box( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    if( ! Diary::d->is_read_only() )
        m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                               create_mark( iter_box ),
                                               1 ) );
    apply_tag( m_tag_checkbox, iter_start, iter_box );
    apply_tag( m_tag_highlight, iter_box, iter_end );
}

void
TextbufferDiary::apply_check_ccl( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current - 1 ) );
    Gtk::TextIter iter_box( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    if( ! Diary::d->is_read_only() )
        m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                               create_mark( iter_box ),
                                               2 ) );
    apply_tag( m_tag_checkbox, iter_start, iter_box );
    apply_tag( m_tag_strikethrough, iter_box, iter_end );
}

void
TextbufferDiary::apply_match( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_search ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current+1 ) );
    apply_tag( m_tag_match, iter_start, iter_end );
}

//#define CALL_MEMBER_FN(object,ptrToMember)  ((object).*(ptrToMember))

inline void
TextbufferDiary::apply_markup( const Glib::RefPtr< Tag > &tag)
{
    Gtk::TextIter iter_start( get_iter_at_offset( m_pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
	Gtk::TextIter iter2( iter_start );
	iter2++;
	apply_tag( m_tag_markup, iter_start, iter2 );
	apply_tag( m_tag_hidden, iter_start, iter2 );
	apply_tag( tag, iter2, iter_current );
	iter2 = iter_current;
	iter2++;
	apply_tag( m_tag_markup, iter_current, iter2 );
	apply_tag( m_tag_hidden, iter_current, iter2 );
}

//EVENT HANDLERS
void
TextbufferDiary::on_insert( const Gtk::TextIter& iterator,
							const Glib::ustring& text,
							int bytes )
{
    const Glib::ustring::size_type offset_itr = iterator.get_offset();

    // UNDO
    if( UndoManager::m->is_freezed() )
    {
        Gtk::TextIter iter_scroll( iterator );	// to remove constness
        m_ptr2textview->scroll_to( iter_scroll );
    }
    else
    {
        UndoInsert * undo_insert = new UndoInsert( offset_itr, text );
        UndoManager::m->add_action( undo_insert );
    }

	Gtk::TextBuffer::on_insert( iterator, text, bytes );

	if( m_flag_ongoing_operation || m_flag_parsing )
		return;

	//  CALCULATING LIMITS & PARSING
    if( m_flag_settextoperation )
    {
        parse( begin(), end() );
    }
    else
    {
        Gtk::TextIter iter_start( get_iter_at_offset( offset_itr ) );
        Gtk::TextIter iter_end( get_iter_at_offset( offset_itr + text.length() ) );
        calculate_para_bounds( iter_start, iter_end );
        parse( iter_start, iter_end );
    }
}

void
TextbufferDiary::process_space( void )
{
    Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
    Gtk::TextIter iter_start( iter_end );
    if( ! iter_start.backward_find_char( s_predicate_nl ) )
        return;

    iter_start++;   // skip the new line char
	Glib::ustring line = get_text( iter_start, iter_end );
	char char_lf = '\t';
	unsigned int size = line.size();

	for( unsigned int i = 0; i < size; i++ )
	{
		switch( line[ i ] )
		{
			case '\t':
				if( char_lf == '\t' )
				    char_lf = 'A';	// any list char like [ or *
				else
				if( char_lf != 'A' )	// multiple tabs are possible (indentation)
					return;
				iter_start++;	// indentation level
				break;
			case '[':
				if( char_lf != 'A' )
					return;
				char_lf = ']';
				break;
			case ']':
				if( char_lf != ']' || i != ( size - 1 ) )
					return;
                {
                    PRINT_DEBUG( "time to insert a checkbox" );
                    int offset( iter_start.get_offset() );  // save the position before erase
                    m_flag_ongoing_operation++;
                    erase( iter_start, iter_end );
                    m_flag_ongoing_operation--;
                    iter_start = get_iter_at_offset( offset );	// refresh the iterator
                    insert( iter_start, "☐" );
                }
				break;
			case '*':
				if( char_lf != 'A' || i != ( size - 1 ) )
					return;
                {
                    int offset( iter_start.get_offset() );  // save the position before erasing
                    m_flag_ongoing_operation++;
                    erase( iter_start, iter_end );
                    m_flag_ongoing_operation--;
                    iter_start = get_iter_at_offset( offset );	// refresh the iterator
                    insert( iter_start, "•" );
                }
				break;
			default:
				return;
		}
	}
}

bool
TextbufferDiary::process_new_line( void )
{
    Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
    Gtk::TextIter iter_start( iter_end );
    if( ! iter_start.backward_find_char( s_predicate_nl ) ||
        iter_end.get_line_offset() < 3 )
        return false;

    iter_start++;	// get rid of the new line char
	const Glib::ustring::size_type offset_start( iter_start.get_offset() );   // save for future

    if( iter_start.get_char() == '\t' )
    {
        Glib::ustring text( "\n\t" );
        int value = 0;
        char char_lf = '*';
        iter_start++;   // first tab is already handled, so skip it

        for( ; iter_start != iter_end; ++iter_start )
        {
            switch( iter_start.get_char() )
            {
                // BULLETED LIST
                case L'•':
                    if( char_lf != '*' )
                        return false;
                    char_lf = ' ';
                    text += "• ";
                    break;
                // CHECK LIST
                case L'☐':
                case L'☑':
                case L'☒':
                    if( char_lf != '*' )
                        return false;
                    char_lf = ' ';
                    text += "☐ ";
                    break;
                // NUMBERED LIST
                case '0': case '1': case '2': case '3': case '4':
                case '5': case '6': case '7': case '8': case '9':
                    if( char_lf != '*' && char_lf != '1' )
                        return false;
                    char_lf = '1';
                    value *= 10;
                    value += iter_start.get_char() - '0';
                    break;
                case '-':
                case '.':
                case ')':
                    if( char_lf != '1' )
                        return false;
                    char_lf = ' ';
                    text += Glib::ustring::compose<int,char>(
                            "%1%2 ", ++value, iter_start.get_char() );
                    break;
                case '\t':
                    if( char_lf != '*' )
                        return false;
                    text += '\t';
                    break;
                case ' ':
                    if( char_lf != ' ' )
                        return false;
                    // remove the last bullet if no text follows it:
                    if( iter_start.get_offset() == iter_end.get_offset() - 1 )
                    {
                        iter_start = get_iter_at_offset( offset_start );
                        m_flag_ongoing_operation++;
                        erase( iter_start, iter_end );
                        m_flag_ongoing_operation--;
                        iter_start = get_iter_at_offset( offset_start );
                        insert( iter_start, "\n" );
                        return true;
                    }
                    else
                    {
                        insert( iter_end, text );
                        return true;
                    }
                    break;
                default:
                    return false;
            }
        }
	}
	return false;
}

void
TextbufferDiary::on_erase(	const Gtk::TextIter& iter_start,
							const Gtk::TextIter& iter_end )
{
    const Glib::ustring::size_type offset_itr = iter_start.get_offset();

    if( UndoManager::m->is_freezed() )
    {
        Gtk::TextIter iter_scroll( iter_start );	// to remove constness
        m_ptr2textview->scroll_to( iter_scroll );
    }
    else
    {
        UndoErase * undo_erase = new UndoErase( offset_itr, get_slice( iter_start, iter_end ) );
        UndoManager::m->add_action( undo_erase );
    }

    Gtk::TextBuffer::on_erase( iter_start, iter_end );

	// set_text() calls on_erase too:
    if( m_flag_settextoperation || m_flag_ongoing_operation || m_flag_parsing )
        return;

    parser_open_tag_begin = parser_open_tag_end = std::string::npos;

    //  CALCULATING LIMITS & PARSING
    Gtk::TextIter iter_start2( get_iter_at_offset( offset_itr ) );
    Gtk::TextIter iter_end2( iter_start2 );
    calculate_para_bounds( iter_start2, iter_end2 );
    parse( iter_start2, iter_end2 );
}

/*void
TextbufferDiary::on_apply_tag( const Glib::RefPtr< TextBuffer::Tag >& tag,
							   const Gtk::TextIter& iter_begin,
							   const Gtk::TextIter& iter_end )
{
	// do not check spelling of links:
    if( ( iter_begin.has_tag( m_tag_link ) || iter_begin.has_tag( m_tag_hidden ) ) &&
        tag == m_tag_misspelled )
		return;
	else
		Gtk::TextBuffer::on_apply_tag( tag, iter_begin, iter_end );
}*/

void
TextbufferDiary::on_mark_set( const TextBuffer::iterator &iter,
							  const Glib::RefPtr< TextBuffer::Mark > &mark )
{
	Gtk::TextBuffer::on_mark_set( iter, mark );

	if( mark == get_insert() )
	{
		PRINT_DEBUG( "on_mark_set()" );
		bool flag_within_open_tag( false );
		if( parser_open_tag_begin != std::string::npos )
		{
			if( iter.get_offset() >= int( parser_open_tag_begin ) &&
				iter.get_offset() <= int( parser_open_tag_end ) )
				flag_within_open_tag = true;
			else
			{
                Gtk::TextIter iter_begin( get_iter_at_offset( parser_open_tag_begin ) );
                Gtk::TextIter iter_end( get_iter_at_offset( parser_open_tag_end ) );

                if( parser_open_tag_is_link )
                {
                    PRINT_DEBUG( "OPEN TAG IS LINK" );
                    parse( iter_begin, iter_end );
                }
                else
                    apply_tag( m_tag_hidden, iter_begin, iter_end );

                parser_open_tag_begin = parser_open_tag_end = std::string::npos;
			}
		}
		if( ! flag_within_open_tag )	// not within an already open tag
		{
			PRINT_DEBUG( "NOT WITHIN OPEN TAG" );
			if( iter.has_tag( m_tag_link_hidden ) )
			{
				PRINT_DEBUG( "WITHIN HIDDEN LINK" );
                Gtk::TextIter iter_begin( iter );
                Gtk::TextIter iter_end( iter );
				if( ! iter.begins_tag( m_tag_link_hidden ) )
				{
					PRINT_DEBUG( "BEGINS TAG" );
					iter_begin.backward_to_tag_toggle( m_tag_link_hidden );
				}
				iter_end.forward_to_tag_toggle( m_tag_link_hidden );
				parser_open_tag_begin = iter_begin.get_offset();
				parser_open_tag_end = iter_end.get_offset();
				parser_open_tag_is_link = true;
				parse( iter_begin, iter_end );
			}
			// STARTING MARKUP
			else
			if( iter.begins_tag( m_tag_markup ) )
			{
				parser_open_tag_begin = iter.get_offset();
				parser_open_tag_end = parser_open_tag_begin + 1;
				parser_open_tag_is_link = false;
				Gtk::TextIter iter_begin( get_iter_at_offset( parser_open_tag_begin ) );
				Gtk::TextIter iter_end( get_iter_at_offset( parser_open_tag_end ) );
				PRINT_DEBUG( "BEGINS MARKUP |" + get_slice( iter_begin, iter_end ) + "|" );

				remove_tag( m_tag_hidden, iter_begin, iter_end );

				if( parser_offset_insert_prev != parser_open_tag_begin - 1 )
				{
					move_mark( get_selection_bound(), iter_end );
					move_mark( mark, iter_end );
				}
			}
			// FINISHING MARKUP
			else
			if( iter.ends_tag( m_tag_bold ) || iter.ends_tag( m_tag_italic ) ||
				iter.ends_tag( m_tag_highlight ) || iter.ends_tag( m_tag_strikethrough ) )
			{
				parser_open_tag_begin = iter.get_offset();
				parser_open_tag_end = parser_open_tag_begin + 1;
				parser_open_tag_is_link = false;
				Gtk::TextIter iter_begin( iter );
				Gtk::TextIter iter_end( get_iter_at_offset( parser_open_tag_end ) );
				PRINT_DEBUG( "ENDS MARKUP |" + get_slice( iter_begin, iter_end ) + "|" );

				remove_tag( m_tag_hidden, iter_begin, iter_end );

				if( parser_offset_insert_prev == parser_open_tag_begin + 2 )
				{
					move_mark( get_selection_bound(), iter_end );
					move_mark( mark, iter_end );
				}
			}
		}

		parser_offset_insert_prev = iter.get_offset();
	}
}

Link*
TextbufferDiary::get_link( int offset ) const
{
	for ( ListLinks::const_iterator iter = m_list_links.begin();
		  iter != m_list_links.end();
		  ++iter )
	{
		if(	offset >= ( *iter )->m_mark_start->get_iter().get_offset() &&
			offset <= ( *iter )->m_mark_end->get_iter().get_offset() )
			return( *iter );
	}

	return NULL;
}

Link*
TextbufferDiary::get_link( const Gtk::TextIter &iter ) const
{
	for( ListLinks::const_iterator link = m_list_links.begin();
		  link != m_list_links.end();
		  ++link )
	{
		if( iter.in_range( ( *link )->m_mark_start->get_iter(),
						   ( *link )->m_mark_end->get_iter() ) )
			return( *link );
	}
	return NULL;
}

void
TextbufferDiary::clear_links( int start, int end )
{
	ListLinks::iterator iter_tmp;
	for( ListLinks::iterator iter = m_list_links.begin(); iter != m_list_links.end(); )
		if(	start <= ( *iter )->m_mark_start->get_iter().get_offset() &&
			end >= ( *iter )->m_mark_end->get_iter().get_offset() )
		{
			iter_tmp = iter;
			++iter;
			delete( *iter_tmp );
			m_list_links.erase( iter_tmp );
		}
		else
			++iter;
}

void
TextbufferDiary::clear_links( void )
{
    for( ListLinks::iterator iter = m_list_links.begin();
         iter != m_list_links.end();
         ++iter )
    {
        delete( *iter );
    }
    m_list_links.clear();
}

void
TextbufferDiary::clear_images( int start, int &end )
{
    assert( m_flag_parsing );   // this function should only be called during parsing

    ListImages::iterator iter_tmp;
    for( ListImages::iterator iter = m_list_images.begin(); iter != m_list_images.end(); )
    {
        const int pos_img( ( *iter )->m_mark->get_iter().get_offset() );
        if( start <= pos_img && end >= pos_img )
        {
            InlineImage *img( *iter );

            // REMOVE IMAGE FROM THE BUFFER
            Gtk::TextIter titer_end( img->m_mark->get_iter() );
            erase( img->m_mark->get_iter(), ++titer_end );

            // REMOVE IMAGE FROM THE LIST
            iter_tmp = iter;
            ++iter;
            m_list_images.erase( img );
            delete( img );
            end--;
        }
        else
            ++iter;
    }
}

void
TextbufferDiary::clear_images( void ) // note that this version does not alter the buffer
{
    for( ListImages::iterator iter = m_list_images.begin();
         iter != m_list_images.end();
         ++iter )
    {
        delete( *iter );
    }
    m_list_images.clear();
}

void
TextbufferDiary::handle_menu( Gtk::Menu *menu )
{
    if( Diary::d->is_read_only() )
        return;

    // SEPARATOR
    Gtk::SeparatorMenuItem *separator( Gtk::manage( new Gtk::SeparatorMenuItem ) );
    separator->show();
    menu->prepend( *separator );

    if( !m_ptr2entry->get_lang_final().empty() )
    {
        // below code is created by using GtkSpell as a template
        // SUGGESTIONS
        Gtk::TextIter iter_begin = get_iter_at_offset( m_spell_suggest_offset );
        if( iter_begin.has_tag( m_tag_misspelled ) )
        {
            Gtk::TextIter iter_end( iter_begin );
            iter_begin.backward_to_tag_toggle( m_tag_misspelled );
            iter_end.forward_to_tag_toggle( m_tag_misspelled );
            add_suggestion_menus( get_text( iter_begin, iter_end ), menu );
        }
    }

    // LANGUAGE SELECTION SUBMENU
    Gtk::MenuItem *mi_langs( Gtk::manage( new Gtk::MenuItem( _("Spell Checking" ) ) ) );
    mi_langs->set_submenu( *build_languages_menu() );
    mi_langs->show_all();
    menu->prepend( *mi_langs );
}

// FORMATTING
void
TextbufferDiary::toggle_format(	Glib::RefPtr< Tag > tag, const Glib::ustring &markup )
{
    if( ! m_ptr2textview->has_focus() )
        return;

	Gtk::TextIter iter_start, iter_end;
	if( get_has_selection() )
	{
        int start( -2 ), end( -1 );
        bool properly_separated( false );

        get_selection_bounds( iter_start, iter_end );
        iter_end--;

        if( ! iter_start.is_start() )
        {
            --iter_start;   // also evaluate the previous character
        }
        else
        {
            properly_separated = true;
            start = -2;
        }


        for( ; ; ++iter_start )
        {
            if( iter_start.has_tag( tag ) )
                return;
            switch( iter_start.get_char() )
            {
                // do nothing if selection spreads over more than one line:
                case '\n':
                    if( start > -2 )
                        return;
                    /* else no break */
				case ' ':
				case '\t':
				    if( start == -2 )
				    {
				        properly_separated = true;
				        start = -1;
				    }
					break;
//				case '*':
//				case '_':
//				case '#':
//				case '=':
//					if( iter_start.get_char() == markup[ 0 ] )
//						break;
					/* else no break */
				default:
                    if( start == -2 )
                        start = -1;
                    else
					if( start == -1 )
						start = iter_start.get_offset();
					end = iter_start.get_offset();
					break;
			}
			if( iter_start == iter_end )
				break;
		}
		// add markup chars to the beginning and end:
		if( start >= 0 )
		{
            if( properly_separated )
            {
                insert( get_iter_at_offset( start ), markup );
                end += 2;
            }
            else
            {
                insert( get_iter_at_offset( start ), " " + markup );
                end += 3;
            }

			insert( get_iter_at_offset( end ), markup );
			place_cursor( get_iter_at_offset( end ) );
		}
	}
	else	// no selection case
	{
		Glib::RefPtr< Gtk::TextMark > mark = get_insert();
		iter_start = mark->get_iter();
        if( Glib::Unicode::isspace( iter_start.get_char() ) )
        {
            if( iter_start.starts_line() )
                return;
            iter_start--;
            if( iter_start.has_tag( TextbufferDiary::m_tag_markup ) )
                iter_start--;
        }
        else if( iter_start.has_tag( TextbufferDiary::m_tag_markup ) )
        {
            if( iter_start.starts_line() )
                return;
            iter_start--;
            if( Glib::Unicode::isspace( iter_start.get_char() ) )
            {
                // ugly!
                iter_start++;
                iter_start++;
            }
        }
		if( iter_start.has_tag( tag ) ) // if already has the tag, remove it
		{
			m_flag_ongoing_operation++;

			// necessary when cursor is between a space char and non-space char:
			if( iter_start.starts_word() )
				iter_start++;

			int offset_start( iter_start.get_line_offset() );

			iter_start.backward_to_tag_toggle( tag );
			backspace( iter_start );

			iter_end = get_iter_at_offset( offset_start );
			iter_end.forward_to_tag_toggle( TextbufferDiary::m_tag_markup );

			m_flag_ongoing_operation--;

			backspace( ++iter_end );
		}
		else
        if( iter_start.has_tag( m_tag_bold ) == false &&
            iter_start.has_tag( m_tag_italic ) == false &&
            iter_start.has_tag( m_tag_strikethrough ) == false &&
            iter_start.has_tag( m_tag_highlight ) == false &&
            iter_start.has_tag( m_tag_heading ) == false &&
            iter_start.has_tag( m_tag_link ) == false ) // formatting tags cannot be nested
		{
			// find word boundaries:
			if( !( iter_start.starts_word() || iter_start.starts_line() ) )
				iter_start.backward_word_start();
			insert( iter_start, markup );

			iter_end = mark->get_iter();
			if( !( iter_end.ends_word() || iter_end.ends_line() ) )
			{
				iter_end.forward_word_end();
				insert( iter_end, markup );
			}
			else
			{
				int offset = iter_end.get_offset();
				insert( iter_end, markup );
				place_cursor( get_iter_at_offset( offset ) );

			}
		}
	}
}

void
TextbufferDiary::toggle_bold( void )
{
	toggle_format( m_tag_bold, "*" );
}

void
TextbufferDiary::toggle_italic( void )
{
	toggle_format( m_tag_italic, "_" );
}

void
TextbufferDiary::toggle_strikethrough( void )
{
	toggle_format( m_tag_strikethrough, "=" );
}

void
TextbufferDiary::toggle_highlight( void )
{
	toggle_format( m_tag_highlight, "#" );
}

void
TextbufferDiary::set_comment_visibility( bool visible )
{
    m_tag_comment->property_invisible() = visible;
}

void
TextbufferDiary::calculate_para_bounds( Gtk::TextIter &iter_begin, Gtk::TextIter &iter_end )
{
    if( ! iter_begin.backward_find_char( s_predicate_nl ) )
        iter_begin.set_offset( 0 );

    if( iter_end.ends_line() )
        iter_end++;
    else if( iter_end.forward_find_char( s_predicate_nl ) )
        iter_end++;
    else
        iter_end.forward_to_end();
}

bool
TextbufferDiary::calculate_multiline_selection_bounds( Gtk::TextIter &iter_begin,
                                                       Gtk::TextIter &iter_end )
{
    bool flag_starts_first_para( false );

    if( get_has_selection() )
        get_selection_bounds( iter_begin, iter_end );
    else
        iter_begin = iter_end = get_iter_at_mark( get_insert() );

    if( ! iter_begin.backward_find_char( s_predicate_nl ) )
    {
        if( ! iter_begin.forward_find_char( s_predicate_nl ) )
            return false;
        else
            flag_starts_first_para = true;
    }
    else
        iter_begin++;

    if( iter_end.get_char() != '\n' )
    {
        if( ! iter_end.forward_find_char( s_predicate_nl ) )
        {
            if( flag_starts_first_para )
                return false; // ignore first line
            iter_end.forward_to_end();
        }
    }

    return( !( iter_begin == iter_end && flag_starts_first_para ) );
}

void
TextbufferDiary::handle_indent( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

    Gtk::TextIter iter, iter_end;
    calculate_multiline_selection_bounds( iter, iter_end );

    if ( iter == iter_end ) // empty line
    {
        insert( iter, "\t" );
        return;
    }

    Glib::RefPtr< Gtk::TextMark > mark_end = create_mark( iter_end );

    iter = insert( iter, "\t" );    // first line

    while( iter != get_iter_at_mark( mark_end ) )
    {
        if( iter.get_char() == '\n' )
        {
            ++iter;
            iter = insert( iter, "\t" );
        }
        else
            ++iter;
    }

    delete_mark( mark_end );
}

void
TextbufferDiary::handle_unindent( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

    Gtk::TextIter iter, iter_end;
    calculate_multiline_selection_bounds( iter, iter_end );

    if ( iter == iter_end ) // empty line
    {
        if( iter.get_char() == '\t' )
            erase( iter, ++iter_end );

        return;
    }

    Glib::RefPtr< Gtk::TextMark > mark_end = create_mark( iter_end );

    if( iter.get_char() == '\t' )
    {
        iter_end = iter;
        iter = erase( iter, ++iter_end ); // first line
    }

    while( iter != get_iter_at_mark( mark_end ) )
    {
        if( iter.get_char() == '\n' )
        {
            ++iter;
            if( iter.get_char() == '\t' )
            {
                iter_end = iter;
                iter = erase( iter, ++iter_end );    // first line
            }
        }
        else
            ++iter;
    }
}

void
TextbufferDiary::add_bullet( void )
{
    Gtk::TextIter iter, iter_end;
    if( ! calculate_multiline_selection_bounds( iter, iter_end ) )
        return;

    if ( iter == iter_end ) // empty line
    {
        insert( iter, "\t• " );
        return;
    }

    Glib::RefPtr< Gtk::TextMark > mark_end = create_mark( iter_end );
	Gtk::TextIter iter_erase_begin( iter );
	char char_lf( 't' );	// tab
	while( iter != get_iter_at_mark( mark_end ) )
	{
		switch( iter.get_char() )
		{
			case L'•':	// remove bullet
				if( char_lf == 'b' )
					char_lf = 's';	// space
				else
					char_lf = 'n'; // new line
				break;
			case ' ':
                if( char_lf == 's' || char_lf == 'S' )
                {
                    char_lf += ( 'g' - 's' ); // g or G depending on the previous value
                    iter = erase( iter_erase_begin, ++iter );
                    continue;
                }
                else
                    break;
			case '\n':
			    char_lf = 't';  // tab
			    break;
			case L'☐':
			case L'☑':
			case L'☒':
                if( char_lf == 'b' )
                    char_lf = 'S';  // capital s: space 2
                else
                    char_lf = 'n'; // new line
                break;
			case '\t':
                if( char_lf == 't'  || char_lf == 'b' )
                {
                    char_lf = 'b';  // bullet
                    iter_erase_begin = iter;
                }
                break;
			case 0:	// end
			default:
			    if( char_lf == 'G' || char_lf == 't' || char_lf == 'b' )
					iter = insert( iter, "\t• " );
			    char_lf = 'n';
				break;
		}
		++iter;
	}

	delete_mark( mark_end );
}

void
TextbufferDiary::add_checkbox( void )
{
    Gtk::TextIter iter, iter_end;
    if( ! calculate_multiline_selection_bounds( iter, iter_end ) )
        return;

    if ( iter == iter_end ) // empty line
    {
        insert( iter, "\t☐ " );
        return;
    }

    Glib::RefPtr< Gtk::TextMark > mark_end = create_mark( iter_end );
	Gtk::TextIter iter_erase_begin( iter );
	char char_lf( 't' );	// tab
	while( iter != get_iter_at_mark( mark_end ) )
	{
		switch( iter.get_char() )
		{
			case L'☐':
			case L'☑':
			case L'☒':	// remove checkbox
				if( char_lf == 'c' )
					char_lf = 's';	// space
                else
                    char_lf = 'n'; // new line
                break;
			case ' ':
                if( char_lf == 's' || char_lf == 'S' )
                {
                    char_lf += ( 'g' - 's' ); // g or G depending on the previous value
                    iter = erase( iter_erase_begin, ++iter );
                    continue;
                }
                else
                    break;
            case '\n':
                char_lf = 't';  // tab
                break;
			case L'•':
                if( char_lf == 'c' )
                    char_lf = 'S';  // capital s: space 2
                else
                    char_lf = 'n'; // new line
                                    break;
			case '\t':
			    if( char_lf == 't'  || char_lf == 'c' )
                {
                    char_lf = 'c';  // bullet
                    iter_erase_begin = iter;
                }
                break;
			case 0:	// end
			default:
				if( char_lf == 'G' || char_lf == 't' || char_lf == 'c' )
					iter = insert( iter, "\t☐ " );
                char_lf = 'n';
                break;
		}
		++iter;
	}

    delete_mark( mark_end );
}

void
TextbufferDiary::add_empty_line_above( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.backward_line() )
		iter.forward_line();
	insert( iter, "\n" );
}

void
TextbufferDiary::remove_empty_line_above( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.backward_line() )
		iter.forward_line();

	if( iter.get_line() < 1 )
		return;

	Gtk::TextIter iter_begin( --iter );
	iter_begin--;
	if( iter_begin.get_char() == '\n' )
		erase( iter_begin, iter );
}

void
TextbufferDiary::move_line_up( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

    Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
    if( iter.get_line() < 2 )
        return;

    if( iter.backward_line() )
    {
        int offset( iter.get_offset() );

        iter.forward_line();
        iter--;

        Gtk::TextIter iter_end( get_iter_at_mark( get_insert() ) );
        if( !iter_end.ends_line() )
            iter_end.forward_to_line_end();

        Glib::ustring text( get_text( iter, iter_end ) );

        // image will vanish by itself:
        if( get_iter_at_offset( iter.get_offset() + 1 ).has_tag( m_tag_image ) )
            iter_end--;

        erase( iter, iter_end );

        iter = get_iter_at_offset( offset - 1 );

        insert( iter, text );

        place_cursor( get_iter_at_offset( offset ) );
    }
}

void
TextbufferDiary::move_line_down( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

    Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );

    if( iter.backward_line() )
    {
        iter.forward_line();
        iter--;

        Gtk::TextIter iter_end( get_iter_at_mark( get_insert() ) );
        if( !iter_end.ends_line() )
            if( !iter_end.forward_to_line_end() )
                return;

        int offset( iter.get_offset() + 1 );

        Glib::ustring text( get_text( iter, iter_end ) );

        // image will vanish by itself:
        if( get_iter_at_offset( iter.get_offset() + 1 ).has_tag( m_tag_image )  )
            iter_end--;

        erase( iter, iter_end );

        iter = get_iter_at_offset( offset );
        if( !iter.ends_line() )
            iter.forward_to_line_end();
        offset = iter.get_offset();

        insert( iter, text );

        place_cursor( get_iter_at_offset( offset + 1 ) );
    }
}

void
TextbufferDiary::insert_link( DiaryElement *element )
{
	// TODO: implement a custom insert function that adds spaces where necessary
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.get_tags().size() > 0 )
		return;
	if( iter.get_offset() > 0 )
		iter--;
	char c( iter.get_char() );
	if( c != ' ' && c != '\n' && c != '\t' )
		insert( get_iter_at_mark( get_insert() ), " ");
	insert( get_iter_at_mark( get_insert() ),
				Glib::ustring::compose( "<deid:%1\t%2>", element->get_id(), element->get_name() ) );
}

void
TextbufferDiary::insert_time_stamp( void )
{
    if( ! m_ptr2textview->has_focus() )
        return;

	// TODO: implement a custom insert function that adds spaces where necessary
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	//if( iter.get_tags().size() > 0 )
		//return;
	if( iter.get_offset() > 0 )
		iter--;
	char c( iter.get_char() );
	if( c != ' ' && c != '\n' && c != '\t' )
		insert( get_iter_at_mark( get_insert() ), " ");

	time_t date( time( NULL ) );
	insert( get_iter_at_mark( get_insert() ), Date::format_string( date ) );
}

void
TextbufferDiary::set_richtext( Entry *entry )
{
    m_flag_settextoperation = true;
    clear_links();
    clear_images();
    parser_open_tag_begin = std::string::npos;
    parser_open_tag_end = std::string::npos;
    parser_offset_insert_prev = std::string::npos;
    UndoManager::m->clear();
    UndoManager::m->freeze();
    m_ptr2entry = entry;
    set_language( entry->get_lang_final() );
    set_theme( entry->get_theme_is_set() ?
            entry->get_theme() : Diary::d->get_default_theme() );
    Gtk::TextBuffer::set_text( entry->get_text() );
    place_cursor( begin() );
    UndoManager::m->thaw();
    m_flag_settextoperation = false;
}

void
TextbufferDiary::set_theme( const Theme *theme )
{
    m_ptr2textview->override_font( theme->font );
    m_ptr2textview->override_background_color( theme->color_base, Gtk::STATE_FLAG_NORMAL );
    m_ptr2textview->override_color( theme->color_text, Gtk::STATE_FLAG_NORMAL );

    m_ptr2textview->override_background_color( theme->color_heading, Gtk::STATE_FLAG_SELECTED );
    m_ptr2textview->override_color( theme->color_base, Gtk::STATE_FLAG_SELECTED );

    m_tag_heading->property_foreground_rgba() = theme->color_heading;
    m_tag_subheading->property_foreground_rgba() = theme->color_subheading;
    m_tag_highlight->property_background_rgba() = theme->color_highlight;
    m_tag_comment->property_background_rgba() = theme->color_base; // to disable highlighting

    Gdk::RGBA color_mid( midtone( theme->color_base, theme->color_text ) );

    m_tag_comment->property_foreground_rgba() = color_mid;
    m_tag_region->property_paragraph_background_rgba() = midtone(
            theme->color_base, theme->color_text, 0.9 );
    m_tag_match->property_background_rgba() = contrast(
            theme->color_base, m_theme_color_match );
    m_tag_markup->property_foreground_rgba() = color_mid;
    m_tag_markup_link->property_foreground_rgba() = color_mid;
    m_tag_link->property_foreground_rgba() = contrast(
            theme->color_base, m_theme_color_link );
    m_tag_link_broken->property_foreground_rgba() = contrast(
            theme->color_base, m_theme_color_linkbroken );
}

// SPELL CHECKING BY DIRECT UTILIZATION OF ENCHANT (code partly copied from GtkSpell library)
void
TextbufferDiary::handle_word( void )
{
    PRINT_DEBUG( "ALPHA_LAST=[" + alpha_last + "]" );

    const gchar *word = alpha_last.c_str();

    if( enchant_dict_check( m_enchant_dict, word, strlen( word ) ) != 0 )
    {
        Gtk::TextIter iter_start( get_iter_at_offset( pos_alpha ) );
        Gtk::TextIter iter_end( get_iter_at_offset( pos_current ) );
        apply_tag( m_tag_misspelled, iter_start, iter_end );
    }
}

static void
set_lang_from_dict_cb( const char * const lang_tag, const char * const provider_name,
                       const char * const provider_desc, const char * const provider_file,
                       void * user_data )
{
    std::string *language = ( std::string* ) user_data;
    ( *language ) = lang_tag;
}

std::string
TextbufferDiary::set_language( std::string lang )
{
    if ( m_enchant_dict )
        enchant_broker_free_dict( Lifeobase::s_enchant_broker, m_enchant_dict );

    if( lang.empty() )  // empty means checking is turned off
    {
        m_enchant_dict = NULL;
        m_flag_handle_word = false;
        return "";
    }

    m_enchant_dict = enchant_broker_request_dict( Lifeobase::s_enchant_broker, lang.c_str() );

    if( !m_enchant_dict )
    {
        print_error( "Enchant error for language: " + lang );
        m_flag_handle_word = false;
        return "";
    }

    m_flag_handle_word = true;
    enchant_dict_describe( m_enchant_dict, set_lang_from_dict_cb, &lang );

    return lang;
}

Gtk::Menu*
TextbufferDiary::build_languages_menu( void )
{
    Gtk::Menu *menu = Gtk::manage( new Gtk::Menu );
    Gtk::RadioButtonGroup group;

    Gtk::RadioMenuItem *mi_lang =
            Gtk::manage( new Gtk::RadioMenuItem( group, _( "Diary Default" ) ) );
    if( m_ptr2entry->get_lang() == LANG_INHERIT_DIARY )
        mi_lang->set_active( true );
    else
        mi_lang->signal_activate().connect(
                sigc::bind( sigc::mem_fun( *this, &TextbufferDiary::handle_language_changed ),
                            LANG_INHERIT_DIARY, mi_lang ) );
    menu->append( *mi_lang );

    mi_lang = Gtk::manage( new Gtk::RadioMenuItem( group, _( STRING::OFF ) ) );
    if( !m_ptr2entry->get_lang().empty() )
        mi_lang->signal_activate().connect(
                sigc::bind( sigc::mem_fun( *this, &TextbufferDiary::handle_language_changed ),
                            "", mi_lang ) );
    else
        mi_lang->set_active( true );
    menu->append( *mi_lang );

    for( LanguageList::iterator iter = Lifeobase::s_lang_list.begin();
         iter != Lifeobase::s_lang_list.end(); ++iter )
    {
        mi_lang = Gtk::manage( new Gtk::RadioMenuItem( group, *iter ) );
        if( m_ptr2entry->get_lang().compare( *iter ) == 0 )
            mi_lang->set_active( true );
        else
            mi_lang->signal_activate().connect(
                    sigc::bind( sigc::mem_fun( *this, &TextbufferDiary::handle_language_changed ),
                                *iter, mi_lang ) );
        menu->append( *mi_lang );
    }

    // SOME OTHER LANGUAGE THAT IS NOT SUPPORTED BY THE CURRENT SYSTEM
    if( !m_ptr2entry->get_lang().empty() && m_ptr2entry->get_lang() != LANG_INHERIT_DIARY &&
        Lifeobase::s_lang_list.find( m_ptr2entry->get_lang() ) == Lifeobase::s_lang_list.end() )
    {
        mi_lang = Gtk::manage( new Gtk::RadioMenuItem( group, m_ptr2entry->get_lang() ) );
        mi_lang->set_active( true );
        menu->append( *mi_lang );
    }

    return menu;
}

void
TextbufferDiary::add_suggestion_menus( const Glib::ustring &word, Gtk::Menu *topmenu )
{
    Gtk::MenuItem *mi;
    char **suggestions;
    size_t n_suggs;
    int menu_position( 0 );

//    if (!spell->speller)
//        return;

    suggestions = enchant_dict_suggest( m_enchant_dict, word.c_str(), word.size(), &n_suggs );

    if ( suggestions == NULL || !n_suggs )
    {
        Gtk::Label *label( Gtk::manage( new Gtk::Label ) );
        label->set_markup( _( "<i>(no suggestions)</i>" ) );

        mi = Gtk::manage( new Gtk::MenuItem );
        mi->add( *label );
        mi->show_all();
        topmenu->insert( *mi, menu_position++ );
    }
    else
    {
        Gtk::Menu *menu_more( NULL );

        for( size_t i = 0; i < n_suggs && i < 20; i++ )
        {
            mi = Gtk::manage( new Gtk::MenuItem( suggestions[ i ] ) );
            mi->show();
            mi->signal_activate().connect( sigc::bind(
                    sigc::mem_fun( this, &TextbufferDiary::replace_misspelled_word ),
                    mi->get_label() ) );

            if( i < 5 )
            {
                topmenu->insert( *mi, menu_position++ );
            }
            else
            {
                if( i == 5 )
                {
                    Gtk::MenuItem *mi_more = Gtk::manage( new Gtk::MenuItem( _("More...") ) );
                    mi_more->show();
                    topmenu->insert( *mi_more, menu_position++ );
                    menu_more = Gtk::manage( new Gtk::Menu );
                    mi_more->set_submenu( *menu_more );
                }
                menu_more->append( *mi );
            }
        }

        enchant_dict_free_string_list( m_enchant_dict, suggestions );
    }

    // ADD TO DICTIONARY
    mi = Gtk::manage( new Gtk::MenuItem(
            Glib::ustring::compose( _("Add \"%1\" to Dictionary" ), word ) ) );
    mi->show();
    topmenu->insert( *mi, menu_position++ );
    mi->signal_activate().connect( sigc::bind(
            sigc::mem_fun( this, &TextbufferDiary::add_word_to_dictionary ), word ) );

    // IGNORE ALL
    mi = Gtk::manage( new Gtk::MenuItem( _("Ignore All" ) ) );
    mi->show();
    topmenu->insert( *mi, menu_position++ );
    mi->signal_activate().connect( sigc::bind(
            sigc::mem_fun( this, &TextbufferDiary::ignore_misspelled_word ), word ) );
}

void
TextbufferDiary::handle_language_changed( const std::string& lang,
                                          const Gtk::RadioMenuItem *mi_lang )
{
    if( mi_lang->get_active() )
    {
        if( lang.empty() )
        {
            m_ptr2entry->set_lang( "" );
            m_flag_handle_word = false;
            remove_tag( m_tag_misspelled, begin(), end() );
        }
        else if( lang == LANG_INHERIT_DIARY )
        {
            m_ptr2entry->set_lang( LANG_INHERIT_DIARY );
            if( set_language( Diary::d->get_lang() ).empty() )
                remove_tag( m_tag_misspelled, begin(), end() );
            else
                parse( begin(), end() );
        }
        else
        {
            m_ptr2entry->set_lang( set_language( lang ) );
            parse( begin(), end() );
        }
    }
}

void
TextbufferDiary::ignore_misspelled_word( const Glib::ustring &word )
{
    enchant_dict_add_to_session( m_enchant_dict, word.c_str(), word.size() );
    parse( begin(), end() );   // check entire buffer again
}

void
TextbufferDiary::replace_misspelled_word( const Glib::ustring &new_word )
{
//    if (!spell->speller)
//        return;

    Gtk::TextIter iter_begin = get_iter_at_offset( m_spell_suggest_offset );

    if( !iter_begin.has_tag( m_tag_misspelled ) )
    {
        PRINT_DEBUG( "No misspelled word found at suggestion offset" );
        return;
    }

    Gtk::TextIter iter_end( iter_begin );
    iter_begin.backward_to_tag_toggle( m_tag_misspelled );
    iter_end.forward_to_tag_toggle( m_tag_misspelled );
    const Glib::ustring old_word( get_text( iter_begin, iter_end ) );

    PRINT_DEBUG( "Old word: \"" + old_word + "\"" );
    PRINT_DEBUG( "New word: \"" + new_word + "\"" );

    // TODO: combine in undo history
    m_flag_ongoing_operation++;
    int offset( iter_begin.get_offset() );
    erase( iter_begin, iter_end );
    iter_begin = get_iter_at_offset( offset );
    m_flag_ongoing_operation--;
    insert( iter_begin, new_word );

    // why?
    enchant_dict_store_replacement( m_enchant_dict,
                                    old_word.c_str(), old_word.size(),
                                    new_word.c_str(), new_word.size() );
}

void
TextbufferDiary::add_word_to_dictionary( const Glib::ustring &word )
{
    enchant_dict_add_to_pwl( m_enchant_dict, word.c_str(), word.size() );
    parse( begin(), end() );   // check entire buffer again
}

// TEXTVIEW ========================================================================================
// STATIC MEMBERS
TextviewDiary::TextviewDiary( BaseObjectType *cobject, const Glib::RefPtr<Gtk::Builder>& )
:	Gtk::TextView( cobject ),
	m_cursor_hand( Gdk::Cursor::create( Gdk::HAND2 ) ),
	m_cursor_xterm( Gdk::Cursor::create( Gdk::XTERM ) ),
	m_ptr2cursor_last( &m_cursor_xterm ), m_link_hovered( NULL )
{
    m_buffer = new TextbufferDiary;
	set_buffer( static_cast< Glib::RefPtr< TextbufferDiary > >( m_buffer ) );
	m_buffer->set_textview( this );
	set_wrap_mode( Gtk::WRAP_WORD );
	set_left_margin( 10 );

	signal_populate_popup().connect(
			sigc::mem_fun( *m_buffer, &TextbufferDiary::handle_menu ) );
}

inline void
TextviewDiary::update_link( void )
{
	Gtk::TextIter		iter;
	const Glib::RefPtr< Gdk::Cursor >
	                    *ptr2cursor = &m_cursor_xterm;
	int					pointer_x, pointer_y;
	int					trailing, buffer_x, buffer_y;
    Gdk::ModifierType	modifiers;

    Gtk::Widget::get_window()->get_pointer( pointer_x, pointer_y, modifiers );
	window_to_buffer_coords( Gtk::TEXT_WINDOW_WIDGET,
							 pointer_x, pointer_y,
							 buffer_x, buffer_y );
	get_iter_at_position( iter, trailing, buffer_x, buffer_y );

	// FIX ITER IF NEEDED:
	if( iter.has_tag( m_buffer->m_tag_hidden ) )
		iter.forward_to_tag_toggle( m_buffer->m_tag_hidden );
	Gtk::TextIter iter2( iter );
	iter2++;
	if( ( iter2.ends_tag( m_buffer->m_tag_link ) ||
		  iter2.ends_tag( m_buffer->m_tag_link_broken ) ) && trailing > 0 )
		iter++;

	m_link_hovered = m_buffer->get_link( iter );

	if( m_link_hovered != NULL )
	{
		if( !( modifiers & Gdk::CONTROL_MASK ) )
			ptr2cursor = &m_cursor_hand;
	}

	if( ptr2cursor != m_ptr2cursor_last )
	{
		m_ptr2cursor_last = ptr2cursor;
		get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( *ptr2cursor );
	}
}

void
TextviewDiary::set_richtext( Entry *entry )
{
    m_link_hovered = NULL;
    m_buffer->set_richtext( entry );
}

bool
TextviewDiary::on_motion_notify_event( GdkEventMotion *event )
{
	update_link();
	return Gtk::TextView::on_motion_notify_event( event );
}

bool
TextviewDiary::on_button_press_event( GdkEventButton *event )
{
    if( event->button == 1 )
    {
        if( m_link_hovered != NULL )
            if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK )
                return true;
    }
    else if( event->button == 3)
    {
        int x, y;
        Gtk::TextIter iter;

        window_to_buffer_coords( Gtk::TEXT_WINDOW_TEXT, event->x, event->y, x, y );
        get_iter_at_location( iter, x, y );
        m_buffer->m_spell_suggest_offset = iter.get_offset();
        //gtk_text_buffer_move_mark(spell->buffer, spell->mark_click, &iter);
    }

	return Gtk::TextView::on_button_press_event( event );
}

bool
TextviewDiary::on_button_release_event( GdkEventButton *event )
{
	if( m_link_hovered != NULL )
		if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK &&
		      event->button == 1 )
		{
			m_link_hovered->go();
			return true;
		}

	return Gtk::TextView::on_button_release_event( event );
}

bool
TextviewDiary::on_key_press_event( GdkEventKey *event )
{
    if( ! Diary::d->is_read_only() )
    {
        if( ( event->state & ( Gdk::CONTROL_MASK|Gdk::MOD1_MASK|Gdk::SHIFT_MASK ) ) == 0 )
        {
            switch( event->keyval )
            {
                case GDK_KEY_space:
                    m_buffer->process_space();
                    break;
                case GDK_KEY_Return:
                    if( m_buffer->process_new_line() )
                        return true;
                    break;
                case GDK_KEY_Control_L:
                case GDK_KEY_Control_R:
                    if( m_link_hovered )
                        update_link();
                    break;
            }
        }
    }

	return Gtk::TextView::on_key_press_event( event );
}

//void
//TextviewDiary::on_style_changed( const Glib::RefPtr< Gtk::Style > &style_prev )
//{
//	Gtk::TextView::on_style_changed( style_prev );
////	TextbufferDiary = get_pango_context()->get_font_description();
//	TextbufferDiary::m_theme_font = get_style()->get_font();
//}

bool
TextviewDiary::on_key_release_event( GdkEventKey *event )
{
	if( event->keyval == GDK_KEY_Control_L || event->keyval == GDK_KEY_Control_R )
		if( m_link_hovered )
			update_link();
		
	return Gtk::TextView::on_key_release_event( event );
}
