/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_ENTRY_HEADER
#define LIFEOGRAPH_ENTRY_HEADER


#include "diarydata.hpp"


namespace LIFEO
{

const static char LANG_INHERIT_DIARY[] = "d";


class Entry : virtual public DiaryElement
{
	public:
        static ElementShower< Entry > *shower;

	typedef sigc::signal< void, Entry* >
								Signal_void_Entry;

                                Entry( DiaryBase * const, const Date::date_t,
                                       const Glib::ustring&, bool = false );
								Entry( DiaryBase * const, const Date::date_t,
								       bool = false );
		virtual					~Entry() {}

		void					show( void );
		void					prepare_for_hiding( void );

		const Glib::ustring*	get_text() const { return &m_text; }
		Glib::ustring&			get_text() { return m_text; }
		void					set_text( const Glib::ustring &text )
		{ m_text = text; calculate_title( text ); }

		Date					get_date( void ) const { return m_date; }
        time_t                  get_date_created( void ) const { return m_date_created; }
        time_t                  get_date_changed( void ) const { return m_date_changed; }
		Glib::ustring			get_date_created_str( void ) const;
		Glib::ustring			get_date_changed_str( void ) const;
		void					set_date( Date date )
		{ m_date = date; }

		int						get_size( void ) const
		{ return m_text.size(); }
		Type					get_type( void ) const
		{ return ET_ENTRY; }
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon( void ) const;
        const Glib::RefPtr< Gdk::Pixbuf >&
                                get_icon32( void ) const;

		void					calculate_title( const Glib::ustring& );

		Glib::ustring			get_list_str( void ) const
		{ return Glib::ustring::compose( "%1 -  %2", m_date.format_string(),
				Glib::Markup::escape_text( m_name ) ); }

		bool					get_filtered_out( void );
		void					set_filtered_out( bool filteredout = true )
		{ m_flag_filteredout = filteredout; }

		bool					is_favored( void ) const { return m_option_favored; }
		void					set_favored( bool favored ) { m_option_favored = favored; }

        std::string             get_lang( void ) const { return m_option_lang; }
        std::string             get_lang_final( void ) const {
            return m_option_lang == LANG_INHERIT_DIARY ?
                    m_ptr2diary->get_lang() : m_option_lang; }
        void                    set_lang( const std::string &lang ) { m_option_lang = lang; }

        bool                    is_trashed( void ) const { return m_option_trashed; }
        void                    set_trashed( bool trashed ) { m_option_trashed = trashed; }

		void					set_tags( const Tagset& );
		const Tagset&			get_tags( void ) const;
		Tagset&					get_tags( void );
		bool					add_tag( Tag* );
		bool					remove_tag( Tag* );

		// NULL means theme is not set
		const Theme*			get_theme( void ) const
		{
            if( m_ptr2theme == NULL )   // FIXME: is this necessary?
                throw LIFEO::Error( "Illegal theme request" );
            return m_ptr2theme;
		}
		void					set_theme( const Theme *theme )
		{ m_ptr2theme = theme; }
		void					unset_theme( void )
		{ m_ptr2theme = NULL; }
		bool					get_theme_is_set( void ) const
		{ return( m_ptr2theme != NULL ); }

	protected:
		Date					m_date;
		time_t					m_date_created;
		time_t					m_date_changed;
		Glib::ustring			m_text;
		Tagset					m_tags;
		const Theme				*m_ptr2theme;

        bool                    m_option_favored;
        std::string             m_option_lang;  // empty means off
        bool                    m_option_trashed;
        bool                    m_flag_filteredout;

	friend class Diary;	// TODO: remove this friendship too??
	friend class DiaryView;	// TODO: remove this friendship too??
	friend class EntryView;
};

// ENTRY SET ===================================================================
class PoolEntries : public std::map< Date::date_t, Entry*, FuncCompareDates >
{
	public:
									PoolEntries( void )
		:	std::map< Date::date_t, Entry*, FuncCompareDates >( compare_dates ) {}
									~PoolEntries();

		void						clear( void );
};

} // end of namespace LIFEO

#endif

