/***********************************************************************************

    Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <sys/stat.h>
#include <utime.h>

#include "helpers.hpp"
#include "lifeobase.hpp"
#include "view_login.hpp"
#include "dialog_password.hpp"


using namespace LIFEO;
using namespace HELPERS;


ViewLogin::Colrec   *ViewLogin::colrec;
std::string         ViewLogin::m_path_cur;


ViewLogin::ViewLogin( void )
:   m_flag_info_is_visible( false ), m_sort_date_type( SDT_SAVE )
{
    Gtk::Button *button_new_diary, *button_browse_diary;
    Gtk::TreeView::Column *col_diary, *col_date;

    try
    {
        colrec = new Colrec;

        Lifeobase::builder->get_widget( "grid_login", m_grid_login );
        Lifeobase::builder->get_widget( "treeview_diaries", m_treeview_diaries );
        Lifeobase::builder->get_widget( "button_new_diary", button_new_diary );
        Lifeobase::builder->get_widget( "button_open_diary", button_browse_diary );
        Lifeobase::builder->get_widget( "button_login_edit", m_button_edit_diary );
        Lifeobase::builder->get_widget( "button_login_read", m_button_read_diary );
        Lifeobase::builder->get_widget( "button_login_remove_diary", m_button_remove_diary );

        col_diary = Gtk::manage( new Gtk::TreeView::Column( _( "Name" ), colrec->name ) );
        col_date = Gtk::manage( new Gtk::TreeView::Column( _( STRING::COLHEAD_BY_LAST_SAVE ),
                                                           colrec->date ) );

        m_infobar = Gtk::manage( new Gtk::InfoBar() );
        m_label_info = Gtk::manage( new Gtk::Label() );
        m_button_info = Gtk::manage( new Gtk::Button() );

        m_treestore_diaries = Gtk::ListStore::create( *colrec );
        m_treesel_diaries = m_treeview_diaries->get_selection();
    }
    catch( ... )
    {
        throw LIFEO::Error( "creation of login view failed" );
    }

    // INFOBAR
    m_label_info->set_line_wrap( true );
    Gtk::Container *infobar_container(
            dynamic_cast< Gtk::Container* >( m_infobar->get_content_area() ) );
    if( infobar_container )
        infobar_container->add( *m_label_info );
    m_grid_login->attach( *m_infobar, 0, 0, 1, 1 );
    m_infobar->add_action_widget( *m_button_info, 0 );
    m_infobar->show_all_children( true );

    // LIST OF DIARIES
    m_treeview_diaries->set_model( m_treestore_diaries );
    col_diary->set_expand( true );
    col_diary->set_clickable( true );
    col_date->set_clickable( true );
    m_treeview_diaries->append_column( *col_diary );
    m_treeview_diaries->append_column( *col_date );
    m_treestore_diaries->set_sort_column( 1, Gtk::SORT_DESCENDING );
    m_treeview_diaries->set_has_tooltip( true );
    m_treeview_diaries->drag_dest_set( Gtk::DEST_DEFAULT_ALL, Gdk::ACTION_COPY );
    m_treeview_diaries->drag_dest_add_uri_targets();

    Lifeobase::base->set_default( *m_button_edit_diary );

    m_button_edit_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::handle_button_edit ) );
    m_button_read_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::handle_button_read ) );

    m_button_remove_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::remove_selected_diary ) );

    m_treesel_diaries->signal_changed().connect(
            sigc::mem_fun( this, &ViewLogin::handle_diary_selection_changed ) );

    m_treeview_diaries->signal_row_activated().connect_notify(
            sigc::mem_fun( this, &ViewLogin::handle_diary_activated ) );
    m_treeview_diaries->signal_button_release_event().connect_notify(
            sigc::mem_fun( this, &ViewLogin::handle_button_release ) );
    m_treeview_diaries->signal_key_release_event().connect_notify(
            sigc::mem_fun( this, &ViewLogin::handle_key_release ) );
    m_treeview_diaries->signal_query_tooltip().connect(
            sigc::mem_fun( this, &ViewLogin::handle_diary_tooltip ) );
    m_treeview_diaries->signal_drag_data_received().connect(
            sigc::mem_fun( this, &ViewLogin::handle_dragged_file ) );

    button_new_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::create_new_diary ) );
    button_browse_diary->signal_clicked().connect(
            sigc::mem_fun( this, &ViewLogin::add_existing_diary ) );

    col_diary->signal_clicked().connect( sigc::mem_fun( this, &ViewLogin::sort_by_name ) );
    col_date->signal_clicked().connect( sigc::mem_fun( this, &ViewLogin::sort_by_date ) );

    m_infobar->signal_response().connect(
            sigc::mem_fun( this, &ViewLogin::handle_infobar_response ) );

    // DETECT STOCK DIARIES
    Gio::init();
    try
    {
        Glib::RefPtr< Gio::File > directory = Gio::File::create_for_path( DIARYDIR );
        if( directory )
        {
            Glib::RefPtr< Gio::FileEnumerator > enumerator = directory->enumerate_children();
            if( enumerator )
            {
                Glib::RefPtr< Gio::FileInfo > file_info = enumerator->next_file();
                while( file_info )
                {
                    if( file_info->get_file_type() != Gio::FILE_TYPE_DIRECTORY )
                    {
                        std::string path( DIARYDIR "/" );
                        path += file_info->get_name();
                        Lifeobase::stock_diaries.insert( path );
                    }
                    file_info = enumerator->next_file();
                }
            }
        }
    }
    catch( const Glib::Exception& ex )
    {
        LIFEO::Error( ex.what() );
    }
}

void
ViewLogin::handle_start( void )
{
    Lifeobase::base->remove();
    Lifeobase::base->add( *m_grid_login );
    Lifeobase::base->set_title( PROGRAM_NAME );
    if( Lifeobase::base->m_flag_open_directly )
        if( open_selected_diary( Lifeobase::base->m_flag_read_only ) == LIFEO::SUCCESS )
            return;
    populate_diaries();
}

void
ViewLogin::handle_logout( void )
{
    handle_start();
    if( Lifeobase::loginstatus == Lifeobase::LOGGED_TIME_OUT )
        show_info( Gtk::MESSAGE_INFO, _( STRING::ENTER_PASSWORD_TIMEOUT ) );
    else if( m_flag_info_is_visible )
        hide_infobar();
}

void
ViewLogin::populate_diaries( void )
{
    Gtk::TreeModel::Row row;

    struct stat fst;    // file date stat

    m_treestore_diaries->clear();
    for( ListPaths::reverse_iterator iter = Lifeobase::settings.recentfiles.rbegin();
         iter != Lifeobase::settings.recentfiles.rend();
         ++iter )
    {
        row = *( m_treestore_diaries->append() );
        row[ colrec->name ] = Glib::filename_display_basename( *iter );
        row[ colrec->path ] = *iter;

        if( stat( iter->c_str(), &fst ) == 0 )
            row[ colrec->date ] = Date::format_string(
                    time_t( m_sort_date_type == SDT_SAVE ? fst.st_mtime : fst.st_atime ) );
        else
            row[ colrec->date ] = "XXXXX";
    }

    // STOCK DIARIES
    for( ListPaths::iterator itr_diary = Lifeobase::stock_diaries.begin();
         itr_diary != Lifeobase::stock_diaries.end();
         ++itr_diary )
    {
        std::string path( *itr_diary );
        row = *( m_treestore_diaries->append() );
        row[ colrec->name ] = "[*] " + Glib::filename_display_basename( path );
        row[ colrec->path ] = path;

        stat( path.c_str(), &fst );
        row[ colrec->date ] = Date::format_string(
                time_t( m_sort_date_type == SDT_SAVE ? fst.st_mtime : fst.st_atime ) );
    }

    if( Lifeobase::settings.recentfiles.size() > 0 )
        m_treesel_diaries->select( m_treestore_diaries->get_iter( "0" ) );
}

void
ViewLogin::sort_by_date( void )
{
    int col_id;
    Gtk::SortType st;
    m_treestore_diaries->get_sort_column_id( col_id, st );
    if( col_id == 1 )
    {
        if( m_sort_date_type == SDT_SAVE )
        {
            m_sort_date_type = SDT_ACCESS;
            m_treeview_diaries->get_column( 1 )->set_title( _( STRING::COLHEAD_BY_LAST_ACCESS ) );
        }
        else
        {
            m_sort_date_type = SDT_SAVE;
            m_treeview_diaries->get_column( 1 )->set_title( _( STRING::COLHEAD_BY_LAST_SAVE ) );

        }
        populate_diaries();
    }
    else
        m_treestore_diaries->set_sort_column( 1, Gtk::SORT_DESCENDING );
}

void
ViewLogin::sort_by_name( void )
{
    m_treestore_diaries->set_sort_column( 0, Gtk::SORT_ASCENDING );
}

LIFEO::Result
ViewLogin::open_selected_diary( bool read_only )
{
    // BEWARE: clear the diary before returning any result but SUCCESS

    // SET PATH
    Result result( Diary::d->set_path( m_path_cur, false, read_only ) );
    switch( result )
    {
        case LIFEO::SUCCESS:
            break;
        case FILE_NOT_FOUND:
            show_info( Gtk::MESSAGE_INFO, _( STRING::DIARY_NOT_FOUND ) );
            break;
        case FILE_NOT_READABLE:
            show_info( Gtk::MESSAGE_INFO, _( STRING::DIARY_NOT_READABLE ) );
            break;
        case FILE_LOCKED:
            show_info( Gtk::MESSAGE_INFO, _( STRING::DIARY_LOCKED ) );
            break;
        default:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::FAILED_TO_OPEN_DIARY ) );
            break;
    }
    if( result != LIFEO::SUCCESS )
    {
        Diary::d->clear();
        return result;
    }

    // FORCE ACCESS TIME UPDATE
    // for performance reasons atime update policy on linux is usually once per day. see: relatime
    if( read_only )
    {
        struct stat fst;    // file date stat
        struct timeval tvs[2];
        stat( m_path_cur.c_str(), &fst );
        tvs[ 0 ].tv_sec = time( NULL );
        tvs[ 0 ].tv_usec = 0;
        tvs[ 1 ].tv_sec = fst.st_mtime;
        tvs[ 1 ].tv_usec = 0;
        utimes( m_path_cur.c_str(), tvs );
    }

    // READ HEADER
    switch( result = Diary::d->read_header() )
    {
        case LIFEO::SUCCESS:
            break;
        case INCOMPATIBLE_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::INCOMPATIBLE_DIARY ) );
            break;
        case CORRUPT_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::CORRUPT_DIARY ) );
            break;
        default:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::FAILED_TO_OPEN_DIARY ) );
            break;
    }
    if( result != LIFEO::SUCCESS )
    {
        Diary::d->clear();
        return result;
    }

    // HANDLE OLD DIARY
    if( Diary::d->is_old() && read_only == false )
    {
        Gtk::MessageDialog *messagedialog
                = new Gtk::MessageDialog( *Lifeobase::base,
                                          "",
                                          false,
                                          Gtk::MESSAGE_WARNING,
                                          Gtk::BUTTONS_CANCEL,
                                          true );
        messagedialog->set_message( _( "Are You Sure You Want to Upgrade The Diary?" ) );
        messagedialog->set_secondary_text( _( STRING::UPGRADE_DIARY_CONFIRM ) );
        messagedialog->add_button( _( "Upgrade The Diary" ), Gtk::RESPONSE_ACCEPT );

        int response( messagedialog->run() );

        delete messagedialog;

        if( response !=  Gtk::RESPONSE_ACCEPT )
        {
            Diary::d->clear();
            return LIFEO::ABORTED;
        }
    }

    // HANDLE ENCRYPTION
    if( Diary::d->is_encrypted() )
    {
        static DialogPasswordPrompt *dialog_password = NULL;
        if( dialog_password == NULL )
            Lifeobase::builder->get_widget_derived(
                    "dialog_import_password", dialog_password );
        dialog_password->set_transient_for( *Lifeobase::base );
        dialog_password->set_path( Diary::d->get_name() );

        if( RESPONSE_GO == ( m_password_attempt_no == 0 ?
                dialog_password->run() : dialog_password->run_again() ) )
        {
            Diary::d->set_passphrase( dialog_password->get_password() );
            dialog_password->hide();
        }
        else
        {
            dialog_password->hide();
            Diary::d->clear();
            return LIFEO::ABORTED;
        }
    }

    // FINALLY READ BODY
    switch( result = Diary::d->read_body() )
    {
        case LIFEO::SUCCESS:
            Lifeobase::base->login();
            break;
        case WRONG_PASSWORD:
            Diary::d->clear();
            sleep( ++m_password_attempt_no ); // wait longer at every failed attempt
            return open_selected_diary( read_only );
        case CORRUPT_FILE:
            show_info( Gtk::MESSAGE_ERROR, _( STRING::CORRUPT_DIARY ) );
            // no break
        default:
            Diary::d->clear();  // clear partially read content if any
            return result;
    }

    return LIFEO::SUCCESS;
}

void
ViewLogin::remove_selected_diary( void )
{
    m_path_removed = m_path_cur;
    Lifeobase::settings.recentfiles.erase( m_path_removed );
    populate_diaries();
    show_info( Gtk::MESSAGE_INFO,
               Glib::ustring::compose( _( "Removed diary: %1" ), m_path_removed ),
               _( "Undo" ), RESP_UNDO );
}

void
ViewLogin::create_new_diary( void )
{
    DialogSaveDiary *dialogsavediary = new DialogSaveDiary;
    dialogsavediary->set_current_folder( Glib::get_home_dir() );

    if( dialogsavediary->run() == Gtk::RESPONSE_ACCEPT )
    {
        Diary::d->init_new(
                dialogsavediary->get_filename_default() );
        delete dialogsavediary;
        Lifeobase::base->login();
    }
    else
        delete dialogsavediary;
}

void
ViewLogin::add_existing_diary( void )
{
    DialogOpenDiary *filechooserdialog = new DialogOpenDiary;

    if( m_treesel_diaries->count_selected_rows() > 0 )
        filechooserdialog->set_current_folder( Glib::path_get_dirname( m_path_cur ) );
    else
        filechooserdialog->set_current_folder( Glib::get_home_dir() );

    int result( filechooserdialog->run() );

    if( result != DialogOpenDiary::RESPONSE_CANCEL )
    {
        m_path_cur = filechooserdialog->get_filename();
        delete filechooserdialog;
        if( open_selected_diary( result == DialogOpenDiary::RESPONSE_READ ) != LIFEO::SUCCESS )
            m_treesel_diaries->unselect_all(); // mostly to clear m_path_cur
    }
    else
        delete filechooserdialog;
}

void
ViewLogin::show_info( Gtk::MessageType type, const Glib::ustring &text,
                      const Glib::ustring &button, InfoResponse response )
{
    m_label_info->set_text( text );
    m_button_info->set_label( button );
    m_resp_cur = response;
    m_infobar->set_message_type( type );
    m_infobar->show();
    m_flag_info_is_visible = true;
}

void
ViewLogin::handle_infobar_response( int )
{
    if( m_resp_cur == RESP_UNDO )
    {
        Lifeobase::settings.recentfiles.insert( m_path_removed );
        populate_diaries();
    }

    hide_infobar();
}

inline void
ViewLogin::hide_infobar( void )
{
    //m_label_info->set_text( "" );
    m_infobar->hide();
    m_flag_info_is_visible = false;
}

void
ViewLogin::handle_diary_selection_changed( void )
{
    m_password_attempt_no = 0;

    if( Lifeobase::m_internaloperation )
        return;

    bool flag_selection( m_treesel_diaries->count_selected_rows() > 0 );

    if( flag_selection )
        m_path_cur = ( *m_treesel_diaries->get_selected() )[ colrec->path ];
    else
        m_path_cur = "";

    bool flag_not_stock( Lifeobase::stock_diaries.find( m_path_cur ) ==
            Lifeobase::stock_diaries.end() );
    m_button_remove_diary->set_visible( flag_selection && flag_not_stock );
    m_button_edit_diary->set_visible( flag_selection && flag_not_stock );
    m_button_read_diary->set_visible( flag_selection );

    m_flag_diary_activated = false;
}

void
ViewLogin::handle_diary_activated( const Gtk::TreePath &path, Gtk::TreeView::Column *col )
{
    // not handled directly to prevent BUTTON RELEASE event to be sent to edit screen widgets
    m_flag_diary_activated = true;
}

void
ViewLogin::handle_button_release( GdkEventButton* )
{
    if( m_flag_diary_activated )
        open_activated_diary();
}

void
ViewLogin::handle_key_release( GdkEventKey *event )
{
    if( m_flag_diary_activated && event->keyval == GDK_KEY_Return )
        open_activated_diary();
}

void
ViewLogin::open_activated_diary( void )
{
    open_selected_diary( Lifeobase::stock_diaries.find( m_path_cur ) !=
            Lifeobase::stock_diaries.end() ); // read only if stock diary

    m_flag_diary_activated = false;
}

bool
ViewLogin::handle_diary_tooltip( int x, int y, bool, const Glib::RefPtr<Gtk::Tooltip> &tooltip )
{
    Gtk::TreeModel::Path path;
    Gtk::TreeView::Column *column;
    int cell_x, cell_y;
    int bx, by;
    m_treeview_diaries->convert_widget_to_bin_window_coords( x, y, bx, by );
    if( ! m_treeview_diaries->get_path_at_pos( bx, by, path, column, cell_x, cell_y ) )
        return false;
    Gtk::TreeIter iter( m_treeview_diaries->get_model()->get_iter( path ) );
    if( !iter )
        return false;
    Gtk::TreeRow row = *( iter );
    Glib::ustring tooltip_string( row[ colrec->path ] );
    tooltip->set_text( tooltip_string );
    m_treeview_diaries->set_tooltip_row( tooltip, path );
    return true;
}

void
ViewLogin::handle_dragged_file( const Glib::RefPtr<Gdk::DragContext>& context,
                                int, int,
                                const Gtk::SelectionData& seldata,
                                guint info, guint time )
{
    if( seldata.get_length() < 0 )
        return;

    Glib::ustring uri = seldata.get_data_as_string();
    std::string filename = uri.substr( 0, uri.find('\n') - 1 );
    filename = Glib::filename_from_uri( filename );

    if( Glib::file_test( filename, Glib::FILE_TEST_IS_DIR ) )
        return;

    if( Lifeobase::settings.recentfiles.find( filename ) != Lifeobase::settings.recentfiles.end() )
        return;

    if( Lifeobase::stock_diaries.find( filename ) != Lifeobase::stock_diaries.end() )
        return;

    Gtk::TreeRow row = *( m_treestore_diaries->append() );
    row[ colrec->name ] = "   +++   " + Glib::filename_display_basename( filename );
    row[ colrec->path ] = filename;
    m_treeview_diaries->get_selection()->select( row );
    m_treeview_diaries->scroll_to_row(  m_treestore_diaries->get_path( row ) );

    Lifeobase::settings.recentfiles.insert( filename );

    PRINT_DEBUG( Glib::ustring::compose( "dropped: %1", filename ) );

    context->drag_finish( true, false, time );
}
