/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "panel_diary.hpp"
#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;


PanelDiary *Lifeobase::panel_diary = NULL;

PanelDiary::PanelDiary( void )
{
    Gtk::Button *button_today;
	try
	{
        Lifeobase::builder->get_widget( "button_logout", m_button_logout );
        Lifeobase::builder->get_widget( "button_today", button_today );

        Menubutton *mb_tools;
        Menubutton::s_builder_name = "menubutton_tools";
        Lifeobase::builder->get_widget_derived( Menubutton::s_builder_name, mb_tools );
        Lifeobase::builder->get_widget( "menuitem_sort_by_date", m_menuitem_sort_by_date );
        Lifeobase::builder->get_widget( "menuitem_sort_by_size", m_menuitem_sort_by_size );
        Lifeobase::builder->get_widget( "menuitem_sort_by_change", m_menuitem_sort_by_change );

        Lifeobase::builder->get_widget( "togglebtn_filter_favorites",
                                        m_togglebtn_filter_favorites );
        Lifeobase::builder->get_widget( "togglebtn_filter_trash", m_togglebtn_filter_trash );
        Lifeobase::builder->get_widget( "button_remove_filters", m_button_remove_filters );

		Lifeobase::builder->get_widget( "calendar_main", m_calendar );
		Lifeobase::builder->get_widget_derived( "treeview_main", m_treeview_entries );
		Lifeobase::builder->get_widget_derived( "entry_filter", m_entry_filter );
		Lifeobase::builder->get_widget_derived( "entry_replace", m_entry_replace );
		Lifeobase::builder->get_widget( "button_match_prev", m_button_match_prev );
		Lifeobase::builder->get_widget( "button_match_next", m_button_match_next );
		Lifeobase::builder->get_widget( "button_replace", m_button_replace );
		Lifeobase::builder->get_widget( "button_replace_all", m_button_replace_all );
		Lifeobase::builder->get_widget( "hbox_toolbar_replace", m_toolbar_replace );

        Menubutton::s_builder_name = "menubutton_diary_add";
        Lifeobase::builder->get_widget_derived( Menubutton::s_builder_name, m_button_add );
        Lifeobase::builder->get_widget( "mnitem_add_entry", m_menuitem_add_entry );
        Lifeobase::builder->get_widget( "mnitem_add_chapter", m_menuitem_add_chapter );
        Lifeobase::builder->get_widget( "mnitem_add_topic", m_menuitem_add_topic );
	}
	catch( ... ) { }

	// FILTER & REPLACE
	m_entry_filter->set_placeholder_text( _( "Filter" ) );
	m_entry_replace->set_placeholder_text( _( "Replace" ) );

	// SIGNALS
    m_button_logout->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( Lifeobase::base, &Lifeobase::logout ), true ) );

    m_menuitem_sort_by_date->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_DATE ) );
    m_menuitem_sort_by_size->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_SIZE ) );
    m_menuitem_sort_by_change->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_CHANGE ) );

    m_togglebtn_filter_favorites->signal_toggled().connect(
            sigc::mem_fun( this, &PanelDiary::handle_filter_favorites_toggled ) );

    m_togglebtn_filter_trash->signal_toggled().connect(
            sigc::mem_fun( this, &PanelDiary::handle_filter_trash_toggled ) );

    m_button_remove_filters->signal_clicked().connect(
            sigc::mem_fun( this, &PanelDiary::remove_filters ) );

	m_calendar->signal_day_selected_double_click().connect(
			sigc::mem_fun( this, &PanelDiary::handle_add_entry ) );
	m_calendar->signal_month_changed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_monthchanged ) );
	m_calendar->signal_day_selected().connect(
			sigc::mem_fun( this, &PanelDiary::handle_calendar_dayselected ) );

	m_entry_filter->signal_changed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_entry_filter_changed ) );
	m_entry_filter->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::go_next_match ) );

	m_entry_replace->signal_activate().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_button_replace->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_match ) );
	m_button_replace_all->signal_clicked().connect(
			sigc::mem_fun( this, &PanelDiary::replace_all_matches ) );

	m_treeview_entries->signal_row_expanded().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );
	m_treeview_entries->signal_row_collapsed().connect(
			sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );

    m_menuitem_add_entry->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_entry ) );
    m_menuitem_add_topic->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_topic ) );
    m_menuitem_add_chapter->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_chapter ) );

    // ACTIONS
    Lifeobase::create_action(
            true, m_action_today, "Today", "x-office-calendar", _( "Today" ),
            _( "Go to today (creates a new entry if there isn't any or when already at today)" ),
            Gtk::AccelKey( GDK_KEY_d, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::show_today ),
            button_today );

    Lifeobase::create_action(
            false, m_action_focusfilter, "FocusFilter",
            Gtk::Stock::FIND, "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_f, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_entry_filter, &Gtk::Widget::grab_focus ) );

    Lifeobase::create_action(
            false, m_action_previous_list, "ListPrevious", Gtk::Stock::GO_UP, _( "Previous" ),
            _( "Go to the previous entry in the list" ),
            Gtk::AccelKey( GDK_KEY_Page_Up, Gdk::CONTROL_MASK ),
            sigc::bind( sigc::mem_fun( m_treeview_entries, &WidgetEntryList::go_up ), false ) );

    Lifeobase::create_action(
            false, m_action_next_list, "ListNext", Gtk::Stock::GO_DOWN, _( "Next" ),
            _( "Go to the next entry in the list" ),
            Gtk::AccelKey( GDK_KEY_Page_Down, Gdk::CONTROL_MASK ),
            sigc::bind( sigc::mem_fun( m_treeview_entries, &WidgetEntryList::go_down ), false ) );

    Lifeobase::create_action(
            false, m_action_previous_match, "MatchPrevious", Gtk::Stock::GO_BACK,
// TRANSLATORS: label and tooltip of the go-to-previous-match-of-filtered-text action
            _( "Previous Match" ),
            _( "Go to previous match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::SHIFT_MASK ),
            sigc::mem_fun( this, &PanelDiary::go_prev_match ),
            m_button_match_prev );

    Lifeobase::create_action(
            false, m_action_next_match, "MatchNext", Gtk::Stock::GO_FORWARD,
// TRANSLATORS: label and tooltip of the go-to-next-match-of-filtered-text action
            _( "Next Match" ),
            _( "Go to next match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::ModifierType( 0 ) ),
            sigc::mem_fun( this, &PanelDiary::go_next_match ),
            m_button_match_next );

    Lifeobase::create_action(
            false, m_action_jump2current, "Jump", Gtk::Stock::JUMP_TO,
            "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_j, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::present_current_row ) );

    Lifeobase::create_action(
            false, m_action_show_prev_session_elem, "PrevSessElem", Gtk::Stock::JUMP_TO,
            "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_l, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::show_prev_session_elem ) );

    m_button_logout->add_accelerator(
            "clicked",
            Lifeobase::base->get_accel_group(),
            GDK_KEY_Escape,
            Gdk::CONTROL_MASK,
            Gtk::ACCEL_VISIBLE );

    // becoming proxy widgets makes them visible:
    m_button_match_prev->set_visible( false );
    m_button_match_next->set_visible( false );
}

void
PanelDiary::add_chapter_category_to_list( const CategoryChapters *chapters )
{
	Gtk::TreeRow	row_container;
	Chapter			*chapter = NULL;

	for( CategoryChapters::const_iterator iter_chapter = chapters->begin();
		 iter_chapter != chapters->end();
		 ++iter_chapter )
	{
		chapter = iter_chapter->second;
		if( ! chapter->is_initialized() )
			continue;

		// show all folders even when they are empty:
		row_container = * m_treeview_entries->m_treestore_entries->append( l_row_diary.children() );
		row_container[ ListData::colrec->ptr ] = chapter;	// ptr must be first:
		row_container[ ListData::colrec->info ] = chapter->get_list_str();
		row_container[ ListData::colrec->icon ] = chapter->get_icon();
		chapter->m_list_data->treepath = m_treeview_entries->m_treestore_entries->get_path( row_container );

		chapter->reset_size();

		if( l_entry == NULL )
			continue;

		while( l_entry->get_date() >= chapter->get_date() )
		{
			chapter->increase_size();
			if( l_entry->get_filtered_out() == false )
			{
				Lifeobase::m_entrycount++;
				l_row_entry = * m_treeview_entries->m_treestore_entries->append( row_container.children() );
				l_row_entry[ ListData::colrec->ptr ] = l_entry;
				l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
				l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();

				l_entry->m_list_data->treepath = m_treeview_entries->m_treestore_entries->get_path( l_row_entry );
			}

			++l_itr_entry;
			if( l_itr_entry != Diary::d->m_entries.end() )
				l_entry = l_itr_entry->second;
			else
			{
				l_entry = NULL;
				break;
			}
		}

		if( chapter->get_expanded() )
			m_treeview_entries->expand_element( chapter );
	}
}

// obviously not needed anymore:
//void
//PanelDiary::update_toolbar( void )
//{
//    bool flag_sensitive =
//            Lifeobase::panel_main->get_cur_elem_type() ==
//                    //DiaryElement::ET_ENTRY ? m_entrycount > 1 :
//                    ( m_treeview_entries->m_treestore_entries->children().size() > 1 );
//
//    m_action_previous_list->set_sensitive( flag_sensitive );
//    m_action_next_list->set_sensitive( flag_sensitive );
//}

void
PanelDiary::update_entry_list( void )
{
	PRINT_DEBUG( "update_entry_list()" );
	Lifeobase::m_internaloperation++;

	Lifeobase::view_entry->sync();

	l_itr_entry = Diary::d->m_entries.begin();
	l_entry = ( l_itr_entry != Diary::d->m_entries.end() ? l_itr_entry->second : NULL );

	m_treeview_entries->m_treestore_entries->clear();
	Lifeobase::m_entrycount = 0;

	// DIARY ITEM
	l_row_diary = * m_treeview_entries->m_treestore_entries->append();
	l_row_diary[ ListData::colrec->ptr ] = Diary::d; // ptr must be first
	l_row_diary[ ListData::colrec->info ] =
			Glib::ustring::compose( "<b>%1</b>", Diary::d->get_name() );
	l_row_diary[ ListData::colrec->icon ] = Lifeobase::icons->diary_16;

	Diary::d->m_list_data->treepath =
			m_treeview_entries->m_treestore_entries->get_path( l_row_diary );

	// CHAPTERS
	//CategoryChapters *chapters = Diary::d->get_current_chapter_ctg();

    if( Diary::d->get_sorting_criteria() == SC_DATE &&
        !( Diary::d->get_filtering_status() & Diary::FS_FILTER_TRASH ) )
	{
		add_chapter_category_to_list( Diary::d->get_topics() );
		add_chapter_category_to_list( Diary::d->get_current_chapter_ctg() );
	}

    for( ; l_itr_entry != Diary::d->m_entries.end(); ++l_itr_entry )
	{
		l_entry = ( *l_itr_entry ).second;
		if( l_entry->get_filtered_out() == false )
		{
			Lifeobase::m_entrycount++;

			l_row_entry = *m_treeview_entries->m_treestore_entries->append(
					l_row_diary.children() );
			l_row_entry[ ListData::colrec->ptr ] = l_entry; // ptr must be first
			l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
			l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();

			l_entry->m_list_data->treepath =
					m_treeview_entries->m_treestore_entries->get_path( l_row_entry );
		}
	}
	// always expand diary
	m_treeview_entries->expand_element( Diary::d );

	// UPDATE DIARY ELEM IF LAST ONE IS NOT IN THE LIST ANYMORE
	DiaryElement *elem = Lifeobase::panel_main->get_cur_elem();
	bool flag_show_elem = true;
	if( Lifeobase::m_entrycount <= 0 )
		elem = Diary::d;
	else
	if( elem != NULL )
	{
		if( elem->get_filtered_out() )
			elem = Diary::d->get_entry_first();
		else
			flag_show_elem = false;
	}
	else	// the case at the start-up
	{
		elem = Diary::d->get_startup_elem();
		if( elem == NULL )	// if last item was empty and dismissed at logout
			elem = Diary::d;
        else if( elem->get_filtered_out() ) // if last item is filtered out (trashed)
            elem = Diary::d->get_entry_first();
	}

	if( flag_show_elem )
		elem->show();
	/*else
	if( elem->get_type() >= DiaryElement::IT_DIARY )	// only entry list elements
		m_treeview_entries->present_row( elem->m_list_data->treepath );*/

	Diary::d->set_filtering_status_applied();

	Lifeobase::m_internaloperation--;
}

void
PanelDiary::update_entry_list_style( void ) // TODO: is this function really needed?
{
    Lifeobase::m_internaloperation++;

    update_entry_list();

    if( Lifeobase::panel_main->get_cur_elem_type() != DiaryElement::ET_ENTRY )
        Diary::d->show();

    m_treeview_entries->columns_autosize();

    Lifeobase::m_internaloperation--;
}

void
PanelDiary::update_calendar( void )
{
	guint year, month, day;

	m_calendar->get_date( year, month, day );
	month++;	// fix month
	Entry *entry;
	m_calendar->clear_marks();

	for( Entryiter itr_entry = Diary::d->get_entries().begin();
		 itr_entry != Diary::d->get_entries().end();
		 ++itr_entry )
	{
		entry = itr_entry->second;
		if( entry->get_filtered_out() || entry->get_date().is_ordinal() )
			continue;

		if( entry->get_date().get_year() == year )
		{
			if( entry->get_date().get_month() == month )
			{
				m_calendar->mark_day( entry->get_date().get_day() );
			}
			else
			if( entry->get_date().get_month() < month )
				break;
		}
		else
		if( entry->get_date().get_year() < year )
			break;
	}
}

void
PanelDiary::update_button_remove_filters( void )
{
    m_button_remove_filters->set_visible(
            Diary::d->get_filtering_status() &
            ( Diary::FS_FILTER_DATE | Diary::FS_FILTER_TAG ) );
}

void
PanelDiary::handle_login( void )
{
    switch( Diary::d->get_sorting_criteria() )
    {
        default: // future-proofing
        case SC_DATE: // no break
            m_menuitem_sort_by_date->set_active();
            break;
        case SC_SIZE:
            m_menuitem_sort_by_size->set_active();
            break;
        case SC_CHANGE:
            m_menuitem_sort_by_change->set_active();
            break;
    }

	// not called automatically because of m_internaloperation:
	m_treeview_entries->set_sorting_criteria();
	update_entry_list();

    m_button_remove_filters->set_visible( false );

	m_button_add->set_visible( ! Diary::d->is_read_only() );
}

void
PanelDiary::handle_logout( void )
{
    Lifeobase::m_internaloperation++;

	m_entry_filter->set_text( "" );
	m_entry_replace->set_text( "" );
	m_togglebtn_filter_favorites->set_active( false );
    m_togglebtn_filter_trash->set_active( false );
	m_treeview_entries->clear();

    m_button_match_prev->hide();
    m_button_match_next->hide();
    m_toolbar_replace->hide();

    Lifeobase::m_internaloperation--;
}

void
PanelDiary::show( const DiaryElement *element )
{
	if( element->get_type() >= DiaryElement::ET_DIARY ) // if a list element
		m_treeview_entries->present_element( element );
	else
		m_treeview_entries->get_selection()->unselect_all();

	if( element->get_type() > DiaryElement::ET_DIARY && ! element->get_date().is_ordinal() )
    {
        select_date_in_calendar( element->get_date() );
        m_menuitem_add_entry->set_sensitive( true );
        m_menuitem_add_chapter->set_sensitive(
                Diary::d->get_current_chapter_ctg()->get_chapter( element->get_date().m_date )
                == NULL );
    }
	else
	{
        m_calendar->select_day( 0 );
        m_menuitem_add_entry->set_sensitive( false );
        m_menuitem_add_chapter->set_sensitive( false );
	}
}

void
PanelDiary::show_today( void )
{
    Entry *entry_today = Diary::d->get_entry_today();
    bool flag_add = false;

    if( entry_today == NULL )   // add new entry if no entry exists on selected date
    {
        if( Diary::d->is_read_only() )
            return;
        else
            flag_add = true;
    }
    else                        // or current entry is already at today
    if( entry_today->get_date().get_pure() ==
                Lifeobase::panel_main->get_cur_elem()->get_date().get_pure() )
        flag_add = true;

    if( flag_add )
    {
        entry_today = Diary::d->add_today();
        update_entry_list();
    }

    entry_today->show();
}

void
PanelDiary::show_prev_session_elem( void )
{
	DiaryElement *elem( Diary::d->get_prev_session_elem() );
	if( elem != NULL )
		elem->show();
	else
		print_info( "Previous sesion element cannot be found!" );
}

void
PanelDiary::present_current_row( void )
{
	Gtk::TreeIter iter = m_treeview_entries->get_selection()->get_selected();
	if( iter == m_treeview_entries->m_treemodelsort_entries->children().end() )
		return;
	Gtk::TreePath path = m_treeview_entries->m_treemodelsort_entries->get_path( iter );

	m_treeview_entries->scroll_to_row( path );
}

void
PanelDiary::handle_entry_filter_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;
	// current entry must be closed here or else it loses...
	// ...changes since it was selected:
	//Lifeobase::view_entry->sync(); update_entry_list() does the syncing

	const std::string filterstring( m_entry_filter->get_text().lowercase() );

	Diary::d->set_filter_text( filterstring );
	update_entry_list();    // syncs current entry

	if( filterstring.size() > 0 )
	{
		if( Lifeobase::m_entrycount > 0 )
		{
            Lifeobase::view_entry->get_buffer()->set_search_str( filterstring );
            if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
                Lifeobase::view_entry->get_buffer()->select_searchstr_next();
		}

		m_entry_filter->set_tooltip_markup(
				Glib::ustring::compose(
						_( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
						Lifeobase::m_entrycount, Diary::d->get_size() ) );
	}
	else
	{
	    Lifeobase::view_entry->get_buffer()->set_search_str( "" );
		m_entry_filter->set_tooltip_text( _( "Enter text to be filtered" ) );
	}

	update_calendar();

	if( Lifeobase::m_entrycount < 1 || filterstring.size() < 1 )
	{
		m_button_match_prev->hide();
		m_button_match_next->hide();
		m_toolbar_replace->hide();
	}
	else
	{
		m_button_match_prev->show();
		m_button_match_next->show();
        if( ! Diary::d->is_read_only() )
            m_toolbar_replace->show();
	}
}

void
PanelDiary::handle_filter_favorites_toggled( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Diary::d->toggle_filter_favorites();	// set filter
	update_entry_list();					// apply the filter
}

void
PanelDiary::handle_filter_trash_toggled( void )
{
    if( Lifeobase::m_internaloperation ) return;

    Diary::d->toggle_filter_trash();    // set filter
    update_entry_list();                // apply the filter
}

void
PanelDiary::remove_filters( void )
{
    if( Diary::d->get_filtering_status() & ( Diary::FS_FILTER_DATE | Diary::FS_FILTER_TAG) )
    {
        const Tag *tag( Diary::d->get_filter_tag() );
        if( tag != NULL )
        {
            Diary::d->set_filter_tag( NULL );
            Lifeobase::view_tag->update_button_filter_tag();
            // UPDATE ICON
            Gtk::TreeRow row = * Lifeobase::base->get_element_row( tag );
            row[ PanelExtra::colrec->icon ] = tag->get_icon();
        }
        m_button_remove_filters->set_visible( false );

        Diary::d->clear_filter_date();

        update_entry_list();
    }
}

void
PanelDiary::handle_calendar_dayselected( void )
{
	if( Lifeobase::m_internaloperation ) return;

	guint year, month, day;
	m_calendar->get_date( year, month, day );
	Date date( year, month + 1, day );

    m_menuitem_add_entry->set_sensitive( day > 0 );
    m_menuitem_add_chapter->set_sensitive( day > 0 &&
            Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) == NULL );

	Entry *entry;
	if( date.m_date == ( Lifeobase::panel_main->get_cur_elem()->get_date().get_pure() ) )
	{
        entry = Diary::d->get_entry_next_in_day(
                Lifeobase::panel_main->get_cur_elem()->get_date() );
	}
	else
	{
        entry = Diary::d->get_entry( date.m_date + 1 ); // 1 increment to fix order
	}

	if( entry )
		entry->show();
}

void
PanelDiary::handle_calendar_monthchanged( void )
{
    if( Lifeobase::m_internaloperation ) return;

	update_calendar();
}

void
PanelDiary::handle_add_entry( void )
{
    if( Diary::d->is_read_only() )
        return;

    guint year, month, day;
    m_calendar->get_date( year, month, day );

    if( day )
    {
        Entry *entry( Diary::d->create_entry( Date::make_date( year, month + 1, day ) ) );

        update_entry_list();
        update_calendar();

        entry->show();
    }

    // this is the only way i could find to give focus to textview
    //FIXME
//  Glib::signal_timeout().connect_once(
//          sigc::mem_fun( m_textviewdiary, &Gtk::Widget::grab_focus ), 200 );
}

void
PanelDiary::handle_add_topic( void )
{
    Chapter *c( Diary::d->add_topic( _( "New topic" ) ) );
    update_entry_list();
    c->show();
    Lifeobase::panel_main->start_title_edit();
}

void
PanelDiary::handle_add_chapter( void )
{
    guint year, month, day;
    m_calendar->get_date( year, month, day );
    Date date( year, month + 1, day );

    if( day > 0 && Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) == NULL )
    {
        Chapter *c( Diary::d->get_current_chapter_ctg()->create_chapter(
                _( "Untitled chapter" ), date ) );

        update_entry_list();
        c->show();
        Lifeobase::panel_main->start_title_edit();
    }
}

void
PanelDiary::select_date_in_calendar( const Date &date )
{
	++Lifeobase::m_internaloperation;

	// CALENDAR
	// the current day has to be deselected first to avoid gtk's complaints
	m_calendar->select_day( 0 );
	if( ! date.is_ordinal() )
	{
		m_calendar->select_month( date.get_month() - 1, date.get_year() );
		update_calendar();
		if( date.get_day() )
			m_calendar->select_day( date.get_day() );
	}

	--Lifeobase::m_internaloperation;
}

void
PanelDiary::go_prev_match( void )
{
    if( Lifeobase::m_entrycount < 1 )
        return;

    // TODO: move to Entry
    if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
        if( Lifeobase::view_entry->get_buffer()->select_searchstr_previous() )
            return;

    int i = 0;
    do
    {
        m_treeview_entries->go_up( true );
        // put cursor to the end:
        Lifeobase::view_entry->get_buffer()->select_range(
                Lifeobase::view_entry->get_buffer()->end(),
                Lifeobase::view_entry->get_buffer()->end() );
        i++;
    }
    while( ! Lifeobase::view_entry->get_buffer()->select_searchstr_previous() &&
           i < Lifeobase::m_entrycount );
}

void
PanelDiary::go_next_match( void )
{
	if( Lifeobase::m_entrycount < 1 )
		return;

	// TODO: move to Entry
	if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
		if( Lifeobase::view_entry->get_buffer()->select_searchstr_next() )
			return;

	int i = 0;
	do
	{
        m_treeview_entries->go_down( true );
		i++;
	}
	while( ! Lifeobase::view_entry->get_buffer()->select_searchstr_next() &&
		   i < Lifeobase::m_entrycount );
}

void
PanelDiary::replace_match( void )
{
	if( ! Lifeobase::view_entry->get_buffer()->get_has_selection() )
		go_next_match();
	if( Lifeobase::view_entry->get_buffer()->get_has_selection() )
	{
		Lifeobase::view_entry->get_buffer()->erase_selection();
		Lifeobase::view_entry->get_buffer()->insert_at_cursor( m_entry_replace->get_text() );
		go_next_match();
	}
}

void
PanelDiary::replace_all_matches( void )
{
	Lifeobase::view_entry->sync();

	Diary::d->replace_text( m_entry_replace->get_text() );
	// update current entry:
	if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
	{
		Entry *entry = dynamic_cast< Entry* >( Lifeobase::panel_main->get_cur_elem() );
		Lifeobase::view_entry->get_buffer()->set_richtext( entry );
	}
}

void
PanelDiary::handle_entry_title_changed( DiaryElement *element )
{
    if( Lifeobase::m_internaloperation ) return;

    Gtk::TreeRow row = * get_row( element->m_list_data->treepath );
    row[ ListData::colrec->info ] = element->get_list_str();
}

Gtk::TreeRow
PanelDiary::get_row( const Gtk::TreePath &path )
{
	return( * m_treeview_entries->m_treestore_entries->get_iter( path ) );
	// TODO: check validity
}

void
PanelDiary::handle_treeview_row_expanded( const Gtk::TreeIter &iter,
										  const Gtk::TreePath &path )
{
	//if( Lifeobase::m_internaloperation )
		//return;

	DiaryElement *element = ( * iter )[ ListData::colrec->ptr ];
	if( element->get_type() == DiaryElement::ET_CHAPTER )
	{
		Chapter *chapter = dynamic_cast< Chapter* >( element );
		chapter->set_expanded( m_treeview_entries->row_expanded( path ) );
	}
}
