#!/usr/bin/python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Jono Bacon <jono@ubuntu.com>
# Copyright (C) 2010 Michael Budde <mbudde@gmail.com>
#
#This program is free software: you can redistribute it and/or modify it
#under the terms of the GNU General Public License version 3, as published
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranties of
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import os
import gtk
import gobject
import ConfigParser
import urllib
import logging

from lernid import ConnectDialog
from lernid.PasswordDialog import PasswordDialog
from lernid.lernidconfig import save_cache_path
from lernid.CouchDBPreferences import Preferences
from lernid.Event import Event
from lernid.LernidOptions import Options
from lernid.Statusbar import Statusbar
from lernid.IrcBackend import Server



_instance_object = None

class EventManager(gobject.GObject):

    __gtype_name__ = 'EventManager'

    __gsignals__ = {
        'event-connect': (
            gobject.SIGNAL_RUN_LAST, None, (gobject.TYPE_PYOBJECT,)
        ),
        'event-disconnect': (
            gobject.SIGNAL_RUN_LAST, None, (gobject.TYPE_PYOBJECT,)
        ),
    }

    def __init__(self):
        gobject.GObject.__init__(self)
        self._cachefolder = save_cache_path('lernid')
        self._cachefolder_events = os.path.join(self._cachefolder, 'events')
        self._isconnected = False
        self._connectdialog = None
        self._passworddialog = None
        self._widgets = {}
        self._read_config()

    @classmethod
    def get_instance(cls):
        global _instance_object
        if not _instance_object:
            _instance_object = cls()
        return _instance_object

    def _read_config(self):
        if Options.get('config'):
            uri = Options.get('config')
            if uri.startswith('http://') or uri.startswith('https://'):
                configfile = urllib.urlopen(uri)
            else:
                configfile = open(uri)
        else:
            configfile = urllib.urlopen('http://www.jonobacon.org/files/lernid/ubuntu.lernid')
        self._config = ConfigParser.ConfigParser()
        self._config.readfp(configfile)

    def _connect_event(self, event, nick):
        if self.is_connected():
            self.disconnect()
        self._event = Event(
            name=event,
            nick=nick,
            homepage=self._config.get(event, 'homepage'),
            icalurl=self._config.get(event, 'icalurl'),
            classroom=self._config.get(event, 'classroom'),
            chat=self._config.get(event, 'chat'),
            eventstart=self._config.get(event, 'eventstart'),
            eventend=self._config.get(event, 'eventend')
        )
        logging.debug('connecting to event {0}'.format(event))
        Statusbar.push_message(_('Connecting to event'), duration=10)
        self._isconnected = True
        self.emit('event-connect', self._event)

    def _identify_nick(self, nick, password):
        server = Server.get_server('irc.freenode.net', nick)
        identifier = server.identify_nick(password)
        def update_statusbar(obj):
            Statusbar.push_message(_('Your nickname is now identified'), duration=60)
        identifier.connect('identified', update_statusbar)
        def retry(obj):
            if not self._passworddialog:
                self._passworddialog = PasswordDialog.new()
            password = self._passworddialog.run()
            if password != None:
                identifier.retry(password)
        identifier.connect('invalid-password', retry)


    def connect_dialog(self, widget=None, parent=None):
        """Show the connect dialog and connect to the resources."""
        if not self._connectdialog:
            self._connectdialog = ConnectDialog.NewConnectDialog()
            self._connectdialog.set_transient_for(parent)
            self._connectdialog.connect('response', self._connect_dialog_reponse)
            self._connectdialog.set_nick(Preferences.get('nick'))
            self._connectdialog.set_transient_for(widget)
        self._connectdialog.show()

    def _connect_dialog_reponse(self, dialog, response):
        if response == gtk.RESPONSE_OK:
            event = self._connectdialog.get_event()
            nick = self._connectdialog.get_nick()
            Preferences.set('nick', nick)
            self._connect_event(event, nick)
            password = self._connectdialog.get_password()
            if password:
                self._identify_nick(nick, password)

    def disconnect(self):
        if not self.is_connected():
            return
        self._isconnected = False
        Statusbar.push_message(_('Disconnecting from event'), duration=10)
        self.emit('event-disconnect', self._event)
        logging.debug('disconnecting')

    def current_event(self):
        return self._event

    def is_connected(self):
        return self._isconnected

    def get_available_events(self):
        return self._config.sections()

    def register_widget(self, widget):
        if widget.name in self._widgets:
            raise ValueError('a widget with that name is already registered')
        self._widgets[widget.name] = widget

    def get_widget_by_name(self, name):
        if name in self._widgets:
            return self._widgets[name]
        return None

