/*
 *  Copyright 1994-2020 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include <inttypes.h>
#include "context.h"
#include "xmlutils.h"
#include "sequences.h"


void
Context_store_bank(Context_t *ctx, const uint8_t bank)
{
  assert(bank < MAX_BANKS);
  ctx->bank = bank;
  ctx->banks[ctx->bank_set][bank] = ctx->sm->cur->id;
}


void
Context_use_sequence_bank(Context_t *ctx, const uint8_t bank)
{
  u_long id = ctx->banks[ctx->bank_set][bank];
  Context_set_sequence(ctx, id);
}


void
Context_clear_bank(Context_t *ctx, const uint8_t bank)
{
  ctx->banks[ctx->bank_set][bank] = 0;
}


void
Context_save_banks(const Context_t *ctx)
{
  const gchar *home_dir = NULL;
  char *filename = NULL;
  json_t *banks = json_array();

  // create ~/.lebiniou directory (be safe if it doesn't exist)
  home_dir = g_get_home_dir();
  filename = g_strdup_printf("%s/." PACKAGE_NAME, home_dir);
  rmkdir(filename);
  g_free(filename);

  filename = g_strdup_printf("%s/." PACKAGE_NAME "/banks.json", home_dir);
  printf("[s] Banks filename: %s\n", filename);

  for (uint8_t bs = 0; bs < MAX_BANKS; bs++)
    for (uint8_t b = 0; b < MAX_BANKS; b++)
      if (ctx->banks[bs][b]) {
        json_t *bank = json_object();

        json_object_set_new(bank, "bank_set", json_integer(bs));
        json_object_set_new(bank, "bank", json_integer(b));
        json_object_set_new(bank, "sequence", json_integer(ctx->banks[bs][b]));
        json_array_append_new(banks, bank);
      }

  json_dump_file(banks, filename, JSON_INDENT(4));
  json_decref(banks);
  g_free(filename);
}


/*
 * There will be some loss when migrating an XML banks file
 * to the new banks/shortcuts model:
 *
 *   - we can map all the bank sets/banks/sequences, but
 *   - only 10 shortcuts are available so we'll try to map
 *     as many colormaps/images found as possible to free slots
 */
enum BankMode { SEQUENCES = 0, COLORMAPS, IMAGES };


static void
Context_load_banks_json(Context_t *ctx, gchar *filename)
{
  json_t *j_banks = json_load_file(filename, 0, NULL);
  g_free(filename);

  for (uint16_t b = 0; b < json_array_size(j_banks); b++) {
    json_t *j_bank_object = json_array_get(j_banks, b);
    json_t *j_bank_set = json_object_get(j_bank_object, "bank_set");
    json_t *j_bank = json_object_get(j_bank_object, "bank");
    json_t *sequence = json_object_get(j_bank_object, "sequence");

    uint8_t bank_set = json_integer_value(j_bank_set);
    uint8_t bank = json_integer_value(j_bank);

    if (NULL != sequence) {
      ctx->banks[bank_set][bank] = json_integer_value(sequence);
#ifdef DEBUG_BANKS
      printf("[b] Sequence %d-%d: %lld\n", bank_set, bank, json_integer_value(sequence));
#endif
    }
  }

  json_decref(j_banks);
}


static void
Context_load_banks_xml(Context_t *ctx, gchar *filename)
{
  xmlDocPtr doc = NULL; /* XmlTree */
  xmlNodePtr node = NULL, bank_node = NULL;

  xmlKeepBlanksDefault(0);
  xmlSubstituteEntitiesDefault(1);

  doc = xmlParseFile(filename);
  if (NULL == doc) {
    return;
  }

  node = xmlDocGetRootElement(doc);
  if (NULL == node) {
    xerror("Banks: xmlDocGetRootElement error\n");
  }

  node = xmlFindElement("banks", node);
  if (NULL == node) {
    xerror("Banks: no <banks> found\n");
  }

  bank_node = node->xmlChildrenNode;
  if (NULL == bank_node) { /* Empty banks.xml file */
    return;
  }

  while (NULL != bank_node) {
    long mode, set, id, seq_id;
    xmlChar *youhou;
    int res;

    youhou = xmlGetProp(bank_node, (const xmlChar *)"mode");
    mode = getintfield(youhou);
    xmlFree(youhou);
    assert(mode >= 0);
    assert(mode <= IMAGES);

    youhou = xmlGetProp(bank_node, (const xmlChar *)"set");
    set = getintfield(youhou);
    xmlFree(youhou);
    assert(set >= 0);
    assert(set < MAX_BANKS);

    youhou = xmlGetProp(bank_node, (const xmlChar *)"id");
    id = getintfield(youhou);
    xmlFree(youhou);
    assert(id >= 0);
    assert(id < MAX_BANKS);

    res = xmlGetOptionalLong(doc, bank_node, &seq_id);
    if (res == -1) {
      xerror("Banks: no value set\n");
    } else {
      if (mode == SEQUENCES) {
        printf("[X] Bankset: %li Bank: %li id: %li\n", set, id, seq_id);
        ctx->banks[set][id] = seq_id;
      } else if (id < MAX_SHORTCUTS) { // colormap/image shortcuts
        if (mode == COLORMAPS) {
          if (!ctx->shortcuts[SH_COLORMAP][id]) {
            printf("[X] Found free shortcut %ld for colormap: %li\n", id, seq_id);
            ctx->shortcuts[SH_COLORMAP][id] = seq_id;
          } else {
            printf("[X] No free shortcut %ld for colormap: %li, dropped\n", id, seq_id);
          }
        } else {
          if (!ctx->shortcuts[SH_IMAGE][id]) {
            printf("[X] Found free shortcut %ld for image: %li\n", id, seq_id);
            ctx->shortcuts[SH_IMAGE][id] = seq_id;
          } else {
            printf("[X] No free shortcut %ld for image: %li, dropped\n", id, seq_id);
          }
        }
      }
    }

    bank_node = bank_node->next;
  }

  xmlFreeDoc(doc);
}


void
Context_load_banks(Context_t *ctx)
{
  const gchar *home_dir = NULL;
  char *filename;
  int res;
  struct stat dummy;

  home_dir = g_get_home_dir();
  filename = g_strdup_printf("%s/." PACKAGE_NAME "/banks.json", home_dir);
  res = stat(filename, &dummy);

  if (-1 == res) {
    g_free(filename);

    // check for old banks.xml file
    filename = g_strdup_printf("%s/." PACKAGE_NAME "/banks.xml", home_dir);
    res = stat(filename, &dummy);

    if (-1 == res) {
      g_free(filename);
    } else {
      int err;
      gchar *backup;

      Context_load_banks_xml(ctx, filename);

      backup = g_strdup_printf("%s/." PACKAGE_NAME "/banks.xml.orig", home_dir);
      if ((err = rename(filename, backup)) == 0) {
        Context_save_banks(ctx); // convert to JSON
        Context_save_shortcuts(ctx);
        printf("[s] Renamed old XML banks %s to %s\n", filename, backup);
      } else {
        fprintf(stderr, "[s] Error renaming %s to %s: %s\n", filename, backup, strerror(errno));
      }
      g_free(filename);
      g_free(backup);
    }
  } else {
    Context_load_banks_json(ctx, filename);
  }
}
