/*
 *  Copyright 1994-2019 Olivier Girondel
 *  Copyright 2019 Laurent Marsac
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * This is based on the "snake" plugin
 *   - The snake shape is not a line now, but is based on sound waveform
 *
 *   - Color: linked to sound amplitude
 */

#include "context.h"
#include "oscillo.h"

static Porteuse_t *P = NULL;
static int connect = 1;
u_short sgn_size = 0 ;

u_long id = 1556272346;

u_long options = BE_GFX|BEQ_UNIQUE|BE_SFX2D;

u_long mode = OVERLAY;

char dname[] = "snake_oscillo";
char desc[] = "Snake oscillo";


static u_short x = 0, y = 0;


void init_oscillo(Context_t *ctx, u_short x, u_short y, u_short length, u_short dir, short inc)
{
  double y_factor = 0.1;
  double x_factor = 0.04;

  Porteuse_delete(P);

  u_short sgn_length = round(5.0f * (float)length);
  if (sgn_length > sgn_size) {
    sgn_length = sgn_size;
  }

  P = Porteuse_new(sgn_length, A_MONO);

  // oscillo
  u_short i = 0;
  Transform_t t;

  memset(&t, 0, sizeof(t));

  if (dir) {
    t.v_j_factor = HMAXY * y_factor;
    t.v_i.x = 1.0 / (float)(P->size - 1) * (float)length;

    if (inc < 0) {
      x = x - length;
    }
  } else {
    t.v_j_factor = HMAXX * x_factor;
    t.v_i.y = 1.0 / (float)(P->size - 1) * (float)length;

    if (inc < 0) {
      y = y - length;
    }
  }

  P->origin.x = x;
  P->origin.y = y;

  double win_avg = 0.0;
  double color_factor = 3.0;
  /* approx */
  u_short win_overlap = ctx->input->size >> 1 ;
  u_short win_size = floor((double)(ctx->input->size - win_overlap) / (double)P->size) + win_overlap;

  double tr = 0.1;
  u_short r = floor((double)P->size * tr);
  u_short factor_orig = t.v_j_factor;

  for (i = 0; i < P->size; i++) {
    /* Use a tukey window for smoother join of successive segments */
    double tc ;
    if (i < r/2) {
      tc = cos(2*M_PI * (i - r/2) / r) / 2.0 + 0.5;
    } else if (i > P->size - r/2) {
      tc = cos(2*M_PI * (i - 1.0 + r/2) / r) / 2.0 + 0.5;
    } else {
      tc = 1.0;
    }

    t.v_j_factor = floor((double)factor_orig * tc);
    P->trans[i] = t;

    /* compute color */
    if (i == P->size-1) {
      win_avg = compute_avg_abs(ctx->input->data[A_MONO], i*(win_size-win_overlap), ctx->input->size);
    } else {
      win_avg = compute_avg_abs(ctx->input->data[A_MONO], i*(win_size-win_overlap), i*(win_size-win_overlap)+win_size);
    }
    win_avg = color_factor * win_avg;
    if (win_avg > 1.0) {
      win_avg = 1.0;
    }
    P->color[i] = win_avg * PIXEL_MAXVAL;
  }

  Porteuse_init_alpha(P);
}


int8_t
create(Context_t *ctx)
{
  sgn_size = ctx->input->size;

  return 1;
}


void
destroy(Context_t *ctx)
{
  Porteuse_delete(P);
}


void
run(Context_t *ctx)
{
  Buffer8_t *dst = passive_buffer(ctx);
  Buffer8_clear(dst);

  u_short original_fft_size = 513; /* FFT size used when below parameters were set */
  u_short length_min = 8 * WIDTH/960;  /* minimum length of the snake, in pixels, scales with WIDTH */
  u_short length_max = 80 * WIDTH/960; /* maximum length of the snake, in pixels, scales with WIDTH */
  double spectrum_id_factor = 2; /* snake length will be length_max - average_frequency * spectrum_id_factor */
  double spectrum_low_treshold_factor = 0.1; /* spectrum value higher than this treshold will be used, between 0 and 1 */
  u_short spectrum_id_orientation_factor = 40; /* smaller means changing orientation more often */

  /* choose direction and increment mode */
  static u_short mode = 1; /* 0: direction changes at each run, 1: direction randomy changes, 2: also randomly changes orientation */
  u_short change_inc_on_hf = 1; /* 0: no change, 1: change orientation more often on high frequency */

  u_short average_freq_id = compute_avg_freq_id(ctx->input, spectrum_low_treshold_factor);

  /* scale average frequency id depending of input->spectrum_size */
  average_freq_id = round((double)average_freq_id * (double)original_fft_size / (double)ctx->input->spectrum_size);

  /* compute snake length based on average frequency */
  u_short length = length_max - average_freq_id * spectrum_id_factor;
  if (length < length_min) {
    length = length_min;
  }
  if (length > length_max) {
    length = length_max;
  }

  static u_short dir = 0; /* direction: 0 is Y and 1 is X */
  short inc = 1;  /* increment: 1 or -1 */

  switch (mode) {
      default:
      case 0:
        dir = !dir;
        break;

      case 1:
        dir = drand48() < .5;
        break;

      /* random dir and inc */
      case 2:
        dir = drand48() < .5;
        inc = drand48() < .5 ? -1 : 1;
        break;
  }

  /* if set, change orientation on high frequency */
  static short inc_hf = 1;
  if (change_inc_on_hf && (drand48() < average_freq_id / (double)spectrum_id_orientation_factor)) {
    inc_hf = -inc_hf;
    inc = inc_hf;
  }

  /* avoid going back on previous path */
  static u_short last_dir = 0;
  static u_short last_inc = -1;
  if (last_dir == dir) {
    inc = last_inc;
  }
  last_dir = dir;
  last_inc = inc;

  /* remove length bias due to different HEIGHT and WIDTH */
  if (!dir) {
    length = (u_short)ceil((double)length * (double)HEIGHT / (double)WIDTH);
  }

  connect = 1;
  init_oscillo(ctx, x, y, length, dir, inc);
  Porteuse_draw(P, ctx, connect);

  /* X direction */
  if (dir) {
    x = (x + inc * length) % WIDTH;
  } else {
  /* Y direction */
    y = (y + inc * length) % HEIGHT;
  }
}
