/*
 *  Copyright 1994-2019 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "biniou.h"

float phase = 1.0;


Input_t *
Input_new(const uint32_t size)
{
  uint32_t c;

  Input_t *input = xcalloc(1, sizeof(Input_t));

  pthread_mutex_init(&input->mutex, NULL);

  input->size = size;
  input->mute = 0;
  input->spectrum_size = input->size / 2 + 1;

  VERBOSE(printf("[w] data size= %d, power spectrum size= %d\n",
		 input->size, input->spectrum_size));

  for (c = 0; c < 3; c++) {
    input->data[c]   = fftw_alloc_real(input->size);
    input->data_u[c] = xcalloc(input->size, sizeof(double));
    input->out[c]    = fftw_alloc_complex(input->spectrum_size);

    input->plan_fft[c] = fftw_plan_dft_r2c_1d(input->size,
					      input->data[c],
					      input->out[c],
					      FFTW_MEASURE);

    input->spectrum[c]     = xcalloc(input->spectrum_size, sizeof(double));
    input->spectrum_log[c] = xcalloc(input->spectrum_size, sizeof(double));
  }

  for (c = 0; c < input->size; c++) {
    input->data[A_LEFT][c]  = input->data_u[A_LEFT][c]  = 0;
    input->data[A_RIGHT][c] = input->data_u[A_RIGHT][c] = 0;
  }

#ifdef DEBUG
  input->timer = b_timer_new();
#endif

  return input;
}


void
Input_delete(Input_t *input)
{
  int c;

  for (c = 0; c < 3; c++) {
    fftw_free(input->data[c]);
    xfree(input->data_u[c]);
    fftw_free(input->out[c]);
    xfree(input->spectrum[c]);
    xfree(input->spectrum_log[c]);
    fftw_destroy_plan(input->plan_fft[c]);
  }

#ifdef DEBUG
  b_timer_delete(input->timer);
#endif

  xfree(input);
  fftw_cleanup();
}


static int
Input_seek_max_spectrum(Input_t *input, int c)
{
  uint32_t i;
  u_short new_max = 0;

  /* Input_reset_max_spectrum(input); */
  input->max_spectrum[c] = -1.0;

  /* start at 1 to avoid power spectrum value at index 0 */
  for (i = 1; i < input->spectrum_size; i++)
    if (input->spectrum[c][i] > input->max_spectrum[c]) {
      input->max_spectrum[c] = input->spectrum[c][i];
      new_max = i;
    }

  return new_max;
}


static void
Input_do_fft(Input_t *input)
{
  /* const int N = input->size; */
  /* const int even = (N % 2 == 0); */
  int c;
  uint32_t k;

#ifdef DEBUG
  b_timer_start(input->timer);
#endif

  for (c = 0; c < 3; c++)
    fftw_execute(input->plan_fft[c]);

  for (c = 0; c < 3; c++) {
    for (k = 0; k < input->spectrum_size; k++) {
      input->out[c][k] /= (float)input->size;
      /* printf("(%d %d %f) ", c, k, input->out[c][k]); */
    }

    for (k = 0; k < input->spectrum_size; k++) {
      // input->spectrum[c][k] = cabs(input->out[c][k]);
      // normalize to [0.0..+1.0]
      input->spectrum[c][k] = cabs(input->out[c][k]) * M_SQRT1_2; // == / sqrtf(2.0);
      assert(input->spectrum[c][k] >= 0);
      assert(input->spectrum[c][k] <= 1.0);
    }

    for (k = 0; k < input->spectrum_size; k++) {
      input->spectrum[c][k] = input->out[c][k];
      // assert(input->spectrum[c][k] >= 0);
    }
  }

  for (c = 0; c < 3; c++) {
    int new_max = Input_seek_max_spectrum(input, c);

    for (k = 0; k < input->spectrum_size; ++k) {
      /* Log toussa */
      /* log1p(x)=logf(x+1) */
      input->spectrum_log[c][k] = log1p(input->spectrum[c][k]) / (M_LN2 / M_LN10);
    }

    input->max_spectrum_log[c] = input->spectrum_log[c][new_max];

#ifdef XDEBUG
    if (c == A_MONO) {
      printf("[s] %d % 3d\tspectrum: %f\tspectrum_log: %f\n",
	     c, new_max,
	     input->max_spectrum[c],
	     input->max_spectrum_log[c]);
    }
#endif
  }

#ifdef XDEBUG
  printf("[i] FFT time: %f ms\n", b_timer_elapsed(input->timer) * 1000);
#endif
}


static inline double
Input_clamp(const double val)
{
  if (val < -1.0) return -1.0;
  else if (val > 1.0) return 1.0;
  else return val;
}


void
Input_set(Input_t *input, u_char mode)
{
  /* mode:
   * A_MONO   => copy A_MONO to A_LEFT and A_RIGHT
   * A_STEREO => compute A_MONO as the average of A_LEFT and A_RIGHT
   */
  uint32_t i;

  if (mode == A_MONO) {
    for (i = 0; i < input->size; i++) {
      /* clamp values */
      input->data[A_MONO][i] = Input_clamp(input->data[A_MONO][i]);

      /* set input from A_MONO source */
      input->data_u[A_MONO][i] = (input->data[A_MONO][i] + 1.0) / 2.0;

      /* copy to A_LEFT and A_RIGHT */
      input->data[A_LEFT][i] = input->data[A_RIGHT][i] = input->data[A_MONO][i];
      input->data_u[A_LEFT][i] = input->data_u[A_RIGHT][i] = input->data_u[A_MONO][i];
    }
  } else {
    assert(mode == A_STEREO);
    for (i = 0; i < input->size; i++) {
      /* clamp values */
      input->data[A_LEFT][i] = Input_clamp(input->data[A_LEFT][i]);
      input->data[A_RIGHT][i] = Input_clamp(input->data[A_RIGHT][i]);

      /* set input from A_LEFT and A_RIGHT */
      input->data_u[A_LEFT][i] = (input->data[A_LEFT][i] + 1.0) / 2.0;
      input->data_u[A_RIGHT][i] = (input->data[A_RIGHT][i] + 1.0) / 2.0;

      /* compute A_MONO from A_LEFT and A_RIGHT */
      input->data[A_MONO][i] = (input->data[A_LEFT][i] + (phase * input->data[A_RIGHT][i])) / 2;
      input->data_u[A_MONO][i] = (input->data[A_MONO][i] + 1.0) / 2.0;
    }
  }

  Input_do_fft(input);
}


static inline void
do_roulette(Input_t *input)
{
  INC(input->roulette, input->size);
}


inline float
Input_random_s_u_float(Input_t *input)
{
  /* random float [-1..+1] */
  float f = input->data[A_MONO][input->roulette];
  do_roulette(input);
  return f;
}


inline float
Input_random_u_u_float(Input_t *input)
{
  /* random float [0..1] */
  float f = input->data_u[A_MONO][input->roulette];
  do_roulette(input);
  return f;
}


inline float
Input_random_float_range(Input_t *input, const float min, const float max)
{
  /* random short */
  float f;
  float rnd = input->data_u[A_MONO][input->roulette];
#ifdef DEBUG
  if (max <= min)
    xerror("Input_random_short_range: max %f <= min %f\n", max, min);
#endif
  f = min + rnd * (max - min);
  do_roulette(input);
  return f;
}


inline short
Input_random_short_range(Input_t *input, const short min, const short max)
{
  /* random short */
  short s;
  float rnd = input->data_u[A_MONO][input->roulette];
#ifdef DEBUG
  if (max <= min)
    xerror("Input_random_short_range: max %d <= min %d\n", max, min);
#endif
  s = min + rnd * (max - min);
  do_roulette(input);
  return s;
}


inline u_char
Input_random_u_char(Input_t *input)
{
  u_char uc = (input->data_u[A_MONO][input->roulette] * 255);

  do_roulette(input);
  return uc;
}


inline float
Input_get_volume(Input_t *input)
{
  uint32_t i;
  float volume = 0;

  pthread_mutex_lock(&input->mutex);

  for (i = 0; i < input->size; i++)
    volume += fabs(input->data[A_MONO][i]);

  volume /= input->size;

  pthread_mutex_unlock(&input->mutex);

  return volume;
}


void
Input_toggle_mute(Input_t *input)
{
  input->mute = !input->mute;
}
