/*
 *  Copyright 1994-2018 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "context.h"
#include "spline.h"
#include "particles.h"


u_long id = 948640645;
u_long options = BE_SFX3D|BEQ_PARTICLES;
u_long mode = OVERLAY;
char desc[] = "Space-phase reconstruction using particles";

#define SPAN_SIZE 9
#define CONNECT   1

static const Point3d_t ORIGIN = { { 0.0, 0.0, 0.0 } };
static Spline_t *s = NULL;
static Particle_System_t *ps = NULL;


static void
Delay1_points(Context_t *ctx)
{
  u_short i;
  Buffer8_t *dst = passive_buffer(ctx);
  const Params3d_t *params3d = &ctx->params3d;
  Input_t *input = ctx->input;

  Buffer8_clear(dst);

  for (i = 0; i < s->nb_spoints; i++) {
    Pixel_t color = Input_random_u_char(input);
    set_pixel_3d(params3d, dst, &s->spoints[i], color);
  }
}


static void
Delay1_lines(Context_t *ctx)
{
  u_short i;
  Buffer8_t *dst = passive_buffer(ctx);
  const Params3d_t *params3d = &ctx->params3d;
  Input_t *input = ctx->input;
  u_long points;

  Buffer8_clear(dst);

  points = s->nb_spoints-1;

  for (i = 0; i < points; i++) {
    Pixel_t color = Input_random_u_char(input);
    draw_line_3d(params3d, dst, &s->spoints[i], &s->spoints[i+1], color);
  }
}


static void
Delay1_draw(Context_t *ctx)
{
  if (CONNECT)
    Delay1_lines(ctx);
  else
    Delay1_points(ctx);
}


static void
Delay1_particles(Context_t *ctx)
{
  u_short i;
  Input_t *input = ctx->input;
  Buffer8_t *dst = passive_buffer(ctx);

  Particle_System_go(ps);

  for (i = 0; (i < s->nb_spoints) && Particle_System_can_add(ps); i++) {
    Particle_t *p = NULL;
    float ttl = Input_random_float_range(input, 0.8, 2.0);
    /* XXX Input_random_Pixel */
    Pixel_t col = Input_random_u_char(input);
    p = Particle_new_indexed(ttl, col, s->spoints[i], p3d_mul(&s->spoints[i], 0.25), ORIGIN, 0.0);

    Particle_System_add(ps, p);
  }

  Particle_System_draw(ps, &ctx->params3d, dst);
}


static void
Delay1_init(Context_t *ctx)
{
  u_short i;

  pthread_mutex_lock(&ctx->input->mutex);

  /* Map cubique (x y z) */
  s->cpoints[0].pos.x = ctx->input->data[A_MONO][0];
  s->cpoints[0].pos.y = ctx->input->data[A_MONO][1];
  s->cpoints[0].pos.z = ctx->input->data[A_MONO][2];

  for (i = 1; i < s->nb_cpoints; i++) {
    s->cpoints[i].pos.x = s->cpoints[i-1].pos.y;
    s->cpoints[i].pos.y = s->cpoints[i-1].pos.z;
    s->cpoints[i].pos.z = ctx->input->data[A_MONO][i+2];
  }

  pthread_mutex_unlock(&ctx->input->mutex);
}


void
create(Context_t *ctx)
{
  if (ctx->input == NULL) {
    options |= BEQ_DISABLED;
    return;
  }

  ps = Particle_System_new(PS_DEFAULT_MAX_PARTICLES * 2);

  s = Spline_new(SPAN_SIZE, ctx->input->size - 2);
#ifdef DEBUG
  Spline_info(s);
#endif
}


void
destroy(Context_t *ctx)
{
  if (ps != NULL)
    Particle_System_delete(ps);
  if (s != NULL)
    Spline_delete(s);
}


void
run(Context_t *ctx)
{
  if (s != NULL) {
    Delay1_init(ctx);
    Spline_compute(s);
    Delay1_draw(ctx);
    Delay1_particles(ctx);
  }
}
