/*
 *  Copyright 1994-2011 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __BINIOU_CONTEXT_H
#define __BINIOU_CONTEXT_H

#include "input.h"
#include "sequencemanager.h"
#include "params3d.h"
#include "particles.h"
#include "pictfader.h"
#include "cmapfader.h"
#include "alarm.h"
#include "brandom.h"

#define NFPS       25 /* to get mean fps */

enum RandomMode { BR_NONE=0, BR_SEQUENCES, BR_SCHEMES, BR_BOTH } RandomMode_e;

#define NSCREENS 3
#define ACTIVE_BUFFER  0
#define PASSIVE_BUFFER 1
#define SAVE_BUFFER    2


typedef struct BKey_s {
  u_short val;
  u_short mod;
} BKey_t;


typedef struct Context_s {
  char      running:1;

  Plugin_t  *input_plugin;
  Input_t   *input;

  /* Buffers */
  Buffer8_t  *buffers[NSCREENS];
  /* 0 = active_buffer
   * 1 = passive_buffer - for double-buffering
   * 2 = save_buffer    - push/pop screen for lens effects
   */

  /* Faders */
  PictFader_t *pf;
  Alarm_t *a_picts;

  CmapFader_t *cf;
  Alarm_t *a_cmaps;

  GSList    *outputs;
	
  SequenceManager_t *sm;

  /* TODO move this to a global */
  Params3d_t params3d;

  enum RandomMode random_mode;
  Alarm_t  *a_random;

  u_long   frames;

  GList    *events;
  u_long   nb_events;

  BTimer_t *timer;

  /* FPS stuff */
  char     sync_fps:1;
  u_short  max_fps;
  float    i_max_fps; /* inverse of the maximum frames per second */
  int      fps[NFPS];
  BTimer_t  *fps_timer;

  enum OSDMode   osd_mode;
  char     fullscreen:1;
  char     display_colormap:1;
  char     take_screenshot:1;

  void (*events_cb)(struct Context_s *, const BKey_t *); /* events callback */

  u_char   bank; /* the bank used */
} Context_t;


Context_t *Context_new();
void Context_delete(Context_t *);

void Context_set(Context_t *);
void Context_run(Context_t *);
void Context_update(Context_t *);
void Context_update_auto(Context_t *);

int  Context_add_rand(Sequence_t *, const enum PluginOptions, const int);
void Context_randomize(Context_t *);

void Context_set_max_fps(Context_t *, const u_short);
void Context_set_random_mode(Context_t *, const enum RandomMode);
void Context_set_osd_mode(Context_t *, const enum OSDMode);

void Context_insert_plugin(Context_t *, Plugin_t *);
void Context_remove_plugin(Context_t *, Plugin_t *);

void Context_screenshot(const Context_t *);
float Context_fps(const Context_t *);

void Context_previous_sequence(Context_t *);
void Context_next_sequence(Context_t *);
void Context_latest_sequence(Context_t *);
void Context_random_sequence(Context_t *);
void Context_set_sequence(Context_t *, const u_long);

/* Events */
int Context_event(Context_t *, const Event_t *); /* TODO void ? */
void Context_process_events(Context_t *);
void Context_add_event(Context_t *, const Event_t *);
void Context_send_event(Context_t *, const enum RcptTo, const enum Command,
		        const enum Arg);

Buffer8_t *active_buffer(const Context_t *);
Buffer8_t *passive_buffer(const Context_t *);


static inline Buffer8_t *
save_buffer(const Context_t *ctx)
{
  return ctx->buffers[SAVE_BUFFER];
}


static inline void
swap_buffers(Context_t *ctx)
{
  Buffer8_t *tmp = ctx->buffers[ACTIVE_BUFFER];
  ctx->buffers[ACTIVE_BUFFER] = ctx->buffers[PASSIVE_BUFFER];
  ctx->buffers[PASSIVE_BUFFER] = tmp;
}


static inline void
push_buffer(const Context_t *ctx)
{
  Buffer8_copy(active_buffer(ctx), save_buffer(ctx));
}


static inline void
pop_buffer(const Context_t *ctx)
{
  Buffer8_copy(save_buffer(ctx), active_buffer(ctx));
}


Pixel_t *
export_RGB_buffer(const Context_t *, const u_char, const u_char);

Pixel_t *
export_RGB_active_buffer(const Context_t *, const u_char);

void Context_use_bank(Context_t *, const u_char);
void Context_clear_bank(Context_t *, const u_char);

#endif /* __BINIOU_CONTEXT_H */
