/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_localaccount.mod voms local account plugin

    \section vomslocalaccountsyn SYNOPSIS
    \b lcmaps_voms_localaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-use_voms_gid]

    \section vomslocalaccountdesc DESCRIPTION

    This localaccount acquisition plugin is a 'VOMS-aware' modification of the 'localaccount' plugin.
    The plugin tries to find a local account (more specifically a UID) based on the VOMS information
    that has been retrieved by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod"
    from the user's grid credential.
    It will try to match a VO-GROUP-ROLE combination from the user's grid credential with an entry
    in a gridmapfile (most likely the traditional gridmapfile, used by the localaccount and
    poolaccount plugins)
    In this file VO-GROUP-ROLE combinations are listed next to the name of an existing account,
    as shown in the following example.

    EXAMPLE:

    \c "/VO=wilma/GROUP=management" \c wilmamgr

    \c "/VO=fred/GROUP=*"  \c fredmgr

    If the first matching VO-GROUP-ROLE combination is \c "/VO=wilma/GROUP=management" the plugin
    will map the user to the account wilmamgr.
    If the first matching VO-GROUP-ROLE combination is
    \c "/VO=fred/GROUP=*" the plugin will map the user to the account fredmgr.

    \section vomslocalaccountnote1 NOTE 1
        This plugin should only be used in combination with the \e 'voms_localgroup'
        and/or \e 'voms_poolgroup' plugins.

    \section vomslocalaccountoptions OPTIONS
    \subsection vomslocalaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie4 -gridmap \<gridmapfile\>
        When this option is set it will override the default path to the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection vomslocalaccountoptie5 -use_voms_gid
        When this option is set the LCMAPS system relies on other VOMS plugins such as
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod" and \ref lcmaps_voms_poolgroup.mod
        "lcmaps_voms_poolgroup.mod" to assign a primary GID based on the VOMS attributes contained in the 
        user proxy instead of taking the default primary GID that comes with the local account.

\section vomslocalaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section vomslocalaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section vomslocalaccountSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_localaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_localaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
/* Try to provide RTLD_DEFAULT */
#define _GNU_SOURCE
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/stat.h>

#include "lcmaps_voms_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define LCMAPS_MAXGIDBUFFER 256
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *gridmapfile                            = NULL;
static int   use_voms_gid                           = 0;
static int   do_not_map_primary_gid                 = 0;
static int   add_primary_gid_from_mapped_account    = 0;
static int   add_primary_gid_as_secondary_gid_from_mapped_account = 0;
static int   add_secondary_gids_from_mapped_account = 0;

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char *  logstr = "lcmaps_plugin_voms_localaccount-plugin_initialize()";
    int i;
    struct stat s;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (gridmap(file))
     * the second will be the path && filename of the gridmapfile
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-gridmap") == 0) ||
              (strcmp(argv[i], "-GRIDMAP") == 0) ||
              (strcmp(argv[i], "-gridmapfile") == 0) ||
              (strcmp(argv[i], "-GRIDMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* check if the setting exists */
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR, "%s: Error: grid-mapfile not accessible at \"%s\"\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }

                 gridmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else if (strcmp(argv[i], "--do-not-add-primary-gid-from-mapped-account") == 0)
        {
            do_not_map_primary_gid = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-from-mapped-account") == 0)
        {
            add_primary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-as-secondary-gid-from-mapped-account") == 0)
        {
            add_primary_gid_as_secondary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-secondary-gids-from-mapped-account") == 0)
        {
            add_secondary_gids_from_mapped_account = 1;
        }
        else if ((strcmp(argv[i], "--use-voms-gid") == 0) ||
                 (strcmp(argv[i], "--use_voms_gid") == 0) ||
                 (strcmp(argv[i], "-use_voms_gid") == 0))
        {
            use_voms_gid = 1;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    /* Post mortum check */
    if (do_not_map_primary_gid && add_primary_gid_from_mapped_account)
    {
        lcmaps_log(LOG_ERR,"%s: Error: can't set both --do-not-add-primary-gid-from-mapped-account and --add-primary-gid-from-mapped-account\n", logstr);
        return LCMAPS_MOD_FAIL;
    }
    if (use_voms_gid && do_not_map_primary_gid)
    {
        lcmaps_log(LOG_ERR,"%s: Error: can't set both --use-voms-gid and --do-not-add-primary-gid-from-mapped-account\n", logstr);
        return LCMAPS_MOD_FAIL;
    }
    if (use_voms_gid && add_primary_gid_from_mapped_account)
    {
        lcmaps_log(LOG_ERR,"%s: Error: can't set both --use-voms-gid and --add-primary-gid-from-mapped-account\n", logstr);
        return LCMAPS_MOD_FAIL;
    }
    if (use_voms_gid && add_secondary_gids_from_mapped_account)
    {
        lcmaps_log(LOG_ERR,"%s: Error: can't set both --use-voms-gid and --add-secondary-gids-from-mapped-account\n", logstr);
        return LCMAPS_MOD_FAIL;
    }



    return LCMAPS_MOD_SUCCESS;
} 

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char *                   logstr = "lcmaps_plugin_voms_localaccount-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"      ,       "char *"        , 0,   NULL},
        {"fqan_list"    ,       "char **"       , 0,   NULL},
        {"nfqan"        ,       "int"           , 0,   NULL},
        {NULL           ,       NULL            , -1,   NULL},
        {NULL           ,       NULL            , -1,   NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    char *errorstr;
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
#warning old LCMAPS versions do not have lcmaps_get_major_version() functions
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log_debug(4,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log_debug(4,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[3].argName="requested_username";
	argList[3].argType="char *";
	argList[3].argInOut=1;
	argList[3].value=NULL;
    }

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *              logstr = "lcmaps_plugin_voms_localaccount-plugin_run()";
    char *              dn                  = NULL; 
    int                 dn_cnt              = 0;
    char *              username            = NULL;
    struct passwd       *user_info          = NULL;
    int                 i                   = 0;
    int                 cnt_sec_gid         = 0;
    gid_t *             sec_gid             = NULL;
    char **             vo_cred_string_list = NULL;
    int                 cnt_vo_cred_string  = 0;
    int                 found               = 0;
    unsigned short      matching_type       = ((unsigned short)0x0000);
    int                 rc                  = 0;
    char **             fqan_list           = NULL;
    int                 nfqan               = -1;
    char *              req_username        = NULL;
    char *              searchstr           = NULL;
    void *              value               = NULL;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_voms_localaccount-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_voms_localaccount-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_voms_localaccount-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_voms_localaccount;
    }
    lcmaps_log_debug(5,"%s:\n", logstr);

    /*
     * Try to get the ordered values:
     */
    if ( (value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
	dn = *(char **)value; 
        lcmaps_log_debug(5,"%s: found dn: %s\n", logstr, dn);

        /* Check if we don't have a DN already registered, if not, add it to the internal registry */
        getCredentialData (DN, &dn_cnt);
        if (dn_cnt == 0)
        {
            lcmaps_log_debug (5, "%s: Adding DN: %s\n", logstr, dn);
            addCredentialData(DN, &dn);
        }
    }
    else    {
	dn = NULL;
        lcmaps_log_debug(1,"%s: could not get value of dn !\n", logstr);
    }

    /*
     * See if we have a requested req_username: make sure not to get value
     * immediately or we'll get a segfault
     */
    if ( (value=lcmaps_getArgValue("requested_username", "char *", argc, argv))!=NULL )
	req_username=*(char **)value;
    else
	req_username=NULL;
    lcmaps_log_debug(1,"%s: requested username is %s\n", logstr,
	    req_username ? req_username : "unset");

    /*
     * Check the gridmapfile
     */

    if ((gridmapfile != NULL) && (strlen(gridmapfile) > 0))
        lcmaps_log_debug(3,"%s: gridmapfile is: %s\n", logstr, gridmapfile);
    else
    {
        if (gridmapfile) free(gridmapfile);
        gridmapfile = NULL;
        lcmaps_log_debug(1,"%s: No gridmapfile assigned, so function must find out for it self\n", logstr);
    }

    /*
     * Get the VO user information.
     * We can either order it by lcmaps_argument_t or use the getCredentialData() function.
     * The latter case requires the voms parsing plugin (lcmaps_voms.mod) to have run beforehand.
     * Unfortunately the formats of the VOMS strings (from getCredentialData()) and
     * FQANs (from lcmaps_argument_t) are not the same. We may have to introduce
     * two-way conversion functions.
     * The VOMS info has to matched against the info in the gridmapfile
     */
    lcmaps_log_debug(5,"%s: First try to get the FQAN list from input credential repository ...\n", logstr);
    if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) )
    {
	nfqan = *(int *)value;
        lcmaps_log_debug(5,"%s: the list of FQANs should contain %d elements\n", logstr, nfqan);
        if ( ( value = lcmaps_getArgValue("fqan_list", "char **", argc, argv)) )   {
	    fqan_list = *(char ***)value;
            lcmaps_log_debug(5, "%s: found list of FQANs\n", logstr);
	}
	else
	{
            lcmaps_log_debug(1, "%s: could not retrieve list of FQANs!\n", logstr);
            goto fail_voms_localaccount;
        }
        for (i = 0; i < nfqan; i++)
        {
            lcmaps_log_debug(3, "%s: FQAN %d: %s\n", logstr, i, fqan_list[i]);
        }
        vo_cred_string_list = fqan_list;
        cnt_vo_cred_string = nfqan;
    }
    else
    {
        lcmaps_log_debug(1,"%s: ... did not find input credentials in input credential repository...\n", logstr);
        lcmaps_log_debug(1,"%s: ... trying the internal credential repository ...\n", logstr);

        vo_cred_string_list = getCredentialData(LCMAPS_VO_CRED_STRING, &cnt_vo_cred_string);
    }

    if (cnt_vo_cred_string == 0)
    {
        lcmaps_log(LOG_NOTICE,"%s: no VOMS group info --> no mapping\n", logstr);
        goto fail_voms_localaccount;
    }
    else if (cnt_vo_cred_string < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of VOMS groups found ! (failure)\n", logstr);
        goto fail_voms_localaccount;
    }


    /*
     * Try to match the VO strings with the gridmapfile info
     * normally the first available VO string should match
     */
    found = 0;

    if ( req_username )	{
	matching_type = MATCH_EXACT|MATCH_WILD_CHARS;
	searchstr=req_username;
    } else {
	matching_type = MATCH_EXCLUDE|MATCH_WILD_CHARS;
	searchstr=".";
    }


    for (i = 0; i < cnt_vo_cred_string; i++)
    {
        /* clean username before each call to lcmaps_gridlist */
        if (username) free(username);
        username = NULL;
        if ( (rc = lcmaps_gridlist(vo_cred_string_list[i], &username, gridmapfile, matching_type, searchstr, NULL)) == 0)
        {
            found = 1;
            lcmaps_log_debug(3,"%s: found username: %s\n", logstr, username);
            break;
        }
        else if (rc == LCMAPS_MOD_NOFILE)
        {
            lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
            goto fail_voms_localaccount;
        }
        else
        {
	    if (req_username)
		lcmaps_log_debug(1, "%s: no localaccount available for group (%s) and requested user %s in %s\n", logstr, vo_cred_string_list[i], req_username, gridmapfile);
	    else
		lcmaps_log_debug(1, "%s: no localaccount available for group (%s) in %s\n", logstr, vo_cred_string_list[i], gridmapfile);
        }
    }
    if (found != 1)
    {
        lcmaps_log(LOG_WARNING, "%s: Could not find a VOMS localaccount in %s (failure)\n", logstr, gridmapfile);
        goto fail_voms_localaccount;
    }

    /*
     * Get userid to pwd_t structure
     */
    if (username && (strlen(username) > 0))
    {
        if ( ( user_info = getpwnam(username) ) )
        {
            lcmaps_log_debug(5,"%s: address user_info: %p\n", logstr, user_info);
            lcmaps_log_debug(3,"%s: username : %s, char ptr: %p, address char ptr: %p\n", logstr, user_info->pw_name, user_info->pw_name, &(user_info->pw_name));
            lcmaps_log_debug(3,"%s: password : %s\n", logstr, user_info->pw_passwd);
            lcmaps_log_debug(3,"%s: user_id  : %d, address uid: %p\n", logstr, user_info->pw_uid, &(user_info->pw_uid));
            lcmaps_log_debug(3,"%s: group_id : %d\n", logstr, user_info->pw_gid);
            lcmaps_log_debug(3,"%s: realname : %s\n", logstr, user_info->pw_gecos);
            lcmaps_log_debug(3,"%s: home dir : %s\n", logstr, user_info->pw_dir);
            lcmaps_log_debug(3,"%s: shellprg : %s\n", logstr, user_info->pw_shell);

            /* Add this credential data to the credential data repository in the plugin manager */
            addCredentialData(UID, &(user_info->pw_uid));

            /* Map primary Unix GID from the account info */
            if ((!do_not_map_primary_gid) &&
                (add_primary_gid_from_mapped_account))
            {
                lcmaps_log_debug(4,"%s: adding primary GID (%d) from local account to CredentialData\n", logstr, user_info->pw_gid);
                addCredentialData(PRI_GID, &(user_info->pw_gid));
            }

            /* Add the primary GID from the mapped account as an secondary GID to the result */
            if (add_primary_gid_as_secondary_gid_from_mapped_account)
            {
                lcmaps_log_debug(4,"%s: adding primary GID (%d) from local account as a secondary GID to CredentialData\n", logstr, user_info->pw_gid);
                addCredentialData(SEC_GID, &(user_info->pw_gid));
            }

            /* Add secondary Unix group IDs from the mapped local account */
            if (add_secondary_gids_from_mapped_account)
            {
                /* Retrieve secondary group id's */
                if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
                {
                    lcmaps_log_debug(4,"%s: adding secondary GIDs (%d) from local account to CredentialData\n", logstr, user_info->pw_gid);
                    for (i = 0; i < cnt_sec_gid; i++)
                    {
                        addCredentialData(SEC_GID, &(sec_gid[i]));
                    }
                    free(sec_gid);
                }
            }

            /* Old and error tolerant setting to set primary and secondary Unix
             * IDs from the /etc/{passwd,groups} info */
            if (use_voms_gid == 0)
            {
                lcmaps_log_debug(4,"%s: adding primary GID (%d) from local account to CredentialData\n", logstr, user_info->pw_gid);
                addCredentialData(PRI_GID, &(user_info->pw_gid));

                /* Retrieve secondary group id's */
                if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
                {
                    for (i = 0; i < cnt_sec_gid; i++)
                    {
                        addCredentialData(SEC_GID, &(sec_gid[i]));
                    }
                    free(sec_gid);
                }
            }
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: no user account found named \"%s\"\n", logstr, username);
            goto fail_voms_localaccount;
        }
    }
    else
    {   // error (msg is already given)
        goto fail_voms_localaccount;
    }

    /* succes */
 success_voms_localaccount:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: voms_localaccount plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_voms_localaccount:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: voms_localaccount plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char * logstr = "lcmaps_plugin_voms_localaccount-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (gridmapfile) free(gridmapfile);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/voms/lcmaps_voms_localaccount.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.8 $
    $Author: okoeroo $
******************************************************************************/
