/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_ldap_enf.mod ldap enforcement plugin

    \section ldapsection1 SYNOPSIS
    \b lcmaps_ldap_enf.mod -maxuid \<maxuid\> -maxpgid \<maxpgid\> -maxsgid \<maxsgid\> -hostname \<hostname\>
    -port \<port\> [-require_all_groups [yes|no]] -dn_manager \<DN\> -ldap_pw \<path/filename\>
    -sb_groups \<seachbase\> -sb_user \<searchbase\> -timeout \<timeout value\>

    \section ldapsection2 DESCRIPTION
     Ldap enforcement plugin will alter the user and group settings in the
     ldap database, using the user and groups settings provided by the credential acquisition plugins.
     Note that LDAP has to be used as the source of account information for PAM or NSS and has to be RFC 2307 complient.
     (see documentation)

    \section ldapsection3 OPTIONS
      \subsection ldapoptie1 -maxuid \<maxuid\>
                  Maximum number of uids to be used. Strongly advised is to set this to 1.
      \subsection ldapoptie2 -maxpgid \<maxpgid\>
                  Maximum number of primary gids to be used.
      \subsection ldapoptie3 -maxsgid \<maxsgid\>
                  Maximum number of (secondary) gids to be used (not including primary group). Advised is to set this to 1.
      \subsection ldapoptie4 -hostname \<hostname\>
                  The hostname on which the LDAP server is running, e.g. asen.nikhef.nl
      \subsection ldapoptie5 -port \<port\>
                  The port number to which to connect, e.g. 389
      \subsection ldapoptie6 -require_all_groups [yes|no]
                  Specify if all groups set by the PluginManager shall be used. Default is'yes'
      \subsection ldapoptie7 -dn_manager \<DN\>
                  DN of the LDAP manager, e.g. "cn=Manager,dc=root"
      \subsection ldapoptie8 -ldap_pw \<path/filename\>
                  Path to the file containing the password of the LDAP manager.
                  Note: the mode of the file containing the password must be read-only for root (400),
                  otherwise the plugin will not run.
      \subsection ldapoptie9 -sb_groups \<seachbase\>
                  Search base for the (secondary) groups, e.g. "ou=LocalGroups, dc=foobar, dc=ough"
      \subsection ldapoptie10 -sb_user \<searchbase\>
                  Search base for the user, e.g. "ou=LocalUsers, dc=foobar, dc=ough"
      \subsection ldapoptie11 -timeout \<timeout value\>
                  timeout (in seconds) that will be applied to the ldap binding

    \section ldapsection4 RETURN VALUE
    \li LCMAPS_MOD_SUCCESS : succes
    \li LCMAPS_MOD_FAIL    : failure

    \section ldapsection5 ERRORS
     See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

    \section ldapsection6 SEE ALSO
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_ldap.c
    \brief  Interface to the LCMAPS plugins
    \author Wim Som de Cerff and Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for the ldap LCMAPS plugin
    The interface consists of the following functions:
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()

    The following internal functions are available:
    -# lcmaps_set_pgid()
    -# lcmaps_add_username_to_ldapgroup()

*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "lcmaps_plugins_basic_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#include "ldap.h"
/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

#define MAX_UNDEFINED -1

#ifndef NGROUPS
    #ifdef NGROUPS_MAX
        #define NGROUPS NGROUPS_MAX
    #else
        #define NGROUPS 32
    #endif
#endif

/*
 * MAX_NGROUPS has also been used. I couldn't see in the source if it was
 * a self made define in the code or that it was OS specific.
 */

#define MAXGROUPNUM ((int)(999999999))


#ifndef MAX_LOG_BUFFER_SIZE
    #define MAX_LOG_BUFFER_SIZE 2048 /*!< Maximum logging buffer size, length of log
                                          may not exceed this number \internal */
#endif

#define WHITESPACE_CHARS " \n"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

static int plugin_run_or_verify(int, lcmaps_argument_t *, int);
static int log_cred(
        char * type,
        char * add_log
);
static int lcmaps_get_ldap_pw(
        const char * path,
        char ** ldap_passwd
);
static int lcmaps_set_pgid(
        const char * username,
        const char * pgroupname,
        gid_t        pgroupnumber,
        LDAP       * ld_handle,
        const char * searchBase
);
static int lcmaps_add_username_to_ldapgroup(
        const char * username,
        const char * groupname,
        gid_t        groupnumber,
        LDAP       * ld_handle,
        const char * searchBase
);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static int            maxuid  = MAX_UNDEFINED;
static int            maxpgid = MAX_UNDEFINED;
static int            maxsgid = MAX_UNDEFINED;
static int            require_all_groups = 1;
static char *         hostname = NULL;
static int            port = -1;
static char *         dn_manager = NULL;
static char *         ldap_pw = NULL;
static char *         sb_groups = NULL;
static char *         sb_user = NULL;
static char *         ldap_dn = NULL;
static char *         ldap_cred_log      = NULL;
static char *         ldap_cred_log_uid  = NULL;
static char *         ldap_cred_log_pgid = NULL;
static char *         ldap_cred_log_sgid = NULL;
static struct timeval timeout =
{
    (time_t) 0,
    (suseconds_t) 0
};



/******************************************************************************
Function:   log_cred
Description:
    Makes a log of the Enforceable credential data
Parameters:
    *secGid    : list of secondary gid_t
    cntSecGid  : amount of secondary gids (0 to (NGROUPS || 32))

Returns:
    0: success
    1: failure
******************************************************************************/
static int log_cred (char * type, char * add_log)
{
    char * logstr = "lcmaps_plugin_ldap-log_cred()";
    char * tmp = NULL;

    if (strcmp(type, "UID") == 0)
    {
        if (ldap_cred_log_uid == NULL)
        {
            ldap_cred_log_uid  = (char *) malloc(MAX_LOG_BUFFER_SIZE * sizeof(char));

            strcpy(ldap_cred_log_uid, "uid=");
            strcat (ldap_cred_log_uid, add_log);
        }
        else
        {
            tmp = (char *) malloc(3 * sizeof(char));
            strcpy (tmp, ",");
            strcat (ldap_cred_log_uid, tmp);
            strcat (ldap_cred_log_uid, add_log);
        }

        lcmaps_log_debug(5,"%s: ldap_cred_log_uid: %s\n", logstr, ldap_cred_log_uid);

    }
    else if (strcmp(type, "PGID") == 0)
    {
        if (ldap_cred_log_pgid == NULL)
        {
            ldap_cred_log_pgid  = (char *) malloc(MAX_LOG_BUFFER_SIZE * sizeof(char));

            strcpy(ldap_cred_log_pgid, "pgid=");
            strcat (ldap_cred_log_pgid, add_log);
        }
        else
        {
            tmp = (char *) malloc(3 * sizeof(char));
            strcpy (tmp, ",");
            strcat (ldap_cred_log_pgid, tmp);
            strcat (ldap_cred_log_pgid, add_log);
        }

        lcmaps_log_debug(5,"%s: ldap_cred_log_pgid: %s\n", logstr, ldap_cred_log_pgid);
    }
    else if (strcmp(type, "SGID") == 0)
    {
        if (ldap_cred_log_sgid == NULL)
        {
            ldap_cred_log_sgid  = (char *) malloc(MAX_LOG_BUFFER_SIZE * sizeof(char));

            strcpy(ldap_cred_log_sgid, "sgid=");
            strcat (ldap_cred_log_sgid, add_log);
        }
        else
        {
            tmp = (char *) malloc(3 * sizeof(char));
            strcpy (tmp, ",");
            strcat (ldap_cred_log_sgid, tmp);
            strcat (ldap_cred_log_sgid, add_log);
        }

        lcmaps_log_debug(5,"%s: ldap_cred_log_sgid: %s\n", logstr, ldap_cred_log_sgid);
    }
    else if (strcmp(type, "LOG") == 0)
    {
        tmp = strdup (":");


        if (ldap_cred_log == NULL)
            ldap_cred_log = (char *) malloc (MAX_LOG_BUFFER_SIZE * sizeof(char));

        if (ldap_cred_log_uid != NULL)
            strcpy(ldap_cred_log, ldap_cred_log_uid);

        if (ldap_cred_log_pgid != NULL)
        {
            strcat(ldap_cred_log, tmp);
            strcat(ldap_cred_log, ldap_cred_log_pgid);
        }

        if (ldap_cred_log_sgid != NULL)
        {
            strcat(ldap_cred_log, tmp);
            strcat(ldap_cred_log, ldap_cred_log_sgid);
        }

        lcmaps_log(LOG_INFO, "%s: %s\n", logstr, ldap_cred_log);
    }
    return 0;
}



/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char *  logstr = "lcmaps_plugin_ldap_enf-plugin_initialize()";
    int i, j;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        // setting maxuid parameter from init plugin arguments
        if ( (((strcmp(argv[i], "-maxuid") == 0) ||
             (strcmp(argv[i], "-MAXUID") == 0)) &&
             (maxuid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"-maxuid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error, maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxuid = -1;
                         goto fail_ldap;
                     }
                 }
                 maxuid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }

        // setting maxpgid parameter from init plugin arguments
        else if ( (((strcmp(argv[i], "-maxpgid") == 0) ||
             (strcmp(argv[i], "-MAXPGID") == 0)) &&
             (maxpgid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(2,"%s: Checking if argument behind \"-maxpgid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error, maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxpgid = -1;
                         goto fail_ldap;
                     }
                 }
                 maxpgid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }

        // setting maxsgid parameter from init plugin arguments
        else if  ( (((strcmp(argv[i], "-maxsgid") == 0) ||
             (strcmp(argv[i], "-MAXSGID") == 0)) &&
             (maxsgid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(2,"%s: Checking if argument behind \"-maxsgid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error, maybe found some digits, but there is atleast one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxsgid = -1;
                         goto fail_ldap;
                     }
                 }
                 maxsgid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        /* settings for LDAP: */
        else if  ( ((strcmp(argv[i], "-port") == 0) ||
             (strcmp(argv[i], "-PORT") == 0))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(2,"%s: Checking if argument behind \"-port\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error maybe found some digits, but there is atleast one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         port = -1;
                         goto fail_ldap;
                     }
                 }
                 port = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( ((strcmp(argv[i], "-hostname") == 0) ||
             (strcmp(argv[i], "-HOSTNAME") == 0))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 hostname = strdup(argv[i+1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-require_all_groups") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     require_all_groups = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     require_all_groups = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     goto fail_ldap;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-dn_manager") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 dn_manager = strdup(argv[i+1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-ldap_pw") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_get_ldap_pw(argv[i+1], &ldap_pw);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-sb_groups") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 sb_groups = strdup(argv[i+1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-sb_user") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
               sb_user = strdup(argv[i+1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        // setting timeout value parameter (in seconds) from init plugin arguments
        else if ( (strcmp(argv[i], "-timeout") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(2,"%s: Checking if argument behind \"-timeout\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error, maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxpgid = -1;
                         goto fail_ldap;
                     }
                 }
                 timeout.tv_sec = (time_t) atoi(argv[i + 1]);
                 timeout.tv_usec = (suseconds_t) 0;
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_ldap;
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            goto fail_ldap;
        }
    }

    if (maxsgid > NGROUPS)
    {
        lcmaps_log(LOG_ERR,"%s: Error, the prefered set maximum of %d Secondary Gid's exceeds the system maximum of NGROUPS witch is set to %d on this system\n", logstr,  maxsgid, NGROUPS);
        goto fail_ldap;
    }
    else if (maxsgid == MAX_UNDEFINED)
    {
        lcmaps_log(LOG_ERR,"%s: Auto set maximum Secondary Gid's to system maximum of NGROUPS witch is set to %d on this system\n", logstr, NGROUPS);
    }
    if (hostname == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: Invalid hostname, use option \"-hostname <hostname>\"\n", logstr);
        goto fail_ldap;
    }
    if (port == -1)
    {
        lcmaps_log(LOG_ERR, "%s: Port not set, use option \"-port <port>\"\n", logstr);
        goto fail_ldap;
    }
    if (dn_manager == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: Invalid dn ldap manager, use option \"-dn_manager <dn manager>\"\n", logstr);
        goto fail_ldap;
    }
    if (ldap_pw == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: No ldap password found in password file, file might be in wrong mode , use option \"-ldap_pw <path/filename>\"\n", logstr);
        goto fail_ldap;
    }
    if (sb_groups == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: Invalid search base for groups, use option \"-sb_groups <search base>\"\n", logstr);
        goto fail_ldap;
    }
    if (sb_user == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: Invalid seach base for user, use option \"-sb_user <search base>\"\n", logstr);
        goto fail_ldap;
    }

    lcmaps_log_debug(2,"%s: Summary init maxuid       : %d\n", logstr, maxuid);
    lcmaps_log_debug(2,"%s: Summary init maxpgid      : %d\n", logstr, maxpgid);
    lcmaps_log_debug(2,"%s: Summary init maxsgid      : %d\n", logstr, maxsgid);
    lcmaps_log_debug(2,"%s: Summary init hostname     : %s\n", logstr, hostname);
    lcmaps_log_debug(2,"%s: Summary init port         : %d\n", logstr, port);
    lcmaps_log_debug(2,"%s: Summary require_all_groups: %d\n", logstr, require_all_groups);
    lcmaps_log_debug(2,"%s: Summary init dn_manager   : %s\n", logstr, dn_manager);
    lcmaps_log_debug(2,"%s: Summary init ldap_pw      : %s\n", logstr, ldap_pw);
    lcmaps_log_debug(2,"%s: Summary init sb_groups    : %s\n", logstr, sb_groups);
    lcmaps_log_debug(2,"%s: Summary init sb_user      : %s\n", logstr, sb_user);
    lcmaps_log_debug(2,"%s: Summary init timeout      : %d seconds and %d microseconds\n", logstr,
    (int) timeout.tv_sec, (int) timeout.tv_usec);

    return LCMAPS_MOD_SUCCESS;

 fail_ldap:
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char *                   logstr    = "lcmaps_plugin_ldap_enf-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {NULL           ,       NULL            , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *           logstr = "lcmaps_plugin_ldap_enf-plugin_run()";
    struct passwd  * user_info   = NULL;
    int              i;
    gid_t          * list        = NULL;

    uid_t          * uid;
    int              cntUid;
    gid_t          * priGid;
    int              cntPriGid;
    gid_t          * secGid;
    int              cntSecGid;
    struct passwd  * root_info   = NULL;
    LDAP           * ld_handle   = NULL;
    int              rc;
    int              ldap_mess_id;
    LDAPMessage    * ldap_bind_msg = NULL;
    struct group   * grgid       = NULL;

    char           * tmplog      = NULL;


    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_ldap_enf-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_ldap_enf-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_ldap_enf-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_ldap;
    }
    lcmaps_log_debug(5,"%s\n", logstr);

    uid    = getCredentialData(UID,     &cntUid);
    priGid = getCredentialData(PRI_GID, &cntPriGid);
    secGid = getCredentialData(SEC_GID, &cntSecGid);
    lcmaps_log_debug(2, "%s: number of uids: %d, priGids: %d, secGids: %d\n", logstr, cntUid, cntPriGid, cntSecGid);


    /* Check amount of uid's, pri-gid's and sec-gid's with the set maximum */
    lcmaps_log_debug(2, "%s: maxuid = %d, MAX_UNDEFINED = %d\n", logstr, maxuid, MAX_UNDEFINED);
    if (maxuid != MAX_UNDEFINED)
    {
        lcmaps_log_debug(2, "%s: max number of uids: %d\n", logstr, maxuid);
        if (cntUid > maxuid)
        {
            lcmaps_log(LOG_ERR, "%s: Error, the set amount of uid's gathered exceeds the maximum of %d uid('s) by %d\n", logstr, maxuid, (cntUid - maxuid));
            goto fail_ldap;
        }
    }
    if (maxpgid != MAX_UNDEFINED)
    {
        lcmaps_log_debug(2, "%s: max number of primary gid('s): %d\n", logstr, maxpgid);
        if (cntPriGid > maxpgid)
        {
            lcmaps_log(LOG_ERR, "%s: Error, The set amount of primary gid's gathered exceeds the maximum of %d primary gid('s) by %d\n", logstr, maxpgid, (cntPriGid - maxpgid));
            goto fail_ldap;
        }
    }
    if (maxsgid != MAX_UNDEFINED)
    {
        lcmaps_log_debug(2, "%s: max number of secondary gid's: %d\n", logstr, maxsgid);
        if (cntSecGid > maxsgid)
        {
            lcmaps_log(LOG_ERR, "%s: Error, The set amount of secondary gid's gathered exceeds the maximum of %d secondary gid's by %d\n", logstr, maxsgid, (cntSecGid - maxsgid));
            goto fail_ldap;
        }
    }


    /* You must be ROOT */
    if (getuid() != 0)
    {
        lcmaps_log(LOG_ERR, "%s: You are not ROOT!  -> %d\n", logstr, getuid());
        goto fail_ldap;
    }

    /* Only done to get name of root (normally "root") */
    if ((root_info=getpwuid(0)) == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: cannot get passwd info for root\n", logstr);
        if (errno==ENOMEM)
            lcmaps_log(LOG_ERR, "%s: %s\n", logstr, strerror(errno));
        goto fail_ldap;
    }

    /* Get a handle to an LDAP connection. */
    if ( (ld_handle = ldap_init( hostname, port)) == NULL )
    {
      lcmaps_log(LOG_ERR, "%s: no handle to LDAP server on %s using port %s -> %s\n", logstr, hostname, port, strerror(errno));
      goto fail_ldap;
    }
    lcmaps_log_debug(3, "%s: handle to LDAP server successfull\n", logstr);

    /* Bind to the LDAP server. check if timeout is specified, if not, do ldap_simple_bind_s() (blocking)*/
    lcmaps_log_debug(3, "%s: dn_manager: %s\n", logstr, dn_manager);
    lcmaps_log_debug(5,"%s: ldap_pw is--->%s<---\n", logstr, ldap_pw);
    if ((timeout.tv_sec == (time_t) 0) && (timeout.tv_usec == (suseconds_t) 0))
    {
      rc = ldap_simple_bind_s( ld_handle, dn_manager, ldap_pw);
      if ( rc != LDAP_SUCCESS )
      {
        lcmaps_log(LOG_ERR,"%s: Error binding to ldap server with ldap_simple_bind_s() -> %s\n", logstr, ldap_err2string(rc));
        goto fail_ldap;
      }
      lcmaps_log_debug(3, "%s: bind to LDAP server successfull\n", logstr);
    }
    else
    {
      /* First get ldap message id for the asynchronous bind ldap_simple_bind()*/
      ldap_mess_id = ldap_simple_bind( ld_handle, dn_manager, ldap_pw);
      lcmaps_log_debug(5,"%s: ldap_simple_bind() returned this message id: %d\n", logstr, ldap_mess_id);
      if (ldap_mess_id == -1)
      {
        lcmaps_log(LOG_ERR,"%s: Error binding to ldap server with ldap_simple_bind() ->\n", logstr);
        ldap_perror(ld_handle, " ");
        goto fail_ldap;
      }

      /*
       * Then get the result with timeout (rc contains the message type (should be LDAP_RES_BIND),
       * 0 if timeout reached ot -1 in case of error
       */
      rc = ldap_result(ld_handle, ldap_mess_id, 0, &timeout, &ldap_bind_msg);
      lcmaps_log_debug(5,"%s: ldap_result() returned this message id  : %d\n", logstr, ldap_msgid(ldap_bind_msg));
      lcmaps_log_debug(5,"%s: ldap_result() returned this message type: 0x%x\n", logstr, ldap_msgtype(ldap_bind_msg));
      if ( rc == LDAP_RES_BIND )
      {
        lcmaps_log_debug(3, "%s: bind to LDAP server successfull\n", logstr);
      }
      else if ( rc == 0 )
      {
        lcmaps_log(LOG_ERR,"%s: Error, time out occurred in binding to ldap server -> (timeout = %d s)\n", logstr, 0);
        goto fail_ldap;
      }
      else
      {
        lcmaps_log(LOG_ERR,"%s: Error getting handle to ldap server  -> %s (rc = %d)\n", logstr, rc);
        goto fail_ldap;
      }
      lcmaps_log_debug(5,"%s: freeing ldap msg\n", logstr);
      if (ldap_bind_msg)
      {
        ldap_msgfree(ldap_bind_msg);
        ldap_bind_msg = NULL;
      }
    }

    /* get username */
    if (cntUid < 1)
    {
        lcmaps_log(LOG_ERR,"%s: Error, found no userid in credential data repository (failure)\n", logstr);
        goto fail_ldap;
    }
    lcmaps_log_debug(3, "%s: uid = %d\n", logstr, (int) *uid );
    user_info = getpwuid( *uid);
    if (user_info == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: Error, no pw_name associated with uid: %d\n", logstr, uid);
        goto fail_ldap;
    }
    lcmaps_log_debug(3, "%s: uidname = %s\n", logstr, user_info->pw_name );

    /* get the primary gid */
    if (cntPriGid >= 1)
    {
        grgid = getgrgid((gid_t) priGid[0]);
        if (grgid != NULL)
        {
            lcmaps_log_debug(3, "%s: priGidName = %s\n", logstr, grgid->gr_name);
            /* set the pgid */
             if ( lcmaps_set_pgid(user_info->pw_name, grgid->gr_name, (gid_t)(int) grgid->gr_gid, ld_handle, sb_user) != 0 )
             {
                 lcmaps_log(LOG_ERR, "%s: lcmaps_set_pgid failed for user=%s, pri group=%s, pri gid=%d\n", logstr,
                            user_info->pw_name, grgid->gr_name, (int) grgid->gr_gid);
                 goto fail_ldap;
             }
             // Add to log if succesfull
             else
             {
                  // Log User ID with it's Username
                  tmplog = (char *) malloc (MAX_LOG_BUFFER_SIZE * sizeof(char));
                  snprintf(tmplog, MAX_LOG_BUFFER_SIZE-1, "%d(%s)", user_info->pw_uid, user_info->pw_name);
                  log_cred("UID", tmplog);
                  if (tmplog) free(tmplog); tmplog = NULL;

                  // Primary Group ID and Group Name
                  tmplog = (char *) malloc (MAX_LOG_BUFFER_SIZE * sizeof(char));
                  snprintf(tmplog, MAX_LOG_BUFFER_SIZE-1, "%d(%s)", grgid->gr_gid, grgid->gr_name);
                  log_cred("PGID", tmplog);
                  if (tmplog) free(tmplog); tmplog = NULL;
             }
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s: Error, No grgid name associated with primary gid number: %d\n", logstr, (int) priGid[0]);
            goto fail_ldap;
        }
    }

    /* loop over secondary gids */
    i=0;
    while (i < cntSecGid)
    {
        lcmaps_log_debug(3, "%s: secGid = %d\n", logstr, (int) secGid[i]);
        grgid = getgrgid((gid_t) secGid[i]);
        if (grgid != NULL)
        {
            lcmaps_log_debug(3, "%s: secGidName = %s\n", logstr, grgid->gr_name);

            if ( lcmaps_add_username_to_ldapgroup(user_info->pw_name, grgid->gr_name, (gid_t)(int) grgid->gr_gid, ld_handle, sb_groups) != 0 )
            {
                if (require_all_groups == 1)
                {
                    lcmaps_log(LOG_ERR, "%s: failure for user=%s, group=%s, gid=%d\n", logstr,
                           user_info->pw_name, grgid->gr_name, (int) grgid->gr_gid);
                    goto fail_ldap;
                }
                else
                {
                    lcmaps_log_debug(1, "%s: failure for user=%s, group=%s, gid=%d\n", logstr,
                           user_info->pw_name, grgid->gr_name, (int) grgid->gr_gid);
                }
            }
            else
            {
                // Add the secondary group IDs & names
                tmplog = (char *) malloc (MAX_LOG_BUFFER_SIZE * sizeof(char));
                snprintf(tmplog, MAX_LOG_BUFFER_SIZE-1, "%d(%s)", grgid->gr_gid, grgid->gr_name);
                log_cred("SGID", tmplog);
                if (tmplog) free(tmplog); tmplog = NULL;
            }
        }
        else
        {
            lcmaps_log(LOG_INFO, "%s: Error, No grgid name associated with gid number: %d\n", logstr, (int) secGid[i]);
        }
        i++;
    }


    /* Log Enforced Credential Data */
    log_cred("LOG", "");


    /* succes */
 success_ldap:
    if (ld_handle) ldap_unbind_s(ld_handle);
    lcmaps_log(LOG_INFO,"%s: ldap_enf plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_ldap:
    // added:
    if (ldap_bind_msg)
    {
      ldap_msgfree(ldap_bind_msg);
      ldap_bind_msg = NULL;
    }
    if (ld_handle) ldap_unbind_s(ld_handle);
    if (list) free(list);
    lcmaps_log(LOG_INFO,"%s: ldap_enf plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char *              logstr = "lcmaps_plugin_ldap_enf-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (sb_user)
    {
        free(sb_user);
        sb_user = NULL;
    }
    if (sb_groups)
    {
        free(sb_groups);
        sb_groups = NULL;
    }
    if (ldap_pw)
    {
        free(ldap_pw);
        ldap_pw= NULL;
    }
    if (dn_manager)
    {
        free(dn_manager);
        dn_manager = NULL;
    }
    if (hostname)
    {
        free(hostname);
        hostname = NULL;
    }
    if (ldap_dn)
    {
        free(ldap_dn);
        ldap_dn = NULL;
    }

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   lcmaps_add_username_to_ldapgroup()

Description:
    Adds the username to the appropriate (LDAP) group.

Parameters:
    username:    the name of the user
    groupname:   the name of the group
    groupnumber: group id number
    ld_handle:   handle to LDAP
    searchBase:  dn search base

Returns:
    0 on success.
   -1 on ldap failure
    1 on failure
******************************************************************************/
/*!
    \fn lcmaps_add_username_to_ldapgroup(
        const char * username,
        const char * groupname,
        gid_t        groupnumber,
        LDAP       * ld_handle,
        const char * searchBase
        )
    \brief Adds the username to the appropriate (LDAP) group.

    This function tries to add the username to the list of usernames belonging
    to the group with name groupname and gid groupnumber in the posixGroup LDAP
    structure.
    If the group does not exist, -1 is returned.

    \param username     the name of the user
    \param groupname    the name of the group
    \param groupnumber  group id number
    \param ld_handle    handle to LDAP
    \param searchBase   dn search base

    \retval 0 success
    \retval -1 ldap failure
    \retval 1 other failure
*/
static int lcmaps_add_username_to_ldapgroup(
        const char * username,
        const char * groupname,
        gid_t        groupnumber,
        LDAP       * ld_handle,
        const char * searchBase
    )
{
    char *         logstr = "lcmaps_plugin_ldap_enf-lcmaps_add_username_to_ldapgroup()";
    int            i = 0;
    int            filtersize;
    char         * filterprefix = "(&(objectClass=posixGroup)(gidNumber=";
    char         * filter = NULL;
    char         * filtersuffix = "))";
    char         * memberUidfilter = ")(memberUid=";
    char         * memberfilter;
    LDAPMessage  * searchResult, *entry;
    char         * attribute, **values;
    struct timeval timeOut = {10,0};
    char           groupstr[10];
    char         * dn = NULL;
    BerElement   * ber;
    int            rc, entryCount, dnsize, membersize;
    char         * dnprefix = "cn=";
    LDAPMod        modGID;
    LDAPMod      * modify[2];
    char         * modifyDN;
    char         * GIDValue[2];

    if (! ld_handle)
    {
        lcmaps_log(LOG_ERR,"%s: empty ldap handle\n", logstr);
        return -1;
    }
    if (groupnumber > MAXGROUPNUM)
    {
        lcmaps_log(LOG_ERR,"%s: groupid (%d) too large, max = %d\n", logstr,
        groupnumber,MAXGROUPNUM);
        return 1;
    }

    snprintf(groupstr,(size_t)10,"%d",groupnumber);
    filtersize = strlen(filterprefix)+strlen(groupstr)+strlen(filtersuffix)+1;
    filter = (char *) malloc(filtersize*sizeof(char));

    snprintf(filter,(size_t)filtersize,"%s%s%s",filterprefix, groupstr, filtersuffix);

    lcmaps_log_debug(3,"%s: groupstr = %s\n", logstr,groupstr);
    lcmaps_log_debug(3,"%s: filter: %s, filtersize = %d\n", logstr,filter, filtersize);
    lcmaps_log_debug(3,"%s: username: %s, groupnumber = %d, ld_handle = %p, searchBase = %s\n", logstr,
                     username, groupnumber, ld_handle, searchBase);

    membersize = strlen(filterprefix) + strlen(groupstr) + strlen(memberUidfilter) + strlen(username) + strlen(filtersuffix) +1;
    memberfilter = (char *) malloc(membersize*sizeof(char));
    snprintf(memberfilter,(size_t)memberfilter,"%s%s%s%s%s",filterprefix, groupstr,memberUidfilter,username ,filtersuffix);
    lcmaps_log_debug(3,"%s: memberefilter: %s, membersize = %d\n", logstr,memberfilter, membersize);

    dnsize = strlen(dnprefix) + strlen(groupname)  + strlen(searchBase) +2;  // '/0' and ','
    modifyDN = (char *) malloc(dnsize*sizeof(char));
    snprintf(modifyDN,(size_t)dnsize,"%s%s,%s",dnprefix,groupname,searchBase );
    lcmaps_log_debug(3,"%s: modifyDN = %s\n", logstr,modifyDN);
    //modifyDN = "cn=tbadmin,ou=LocalGroups,dc=foobar,dc=ough";


    modGID.mod_op     = LDAP_MOD_ADD;
    modGID.mod_type   = "memberUid";
    GIDValue[0]       = strdup(username);
    GIDValue[1]       = NULL;
    modGID.mod_values = GIDValue;

    modify[0] = &modGID;
    modify[1] = NULL;


   /* Search the directory  to test if entry exists */
    rc = ldap_search_ext_s(
                    ld_handle,             /* LDAP session handle */
                    searchBase,            /* container to search */
                    LDAP_SCOPE_SUBTREE,
                    memberfilter,
                    NULL,                  /* return all attributes */
                    0,                     /* return attributes and values */
                    NULL,                  /* server controls */
                    NULL,                  /* client controls */
                    &timeOut,              /* search timeout */
                    LDAP_NO_LIMIT,         /* no size limit */
                    &searchResult );       /* returned results */

    if ( rc != LDAP_SUCCESS )
    {
        lcmaps_log(LOG_ERR,"%s: ldap_search_ext_s error: %s\n", logstr, ldap_err2string( rc ));
        free(GIDValue[0]);
        free(filter);
        free(modifyDN);
        free(memberfilter);
        return -1;
    }

    entryCount = ldap_count_entries( ld_handle, searchResult );
    lcmaps_log_debug(5,"%s: Search completed successfully. Entries returned: %d\n", logstr, entryCount);

    if ( entryCount != 0)
    {
        lcmaps_log_debug(3,"%s: No add, allready exists:  modifyDN = %s\n", logstr,modifyDN);
        free(GIDValue[0]);
        free(filter);
        free(modifyDN);
        free(memberfilter);
        ldap_msgfree( searchResult );
        return 0;
    }

    /* if not exists, add the entry  */
    rc= ldap_modify_ext_s( ld_handle,  /* LDAP session handle */
                           modifyDN,   /* the object to modify */
                           modify,     /* array of LDAPMod structures */
                           NULL,       /* server controls */
                           NULL);      /* client controls */

    if ( rc != LDAP_SUCCESS )
    {
        lcmaps_log(LOG_ERR,"%s: ldap_modify_ext_s failed for group %s(%d): %s\n", logstr, groupname, groupnumber, ldap_err2string( rc ));
        free(GIDValue[0]);
        free(filter);
        free(modifyDN);
        free(memberfilter);
        return -1;
    }
    lcmaps_log_debug(3,"%s: %s modified successfully.\n", logstr, modifyDN);


   /* Search the directory */
   /* this is an extra test, could be removed */
    rc = ldap_search_ext_s(
                    ld_handle,             /* LDAP session handle */
                    searchBase,            /* container to search */
                    LDAP_SCOPE_SUBTREE,
                    filter,
//                    &attrstr,              /* return all attributes */
                    NULL,                  /* return all attributes */
                    0,                     /* return attributes and values */
                    NULL,                  /* server controls */
                    NULL,                  /* client controls */
                    &timeOut,              /* search timeout */
                    LDAP_NO_LIMIT,         /* no size limit */
                    &searchResult );       /* returned results */

    if ( rc != LDAP_SUCCESS )
    {
        lcmaps_log(LOG_ERR,"%s: ldap_search_ext_s error: %s\n", logstr, ldap_err2string( rc ));
        free(GIDValue[0]);
        free(filter);
        free(modifyDN);
        free(memberfilter);
        ldap_msgfree( searchResult );
        return -1;
    }

    /* Go through the search results by checking entries */
    for (   entry   =   ldap_first_entry( ld_handle, searchResult );
            entry   !=  NULL;
            entry   =   ldap_next_entry( ld_handle, entry ) )
    {
        if (( dn = ldap_get_dn( ld_handle, entry )) != NULL )
        {
            lcmaps_log_debug(3,"%s: -->  dn: %s\n", logstr, dn);
            ldap_memfree( dn );
        }
       for (   attribute = ldap_first_attribute( ld_handle, entry, &ber );
                attribute != NULL;
                attribute = ldap_next_attribute( ld_handle, entry, ber ) )
        {
            /* Get values and print.  Assumes all values are strings. */
            values = ldap_get_values( ld_handle, entry, attribute);
            if ( values != NULL )
            {
                for ( i = 0; values[i] != NULL; i++ )
                    lcmaps_log_debug(3,"%s:        %s: %s\n", logstr, attribute, values[i]);
                ldap_value_free( values );
            }
            ldap_memfree( attribute );
        }

        ber_free(ber, 0);
    }

    entryCount = ldap_count_entries( ld_handle, searchResult );
    lcmaps_log_debug(3,"%s: Search completed successfully. Entries returned: %d\n", logstr, entryCount);

    ldap_msgfree( searchResult );
    free(GIDValue[0]);
    free(filter);
    free(modifyDN);
    free(memberfilter);


    return 0;
}

/******************************************************************************
Function:   lcmaps_set_pgid()

Description:
    Sets the primary group ID

Parameters:
    username:     the name of the user
    pgroupname:   the name of the primary group
    pgroupnumber: primary group id number
    ld_handle:    handle to LDAP
    searchBase:   dn search base

Returns:
    0 on success.
   -1 on ldap failure
    1 on failure
******************************************************************************/
/*!
    \fn lcmaps_set_pgid(
        const char * username,
        const char * pgroupname,
        gid_t        pgroupnumber,
        LDAP       * ld_handle,
        const char * searchBase
    )
    \brief Sets the primary group ID

    This function tries to set the primary group in the posixAccount LDAP structure
    for the user "username".

    \param username      the name of the user
    \param pgroupname    the name of the primary group
    \param pgroupnumber  primary group id number
    \param ld_handle     handle to LDAP
    \param searchBase    dn search base

    \retval 0 success
    \retval -1 ldap failure
    \retval 1 other failure
*/
static int lcmaps_set_pgid(
        const char * username,
        const char * pgroupname,
        gid_t        pgroupnumber,
        LDAP       * ld_handle,
        const char * searchBase
    )
{
    char *         logstr = "lcmaps_plugin_ldap_enf-lcmaps_set_pgid()";
    int            rc, dnsize;
    char           prigroupstr[10];
    LDAPMod        modGID;
    LDAPMod      * modify[2];
    char         * modifyDN = NULL;
    char         * GIDValue[2];
    char         * dnprefix = "uid=";


    /* print incoming vars: */
    lcmaps_log_debug(5, "%s: parameters: user=%s, pri group=%s, pri gid=%d\n", logstr,
               username, pgroupname, (int) pgroupnumber);

    if (! ld_handle)
    {
        lcmaps_log(LOG_ERR,"%s: empty ldap handle\n", logstr);
        return -1;
    }
    if (pgroupnumber > MAXGROUPNUM)
    {
        lcmaps_log(LOG_ERR,"%s: pgroupid (%d) too large, max = %d\n", logstr,
        pgroupnumber,MAXGROUPNUM);
        return 1;
    }

    snprintf(prigroupstr,(size_t)10,"%d", pgroupnumber);

    /* modifyDN    the object to modify */
    dnsize = strlen(dnprefix) + strlen(username)  + strlen(searchBase) +3;  // '/0' and ','
    modifyDN = (char *) malloc(dnsize*sizeof(char));
    snprintf(modifyDN,(size_t)dnsize,"%s%s, %s",dnprefix,username,searchBase );
    lcmaps_log_debug(3,"%s: modifyDN = %s\n", logstr,modifyDN);


    modGID.mod_op     = LDAP_MOD_REPLACE;
    modGID.mod_type   = "gidNumber";
    GIDValue[0]       = prigroupstr;
    GIDValue[1]       = NULL;
    modGID.mod_values = GIDValue;

    modify[0] = &modGID;
    modify[1] = NULL;


    /* modify the entry  */
    rc= ldap_modify_ext_s( ld_handle,  /* LDAP session handle */
                           modifyDN,   /* the object to modify */
                           modify,     /* array of LDAPMod structures */
                           NULL,       /* server controls */
                           NULL);      /* client controls */

    if ( rc != LDAP_SUCCESS )
    {
        lcmaps_log(LOG_ERR,"%s: ldap_modify_ext_s failed: %s\n", logstr, ldap_err2string( rc ));
        /* free mem */
        free(modifyDN);
        return -1;
    }
    lcmaps_log_debug(3,"%s: %s modified successfully.\n", logstr, modifyDN);

    /* free mem */
    free(modifyDN);

    return 0;
}


/******************************************************************************
Function:   lcmaps_get_ldap_pw()

Description:
    Get the LDAP password from file

Parameters:
    path:   path to the ldap_pw file containing the password
    ldap_passwd : variable to set the password in

Returns:
    0 on success.
    1 on failure
******************************************************************************/
/*!
    \fn lcmaps_get_ldap_pw(
        const char * path,
              char ** ldap_passwd
    )
    \brief Get the LDAP password from file

    This function tries to read the LDAP password from the ldap_pw file.
    It also tests if the access bits of the file are correctly set.


    \param path     the path to the ldap_pw file containing the password.
    \param ldap_passwd variable to set the password in

    \retval 0 success
    \retval 1 other failure
*/

static int lcmaps_get_ldap_pw(
        const char * path,
        char       ** ldap_passwd
    )
{
    char *        logstr = "lcmaps_plugin_ldap_enf-lcmaps_get_ldap_pw()";
    FILE        * fp = NULL;
    struct stat   buf;
    int           c;
    int           count = 0;
    char        * temppwd = NULL;

    lcmaps_log_debug(3,"%s: path: %s \n", logstr, path);

    /* get the stats of the file */
    if ((stat(path, &buf)) == -1)
    {
        lcmaps_log(LOG_ERR, "%s: unable to get stat of ldap_pw file: %s\n", logstr, path);
        return 1;
    }

    /* test if it's a file */
    if ( !(S_ISREG(buf.st_mode)))
    {
        lcmaps_log(LOG_ERR, "%s: ldap_pw file is not a file: %s\n", logstr, path);
        return 1;
    }
    /* test if root is owner */
    if (buf.st_uid != 0) {
        lcmaps_log(LOG_ERR, "%s: ldap_pw file is not owned by root (readonly for root) : %s\n", logstr, path);
        return 1;
    }

    /* test if file has the access bits set correctly */
    if ((0000777  & (buf.st_mode) ) != S_IRUSR )
    {
        lcmaps_log(LOG_ERR, "%s: ldap_pw file has incorrect accessibility (readonly for root) : %s\n", logstr, path);
        return 1;
    }

    /* try to open the file */
    if ((fp = fopen(path, "r")) == NULL)
    {
        lcmaps_log(LOG_ERR, "%s: unable to open ldap_pw file: %s\n", logstr, path);
        return 1;
    }

    /* read the passwd from the file */
    /* I thought tabs and spaces were not allowed in the passwd, but they are, so
     * restore original line
     */
//    while ((c = getc(fp)) != EOF && (strchr(WHITESPACE_CHARS, c) == NULL)) count++;
    while ((c = getc(fp)) != EOF && (c != '\n')) count++;
    temppwd = (char *) malloc(count * sizeof(char) + 2);

    if ((fseek(fp, 0L, SEEK_SET)) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: unable to reset the fp\n", logstr);
        fclose(fp);
        return 1;
    }

    if ((fgets(temppwd, count+1, fp)) == NULL )
    {
        lcmaps_log(LOG_ERR, "%s: unable to read the password: fgets failed\n", logstr);
        fclose(fp);
        return 1;
    }

    fclose(fp);
    lcmaps_log_debug(3,"%s: password: %s\n", logstr, temppwd);
//    *ldap_passwd = strdup(" s ts   ");
    *ldap_passwd = strdup(temppwd);

    /* free mem */
    if (temppwd) free(temppwd);

    return 0;
}
/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-basic/src/ldap_enf/lcmaps_ldap.c,v $
    $Date: 2010-02-19 06:08:47 $
    $Revision: 1.8 $
    $Author: okoeroo $
******************************************************************************/
